/*
 * dproc.c - SGI IRIX process access functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dproc.c,v 1.3 98/02/13 12:49:42 abe Exp $";
#endif

#include "lsof.h"


_PROTOTYPE(static void get_kernel_access,(void));
_PROTOTYPE(static void process_text,(struct pregion *prp, struct pregion *sprp));
_PROTOTYPE(static void readfsinfo,(void));

#if	IRIXV==50300 || !defined(IRIX32BITS) || IRIXV>=60200
/*
 * Handle lseek complications:
 *
 * o  For Irix 5.3 and its 64 bit descendants with _KERNEL defined, declare
 *    an alternate lseek prototype that avoids 32 vs. 64 bit length confusion
 *    about off_t.
 *
 * o  For IRIX 6. and above, use lseek64().
 */

# if	IRIXV>=60200
#define	lseek	lseek64
# else	/* IRIXV<60200 */
_PROTOTYPE(int _lseek,(int, long, int));
#define lseek _lseek
# endif	/* IRIXV>=60200 */
#endif	/* IRIXV==50300 && !defined(IRIX32BITS) */

#if	IRIXV>=60400
_PROTOTYPE(static void read_proc,(void));
#endif	/* IRIXV>=60400 */


/*
 * Local static values
 */

static KA_T Kp;				/* kernel process table address */
static int Kprl = PROCSIZE;		/* kernel proc struct read length */
static int Kpst = PROCSIZE;		/* kernel proc struct stride size */
static caddr_t *Ndc = NULL;		/* node cache */
static MALLOC_S Nn = 0;			/* number of Ndc[] entries allocated */
static struct var Var;			/* kernel variables */

#if	IRIXV>=60400
static int Np;				/* occupied P[] count */
static int Npa = 0;			/* allocated P[] count */
static struct proc *P = (struct proc *)NULL;
					/* IRIX 6.4 and above proc table */
#define	PROCINCR	32		/* increment for increasing P[] */
#endif	/* IRIXV>=60400 */


/*
 * gather_proc_info() -- gather process information
 */

void
gather_proc_info()
{
	struct vnode *cdir, *rdir;
	char *comm;
	int i, j, nf, sgs;
	struct proc *p;
	KA_T pa;
	struct proc ps;
	short pss, sf;
	int px;
	uid_t uid;

#if	IRIXV<50101
	static struct file **fp;
	int ul;
	static char *uaf = NULL;
	static struct user *u;
	static MALLOC_S uafl;
#else	/* IRIXV>=50101 */
# if	IRIXV<60200
	int dl;
	struct sat_wd sw;
	KA_T ta;
# endif	/* IRIXV<60200 */
	struct file *f;
	struct cred pcred;
# if	IRIXV>=50300
	struct shaddr_s sg;
	struct ufchunk uc;
# endif	/* IRIXV>=50300 */
# if	IRIXV<60400
	static struct user *u;
	struct user us;
# else	/* IRIXV>=60400 */
	struct fdt fdt, *pf;
# endif	/* IRIXV<60400 */
#endif	/* IRIXV<50101 */

#if	IRIXV<50101
/*
 * Allocate space for user area and file pointers for IRIX version
 * below 5.1.1.
 */
	if (uaf == (char *)NULL) {
		uafl = (MALLOC_S) (sizeof(struct user)
		     +  (Var.v_nofiles * sizeof(struct file *)));
		if ((uaf = (char *)malloc(uafl)) == NULL) {
			(void) fprintf(stderr,
				"%s: no space for uarea and file pointers\n",
				Pn);
			Exit(1);
		}
		fp = (struct file **)(uaf + sizeof(struct user));
		u = (struct user *)uaf;
	}
#else	/* IRIXV>=50101 */

# if	IRIXV<60400
/*
 * Set user structure pointer for IRIX 5.1.1 and below 6.4.
 */
	u = &us;
# endif	/* IRIXV<60400 */

# if	 IRIXV<60200
/*
 * Allocate space for current working and root directory buffers for
 * IRIX versions 5.1.1 through 6.1.
 */
	if (Cwd == (char *)NULL) {
		if ((Cwd = (char *)malloc(MAXPATHLEN)) == NULL) {
			(void) fprintf(stderr,
				"%s: no space for cwd buffer\n", Pn);
			Exit(1);
		}
	}
	if (Rtd == (char *)NULL) {
		if ((Rtd = (char *)malloc(MAXPATHLEN)) == NULL) {
			(void) fprintf(stderr,
				"%s: no space for root directory buffer\n",
				Pn);
			Exit(1);
		}
	}
	*Cwd = *Rtd = '\0';
# endif	/* IRIXV<60200 */
#endif	/* IRIXV<50101 */

#if     defined(HASNCACHE)
/*
 * Read the kernel name cache.
 */
	ncache_load();
#endif  /* defined(HASNCACHE) */

/*
 * Examine proc structures and their associated information.
 */

#if	IRIXV<60400
	for (p = &ps, px = 0; px < Var.v_proc; px++)
#else	/* IRIXV>=60400 */
	(void) read_proc();
	for (p = P, px = 0; px < Np; p++, px++)
#endif	/* IRIXV<60400 */

	{
		sgs = 0;

#if	IRIXV<60400
		pa = Kp + (KA_T)(px * Kpst);
		if (kread(pa, (char *)&ps, Kprl))
			continue;
#endif	/* IRIXV<60400 */

		if (p->p_stat == 0 || p->p_stat == SZOMB)
			continue;
	/*
	 * Get User ID.
	 */

#if	IRIXV>=50101
		if (p->p_cred == NULL
		||  kread((KA_T)p->p_cred, (char *)&pcred, sizeof(pcred)))
			continue;
		uid = (uid_t)pcred.cr_ruid;
#else	/* IRIXV<50101 */
		uid = (uid_t)p->p_uid;
#endif	/* IRIXV>=50101 */

		if (is_proc_excl(p->p_pid, p->p_pgrp, (UID_ARG)uid, &pss, &sf))
			continue;
#if	IRIXV<60400
	/*
	 * Get the user area associated with the process.
	 */

# if	IRIXV>=50101
#  if	IRIXV>=60200
		if (!p->p_user
		||  kread((KA_T)p->p_user, (char *)&us, sizeof(us)))
#  else	/* IRIXV<60200 */
		if (syssgi(SGI_RDUBLK, p->p_pid, (char *)&us, sizeof(us))
		!= sizeof(us))
#  endif	/* IRIXV>=60200 */

			continue;
# else	/* IRIXV<50101 */
		if ((ul = syssgi(SGI_RDUBLK, p->p_pid, uaf, uafl))
		< (int)sizeof(struct user))
			continue;
		if (ul <= (int)sizeof(struct user)
		||  ((long)u->u_ofile - UADDR) != sizeof(struct user))
			nf = 0;
		else
			nf = (ul - sizeof(struct user)) / sizeof(struct file *);
# endif	/* IRIXV>=50101 */
#endif	/* IRIXV<60400 */

#if	IRIXV==50300 || IRIXV>=60100
	/*
	 * Read the IRIX 5.3, and IRIX 6.1 and above shared group descriptor.
	 */
		if (p->p_shaddr
		&&  kread((KA_T)p->p_shaddr, (char *)&sg, sizeof(sg)) == 0)
			sgs = 1;
#endif	/* IRIXV==50300 || IRIXV>=60100 */

#if	IRIXV>=50101 && IRIXV<60200
	/*
	 * Read the IRIX 5.1.1 through 6.1 current working directory and
	 * root directory paths.
	 */
		*Cwd = *Rtd = '\0';
		if ((ta = (KA_T)u->u_satwd)
		&& kread((KA_T)ta, (char *)&sw, sizeof(sw)) == 0) {
			ta += offsetof(struct sat_wd, data);
			if ((dl = sw.cwd_len) > 0) {
				if (dl >= MAXPATHLEN)
					dl = MAXPATHLEN - 1;
				if (dl < 2 || kread(ta, Cwd, dl))
					*Cwd = '\0';
				else
					Cwd[dl] = '\0';
			}
			if ((dl = sw.root_len) > 0) {
				if (dl >= MAXPATHLEN)
					dl = MAXPATHLEN - 1;
				ta += sw.cwd_len;
				if (dl < 2 || kread(ta, Rtd, dl))
					*Rtd = '\0';
				else
					Rtd[dl] = '\0';
			}
		}
#endif	/* IRIXV>=50101 && IRIXV<60200 */

	/*
	 * Allocate a local process structure.
	 */

#if	IRIXV<60400
		cdir = u->u_cdir;
		comm = u->u_comm;
		rdir = u->u_rdir;
#else	/* IRIXV>=60400 */
		cdir = p->p_cdir;
		comm = p->p_ui.u_comm;
		rdir = p->p_rdir;
#endif	/* IRIXV<60400 */

		if (is_cmd_excl(comm, &pss, &sf))
			continue;
		alloc_lproc((int)p->p_pid, (int)p->p_pgrp, (int)p->p_ppid,
			(UID_ARG)uid, comm, (int)pss, (int)sf);
		Plf = NULL;
	/*
	 * Save current working directory information.
	 */
		if (cdir) {
			alloc_lfile(CWD, -1);
			process_node((caddr_t)cdir);
			if (Lf->sf)
				link_lfile();
		}
	/*
	 * Save root directory information.
	 */
		if (rdir) {
			alloc_lfile(RTD, -1);
			process_node((caddr_t)rdir);
			if (Lf->sf)
				link_lfile();
		}
	/*
	 * Print information on the text file.
	 */

#if	IRIXV<50300 || (IRIXV>50300 && IRIXV<60100)
		if (p->p_region)
			process_text(p->p_region, (struct pregion *)NULL);
#else	/* IRIXV==50300 || IRIXV>=60100 */
		if (PREG_FIRST(p->p_region) || (sgs && PREG_FIRST(sg.s_region)))
			process_text(PREG_FIRST(p->p_region),
				     sgs ? PREG_FIRST(sg.s_region)
					 : (struct pregion *)NULL);
#endif	/* IRIXV<50300 || (IRIXV>50300 && IRIXV<60100) */

	/*
	 * Save information on file descriptors.
	 */

#if	IRIXV<50101
	/*
	 * The file pointers for IRIX versions below 5.1.1 are stored in a
	 * dynamically allocated array that follows the user area segment.
	 */
		for (i = 0; i < nf; i++) {
		     if (fp[i]) {
			alloc_lfile(NULL, i);
			process_file(fp[i]);
			if (Lf->sf)
			    link_lfile();
		     }
		}
#endif	/* IRIXV<50101 */

#if	(IRIXV>=50101 && IRIXV<50300) || (IRIXV>50300 && IRIXV<60100)
	/*
	 * Under IRIX 5.1.1 and below IRIX 5.3, or above 5.3 and below 6.1,
	 * the file pointers are stored in dynamically-linked ufchunk
	 * structures, each containing NFPCHUNK file pointers.  The first
	 * ufchunk structure is in the user area.
	 */
		for (i = 0, j = 0; i < u->u_nofiles; i++) {
		    if (++j > NFPCHUNK) {
			if ( ! u->u_flist.uf_next)
			    break;
			if (kread((KA_T)u->u_flist.uf_next, (char *)&u->u_flist,
				  sizeof(struct ufchunk)))
			    break;
			j = 1;
		    }
		    if ((f = u->u_flist.uf_ofile[j-1])) {
			alloc_lfile(NULL, i);
			process_file(f);
			if (Lf->sf)
			    link_lfile();
		    }
		}
#endif	/* (IRIXV>=50101 && IRIXV<50300) || (IRIXV>50300 && IRIXV<60100) */

#if	IRIXV==50300 || (IRIXV>=60100 && IRIXV<60400)
	/*
	 * Under IRIX 5.3 and (6.1 and below 6.4), the file pointers are stored
	 * in dynamically-linked ufchunk structures, each containing NFPCHUNK
	 * file pointers.  The first ufchunk structure is addressed via the
	 * u_flist pointer in the user structure.
	 */
		uc.uf_next = u->u_flist;
		if ((nf = u->u_nofiles) == 0 && sgs) {
		    nf = sg.s_nofiles;
		    uc.uf_next = sg.s_flist;
		}
		for (i = 0, j = NFPCHUNK; i < nf; i++) {
		    if (++j > NFPCHUNK) {
			if( ! uc.uf_next)
			    break;
			if (kread((KA_T)uc.uf_next, (char *)&uc, sizeof(uc)))
			    break;
			j = 1;
		    }
		    f = uc.uf_ofile[j-1];
		    if (f && f != FDALLOCED) {
			alloc_lfile(NULL, i);
			process_file(f);
			if (Lf->sf)
			    link_lfile();
		    }
		}
#endif	/* IRIXV==50300 || (IRIXV>=60100 && IRIXV<60400) */

#if	IRIXV>=60400
	/*
	 * Under IRIX 6.4 and above, the head of the file pointers is an fdt
	 * struct, addressed from the proc struct.  The fdt struct addresses
	 * a chain of dynamically-linked ufchunk structures, each containing
	 * NFPCHUNK file pointers.  The first ufchunk struct is in the fdt
	 * struct
	 */
	    if (!(pf = p->p_fdt) && sgs)
		pf = sg.s_fdt;
	    if (!pf || kread((KA_T)pf, (char *)&fdt, sizeof(fdt)))
		continue;
	    for (i = j = 0; i < fdt.fd_nofiles; i++, j++) {
		if (j >= NFPCHUNK) {
		    if (!fdt.fd_flist.uf_next
		    ||  kread((KA_T)fdt.fd_flist.uf_next, (char *)&fdt.fd_flist,
			      sizeof(fdt.fd_flist)))
		    {
			break;
		    }
		    j = 0;
		}
		if ((f = fdt.fd_flist.uf_ofile[j])) {
		    alloc_lfile(NULL, i);
		    process_file(f);
		    if (Lf->sf)
			link_lfile();
		}
	    }
#endif	/* IRIXV>=60400 */

	/*
	 * Examine results.
	 */
		if (examine_lproc())
			return;
	}
}


/*
 * get_kernel_access() - get access to kernel memory
 */

static void
get_kernel_access()
{
	KA_T kvar;
	short no_proc = 0;
	short no_var = 0;
	unsigned long v;

#if	defined(WILLDROPGID)
/*
 * If kernel memory isn't coming from KMEM, drop setgid permission
 * before attempting to open the (Memory) file.
 */
	if (Memory)
		(void) dropgid();
#else	/* !defined(WILLDROPGID) */
/*
 * See if the non-KMEM memory file is readable.
 */
	if (Memory && !is_readable(Memory, 1))
		Exit(1);
#endif	/* defined(WILLDROPGID) */

/*
 * Open kernel memory access.
 */
	if ((Kd = open(Memory ? Memory : KMEM, O_RDONLY, 0)) < 0) {
	    (void) fprintf(stderr, "%s: can't open %s: %s\n", Pn,
		Memory ? Memory : KMEM, strerror(errno));
	    Exit(1);
	}	

#if	defined(WILLDROPGID)
/*
 * Drop setgid permission, if necessary.
 */
	if (!Memory)
		(void) dropgid();
#else	/* !defined(WILLDROPGID) */
/*
 * See if the name list file is readable.
 */
	if (Nmlst && !is_readable(Nmlst, 1))
		Exit(1);
#endif	/* defined(WILLDROPGID) */

/*
 * Read kernel addresses from the name list file.
 */
	(void) build_Nl(Drive_Nl);
	if (NLIST_TYPE(Nmlst ? Nmlst : N_UNIX, Nl) == -1) {
	    (void) fprintf(stderr,
		"%s: can't read kernel name list from %s\n",
		Pn, Nmlst ? Nmlst : N_UNIX);
	    Exit(1);
	}
/*
 * Set process table and var structure addresses.
 */
	if (Nmlst) {

	/*
	 * If the kernel name list file is specified, get the addresses
	 * from it.
	 */
	    if (get_Nl_value("proc", Drive_Nl, &v) < 0 || !v

#if     IRIXV>=50101
	    ||  kread((KA_T)v, (char *)&Kp, sizeof(Kp))
#endif	/* IRIXV>=50101 */

	    )
		no_proc++;

#if     IRIXV<50101
	    Kp = (KA_T)(v & 0x7fffffff);
#endif	/* IRIXV<50101 */

	    if (get_Nl_value("var", Drive_Nl, (unsigned long *)&kvar) < 0
	    ||  !kvar)
		no_var++;
	} else {

	/*
	 * If no kernel name list file is specified, get the addresses
	 * via sysmp() calls.  In the case of IRIX 6.4, get the proc
	 * hash table address via /dev/kmem.
	 *
	 * Verify that the local notion of the size of a proc structure
	 * agrees with the kernel's.
	 */

#if	IRIXV<60400
	    if ((Kp = (KA_T)sysmp(MP_KERNADDR, MPKA_PROC)) == (KA_T)-1)
#else	/* IRIXV>=60400 */
	    if (get_Nl_value("proc", Drive_Nl, &v) < 0 || !v
	    ||  kread((KA_T)v, (char *)&Kp, sizeof(Kp)) | !Kp)
#endif	/* IRIXV<60400 */

		no_proc++;

#if	IRIXV<50101
	    Kp &= 0x7fffffff;
#endif	/* IRIXV<50101 */

	    if ((kvar = (KA_T)sysmp(MP_KERNADDR, MPKA_VAR)) == (KA_T)-1)
		no_var++;
	    if ((Kpst = sysmp(MP_KERNADDR, MPKA_PROCSIZE)) != PROCSIZE) {
		(void) fprintf(stderr,
		    "%s: WARNING: kernel proc struct size differs;", Pn);
		(void) fprintf(stderr,
		    " output may not be correct.\n");
		(void) fprintf(stderr,
		    "      kernel size = %d, sizeof(struct proc) = %d\n",
		    Kpst, (int)PROCSIZE);
		if (Kpst < (PROCSIZE - (PROCSIZE >> 3))
		||  Kpst > (PROCSIZE + (PROCSIZE >> 3)))
		    Kpst = PROCSIZE;
		if (Kpst < PROCSIZE)
		    Kprl = Kpst;
		(void) fprintf(stderr,
		    "      using %d as kernel length\n", Kpst);
	    }
	}
/*
 * Notify about proc and var address errors, then exit.
 */
	if (no_proc)
	    (void) fprintf(stderr, "%s: can't get proc table address\n", Pn);
	if (no_var)
	    (void) fprintf(stderr, "%s: can't get var struct address\n", Pn);
	if (no_proc || no_var)
	    Exit(1);
/*
 * Read var structure.
 */
	if (kread(kvar, (char *)&Var, sizeof(Var))) {
	    (void) fprintf(stderr, "%s: can't read var struct\n", Pn);
	    Exit(1);
	}
}


/*
 * initialize() - perform all initialization
 */

void
initialize()
{
	get_kernel_access();
	readfsinfo();
}


/*
 * kread() - read from kernel memory
 */

int
kread(addr, buf, len)
	KA_T addr;			/* kernel memory address */
	char *buf;			/* buffer to receive data */
	READLEN_T len;			/* length to read */
{
	int br;

#if	defined(IRIX32BITS) && IRIXV<60200
	if (lseek(Kd, (KA_T)(addr & 0x7fffffff), L_SET) == (KA_T)-1L)
#else	/* !defined(IRIX32BITS) || IRIXV>=60200 */
	if (lseek(Kd, (KA_T)addr, L_SET) == (KA_T)-1L)
#endif	/* defined(IRIX32BITS) && IRIXV<60200 */

		return(-1);
	br = read(Kd, buf, len);
	return((br == len) ? 0 : 1);
}



/*
 * process_text() - process text access information
 */

static void
process_text(prp, sprp)
	preg_t *prp;		/* process region pointer */
	preg_t *sprp;		/* shared process region pointer */
{
	int i, j, lm, rx;
	struct pregion p;
	preg_t *pp;
	struct region r;
	caddr_t na;
/*
 * Loop through both process regions.
 */
	for (rx = 0; rx < 2; rx++) {
	    pp = rx ? sprp : prp;
	/*
	 * Follow the virtual address space pregion structure chain.
	 */

#if	IRIXV<50300 || (IRIXV>50300 && IRIXV<60100)
	    for (i = lm = 0; pp; pp = p.p_forw, lm++)
#else	/* IRIXV==50300 || IRIXV>=60100 */
	for (i = lm = 0;
	     pp; 
	     pp = (preg_t *)p.p_avlnode.avl_nextino, lm++)
#endif	/* IRIXV<50300 || (IRIXV>50300 && IRIXV<60100) */

	    {

	    /*
	     * Avoid infinite loop.
	     */
		if (lm > 1000) {
		    if (!Fwarn)
			(void) fprintf(stderr, "%s: too many regions, PID %d\n",
			    Pn, Lp->pid);
		    return;
		}
	    /*
	     * Read the pregion; then read the region.
	     */
		if (kread((KA_T)pp, (char *)&p, sizeof(p))
		||  kread((KA_T)p.p_reg, (char *)&r, sizeof(r)))
		    return;
	    /*
	     * Skip file entries with no file pointers and duplicate entries.
	     */

#if	IRIXV<50101
		if ((na = (caddr_t)r.r_iptr) == NULL)
#else	/* IRIXV>=50101 */
		if ((na = (caddr_t)r.r_vnode) == NULL)
#endif	/* IRIXV<50101 */

		    continue;
		for (j = 0; j < i; j++) {
		    if (Ndc[j] == na)
			break;
		}
		if (j < i)
		    continue;
	    /*
	     * Cache the vnode address for duplicate checking.
	     */
		if (!Ndc) {
		    if (!(Ndc = (caddr_t *)malloc((MALLOC_S)(sizeof(KA_T)*32))))
		    {
			(void) fprintf(stderr, "%s: no txt ptr space, PID %d\n",
			    Pn, Lp->pid);
			Exit(1);
		    }
		    Nn = 32;
		} else if (i >= Nn) {
		    Nn += 32;
		    if (!(Ndc = (caddr_t *)realloc((MALLOC_P *)Ndc,
			       (MALLOC_S)(Nn * sizeof(KA_T)))))
		    {
			(void) fprintf(stderr,
			    "%s: no more txt ptr space, PID %d\n", Pn, Lp->pid);
			Exit(1);
		    }
		}
		Ndc[i++] = na;
	    /*
	     * Save text node information.
	     */

#if	IRIXV<50200
		switch (p.p_type) {
		    case PT_MAPFILE:
			alloc_lfile(" mem", -1);
			break;
		    case PT_LIBTXT:
		    case PT_LIBDAT:
			alloc_lfile(" ltx", -1);
			break;
		    case PT_DATA:
		    case PT_TEXT:
		    default:
			alloc_lfile(" txt", -1);
		}

#else	/* IRIXV>=50200 */
		if (r.r_flags & RG_TEXT)
		    alloc_lfile(" txt", -1);
		else
		    alloc_lfile(" mem", -1);
#endif	/* IRIXV<50200 */

		process_node(na);
		if (Lf->sf)
		    link_lfile();
	    }
	}
}


/*
 * readfsinfo() - read file system information
 */

static void
readfsinfo()
{

#if	IRIXV>=50101
	char buf[FSTYPSZ+1];
	int i;
	unsigned long v;

	if ((Fsinfomax = sysfs(GETNFSTYP)) == -1) {
		(void) fprintf(stderr, "%s: sysfs(GETNFSTYP) error: %s\n",
			Pn, strerror(errno));
		Exit(1);
	} 
	if (Fsinfomax == 0)
		return;
	if ((Fsinfo = (char **)malloc((MALLOC_S)(Fsinfomax * sizeof(char *))))
	== NULL) {
		(void) fprintf(stderr, "%s: no space for sysfs info\n", Pn);
		Exit(1);
	}
	for (i = 1; i <= Fsinfomax; i++) {
		if (sysfs(GETFSTYP, i, buf) == -1) {
			(void) fprintf(stderr,
				"%s: sysfs(GETFSTYP) error: %s\n",
				Pn, strerror(errno));
			Exit(1);
		}
		buf[FSTYPSZ] = '\0';
		if ((Fsinfo[i-1] = (char *)malloc((MALLOC_S)(strlen(buf) + 1)))
		== NULL) {
			(void) fprintf(stderr,
				"%s: no space for file system entry %s\n",
				Pn, buf);
			Exit(1);
		}
		(void) strcpy(Fsinfo[i-1], buf);
	}

#else	/* IRIXV<50101 */

	char *cp;
	int i;
	MALLOC_S len;
/*
 * Access the file system information table in the kernel.
 */
	v = (unsigned long)0;
	if (get_Nl_value("nfstyp", Drive_Nl, &v) < 0 || !v
	||  kread((KA_T)v, (char *)&Nfstyp, sizeof(Nfstyp)) || Nfstyp < 1) {
		(void) fprintf(stderr,
			"%s: nfstyp value (%d) error\n", Pn, Nfstyp);
		Exit(1);
	}
	len = (MALLOC_S)(Nfstyp * sizeof(struct fsinfo));
	v = (unsigned long)0;
	if (get_Nl_value("fsinfo", Drive_Nl, &v) < 0 || !v
	||  (Fsinfo = (struct fsinfo *)malloc(len)) == NULL
	||  kread((KA_T)v, (char *)Fsinfo, len)) {
		(void) fprintf(stderr,
			"%s: fstyp read error, length = %d\n", Pn, len);
		Exit(1);
	}
/*
 * Get the file system names.
 */
	for (i = 0; i < Nfstyp; i++) {
	    if (Fsinfo[i].fs_name) {
		if ((cp = malloc(FSNAMEL)) == NULL) {
		    (void) fprintf(stderr,
			"%s: no space for file system name[%d]\n", Pn, i);
		    Exit(1);
		}
		if (kread((KA_T)Fsinfo[i].fs_name, cp, FSNAMEL)) {
		    (void) fprintf(stderr,
			"%s: can't read file system name[%d]\n", Pn, i);
		    Exit(1);
		}
		cp[FSNAMEL - 1] = '\0';
		Fsinfo[i].fs_name = cp;
	    }
	}
#endif	/* IRIXV>=50101 */

}


#if	IRIXV>=60400
/*
 * read_proc() - read IRIX 6.4 and above processes
 */

static void
read_proc()
{
	int i;
	MALLOC_S l;
	struct proc *p;
	KA_T pa;
	static struct proc **ph = (struct proc **)NULL;
	static int pm;
	unsigned long v;

	if (!ph) {

	/*
	 * Allocate space for kernel's proc hash table.
	 */
	    v = (unsigned long)0;
	    if (get_Nl_value("phash", Drive_Nl, &v) < 0 || !v
	    ||  kread((KA_T)v, (char *)&pm, sizeof(pm))) {
		(void) fprintf(stderr,
		    "%s: can't get proc hash table size\n", Pn);
		Exit(1);
	    }
	    pm++;
	    l = (MALLOC_S)(pm * sizeof(struct proc *));
	    if ((ph = (struct proc **)malloc(l)) == (struct proc **)NULL) {
		(void) fprintf(stderr,
		    "%s: can't allocate %d bytes for proc hash table\n",
		    Pn, (int)l);
		Exit(1);
	    }
	}
	if (!P) {

	/*
	 * Allocate initial space for local proc table.
	 */
	    if ((Npa = Var.v_proc) < 1) {
		(void) fprintf(stderr, "%s: bad proc table size: %d\n",
		    Pn, Var.v_proc);
		Exit(1);
	    }
	    l = (MALLOC_S)((Npa * sizeof(struct proc)) + PROCINCR);
	    if ((P = (struct proc *)malloc(l)) == (struct proc *)NULL) {
		(void) fprintf(stderr, "%s: no space for %d proc structures\n",
		    Pn, Npa);
		Exit(1);
	    }
	}
/*
 * Read the proc hash table.
 */
	if (kread(Kp, (char *)ph, (pm * sizeof(struct proc *)))) {
	    (void) fprintf(stderr, "%s: can't read proc hash table\n", Pn);
	    Exit(1);
	}
/*
 * Loop thru the proc hash table, reading the active chains.
 */
	for (i = Np = 0, p = P; i < pm; i++) {
	    if (!(pa = (KA_T)ph[i]))
		continue;
	    do {
		if (Np >= Npa) {
		    Npa += PROCINCR;
		    l = (MALLOC_S)(Npa * sizeof(struct proc));
		    if ((P = (struct proc *)realloc((MALLOC_P *)P, l))
		    == (struct proc *)NULL)
		    {
			(void) fprintf(stderr,
			    "%s: no more space for %d proc structures\n",
			    Pn, Npa);
			Exit(1);
		    }
		}
		if (kread(pa, (char *)p, Kprl))
		    break;
		pa = (KA_T)p->p_active;
		if (p->p_stat == 0 || p->p_stat == SZOMB)
		    continue;
		Np++;
		p++;
	    } while (pa && pa != (KA_T)ph[i]);
	}
}
#endif	/* IRIXV>=60400 */
