/*
 * print.c - common print support functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: print.c,v 1.5 97/09/23 09:13:13 abe Exp $";
#endif


#include "lsof.h"

#if	defined(EMPTY)
#undef	EMPTY
#endif	/* defined(EMPTY) */

#include <utmp.h>

struct utmp dutmp;			/* dummy entry to get login name size */

#define HCINC		64		/* host cache size increase chunk */
#define LNML sizeof(dutmp.ut_name)
#define PORTHASHBUCKETS	128		/* port hash bucket count
					 * !!MUST BE A POWER OF 2!! */
#define	PORTTABTHRESH	10		/* threshold at which we will switch
					 * from using getservbyport() to
					 * getservent() -- see lkup_port()
					 * and fill_porttab() */

struct hostcache {
	unsigned char addr[4];
	char *name;
};

struct porttab {
	int port;
	char *name;
	struct porttab *next;
};


static struct porttab **Pth[2] = { NULL, NULL };
						/* port hash buckets: Pth[0]
						   for TCP, Pth[1] for UDP */
#define HASHPORT(p)	(((((int)(p)) * 31415) >> 3) & (PORTHASHBUCKETS - 1))


_PROTOTYPE(static void fill_porttab,(void));
_PROTOTYPE(static char *lkup_port,(int p, int pr));
_PROTOTYPE(static int printinaddr,(void));


/*
 * endnm() - locate end of Namech
 */

char *
endnm()
{
	char *s;

	for (s = Namech; *s; s++)
		;
	return(s);
}


/*
 * fill_porttab() -- fill the port table with a getservent() scan
 */

static void
fill_porttab()
{
	int h, p, pr;
	MALLOC_S nl;
	char *nm;
	struct porttab *pt;
	struct servent *se;

	(void) endservent();
/*
 * Scan the services data base for TCP and UDP entries that have a non-null
 * name associated with them.
 */
	(void) setservent(1);
	while ((se = getservent())) {
	    if (!se->s_name || !se->s_proto)
		continue;
	    if (strcasecmp(se->s_proto, "TCP") == 0)
		pr = 0;
	    else if (strcasecmp(se->s_proto, "UDP") == 0)
		pr = 1;
	    else
		continue;
	    if (!(nl = (MALLOC_S)strlen(se->s_name)))
		continue;
	    p = ntohs(se->s_port);
	/*
	 * See if a port->service entry is already cached for this port and
	 * prototcol.  If it is, leave it alone.
	 */
	    h = HASHPORT(p);
	    for (pt = Pth[pr][h]; pt; pt = pt->next) {
		if (pt->port == p)
		    break;
	    }
	    if (pt)
		continue;
	/*
	 * Add a new entry to the cache for this port and protocol.
	 */
	    if (!(pt = (struct porttab *)malloc(sizeof(struct porttab)))) {
		(void) fprintf(stderr,
		    "%s: can't allocate porttab entry for port %d:\n",
		    Pn, p, se->s_name);
		Exit(1);
	    }
	    nl++;
	    if (!(nm = (char *)malloc(nl))) {
		(void) fprintf(stderr,
		    "%s: can't allocate %d bytes for port %d name: %s\n",
		    Pn, nl, p, se->s_name);
		Exit(1);
	    }
	    (void) strcpy(nm, se->s_name);
	    pt->name = nm;
	    pt->port = p;
	    pt->next = Pth[pr][h];
	    Pth[pr][h] = pt;
	}
	(void) endservent();
}


/*
 * gethostnm() - get host name
 */

char *
gethostnm(ia)
	struct in_addr *ia;		/* pointer to Internet address */
{
	char hbuf[32];
	static struct hostcache *hc = NULL;
	static int hcx = 0;
	char *hn, *np;
	int i;
	struct hostent *he = NULL;
	static int nhc = 0;
	unsigned char *u;

	u = (unsigned char *)ia;
/*
 * Search cache.
 */
	for (i = 0; i < hcx; i++) {
		if (u[3] != hc[i].addr[3] || u[2] != hc[i].addr[2]
		||  u[1] != hc[i].addr[1] || u[0] != hc[i].addr[0])
			continue;
		return(hc[i].name);
	}
/*
 * If -n has been specified, construct a dot-form address.  Otherwise,
 * look up host name by address.  If that fails, or if there is no name
 * in the returned hostent structure, construct a dot-form version of the
 * address.
 */
	if (Fhost)
		he = gethostbyaddr((char *)ia, sizeof(struct in_addr), AF_INET);
	if (he == NULL || he->h_name == NULL) {
		(void) sprintf(hbuf, "%u.%u.%u.%u", u[0], u[1], u[2], u[3]);
		hn = hbuf;
	} else
		hn = (char *)he->h_name;
/*
 * Allocate space for name and copy name to it.
 */
	if ((np = (char *)malloc((MALLOC_S)(strlen(hn) + 1))) == NULL) {
		(void) fprintf(stderr, "%s: no space for host name: %s\n",
			Pn, hn);
		Exit(1);
	}
	(void) strcpy(np, hn);
/*
 * Add address/name entry to cache.  Allocate cache space in HCINC chunks.
 */
	if (hcx >= nhc) {
		nhc += HCINC;
		if (hc == NULL)
			hc = (struct hostcache *)malloc(
				(MALLOC_S)(sizeof(struct hostcache) * nhc));
		else
			hc = (struct hostcache *)realloc((MALLOC_P *)hc,
				(MALLOC_S)(sizeof(struct hostcache) * nhc));
		if (hc == NULL) {
			(void) fprintf(stderr, "%s: no space for host cache\n",
				Pn);
			Exit(1);
		}
	}
	hc[hcx].addr[0] = u[0];
	hc[hcx].addr[1] = u[1];
	hc[hcx].addr[2] = u[2];
	hc[hcx].addr[3] = u[3];
	hc[hcx++].name = np;
	return(np);
}


/*
 * lkup_port() - look up port for protocol
 */

static char *
lkup_port(p, pr)
	int p;				/* port number */
	int pr;				/* protocol index: 0 = tcp, 1 = udp */
{
	static int gsbp = 0;
	int h;
	MALLOC_S len;
	char *nm, pb[128], *pn;
	struct porttab *pt;
	static int ptf = 0;
	struct servent *s;
/*
 * If the hash buckets haven't been allocated, do so now.
 */
	if (!Pth[0]) {
	    for (h = 0; h < 2; h++) {
		if (!(Pth[h] = (struct porttab **)calloc(PORTHASHBUCKETS,
				sizeof(struct porttab *))))
		{
		    (void) fprintf(stderr,
		      "%s: can't allocate %d bytes for %s port hash buckets\n",
		      Pn, (2 * (PORTHASHBUCKETS * sizeof(struct porttab *))),
		      (h == 0) ? "TCP" : "UDP");
		    Exit(1);
		}
	    }
	}
/*
 * Don't do any lookups if the -P option has been specified.  In that case,
 * just cache the %d conversion of the port number.
 */
	if (Fport) {

	/*
	 * Hash the port and see if its name has been cached.
	 */
	    h = HASHPORT(p);
	    for (;;) {
		for (pt = Pth[pr][h]; pt; pt = pt->next) {
		    if (pt->port == p)
			return(pt->name);
		}
	    /*
	     * The port has no cache entry.  If the port table hasn't already
	     * been filled by a getservent() scan of the services data base,
	     * and if the count of getservbyport() calls has reached its
	     * tolerance limit (PORTTABTHRESH), call fill_porttab() to do the
	     * getservent() scan and look for the port in the cache once more.
	     */
		if (ptf || gsbp < PORTTABTHRESH)
		    break;
		(void) fill_porttab();
		ptf++;
	    }
	}
/*
 * Allocate a new porttab entry.
 */
	if (!(pt = (struct porttab *)malloc(sizeof(struct porttab)))) {
	    (void) fprintf(stderr,
		"%s: can't allocate porttab entry for port %d\n", Pn, p);
	    Exit(1);
	}
/*
 * If the port cache has been filled via getservent(), then this lookup has
 * failed completely; enter a %d cache entry for it.
 *
 * If the port cache hasn't been filled via getservent(), look up the port's
 * service name via getservbyport().  Count the getservbyport() calls.
 *
 * Don't do a getservbyport() call if the -P option has been specified; just
 * enter a %d conversion of the port number.
 */
	pn = (char *)NULL;
	if (Fport && !ptf && p) {
	    gsbp++;
	    if ((s = getservbyport(htons(p), (pr == 0) ? "tcp" : "udp")))
		pn = s->s_name;
	}
	if (!pn || !strlen(pn)) {
	    if (p) {
		(void) sprintf(pb, "%d", p);
		pn = pb;
	    } else
		pn = "*";
	}
/*
 * Allocate space for the name; copy it to the porttab entry; and link the
 * porttab entry to its hash bucket.
 *
 * Return a pointer to the name.
 */
	len = strlen(pn) + 1;
	if (!(nm = malloc(len))) {
	    (void) fprintf(stderr,
		"%s: can't allocate %d bytes for port name: %s\n",
		Pn, len, pn);
	    Exit(1);
	}
	(void) strcpy(nm, pn);
	pt->name = nm;
	pt->port = p;
	pt->next = Pth[pr][h];
	Pth[pr][h] = pt;
	return(nm);
}


/*
 * printinaddr() - print Internet addresses
 */

static int
printinaddr()
{
	int i, j, len, ty;
	char *host, *port;
	int nl = MAXPATHLEN - 1;
	char *np = Namech;
	char pbuf[32];
/*
 * Process local network address first.  If there's a foreign address,
 * separate it from the local address with "->".
 */
	for (i = 0, *np = '\0'; i < 2; i++) {
	    if (!Lf->li[i].as)
		continue;
	    host = port = (char *)NULL;
	    if (i) {

	    /*
	     * If this is the foreign address, insert the separator.
	     */
		if (nl < 2)

addr_too_long:

		    {
			(void) fputs("network addresses too long", stdout);
			return(1);
		    }
		(void) strcpy(np, "->");
		np += 2;
		nl -= 2;
	    }
	/*
	 * Convert the address to a host name.
	 */
	    if (Lf->li[i].ia.s_addr == INADDR_ANY)
		host ="*";
	    else
		host = gethostnm(&Lf->li[i].ia);
	/*
	 * Process the port number.
	 */
	    if (Lf->li[i].p > 0) {
		if (Fport) {

		/*
		 * If converting port numbers to service names, check the
		 * protocol.
		 */
		    if (strcasecmp(Lf->iproto, "TCP") == 0)
			port = lkup_port(Lf->li[i].p, 0);
		    else if (strcasecmp(Lf->iproto, "UDP") == 0)
			port = lkup_port(Lf->li[i].p, 1);
		}
		if (!port) {
		    (void) sprintf(pbuf, "%d", Lf->li[i].p);
		    port = pbuf;
		}
	    } else if (Lf->li[i].p == 0)
		port = "*";
	/*
	 * Enter the host name.
	 */
	    if (host) {
		if ((len = strlen(host)) > nl)
		    goto addr_too_long;
		if (len) {
		    (void) strcpy(np, host);
		    np += len;
		    nl -= len;
		}
	    }
	/*
	 * Enter the port number, preceded by a colon.
	 */
	    if (port) {
		if (((len = strlen(port)) + 1) >= nl)
		    goto addr_too_long;
		*np++ = ':';
		(void) strcpy(np, port);
		np += len;
		nl -= (len + 1);
	    }
	}
	if (Namech) {
	    (void) fputs(Namech, stdout);
	    return(1);
	}
	return(0);
}


/*
 * printiproto() - print Internet protocol name
 */

void
printiproto(p)
	int p;				/* protocol number */
{
	int i;
	static int m = -1;
	char *s;

	switch(p) {

#if	defined(IPPROTO_TCP)
	case IPPROTO_TCP:
		s = "TCP";
		break;
#endif

#if	defined(IPPROTO_UDP)
	case IPPROTO_UDP:
		s = "UDP";
		break;
#endif

#if	defined(IPPROTO_IP)
	case IPPROTO_IP:
		s = "IP";
		break;
#endif

#if	defined(IPPROTO_ICMP)
	case IPPROTO_ICMP:
		s = "ICMP";
		break;
#endif

#if	defined(IPPROTO_IGMP)
	case IPPROTO_IGMP:
		s = "IGMP";
		break;
#endif

#if	defined(IPPROTO_GGP)
	case IPPROTO_GGP:
		s = "GGP";
		break;
#endif

#if	defined(IPPROTO_EGP)
	case IPPROTO_EGP:
		s = "EGP";
		break;
#endif

#if	defined(IPPROTO_PUP)
	case IPPROTO_PUP:
		s = "PUP";
		break;
#endif

#if	defined(IPPROTO_IDP)
	case IPPROTO_IDP:
		s = "IDP";
		break;
#endif

#if	defined(IPPROTO_ND)
	case IPPROTO_ND:
		s = "ND";
		break;
#endif

#if	defined(IPPROTO_RAW)
	case IPPROTO_RAW:
		s = "RAW";
		break;
#endif

#if	defined(IPPROTO_HELLO)
	case IPPROTO_HELLO:
		s = "HELLO";
		break;
#endif

#if	defined(IPPROTO_PXP)
	case IPPROTO_PXP:
		s = "PXP";
		break;
#endif

#if	defined(IPPROTO_RAWIP)
	case IPPROTO_RAWIP:
		s = "RAWIP";
		break;
#endif

#if	defined(IPPROTO_RAWIF)
	case IPPROTO_RAWIF:
		s = "RAWIF";
		break;
#endif

	default:
		s = NULL;
	}
	if (s)
		(void) sprintf(Lf->iproto, "%.*s", IPROTOL-1, s);
	else {	
		if (m < 0) {
			for (i = 0, m = 1; i < IPROTOL-2; i++)
				m *= 10;
		}
		if (m > p)
			(void) sprintf(Lf->iproto, "%d?", p);
		else
			(void) sprintf(Lf->iproto, "*%d?", p % (m/10));
	}
}


/*
 * printname() - print output name field
 */

void
printname(nl)
	int nl;				/* NL status */
{

#if	defined(HASNCACHE)
	char buf[MAXPATHLEN];
	char *cp, *cp1;
	int fp;
#endif	/* defined(HASNCACHE) */

	int ps = 0;

	if (Lf->nm && Lf->nm[0]) {

	/*
	 * Print the name characters, if there are some.
	 */
	    (void) fputs(Lf->nm, stdout);
	    ps++;
	    if (!Lf->li[0].as && !Lf->li[1].as)
		goto print_nma;
	}
	if (Lf->li[0].as || Lf->li[1].as) {
	    if (ps)
		putchar(' ');
	/*
	 * If the file has Internet addresses, print them.
	 */
	    if (printinaddr())
		ps++;
	    goto print_nma;
	}
	if (Lf->ntype == N_CHR && Lf->dev_def && printchdevname(&Lf->dev, 0)) {

	/*
	 * If this is a character device and it has a name, print it.
	 */
	    ps++;
	    goto print_nma;
	}

#if	defined(HASBLKDEV)
	if (Lf->ntype == N_BLK && printbdevname(&Lf->dev, 0)) {

	/*
	 * If this is a block device and it has a name, print it.
	 */
	    ps++;
	    goto print_nma;
	}
#endif	/* defined(HASBLKDEV) */

	if (Lf->is_com) {

	/*
	 * If this is a common node, print that fact.
	 */
	    (void) fputs("COMMON: ", stdout);
	    ps++;
	    goto print_nma;
	}
	if (Lf->fsdir || Lf->fsdev) {

	/*
	 * Print the file system directory name, device name, and
	 * possible path name components.
	 */
	    if (Lf->fsdir) {
		(void) fputs(Lf->fsdir, stdout);
		ps++;
	    }

#if	defined(HASNCACHE)
	    if (Lf->na
	    &&  (cp = ncache_lookup(buf, sizeof(buf), &fp))) {
		if (*cp == '\0')
		    goto print_nma;
		if (fp) {
		    if (*cp != '/') {
			cp1 = strrchr(Lf->fsdir, '/');
			if (cp1 == (char *)NULL ||  *(cp1 + 1) != '\0')
			    putchar('/');
		    }
		} else
		    (void) fputs(" -- ", stdout);
		(void) fputs(cp, stdout);
		ps++;
		goto print_nma;
	    }
#endif	/* defined(HASNCACHE) */

	    if (Lf->fsdev) {
		if (Lf->fsdir)
		    (void) fputs(" (", stdout);
		else
		    (void) putchar('(');
		(void) fputs(Lf->fsdev, stdout);
		(void) putchar(')');
		ps++;
	    }
	}
/*
 * Print the NAME column addition, if there is one.  If there isn't
 * make sure a NL is printed, as requested.
 */

print_nma:

	if (Lf->nma) {
	    if (ps)
		putchar(' ');
	    (void) fputs(Lf->nma, stdout);
	    ps++;
	}
/*
 * If this file has TCP/IP state information, print it.
 */
	if (!Ffield && Ftcptpi
	&&  (Lf->lts.type >= 0

#if	defined(HASTCPTPIQ)
	||   Lf->lts.rqs || Lf->lts.sqs
#endif	/* defined(HASTCPTPIQ) */

#if	defined(HASTCPTPIW)
	||   Lf->lts.rws || Lf->lts.wws
#endif	/* defined(HASTCPTPIW) */

	)) {
	    if (ps)
		putchar(' ');
	    (void) print_tcptpi(1);
	    return;
	}
	if (nl)
	    putchar('\n');
}


/*
 * printrawaddr() - print raw socket address
 */

void
printrawaddr(sa)
	struct sockaddr *sa;		/* socket address */
{
	(void) sprintf(endnm(), "%u/%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u",
		sa->sa_family,
		(unsigned char)sa->sa_data[0],
		(unsigned char)sa->sa_data[1],
		(unsigned char)sa->sa_data[2],
		(unsigned char)sa->sa_data[3],
		(unsigned char)sa->sa_data[4],
		(unsigned char)sa->sa_data[5],
		(unsigned char)sa->sa_data[6],
		(unsigned char)sa->sa_data[7],
		(unsigned char)sa->sa_data[8],
		(unsigned char)sa->sa_data[9],
		(unsigned char)sa->sa_data[10],
		(unsigned char)sa->sa_data[11],
		(unsigned char)sa->sa_data[12],
		(unsigned char)sa->sa_data[13]);
}


/*
 * printuid() - print User ID or login name
 */

char *
printuid(u, ty)
	UID_ARG u;			/* User IDentification number */
	int *ty;			/* returned UID type pointer (NULL
					 * (if none wanted).  If non-NULL
					 * then: *ty = 0 = login name
					 *	     = 1 = UID number */
{
	int i;
	struct passwd *pw;
	struct stat sb;
	static struct stat sbs;
	static struct uidcache {
	    uid_t u;
	    char nm[LNML+1];
	    struct uidcache *next;
	} **uc = (struct uidcache **)NULL;
	struct uidcache *up, *upn;
	static char user[USERPRTL+1];

	if (Futol) {
	    if (CkPasswd) {

	    /*
	     * Get the mtime and ctime of /etc/passwd, as required.
	     */
		if (stat("/etc/passwd", &sb) != 0) {
		    (void) fprintf(stderr, "%s: can't stat(/etc/passwd): %s\n",
			Pn, strerror(errno));
		    Exit(1);
		}
	    }
	/*
	 * Define the UID cache, if necessary.
	 */
	    if (!uc) {
		if ((uc = (struct uidcache **)calloc(UIDCACHEL,
						sizeof(struct uidcache *)))
		== (struct uidcache **)NULL) {
		    (void) fprintf(stderr,
			"%s: no space for %d byte UID cache hash buckets\n",
			Pn, UIDCACHEL * (sizeof(struct uidcache *)));
		    Exit(1);
		}
		if (CkPasswd) {
		    sbs = sb;
		    CkPasswd = 0;
		}
	    }
	/*
	 * If it's time to check /etc/passwd and if its the mtime/ctime has
	 * changed, destroy the existing UID cache.
	 */
	    if (CkPasswd) {
		if (sbs.st_mtime != sb.st_mtime || sbs.st_ctime != sb.st_ctime)
		{
		    for (i = 0; i < UIDCACHEL; i++) {
			if ((up = uc[i]) != (struct uidcache *)NULL) {
			    do {
				upn = up->next;
				(void) free((FREE_P *)up);
			    } while ((up = upn) != (struct uidcache *)NULL);
			    uc[i] = (struct uidcache *)NULL;
			}
		    }
		    sbs = sb;
	        }
		CkPasswd = 0;
	    }
	/*
	 * Search the UID cache.
	 */
	    i = (int)((((unsigned long)u * 31415L) >> 7) & (UIDCACHEL - 1));
	    for (up = uc[i]; up; up = up->next) {
		if (up->u == (uid_t)u) {
		    if (ty)
			*ty = 0;
		    return(up->nm);
		}
	    }
	/*
	 * The UID is not in the cache.
	 *
	 * Look up the login name from the UID for a new cache entry.
	 */
	    if ((pw = getpwuid((uid_t)u)) == (struct passwd *)NULL) {
		if (!Fwarn) {
		    (void) fprintf(stderr, "%s: no pwd entry for UID %d\n",
			Pn, u);
		}
	    } else {

	    /*
	     * Allocate and fill a new cache entry.  Link it to its hash bucket.
	     */
		if ((upn = (struct uidcache *)malloc(sizeof(struct uidcache)))
		== (struct uidcache *)NULL) {
		    (void) fprintf(stderr,
			"%s: no space for UID cache entry for: %lu, %s)\n",
			(unsigned long)u, pw->pw_name);
		    Exit(1);
		}
		(void) strncpy(upn->nm, pw->pw_name, LNML);
		upn->nm[LNML] = '\0';
		upn->u = (uid_t)u;
		upn->next = uc[i];
		uc[i] = upn;
		if (ty)
		    *ty = 0;
		return(upn->nm);
	    }
	}
/*
 * Produce a numeric conversion of the UID.
 */
	(void) sprintf(user, "%*u", USERPRTL, u);
	if (ty)
	    *ty = 1;
	return(user);
}


/*
 * printunkaf() - print unknown address family
 */

void
printunkaf(fam)
	int fam;			/* unknown address family */
{
	char *p, *s;

	p = "";
	switch (fam) {

#if	defined(AF_UNSPEC)
	case AF_UNSPEC:
		s = "UNSPEC";
		break;
#endif	/* defined(AF_UNSPEC) */

#if	defined(AF_UNIX)
	case AF_UNIX:
		s = "UNIX";
		break;
#endif	/* defined(AF_UNIX) */

#if	defined(AF_INET)
	case AF_INET:
		s = "INET";
		break;
#endif	/* defined(AF_INET) */

#if	defined(AF_IMPLINK)
	case AF_IMPLINK:
		s = "IMPLINK";
		break;
#endif	/* defined(AF_IMPLINK) */

#if	defined(AF_PUP)
	case AF_PUP:
		s = "PUP";
		break;
#endif	/* defined(AF_PUP) */

#if	defined(AF_CHAOS)
	case AF_CHAOS:
		s = "CHAOS";
		break;
#endif	/* defined(AF_CHAOS) */

#if	defined(AF_NS)
	case AF_NS:
		s = "NS";
		break;
#endif	/* defined(AF_NS) */

#if	defined(AF_ISO)
	case AF_ISO:
		s = "ISO";
		break;
#endif	/* defined(AF_ISO) */

#if	defined(AF_NBS)
# if	!defined(AF_ISO) || AF_NBS!=AF_ISO
	case AF_NBS:
		s = "NBS";
		break;
# endif	/* !defined(AF_ISO) || AF_NBS!=AF_ISO */
#endif	/* defined(AF_NBS) */

#if	defined(AF_ECMA)
	case AF_ECMA:
		s = "ECMA";
		break;
#endif	/* defined(AF_ECMA) */

#if	defined(AF_DATAKIT)
	case AF_DATAKIT:
		s = "DATAKIT";
		break;
#endif	/* defined(AF_DATAKIT) */

#if	defined(AF_CCITT)
	case AF_CCITT:
		s = "CCITT";
		break;
#endif	/* defined(AF_CCITT) */

#if	defined(AF_SNA)
	case AF_SNA:
		s = "SNA";
		break;
#endif	/* defined(AF_SNA) */

#if	defined(AF_DECnet)
	case AF_DECnet:
		s = "DECnet";
		break;
#endif	/* defined(AF_DECnet) */

#if	defined(AF_DLI)
	case AF_DLI:
		s = "DLI";
		break;
#endif	/* defined(AF_DLI) */

#if	defined(AF_LAT)
	case AF_LAT:
		s = "LAT";
		break;
#endif	/* defined(AF_LAT) */

#if	defined(AF_HYLINK)
	case AF_HYLINK:
		s = "HYLINK";
		break;
#endif	/* defined(AF_HYLINK) */

#if	defined(AF_APPLETALK)
	case AF_APPLETALK:
		s = "APPLETALK";
		break;
#endif	/* defined(AF_APPLETALK) */

#if	defined(AF_BSC)
	case AF_BSC:
		s = "BSC";
		break;
#endif	/* defined(AF_BSC) */

#if	defined(AF_DSS)
	case AF_DSS:
		s = "DSS";
		break;
#endif	/* defined(AF_DSS) */

#if	defined(AF_ROUTE)
	case AF_ROUTE:
		s = "ROUTE";
		break;
#endif	/* defined(AF_ROUTE) */

#if	defined(AF_RAW)
	case AF_RAW:
		s = "RAW";
		break;
#endif	/* defined(AF_RAW) */

#if	defined(AF_LINK)
	case AF_LINK:
		s = "LINK";
		break;
#endif	/* defined(AF_LINK) */

#if	defined(pseudo_AF_XTP)
	case pseudo_AF_XTP:
		p = "pseudo_";
		s = "XTP";
		break;
#endif	/* defined(pseudo_AF_XTP) */

#if	defined(AF_RMP)
	case AF_RMP:
		s = "RMP";
		break;
#endif	/* defined(AF_RMP) */

#if	defined(AF_COIP)
	case AF_COIP:
		s = "COIP";
		break;
#endif	/* defined(AF_COIP) */

#if	defined(AF_CNT)
	case AF_CNT:
		s = "CNT";
		break;
#endif	/* defined(AF_CNT) */

#if	defined(pseudo_AF_RTIP)
	case pseudo_AF_RTIP:
		p = "pseudo_";
		s = "RTIP";
		break;
#endif	/* defined(pseudo_AF_RTIP) */

#if	defined(AF_NETMAN)
	case AF_NETMAN:
		s = "NETMAN";
		break;
#endif	/* defined(AF_NETMAN) */

#if	defined(AF_INTF)
	case AF_INTF:
		s = "INTF";
		break;
#endif	/* defined(AF_INTF) */

#if	defined(AF_NETWARE)
	case AF_NETWARE:
		s = "NETWARE";
		break;
#endif	/* defined(AF_NETWARE) */

#if	defined(AF_NDD)
	case AF_NDD:
		s = "NDD";
		break;
#endif	/* defined(AF_NDD) */

#if	defined(AF_NIT)
# if	!defined(AF_ROUTE) || AF_ROUTE!=AF_NIT
	case AF_NIT:
		s = "NIT";
		break;
# endif	/* !defined(AF_ROUTE) || AF_ROUTE!=AF_NIT */
#endif	/* defined(AF_NIT) */

#if	defined(AF_802)
# if	!defined(AF_RAW) || AF_RAW!=AF_802
	case AF_802:
		s = "802";
		break;
# endif	/* !defined(AF_RAW) || AF_RAW!=AF_802 */
#endif	/* defined(AF_802) */

#if	defined(AF_X25)
	case AF_X25:
		s = "X25";
		break;
#endif	/* defined(AF_X25) */

#if	defined(AF_CTF)
	case AF_CTF:
		s = "CTF";
		break;
#endif	/* defined(AF_CTF) */

#if	defined(AF_WAN)
	case AF_WAN:
		s = "WAN";
		break;
#endif	/* defined(AF_WAN) */

#if	defined(AF_OSINET)
# if	defined(AF_INET) && AF_INET!=AF_OSINET
	case AF_OSINET:
		s = "OSINET";
		break;
# endif	/* defined(AF_INET) && AF_INET!=AF_OSINET */
#endif	/* defined(AF_OSINET) */

#if	defined(AF_GOSIP)
	case AF_GOSIP:
		s = "GOSIP";
		break;
#endif	/* defined(AF_GOSIP) */

#if	defined(AF_SDL)
	case AF_SDL:
		s = "SDL";
		break;
#endif	/* defined(AF_SDL) */

#if	defined(AF_IPX)
	case AF_IPX:
		s = "IPX";
		break;
#endif	/* defined(AF_IPX) */

#if	defined(AF_SIP)
	case AF_SIP:
		s = "SIP";
		break;
#endif	/* defined(AF_SIP) */

#if	defined(psuedo_AF_PIP)
	case psuedo_AF_PIP:
		p = "pseudo_";
		s = "PIP";
		break;
#endif	/* defined(psuedo_AF_PIP) */

#if	defined(AF_OTS)
	case AF_OTS:
		s = "OTS";
		break;
#endif	/* defined(AF_OTS) */

#if	defined(AF_USER)
	case AF_USER:
		s = "USER";
		break;
#endif	/* defined(AF_USER) */

	default:
		(void) sprintf(Namech, "no further information on family %#x",
			fam);
		return;
	}
	(void) sprintf(Namech, "no further information on %sAF_%s", p, s);
	return;
}
