/*
 * dnode.c - SCO UnixWare node functions for lsof
 */


/*
 * Copyright 1996 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1996 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dnode.c,v 1.3 97/06/07 20:22:30 abe Exp $";
#endif


#include "lsof.h"
#include <sys/fs/namenode.h>
#include <sys/fs/ufs_inode.h>
#include <sys/fs/xnamnode.h>


_PROTOTYPE(static void ent_fa,(caddr_t *a1, caddr_t *a2, char *d));
_PROTOTYPE(static int get_vty,(struct vnode *v, caddr_t va, struct vfs *kv, int *fx));
_PROTOTYPE(static int examine_stream,(struct stdata *vs, struct queue *q, char *mn, char *sn, KA_T *sqp));
_PROTOTYPE(static struct l_dev * findstrdev,(dev_t *dev));
_PROTOTYPE(static char isvlocked,(struct vnode *va));
_PROTOTYPE(static int readlino,(int fx, struct vnode *v, struct l_ino *i));


static struct protos {
	char *module;			/* stream module name */
	char *proto;			/* TCP/IP protocol name */
} Protos[] = {
	{ "tcpu",	"TCP"	},
	{ "udpu",	"UDP"	},
	{ "icmp",	"ICMP"	},
	{ "ipu",	"IP"	},
	{ "tcpl",	"TCP"	},
	{ "tcp",	"TCP"	},
	{ "udpl",	"UDP"	},
	{ "udp",	"UDP"	},
	{ "ipl",	"IP"	},
	{ "ip",		"IP"	}
};
#define	NPROTOS	(sizeof(Protos)/sizeof(struct protos))


/*
 * ent_fa() - enter fattach addresses in NAME column addition
 */

static void
ent_fa(a1, a2, d)
	caddr_t *a1;			/* first fattach address (NULL OK) */
	caddr_t *a2;			/* second fattach address */
	char *d;			/* direction ("->" or "<-") */
{
	char buf[64], *cp;
	MALLOC_S len;

	if (Lf->nma)
		return;
	if (!a1)
	    (void) sprintf(buf, "(FA:%s%#x)", d, *a2);
	else
	    (void) sprintf(buf, " (FA:%#x%s%#x)", *a1, d, *a2);
	len = strlen(buf) + 1;
	if ((cp = (char *)malloc(len)) == NULL) {
	    (void) fprintf(stderr,
		"%s: no space for fattach addresses at PID %d, FD %s\n",
		Pn, Lp->pid, Lf->fd);
	    Exit(1);
	}
	(void) strcpy(cp, buf);
	Lf->nma = cp;
}


/*
 * examine_stream() - examine stream
 */

static int
examine_stream(vs, q, mn, sn, sqp)
	struct stdata *vs;		/* stream head's stdata pointer */
	struct queue *q;		/* queue structure buffer */
	char *mn;			/* module name receiver */
	char *sn;			/* special module name */
	KA_T *sqp;			/* special module's q_ptr */
{
	struct module_info mi;
	struct queue *qp;
	struct qinit qi;
	struct stdata sd;
/*
 * Read stream's head.
 */
	if (!vs || readstdata(vs, &sd)) {
	    (void) sprintf(Namech, "can't read stream head from %x", vs);
	    enter_nm(Namech);
	    return(1);
	}
	if (!sd.sd_wrq) {
	    enter_nm("no stream write queue");
	    return(1);
	}
/*
 * Examine the write queue.
 */
	for (qp = sd.sd_wrq, *mn = '\0'; qp; qp = q->q_next) {

	/*
	 * Read stream queue entry.
	 */
	    if (kread((KA_T)qp, (char *)q, sizeof(struct queue))) {
		(void) sprintf(Namech, "can't read stream queue from %x", qp);
		enter_nm(Namech);
		return(1);
	    }
	/*
	 * Read queue's information structure.
	 */
	    if (!q->q_qinfo || readstqinit(q->q_qinfo, &qi)) {
		(void) sprintf(Namech, "can't read qinit for %x from %x",
		    qp, q->q_qinfo);
		enter_nm(Namech);
		return(1);
	    }
	/*
	 * Read module information structure.
	 */
	    if (!qi.qi_minfo || readstmin(qi.qi_minfo, &mi)) {
		(void) sprintf(Namech, "can't read module info for %x from %x",
		    q->q_qinfo, qi.qi_minfo);
		enter_nm(Namech);
		return(1);
	    }
	/*
	 * Read module name.
	 */
	    if (!mi.mi_idname || kread((KA_T)mi.mi_idname, mn, STRNML-1)) {
		(void) sprintf(Namech, "can't read module name for %x from %x",
		    qi.qi_minfo, mi.mi_idname);
		enter_nm(Namech);
		return(1);
	    }
	    *(mn + STRNML - 1) = '\0';
	/*
	 * Save the q_ptr of the first special module.
	 */
	    if (!sn || *sqp || !q->q_ptr)
		continue;
	    if (strcmp(mn, sn) == 0)
		*sqp = (KA_T)q->q_ptr;
	}
	return(0);
}


/*
 * findstrdev() - look up stream device by device number
 */

static struct l_dev *
findstrdev(dev)
	dev_t *dev;			/* device */
{
	struct clone *c;
	struct l_dev *dp;
/*
 * Search device table for match.
 */

#if	HASDCACHE

findstrdev_again:

#endif	/* HASDCACHE */

	if ((dp = lkupdev(dev, 0, 0)))
	    return(dp);
/*
 * Search for clone.
 */
	if (Clone) {
	    for (c = Clone; c; c = c->next) {
		if (major(*dev) == minor(Devtp[c->dx].rdev)) {

#if	HASDCACHE
		    if (DCunsafe && !Devtp[c->dx].v && !vfy_dev(&Devtp[c->dx]))
			goto findstrdev_again;
#endif	/* HASDCACHE */

		    return(&Devtp[c->dx]);
		}
	    }
	}
	return((struct l_dev *)NULL);
}


/*
 * get_vty() - get vnode type
 *
 * return: vnode type as an N_* symbol value (-1 if unrecognized)
 */

static int
get_vty(v, va, kv, fx)
	struct vnode *v;		/* vnode to test */
	caddr_t va;			/* vnode's kernel address */
	struct vfs *kv;			/* vnode's kernel vfs struct */
	int *fx;			/* file system type index */
{
	int fxt;
	int nty = N_REGLR;

	if (!v->v_vfsp)
	    return(N_REGLR);
	if (kread((KA_T)v->v_vfsp, (char *)kv, sizeof(struct vfs)) == 0) {

	/*
	 * Check the file system type.
	 */
	    fxt = kv->vfs_fstype;
	    if (fxt > 0 && fxt <= Fsinfomax) {
		if (strcmp(Fsinfo[fxt-1], "fifofs") == 0)
		    nty = N_FIFO;
		else if (strcmp(Fsinfo[fxt-1], "nfs") == 0)
		    nty = N_NFS;
		else if (strcmp(Fsinfo[fxt-1], "namefs") == 0)
		    nty = N_NM;

#if	defined(HASPROCFS)
		else if (strcmp(Fsinfo[fxt-1], "proc") == 0)
		    nty = N_PROC;
#endif	/* HASPROCFS */

	    } else {
		(void) sprintf(Namech,
		    "vnode@%#x: bad file system index (%d)", Pn, va, fxt);
		enter_nm(Namech);
		return(-1);
	    }
	} else {
	    (void) sprintf(Namech, "vnode@%#x: bad vfs pointer (%#x)",
		va, v->v_vfsp);
	    enter_nm(Namech);
	    return(-1);
	}
	if (nty == N_REGLR) {
	    if (v->v_type == VFIFO)
		nty = N_FIFO;
	    else if (v->v_stream)
		nty = N_STREAM;
	}
	*fx = fxt;
	return(nty);
}


/*
 * isvlocked() - is a vnode locked
 */

static char
isvlocked(va)
	struct vnode *va;		/* local vnode address */
{
	struct filock f, *flf, *flp;
	int i, l;

	if (!(flf = va->v_filocks))
	    return(' ');
	flp = flf;
	i = 0;
	do {
	    if (i++ > 1000)
		break;
	    if (kread((KA_T)flp, (char *)&f, sizeof(f)))
		break;
	    if (f.set.l_sysid || f.set.l_pid != (pid_t)Lp->pid)
		continue;
	    if (f.set.l_whence == 0 && f.set.l_start == 0
	    &&  (f.set.l_len == 0 || f.set.l_len == 0x7fffffff))
		l = 1;
	    else
		l = 0;
	    switch (f.set.l_type & (F_RDLCK | F_WRLCK)) {
	    case F_RDLCK:
		return((l) ? 'R' : 'r');
	    case F_WRLCK:
		return((l) ? 'W' : 'w');
	    case (F_RDLCK + F_WRLCK):
		return('u');
	    default:
		return(' ');
	    }
	} while (flp != f.next && (flp = f.next) && flp != flf);
	return(' ');
}


/*
 * process_node() - process node
 */

void
process_node(na)
	caddr_t na;			/* vnode kernel space address */
{
	char *cp;
	dev_t dev;
	char dev_ch[32];
	struct l_dev *dp;
	struct fifonode f;
	int fx, rfx;
	struct l_ino i;
	int j, k;
	struct vfs kv, rkv;
	struct vfs *la = NULL;
	struct module_info mi;
	char mn[STRNML];
	struct mnode mno;
	int ni = 0;
	struct namenode nn;
	int px;
	struct queue q;
	struct rnode r;
	struct snode rs;
	struct vnode rv, v;
	struct snode s;
	unsigned char sd = 1;
	struct so_so so;
	KA_T sqp = (KA_T)NULL;
	char *ty;
	enum vtype type;
	struct sockaddr_un ua;
	struct l_vfs *lvfs;

#if	defined(HASPROCFS)
	struct as as;
	struct proc p;
	struct procfsid *pfi;
	struct pid pids;
	struct prnode pr;
#endif	/* HASPROCFS */

/*
 * Read the vnode.
 */
	if (!na) {
	    enter_nm("no vnode address");
	    return;
	}
	if (readvnode((caddr_t)na, &v)) {
            enter_nm(Namech);
            return;
        }

# if    defined(HASNCACHE)
	Lf->na = (unsigned long)na;
# endif /* defined(HASNCACHE) */

/*
 * Determine the vnode type.
 */
	if ((Ntype = get_vty(&v, na, &kv, &fx)) < 0)
		return;
/*
 * Determine the lock state.
 */

get_lock_state:

	Lf->lock = isvlocked(&v);
/*
 * Establish the local virtual file system structure.
 */
	if (!v.v_vfsp)
	    lvfs = (struct l_vfs *)NULL;
	else {
	    if ((lvfs = readvfs(v.v_vfsp, &kv)) == NULL) {
		(void) sprintf(Namech, "bad vfs for %#x at %#x", na, v.v_vfsp);
		enter_nm(Namech);
		return;
	    }
	}
/*
 * Read the fifonode, inode, namenode, prnode, rnode, snode, ...
 */
	switch (Ntype) {
	case N_FIFO:
	    if (!v.v_data || readfifonode(v.v_data, &f)) {
		(void) sprintf(Namech,
		    "vnode@%#x: can't read fifonode (%#x)", na, v.v_data);
		enter_nm(Namech);
		return;
	    }
	    if (f.fn_realvp) {
		if (readvnode((caddr_t)f.fn_realvp, &rv)) {
		    (void) sprintf(Namech,
			"fifonode@%#x: can't read real vnode (%#x)",
			v.v_data, f.fn_realvp);
		    enter_nm(Namech);
		    return;
		}
		if (!rv.v_data || readlino(fx, &rv, &i)) {
		    (void) sprintf(Namech,
			"fifonode@%#x: can't read inode (%#x)",
			v.v_data, rv.v_data);
		    enter_nm(Namech);
		    return;
		}
	    } else
		ni = 1;
	    break;
	case N_NFS:
	    if (!v.v_data || readrnode((caddr_t)v.v_data, &r)) {
		(void) sprintf(Namech, "vnode@%#x: can't read rnode (%#x)",
		    na, v.v_data);
		enter_nm(Namech);
		return;
	    }
	    break;
	case N_NM:
	    if (!v.v_data || kread((KA_T)v.v_data, (char *)&nn, sizeof(nn))) {
		(void) sprintf(Namech, "vnode@%#x: no namenode (%#x)",
		    na, v.v_data);
		enter_nm(Namech);
		return;
	    }
	    i.dev = nn.nm_vattr.va_fsid;
	    i.rdev = nn.nm_vattr.va_rdev;
	    i.number = nn.nm_vattr.va_nodeid;
	    i.size = nn.nm_vattr.va_size;
	    if (!nn.nm_mountpt)
		break;
	/*
	 * The name node is mounted over/to another vnode.  Process that node.
	 */
	    (void) ent_fa(&na, (caddr_t *)nn.nm_mountpt, "->");
	    if (kread((KA_T)nn.nm_mountpt, (char *)&rv, sizeof(rv))) {
		(void) sprintf(Namech,
		    "vnode@%#x: can't read namenode's mounted vnode (%#x)",
		    na, nn.nm_mountpt);
		return;
	    }
	    if ((Ntype = get_vty(&rv, (caddr_t)nn.nm_mountpt, &rkv, &rfx)) < 0)
		return;
	/*
	 * Unless the mounted-over/to node is another "namefs" node, promote
	 * it to the vnode of interest.
	 */
	    if (Ntype == N_NM)
		break;
	    fx = rfx;
	    kv = rkv;
	    v = rv;
	    if (lvfs) {
		(void) free((MALLOC_P *)lvfs);
		lvfs = (struct l_vfs *)NULL;
	    }
	    goto get_lock_state;

#if	defined(HASPROCFS)
	case N_PROC:
	    ni = 1;
	    if (!v.v_data || kread((KA_T)v.v_data, (char *)&pr, sizeof(pr))) {
		(void) sprintf(Namech, "vnode@%#x: can't read prnode (%#x)",
		    na, v.v_data);
		enter_nm(Namech);
		return;
	    }
	    if (pr.pr_proc == NULL) {
		sd = 0;
		if (v.v_type == VDIR) {
		    (void) sprintf(Namech, "/%s", HASPROCFS);
		    enter_nm(Namech);
		    i.number = PR_ROOTINO;
		} else {
		    (void) sprintf(Namech, "/%s/???", HASPROCFS);
		    enter_nm(Namech);
		    i.number = -1L;
		}
		break;
	    }
	    if (kread((KA_T)pr.pr_proc, (char *)&p, sizeof(p))) {
		(void) sprintf(Namech, "prnode@%#x: can't read proc (%#x)",
		    v.v_data, pr.pr_proc);
		enter_nm(Namech);
		return;
	    }
	    if (!p.p_pidp || kread((KA_T)p.p_pidp, (char *)&pids, sizeof(pids)))
	    {
		(void) sprintf(Namech,
		    "proc struct at %#x: can't read pid (%#x)",
		    pr.pr_proc, p.p_pidp);
		enter_nm(Namech);
		return;
	    }
	    if (!p.p_as ||  kread((KA_T)p.p_as, (char *)&as, sizeof(as)))
		sd = 0;
	    else
		i.size = as.a_size;
	    (void) sprintf(Namech, "/%s/%0*d", HASPROCFS, PNSIZ, pids.pid_id);
	    i.number = (long)pids.pid_id + PR_INOBIAS;
	    enter_nm(Namech);
	    break;
#endif	/* HASPROCFS */

	case N_STREAM:
	    if (v.v_stream) {
		Lf->is_stream = 1;
	    /*
	     * Get the queue pointer and module name at the end of the stream.
	     * The module name indentifies socket streams.
	     */
		if (examine_stream(v.v_stream, &q, mn, "sockmod", &sqp))
		    return;
		for (px = 0; px < NPROTOS; px++) {
		    if (strcmp(mn, Protos[px].module) == 0) {
			process_socket(Protos[px].proto, &q);
			return;
		    }
		}
	    /*
	     * If this stream has a "sockmod" module with a non-NULL q_ptr,
	     * try to use it to read an so_so structure.
	     */
		if (sqp && kread(sqp, (char *)&so, sizeof(so)) == 0)
		    break;
		sqp = (KA_T)NULL;
		(void) strcpy(Namech, "STR");
		j = strlen(Namech);
		if (v.v_type == VCHR) {
	    /*
	     * If this is a VCHR stream, look up the device name and record it.
	     *
	     * Because the stream is also represented by a VCHR device, record
	     * the device inode number and clear the stream flag so the file
	     * can be located by the device name.
	     */
		    if ((dp = findstrdev((dev_t *)&v.v_rdev))) {
			Lf->inode = (unsigned long)dp->inode;
			Lf->inp_ty = 1;
			Lf->is_stream = 0;
			Namech[j++] = ':';
			k = strlen(dp->name);
			if ((j + k) <= (MAXPATHLEN - 1)) {
			    (void) strcpy(&Namech[j], dp->name);
			    j += k;
			    if ((cp = strrchr(Namech, '/'))
			    &&  *(cp + 1) == '\0')
			    {
				*cp = '\0';
				j--;
			    }
			}
		    }
		}
	    /*
	     * Follow the "STR" and possibly the device name with "->" and
	     * the module name.
	     */
		if ((j + 2) <= (MAXPATHLEN - 1)) {
		    (void) strcpy(&Namech[j], "->");
		    j += 2;
		}
		if (mn[0]) {
		    if ((j + strlen(mn)) <= (MAXPATHLEN - 1))
			(void) strcpy(&Namech[j], mn);
		} else {
		    if ((j + strlen("none")) <= (MAXPATHLEN - 1))
			(void) strcpy(&Namech[j], "none");
	        }
	    }
	    break;
	case N_REGLR:
	default:

	/*
	 * Follow a VCHR vnode to its snode, then to its real vnode, finally
	 * to its inode.
	 */
	    if (v.v_type == VCHR) {
		if (!v.v_data || readsnode(v.v_data, &s)) {
		    (void) sprintf(Namech,
			"vnode@%#x: can't read snode (%#x)", na, v.v_data);
		    enter_nm(Namech);
		    return;
		}
		if (s.s_realvp) {
		    if (readvnode((caddr_t)s.s_realvp, &rv)) {
			(void) sprintf(Namech,
			    "snode@%#x: can't read real vnode (%#x)",
			    v.v_data, s.s_realvp);
			enter_nm(Namech);
			return;
		    }
		    if (!rv.v_data || readlino(fx, &rv, &i)) {
			(void) sprintf(Namech,
			    "snode@%#x: can't read inode (%#x)",
			     v.v_data, rv.v_data);
			enter_nm(Namech);
			return;
		     }
		}
	    /*
	     * If there's no real vnode, look for a common vnode and a
	     * common snode.
	     */
		else if (s.s_commonvp) {
		    if (readvnode((caddr_t)s.s_commonvp, &rv)) {
			(void) sprintf(Namech,
			    "snode@%#x: can't read real vnode (%#x)",
			    v.v_data, s.s_commonvp);
			enter_nm(Namech);
			return;
		    }
		    if (!rv.v_data || readsnode(rv.v_data, &rs)) {
			(void) sprintf(Namech,
				"vnode@%#x: can't read snode(%#x)",
				s.s_commonvp, rv.v_data);
			enter_nm(Namech);
			return;
		    }
		    ni = 1;
		} else
		    ni = 1;
		break;
	    }
	    if (v.v_data == NULL) {
		(void) sprintf(Namech, "vnode@%#x: no further information",
		    na);
		enter_nm(Namech);
		return;
	    }
	/*
	 * Read inode information.
	 */
	    if (readlino(fx, &v, &i)) {
		(void) sprintf(Namech,
		    "vnode@%#x: can't read successor node from %#x",
		    na, v.v_data);
		enter_nm(Namech);
		return;
	    }
	}
/*
 * Get device and type for printing.
 */
	switch (Ntype) {
	case N_NFS:
	    dev = r.r_attr.va_fsid;
	    break;

#if	defined(HASPROCFS)
	case N_PROC:
	    dev = kv.vfs_dev;
	    break;
#endif	/* HASPROCFS */

	case N_STREAM:
	    if (sqp) {
		dev = so.lux_dev.addr.tu_addr.dev;
		break;
	    }
	    /* fall through */
	default:
	    dev = (v.v_type == VCHR || ni) ? v.v_rdev : i.dev;
	} type = v.v_type;
	if (lvfs && lvfs->dir == NULL)
	    (void) completevfs(lvfs, &dev);
/*
 * Obtain the inode number.
 */
	switch (Ntype) {
	case N_NFS:
	    Lf->inode = (unsigned long)r.r_attr.va_nodeid;
	    Lf->inp_ty = 1;
	    break;

#if	defined(HASPROCFS)
	case N_PROC:
	    Lf->inode = (unsigned long)i.number;
	    Lf->inp_ty = 1;
	    break;
#endif	/* HASPROCFS */

	case N_FIFO:
	    if (!f.fn_realvp) {
		(void) sprintf(dev_ch, "0x%08x", v.v_data);
		enter_dev_ch(dev_ch);
		Lf->inode = (unsigned long)f.fn_ino;
		Lf->inp_ty = 1;
		if (f.fn_flag & ISPIPE)
		    (void) strcpy(Namech, "PIPE");
		if (f.fn_mate)
		    (void) sprintf(endnm(), "->0x%08x", f.fn_mate);
		break;
	    }
	    /* fall through */
	case N_REGLR:
	    if (!ni) {
		Lf->inode = (unsigned long)i.number;
		Lf->inp_ty = 1;
	    }
	    break;
	case N_STREAM:
	    if (sqp) {
		Lf->inode = (unsigned long)so.lux_dev.addr.tu_addr.ino;
		Lf->inp_ty = 1;
	    }
	}
/*
 * Obtain the file size.
 */
	if (Foffset)
	    Lf->off_def = 1;
	else {
	    switch (Ntype) {
	    case N_FIFO:
	    case N_STREAM:
		if (!Fsize)
		    Lf->off_def = 1;
		break;
	    case N_NFS:
		Lf->sz = (unsigned long)r.r_attr.va_size;
		Lf->sz_def = sd;
		break;

#if	defined(HASPROCFS)
	    case N_PROC:
		Lf->sz = (unsigned long)i.size;
		Lf->sz_def = sd;
		break;
#endif	/* HASPROCFS */

	    case N_REGLR:
		if (type == VREG || type == VDIR) {
		    if (!ni) {
			Lf->sz = (unsigned long)i.size;
			Lf->sz_def = sd;
		    }
		} else if (type == VCHR && !Fsize)
		    Lf->off_def = 1;
		break;
	    }
	}
/*
 * Record an NFS file selection.
 */
	if (Ntype == N_NFS && Fnfs)
	    Lf->sf |= SELNFS;
/*
 * Save the file system names.
 */
	if (lvfs) {
	    Lf->fsdir = lvfs->dir;
	    Lf->fsdev = lvfs->fsname;
	}
/*
 * Format the vnode type, and possibly the device name.
 */
	switch (type) {
	case VNON:
	    ty ="VNON";
	    break;
	case VREG:
	case VDIR:
	    ty = (type == VREG) ? "VREG" : "VDIR";
	    Lf->dev_def = 1;
	    Lf->dev = dev;
	    break;
	case VBLK:
	    ty = "VBLK";
	    Lf->dev = dev;
	    Lf->dev_def = 1;
	    Ntype = N_BLK;
	    break;
	case VCHR:
	    Lf->dev = dev;
	    Lf->dev_def = 1;
	    ty = "VCHR";
	    if (Lf->is_stream == 0)
		Ntype = N_CHR;
	    break;
	case VLNK:
	    ty = "VLNK";
	    break;

#if	defined(VSOCK)
	case VSOCK:
	    ty = "SOCK";
	    break;
#endif	/* VSOCK */

	case VBAD:
	    ty = "VBAD";
	    break;
	case VFIFO:
	    if (!Lf->dev_ch || Lf->dev_ch[0] == '\0') {
		Lf->dev = dev;
		Lf->dev_def = 1;
	    }
	    ty = "FIFO";
	    break;
	case VUNNAMED:
	    ty = "UNNM";
	    break;
	default:
	    if (type > 9999)
		(void) sprintf(Lf->type, "*%03d", type % 1000);
	    else
		(void) sprintf(Lf->type, "%4d", type);
	    (void) strcpy(Namech, "unknown type");
	    ty = NULL;
	}
	if (ty)
	    (void) strcpy(Lf->type, ty);
	Lf->ntype = Ntype;
/*
 * If this is a VBLK file and it's missing an inode number, try to
 * supply one.
 */
	if (Lf->inp_ty == 0 && Lf->ntype == N_BLK && Lf->dev_def)
	    find_bl_ino();
/*
 * If this is a VCHR file and it's missing an inode number, try to
 * supply one.
 */
	if (Lf->inp_ty == 0 && type == VCHR && Lf->dev_def)
	    find_ch_ino();
/*
 * If this is a stream with a "sockmod" module whose q_ptr leads to an
 * so_so structure, assume it's a UNIX domain socket and try to get
 * the path.  Clear the is_stream status.
 */
	if (Ntype == N_STREAM && sqp) {
	    if (Funix)
		Lf->sf |= SELUNX;
	    (void) strcpy(Lf->type, "unix");
	    if (!Namech[0]
	    &&  so.laddr.buf && so.laddr.len == sizeof(ua)
	    &&  kread((KA_T)so.laddr.buf, (char *)&ua, sizeof(ua)) == 0) {
		ua.sun_path[sizeof(ua.sun_path) - 1] = '\0';
		(void) strcpy(Namech, ua.sun_path);
	    }
	    Lf->is_stream = 0;
	}
/*
 * Test for specified file.
 */

#if	defined(HASPROCFS)
	if (Ntype == N_PROC) {
	    if (Procsrch)
		Lf->sf |= SELNM;
	    else {
		for (pfi = Procfsid; pfi; pfi = pfi->next) {
		    if (pfi->pid == pids.pid_id) {
			Lf->sf |= SELNM;
			break;
		    }
		}
	    }
	} else
#endif	/* HASPROCFS */

	{
	    if (Sfile && is_file_named(NULL, type))
		Lf->sf |= SELNM;
	}
/*
 * Enter name characters.
 */
	if (Namech[0])
	    enter_nm(Namech);
}


/*
 * readlino() - read local inode information
 */

static int
readlino(fx, v, i)
	int fx;				/* file system index */
	struct vnode *v;		/* vnode pointing to inode */
	struct l_ino *i;		/* local inode */
{
	struct vnode fa;
	struct vfs kv;
	struct mnode mn;
	struct inode sn;
	struct xnamnode xn;

	if (fx < 1 || fx > Fsinfomax || v->v_data == NULL)
	    return(1);
	if (strcmp(Fsinfo[fx-1], "fifofs") == 0
	||  strcmp(Fsinfo[fx-1], "ufs") == 0) {
	    if (kread((KA_T)v->v_data, (char *)&sn, sizeof(sn)))
		return(1);
	    i->dev = sn.i_dev;
	    i->rdev = v->v_rdev;
	    i->number = sn.i_number;
	    i->size = sn.i_size;
	    return(0);
	} else if (strcmp(Fsinfo[fx-1], "s5") == 0)
	    return(reads5lino(v, i));
 	else if (strcmp(Fsinfo[fx-1], "vxfs") == 0)
	    return(readvxfslino(v, i));
	else if (strcmp(Fsinfo[fx-1], "xnamfs") == 0
	     ||  strcmp(Fsinfo[fx-1], "XENIX") == 0)
	{
	    if (kread((KA_T)v->v_data, (char *)&xn, sizeof(xn)))
		return(1);
	    i->dev = xn.x_dev;
	    i->rdev = xn.x_fsid;
	    i->size = xn.x_size;
	    return(0);
	} else if (strcmp(Fsinfo[fx-1], "memfs") == 0) {
	    if (kread((KA_T)v->v_data, (char *)&mn, sizeof(mn)))
		return(1);
	    i->dev = mn.mno_fsid;
	    i->number = mn.mno_nodeid;
	    i->rdev = mn.mno_rdev;
	    i->size = mn.mno_size;
	    return(0);
	}
	return(1);
}
