/*
 * dfile.c - SunOS (Solaris 1.x and 2.x) file processing functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dfile.c,v 1.2 97/04/23 15:55:52 abe Exp $";
#endif


#include "lsof.h"


#if	defined(solaris) && solaris<20500
/*
 * get_max_fd() - get maximum file descriptor plus one
 */

int
get_max_fd()
{
	struct rlimit r;

	if (getrlimit(RLIMIT_NOFILE, &r))
	    return(-1);
	return(r.rlim_cur);
}
#endif	/* defined(solaris) && solaris<20500 */


/*
 * is_file_named() - is this file named?
 */

int
is_file_named(p, nt, vt, ps)
	char *p;			/* path name; NULL = search by device
					 * and inode (from *Lf) */
	int nt;				/* node type -- e.g., N_* */
	enum vtype vt;			/* vnode type */
	int ps;				/* print status: 0 = don't copy name
					 * to Namech */
{
	int f;
	struct sfile *s;
/*
 * Search the file chain.
 */
	for (f = 0, s = Sfile; s; s = s->next) {

	/*
	 * Search by path name, as requested.
	 */
	    if (p != NULL) {
	    	if (strcmp(p, s->aname) == 0 || strcmp(p, s->name)  == 0) {
		    f = 2;
		    break;
		}
		continue;
	    }
	/*
	 * Check for a Sun clone file.
	 */
	    if (nt == N_STREAM

#if	defined(solaris)
	    &&  HaveCloneMaj && major(s->dev) == CloneMaj
#else	/* !defined(solaris) */
	    && major(s->dev) == CLONEMAJ
#endif	/* defined(solaris) */

	    && major(Lf->dev) == minor(s->dev)) {
		f = 1;
		break;
	    }
	/*
	 * Check for a regular file or directory -- the device and
	 * inode numbers must match.
	 */
	    if (s->type) {
		if (Lf->dev == s->dev && (ino_t)Lf->inode == s->i) {
		    f = 1;
		    break;
		}
		continue;
	    }
	/*
	 * Check for a file system match.
	 * Try to avoid matching character files to non-character devices.
	 */
	    if (Lf->dev == s->dev) {
		if (!(vt == VCHR && s->mode != S_IFCHR)) {
		    f = 1;
		    break;
		}
	    }
	}
/*
 * Convert the name if a match occurred.
 */
	if (f) {
	    if (f == 2) {
		if (ps)
		    (void) strcpy(Namech, p);
	    } else {
		if (ps) {
		    (void) strcpy(Namech, s->name);
		    if (s->devnm)
			(void) sprintf(endnm(), " (%s)", s->devnm);
		}
	    }
	    s->f = 1;
	    return(1);
	}
	return(0);
}


/*
 * print_file() - print file
 */

void
print_file()
{
	char buf[16];
	int mjd, mnd;

	if (Hdr == 0) {
	    (void) printf(
		"COMMAND     PID%s%s     USER   FD   TYPE       DEVICE ",
		Fpgrp ? "   PGRP" : "",

#if	defined(HASPPID)
		Fppid ? "   PPID" : ""
#else	/* !defined(HASPPID) */
		""
#endif	/* defined(HASPPID) */

		);
	    (void) printf("%10s", Foffset ? "OFFSET"
					  : Fsize ? "SIZE"
						  : "SIZE/OFF");
	    (void) puts("      INODE NAME");
	    Hdr++;
	}
	(void) printf("%-*.*s%6d", CMDL, CMDL,
	    Lp->cmd ? Lp->cmd : "(unknown)", Lp->pid);
	if (Fpgrp)
	    printf(" %6d", Lp->pgrp);

#if	defined(HASPPID)
	if (Fppid)
	    printf(" %6d", Lp->ppid);
#endif	/* defined(HASPPID) */

	(void) printf(" %8.8s %4s%c%c %4.4s ",
			printuid((UID_ARG)Lp->uid, NULL),
			Lf->fd,
			Lf->access,
			Lf->lock,
			Lf->type
		);
	if (Lf->dev_def) {

#if	defined(solaris)
	/*
	 * Avoid the Solaris major() and minor() functions from
	 * makedev(3C) to get printable major/minor numbers.
	 *
	 * We would like to use the L_MAXMAJ definition from
	 * <sys/sysmacros.h>, but it's incorrect in Solaris 2.3.
	 */
		mjd = (Lf->dev >> L_BITSMINOR) & 0x3fff;
		mnd = Lf->dev & L_MAXMIN;
#else
		mjd = major(Lf->dev);
		mnd = minor(Lf->dev);
#endif

		if (mjd < 10000)
			printf(" %4d,", mjd);
		else
			printf(" *%03d,", mjd % 1000);
		if (mnd < 1000000)
			printf("%6d ", mnd);
		else
			printf("*%05d ", mnd % 100000);
	} else {
		if (Lf->dev_ch) {
			(void) fputs("  ", stdout);
			(void) fputs(Lf->dev_ch, stdout);
			putchar(' ');
		} else
			(void) fputs("             ", stdout);
	}
	if (Lf->sz_def)
		(void) printf("%10lu", Lf->sz);
	else if (Lf->off_def) {
		if (Lf->off < 100000000l) {
			(void) sprintf(buf, "0t%ld", Lf->off);
			(void) printf("%10.10s", buf);
		} else
			(void) printf("%#10x", (Lf->off & 0xffffffff));
	} else
		    (void) fputs("          ", stdout);
	switch(Lf->inp_ty) {
	case 1:

#if	!defined(solaris)
		if ((long)Lf->inode < 0L)
			(void) printf(" %10ld ", (long)Lf->inode);
		else
#endif	/* !solaris */

			(void) printf(" %10lu ", Lf->inode);
		break;
	case 2:
		if (Lf->iproto[0] == '\0')
			(void) fputs("            ", stdout);
		else
			(void) printf(" %10.10s ", Lf->iproto);
		break;
	default:
		(void) fputs("            ", stdout);
	}
	printname(1);
}


/*
 * process_file() - process file
 */

void
process_file(fp)
	struct file *fp;		/* kernel file structure address */
{
	struct file f;
	int flag;

#if	defined(FILEPTR)
	FILEPTR = &f;
#endif	/* defined(FILEPTR) */

	if (kread((KA_T)fp, (char *)&f, sizeof(f))) {
		(void) sprintf(Namech, "can't read file struct from %#x", fp);
		enter_nm(Namech);
		return;
	}
	Lf->off = (unsigned long)f.f_offset;

	if (f.f_count) {

	/*
	 * Construct access code.
	 */
		if ((flag = (f.f_flag & (FREAD | FWRITE))) == FREAD)
			Lf->access = 'r';
		else if (flag == FWRITE)
			Lf->access = 'w';
		else if (flag == (FREAD | FWRITE))
			Lf->access = 'u';

#if	defined(solaris)
	/*
	 * Solaris file structures contain a vnode pointer.  Process it.
	 */
		process_node((caddr_t)f.f_vnode);
		return;
#else	/* !solaris */

	/*
	 * Process non-Solaris file structure by its type.
	 */
		switch (f.f_type) {

		case DTYPE_VNODE:
			if (!Selinet)
				process_node(f.f_data);
			return;
		case DTYPE_SOCKET:
			process_socket((caddr_t)f.f_data);
			return;
		default:
			if (f.f_type != 0 || f.f_ops != NULL) {
				(void) sprintf(Namech,
					"%#x file struct, ty=%#x, op=%#x",
					fp, f.f_type, f.f_ops);
				enter_nm(Namech);
				return;
			}
		}
#endif	/* solaris */

	}
	enter_nm("no more information");
}
