/*
 * dnode.c - NetBSD and OpenBSD node functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dnode.c,v 1.6 97/07/09 15:52:57 abe Exp $";
#endif


#include "lsof.h"


#if	defined(HASFDESCFS)
_PROTOTYPE(static int lkup_dev_tty,(dev_t *dr, unsigned long *ir));
#endif	/* defined(HASFDESCFS) */

#if	defined(HASPROCFS)
#define PROCFSNAME_TMP(x)	#x
#define	PROCFSNAME_STR(x)	PROCFSNAME_TMP(x)
#define	PROCFSNAME		PROCFSNAME_STR(HASPROCFS)

_PROTOTYPE(static void getmemsz,(pid_t pid));

# if	!defined(PGSHIFT)
#define	PGSHIFT	pgshift
# endif	/* !defined(PGSHIFT) */


/*
 * getmemsz() - get memory size of a /proc/<n>/mem entry
 */

static void
getmemsz(pid)
	pid_t pid;
{
	int n;
	struct kinfo_proc *p;
	struct vmspace vm;

	for (n = 0, p = P; n < Np; n++, p++) {
		if (p->P_PID == pid) {
			if (p->P_VMSPACE == NULL
			||  kread((KA_T)p->P_VMSPACE, (char *)&vm, sizeof(vm)))
				return;
			Lf->sz = (unsigned long)ctob(vm.vm_tsize + vm.vm_dsize
			       +		     vm.vm_ssize);
			Lf->sz_def = 1;
			return;
		}
	}
}
#undef	PGSHIFT
#endif	/* defined(HASPROCFS) */


#if	defined(HASFDESCFS)
/*
 * lkup_dev_tty() - look up /dev/tty
 */

static int
lkup_dev_tty(dr, ir)
	dev_t *dr;			/* place to return device number */
	unsigned long *ir;		/* place to return inode number */
{
	int i;

	readdev(0);

# if	defined(HASDCACHE)

lkup_dev_tty_again:

# endif	/* defined(HASDCACHE) */

	for (i = 0; i < Ndev; i++) {
	    if (strcmp(Devtp[i].name, "/dev/tty") == 0) {

# if	defined(HASDCACHE)
		if (DCunsafe && !Devtp[i].v && !vfy_dev(&Devtp[i]))
		    goto lkup_dev_tty_again;
# endif	/* defined(HASDCACHE) */

		*dr = Devtp[i].rdev;
		*ir = (unsigned long)Devtp[i].inode;
		return(1);
	    }
	}

# if	defined(HASDCACHE)
	if (DCunsafe) {
	    (void) rereaddev();
	    goto lkup_dev_tty_again;
	}
# endif	/* defined(HASDCACHE) */

	return(-1);
}
#endif	/* defined(HASFDESCFS) */


/*
 * process_node() - process vnode
 */

void
process_node(va)
	caddr_t va;			/* vnode kernel space address */
{

	dev_t dev;
	char dev_ch[32];
	int devs = 0;
	struct lockf lf, *lff, *lfp;
	int lt;
	struct inode i;
	struct mfsnode m;
	struct nfsnode n;
	int ns;
	enum nodetype {NONODE, CDFSNODE, EXT2NODE, FDESCNODE, INODE,
		KERNFSNODE, MFSNODE, NFSNODE, PFSNODE} nty = NONODE;
	char *ty;
	enum vtype type;
	struct vnode *v, vb;
	struct l_vfs *vfs;

#if	defined(HAS9660FS)
	dev_t iso_dev;
	unsigned long iso_ino, iso_sz;
	int iso_stat = 0;
#endif	/* defined(HAS9660FS) */

#if	defined(HASFDESCFS)
	struct fdescnode f;
	static dev_t f_tty_dev;
	static unsigned long f_tty_ino;
	static int f_tty_s = 0;
#endif	/* defined(HASFDESCFS) */

#if	defined(HASKERNFS)
	struct kernfs_node kn;
	struct stat ksb;
	int ksbs = 0;
	struct kern_target kt;
	int ktnl;
	char ktnm[MAXPATHLEN+1];
#endif	/* defined(HASKERNFS) */

#if	defined(HASPROCFS)
	struct pfsnode p;
	struct procfsid *pfi;
#endif	/* defined(HASPROCFS) */

/*
 * Read the vnode.
 */
	if ( ! va) {
		enter_nm("no vnode address");
		return;
	}
	v = &vb;
	if (readvnode((caddr_t)va, v)) {
                enter_nm(Namech);
                return;
        }

# if	defined(HASNCACHE)
	Lf->id = v->v_id;
	Lf->na = (unsigned long)va;
# endif	/* defined(HASNCACHE) */

/*
 * Get the vnode type.
 */
	if (v->v_mount == NULL)
		vfs = NULL;
	else {
		vfs = readvfs(v->v_mount);
		if (vfs) {
			if (strcmp(vfs->type, MOUNT_NFS) == 0)
				Ntype = N_NFS;

#if	defined(HASKERNFS)
			else if (strcmp(vfs->type, MOUNT_KERNFS) == 0)
				Ntype = N_KERN;
#endif	/* defined(HASKERNFS) */

#if	defined(HASPROCFS)
			else if (strcmp(vfs->type, MOUNT_PROCFS) == 0)
				Ntype = N_PROC;
#endif	/* defined(HASPROCFS) */

#if	defined(HAS9660FS)
			else if (strcmp(vfs->type, MOUNT_CD9660) == 0)
				Ntype = N_CDFS;
#endif	/* defined(HAS9660FS) */

		}
	}
	if (Ntype == N_REGLR) {
		switch (v->v_type) {
		case VFIFO:
			Ntype = N_FIFO;
			break;
		}
	}
/*
 * Read the successor node.
 */
	switch (v->v_tag) {

#if	defined(HAS9660FS)
	case VT_ISOFS:
		if (read_iso_node(v, &iso_dev, &iso_ino, &iso_sz)) {
			(void) sprintf(Namech, "can't read iso_node at: %x",
			    v->v_data);
			enter_nm(Namech);
			return;
		}
		iso_stat = 1;
		nty = CDFSNODE;
		break;
#endif	/* defined(HAS9660FS) */

#if	defined(HASKERNFS)
	case VT_KERNFS:
	
	/*
	 * Read the kernfs_node.
	 */
		if (!v->v_data
		||  kread((KA_T)v->v_data, (char *)&kn, sizeof(kn))) {
		    if (v->v_type != VDIR || !(v->v_flag && VROOT)) {
			(void) sprintf(Namech, "can't read kernfs_node at: %x",
			    v->v_data);
			enter_nm(Namech);
			return;
		    } else
			kn.kf_kt = (struct kern_target *)NULL;
		}
	/*
	 * Generate the /kern file name by reading the kern_target to which
	 * the kernfs_node points.
	 */
		if (kn.kf_kt
		&&  kread((KA_T)kn.kf_kt, (char *)&kt, sizeof(kt)) == 0
		&&  (ktnl = (int)kt.kt_namlen) > 0
		&&  kt.kt_name)
		{
		    if (ktnl > (sizeof(ktnm) - 1))
			ktnl = sizeof(ktnm) - 1;
		    if (kread((KA_T)kt.kt_name, ktnm, ktnl) == 0) {
			ktnm[ktnl] = 0;
			ktnl = strlen(ktnm);
			if (ktnl > (MAXPATHLEN - strlen(_PATH_KERNFS) - 2)) {
			    ktnl = MAXPATHLEN - strlen(_PATH_KERNFS) - 2;
			    ktnm[ktnl] = '\0';
			}
			(void) sprintf(Namech, "%s/%s", _PATH_KERNFS, ktnm);
		    }
		}
	    /*
	     * If this is the /kern root directory, its name, inode number and
	     * size are fixed; otherwise, safely stat() the file to get the
	     * inode number and size.
	     */
		if (v->v_type == VDIR && (v->v_flag & VROOT)) {
		    (void) strcpy(Namech, _PATH_KERNFS);
		    ksb.st_ino = 2;
		    ksb.st_size = DEV_BSIZE;
		    ksbs = 1;
		} else if (Namech[0] && statsafely(Namech, &ksb) == 0)
		    ksbs = 1;
		nty = KERNFSNODE;
		break;
#endif	/* defined(HASKERNFS) */

	case VT_MFS:
		if (!v->v_data
		||  kread((KA_T)v->v_data, (char *)&m, sizeof(m))) {
			(void) sprintf(Namech, "can't read mfsnode at: %x",
				v->v_data);
			enter_nm(Namech);
			return;
		}
		nty = MFSNODE;
		break;
	case VT_NFS:
		if (!v->v_data
		||  kread((KA_T)v->v_data, (char *)&n, sizeof(n))) {
			(void) sprintf(Namech, "can't read nfsnode at: %x",
				v->v_data);
			enter_nm(Namech);
			return;
		}
		nty = NFSNODE;
		break;

#if	defined(HASFDESCFS)
	case VT_FDESC:
		if (!v->v_data
		||  kread((KA_T)v->v_data, (char *)&f, sizeof(f))) {
			(void) sprintf(Namech, "can't read fdescnode at: %x",
				v->v_data);
			enter_nm(Namech);
			return;
		}
		nty = FDESCNODE;
		break;
#endif	/* defined(HASFDESCFS) */

#if	defined(HASPROCFS)
	case VT_PROCFS:
		if (!v->v_data
		||  kread((KA_T)v->v_data, (char *)&p, sizeof(p))) {
			(void) sprintf(Namech, "can't read pfsnode at: %x",
				v->v_data);
			enter_nm(Namech);
			return;
		}
		nty = PFSNODE;
		break;
#endif	/* defined(HASPROCFS) */

#if	defined(HASEXT2FS)
	case VT_EXT2FS:
#endif	/* defined(HASEXT2FS) */

	case VT_UFS:
		if (!v->v_data
		||  kread((KA_T)v->v_data, (char *)&i, sizeof(i))) {
			(void) sprintf(Namech, "can't read inode at: %x",
				v->v_data);
			enter_nm(Namech);
			return;
		}

#if	defined(HASEXT2FS)
		if (v->v_tag == VT_EXT2FS)
			nty = EXT2NODE;
		else
#endif	/* defined(HASEXT2FS) */

			nty = INODE;
		if ((lff = i.i_lockf)) {

		/*
		 * Determine the lock state.
		 */
		    lfp = lff;
		    do {
			if (kread((KA_T)lfp, (char *)&lf, sizeof(lf)))
			    break;
			lt = 0;
			switch(lf.lf_flags & (F_FLOCK|F_POSIX)) {
			case F_FLOCK:
			    if (Cfp && (struct file *)lf.lf_id == Cfp)
				lt = 1;
			    break;
			case F_POSIX:
			    if ((KA_T)lf.lf_id == Kpa)
				lt = 1;
			    break;
			}
			if (!lt)
			    continue;
			if (lf.lf_start == (off_t)0
			&&  lf.lf_end == 0xffffffffffffffffLL)
			    lt = 1;
			else
			    lt = 0;
			if (lf.lf_type == F_RDLCK)
			    Lf->lock = lt ? 'R' : 'r';
			else if (lf.lf_type == F_WRLCK)
			    Lf->lock = lt ? 'W' : 'w';
			else if (lf.lf_type == (F_RDLCK | F_WRLCK))
			    Lf->lock = 'u';
			break;
		    } while ((lfp = lf.lf_next) && lfp != lff);
		}
		break;
	default:
		if (v->v_type == VBAD || v->v_type == VNON)
			break;
		(void) sprintf(Namech, "unknown file system type: %d",
			v->v_tag);
		enter_nm(Namech);
		return;
	}
/*
 * Get device and type for printing.
 */
	type = v->v_type;
	switch (nty) {

#if	defined(HASFDESCFS)
	case FDESCNODE:
		if (f.fd_link
		&&  kread((KA_T)f.fd_link, Namech, sizeof(Namech) - 1) == 0)
		{
		    Namech[sizeof(Namech) - 1] = '\0';
		    break;
		}
		if (f.fd_type == Fctty) {
		    if (f_tty_s == 0)
			f_tty_s = lkup_dev_tty(&f_tty_dev, &f_tty_ino);
		    if (f_tty_s == 1) {
			dev = f_tty_dev;
			Lf->inode = f_tty_ino;
			devs = Lf->inp_ty = 1;
		    }
		}
		break;
#endif	/* defined(HASFDESCFS) */

#if	defined(HASEXT2FS)
	case EXT2NODE:

# if	defined(HASI_FFS)
		dev = i.i_dev;
# else	/* !defined(HASI_FFS) */
		dev = (type == VCHR || type == VBLK) ? i.i_rdev : i.i_dev;
		break;
# endif	/* defined(HASI_FFS) */
#endif	/* defined(HASEXT2FS) */

	case INODE:
		dev = (type == VCHR || type == VBLK) ?

#if	defined(HASI_FFS)
						       i.i_ffs_rdev
#else	/* !defined(HASI_FFS) */
						       i.i_rdev
#endif	/* defined(HASI_FFS) */

						     : i.i_dev;
		devs = 1;
		break;

#if	defined(HASKERNFS)
	case KERNFSNODE:
		if (vfs) {
			dev = (dev_t)vfs->fsid.val[0];
			devs = 1;
		}
		break;
#endif	/* defined(HASKERNFS) */


#if	defined(HAS9660FS)
	case CDFSNODE:
		if (iso_stat) {
			dev = iso_dev;
			devs = 1;
		}
		break;
#endif	/* defined(HAS9660FS) */

	case NFSNODE:
		dev = n.n_vattr.va_fsid;
		devs = 1;
	}
/*
 * Obtain the inode number.
 */
	switch (nty) {

#if	defined(HASEXT2FS)
	case EXT2NODE:
#endif	/* defined(HASEXT2FS) */

	case INODE:
		if (type != VBLK) {
			Lf->inode = (unsigned long)i.i_number;
			Lf->inp_ty = 1;
		}
		break;

#if	defined(HASKERNFS)
	case KERNFSNODE:
		if (ksbs) {
			Lf->inode = (unsigned long)ksb.st_ino;
			Lf->inp_ty = 1;
		}
		break;
#endif	/* defined(HASKERNFS) */

#if	defined(HAS9660FS)
	case CDFSNODE:
		if (iso_stat) {
			Lf->inode = iso_ino;
			Lf->inp_ty = 1;
		}
		break;
#endif	/* defined(HAS9660FS) */

	case NFSNODE:
		Lf->inode = (unsigned long)n.n_vattr.va_fileid;
		Lf->inp_ty = 1;
		break;

#if	defined(HASPROCFS)
	case PFSNODE:
		Lf->inode = (unsigned long)p.pfs_fileno;
		Lf->inp_ty = 1;
		break;
#endif	/* defined(HASPROCFS) */

	}

/*
 * Obtain the file size.
 */
	if (Foffset)
		Lf->off_def = 1;
	else {
		switch (Ntype) {

#if	defined(HAS9660FS)
		case N_CDFS:
			if (iso_stat) {
				Lf->sz = iso_sz;
				Lf->sz_def = 1;
			}
			break;
#endif	/* defined(HAS9660FS) */

		case N_FIFO:
			if (!Fsize)
				Lf->off_def = 1;
			break;

#if	defined(HASKERNFS)
		case N_KERN:
			if (ksbs) {
				Lf->sz = (unsigned long)ksb.st_size;
				Lf->sz_def = 1;
			}
			break;
#endif	/* defined(HASKERNFS) */

		case N_NFS:
			if (nty == NFSNODE) {
				Lf->sz = (unsigned long)n.n_vattr.va_size;
				Lf->sz_def = 1;
			}
			break;

#if	defined(HASPROCFS)
		case N_PROC:
			if (nty == PFSNODE) {
				switch (p.pfs_type) {
				case Proot:
				case Pproc:
					Lf->sz = (unsigned long)DEV_BSIZE;
					Lf->sz_def = 1;
					break;
				case Pcurproc:
					Lf->sz = (unsigned long)
						 sprintf(dev_ch, "%ld",
						(long)p.pfs_pid);
					Lf->sz_def = 1;
					break;
				case Pmem:
					(void) getmemsz(p.pfs_pid);
					break;
				case Pregs:
					Lf->sz = (unsigned long)
						  sizeof(struct reg);
					Lf->sz_def = 1;
					break;

# if	defined(FP_QSIZE)
				case Pfpregs:
					Lf->sz = (unsigned long)
						 sizeof(struct fpreg);
					Lf->sz_def = 1;
					break;
# endif	/* defined(FP_QSIZE) */

				}
			}
			break;
#endif	/* defined(HASPROCFS) */

		case N_REGLR:
			if (type == VREG || type == VDIR) {
				if (nty == INODE) {

#if	defined(HASI_FFS)
					Lf->sz = (unsigned long)i.i_ffs_size;
#else	/* !defined(HASI_FFS) */
					Lf->sz = (unsigned long)i.i_size;
#endif	/* defined(HASI_FFS) */

					Lf->sz_def = 1;
				} else if (nty == MFSNODE) {
					Lf->sz = (unsigned long)m.mfs_size;
					Lf->sz_def = 1;
				}

#if	defined(HASEXT2FS)
				else if (nty == EXT2NODE) {

# if	defined(HASI_E2FS)
					Lf->sz = (unsigned long)i.i_e2fs_size;
# else	/* !defined(HASI_E2FS) */
					Lf->sz = (unsigned long)i.i_size;
# endif	/* defined(HASI_E2FS) */

					Lf->sz_def = 1;
				}
#endif	/* defined(HASEXT2FS) */

			}
			else if ((type == VCHR || type == VBLK) && !Fsize)
				Lf->off_def = 1;
			break;
		}
	}
/*
 * Record an NFS file selection.
 */
	if (Ntype == N_NFS && Fnfs)
		Lf->sf |= SELNFS;
/*
 * Save the file system names.
 */
	if (vfs) {
		Lf->fsdir = vfs->dir;
		Lf->fsdev = vfs->fsname;
	}
/*
 * Format the vnode type, and possibly the device name.
 */
	switch (type) {

	case VNON:
		ty ="VNON";
		break;
	case VREG:
	case VDIR:
		ty = (type == VREG) ? "VREG" : "VDIR";
		if (devs) {
			Lf->dev_def = 1;
			Lf->dev = dev;
		}
		break;
	case VBLK:
		ty = "VBLK";
		if (devs) {
			Lf->dev_def = 1;
			Lf->dev = dev;
		}
		Ntype = N_BLK;
		break;
	case VCHR:
		ty = "VCHR";
		if (devs) {
			Lf->dev = dev;
			Lf->dev_def = 1;
		}
		Ntype = N_CHR;
		break;
	case VLNK:
		ty = "VLNK";
		break;

#if	defined(VSOCK)
	case VSOCK:
		ty = "SOCK";
		break;
#endif	/* defined(VSOCK) */

	case VBAD:
		ty = "VBAD";
		break;
	case VFIFO:
		if (devs) {
			Lf->dev = dev;
			Lf->dev_def = 1;
		}
		ty = "FIFO";
		break;
	default:
		if (type > 9999)
			(void) sprintf(Lf->type, "*%03d", type % 1000);
		else
			(void) sprintf(Lf->type, "%4d", type);
		(void) strcpy(Namech, "unknown type");
		ty = NULL;
	}
	if (ty)
		(void) strcpy(Lf->type, ty);
	Lf->ntype = Ntype;
/*
 * Handle some special cases:
 *
 * 	ioctl(fd, TIOCNOTTY) files;
 *	/kern files
 *	memory node files;
 *	/proc files.
 */

	if (type == VBAD)
		(void) strcpy(Namech, "(revoked)");
	else if (nty == MFSNODE) {
		Lf->dev_def = 0;
		(void) sprintf(Namech, "%#x", m.mfs_baseoff);
		(void) strcpy(dev_ch, "    memory");
		enter_dev_ch(dev_ch);
	}

#if	defined(HASPROCFS)
	else if (nty == PFSNODE) {
		Lf->dev_def = 0;
		ty = NULL;
		(void) sprintf(Namech, "/%s", PROCFSNAME);
		switch (p.pfs_type) {
		case Proot:
			ty = "PDIR";
			break;
		case Pcurproc:
			(void) strcpy(endnm(), "/curproc");
			ty = "PCUR";
			break;
		case Pproc:
			(void) sprintf(endnm(), "/%d", p.pfs_pid);
			ty = "PDIR";
			break;
		case Pfile:
			(void) sprintf(endnm(), "/%d/file", p.pfs_pid);
			ty = "PFIL";
			break;
		case Pmem:
			(void) sprintf(endnm(), "/%d/mem", p.pfs_pid);
			ty = "PMEM";
			break;
		case Pregs:
			(void) sprintf(endnm(), "/%d/regs", p.pfs_pid);
			ty = "PREG";
			break;
		case Pfpregs:
			(void) sprintf(endnm(), "/%d/fpregs", p.pfs_pid);
			ty = "PFPR";
			break;
		case Pctl:
			(void) sprintf(endnm(), "/%d/ctl", p.pfs_pid);
			ty = "PCTL";
			break;
		case Pstatus:
			(void) sprintf(endnm(), "/%d/status", p.pfs_pid);
			ty = "PSTA";
			break;
		case Pnote:
			(void) sprintf(endnm(), "/%d/note", p.pfs_pid);
			ty = "PNTF";
			break;
		case Pnotepg:
			(void) sprintf(endnm(), "/%d/notepg", p.pfs_pid);
			ty = "PGRP";
			break;
		}
		if (ty)
			(void) strcpy(Lf->type, ty);
		enter_nm(Namech);
	}
#endif	/* defined(HASPROCFS) */

#if	defined(HASBLKDEV)
/*
 * If this is a VBLK file and it's missing an inode number, try to
 * supply one.
 */
	if (Lf->inp_ty == 0 && type == VBLK && Lf->dev_def)
		find_bl_ino();
#endif	/* defined(HASBLKDEV) */

/*
 * If this is a VCHR file and it's missing an inode number, try to
 * supply one.
 */
	if (Lf->inp_ty == 0 && type == VCHR && Lf->dev_def)
		find_ch_ino();
/*
 * Test for specified file.
 */

#if	defined(HASPROCFS)
	if (Ntype == N_PROC) {
		if (Procsrch)
			Lf->sf |= SELNM;
		else if (nty == PFSNODE) {
			for (pfi = Procfsid; pfi; pfi = pfi->next) {

# if	defined(HASPINODEN)
				if (pfi->inode == p.pfs_fileno)
# else	/* !defined(HASPINODEN) */
				if (pfi->pid == p.pfs_pid)
# endif	/* defined(HASPINODEN) */

				{
					Lf->sf |= SELNM;
					break;
				}
			}
		}
	} else
#endif	/* defined(HASPROCFS) */

	{
		if (Namech[0]) {
			enter_nm(Namech);
			ns = 1;
		} else
			ns = 0;
		if (Sfile && is_file_named(NULL, type))
			Lf->sf |= SELNM;
		if (ns)
			Namech[0] = '\0';
	}
/*
 * Enter name characters.
 */
	if (Namech[0])
		enter_nm(Namech);
}


#if	defined(OPENBSDV)
/*
 * process_pipe() - process a file structure whose type is DTYPE_PIPE
 */

void
process_pipe(pa)
	caddr_t pa;			/* pipe structure address */
{
	char dev_ch[32];
	struct pipe p;

	if (!pa || kread((KA_T)pa, (char *)&p, sizeof(p))) {
		(void) sprintf(Namech,
			"can't read DTYPE_PIPE pipe struct: %#x", pa);
		enter_nm(Namech);
		return;
	}
	(void) strcpy(Lf->type, "PIPE");
	(void) sprintf(dev_ch, "%#x", pa);
	enter_dev_ch(dev_ch);
	if (Foffset)
		Lf->off_def = 1;
	else {
		Lf->sz = p.pipe_buffer.size;
		Lf->sz_def = 1;
	}
	if (p.pipe_peer)
		(void) sprintf(Namech, "->%#x", p.pipe_peer);
	else
		Namech[0] = '\0';
	if (p.pipe_buffer.cnt)
		(void) sprintf(endnm(), "%, cnt=%d", p.pipe_buffer.cnt);
	if (p.pipe_buffer.in)
		(void) sprintf(endnm(), "%, in=%d", p.pipe_buffer.in);
	if (p.pipe_buffer.out)
		(void) sprintf(endnm(), "%, out=%d", p.pipe_buffer.out);
/*
 * Enter name characters.
 */
	if (Namech[0])
		enter_nm(Namech);
}
#endif	/* defined(OPENBSDV) */
