/*
 * dsock.c - Linux socket processing functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dsock.c,v 1.6 97/10/14 07:53:55 abe Exp $";
#endif


#include "lsof.h"

#if	!defined(UNIX_PATH_MAX)
/*
 * If UNIX_PATH_MAX isn't defined, glibc2 is probably in use.  Derive
 * UNIX_PATH_MAX from the size of the sun_path member of its sockaddr_un
 * structure.
 */

struct sockaddr_un dsockaddr_un;	/* dummy */
#define	UNIX_PATH_MAX	sizeof(dsockaddr_un.sun_path)
#endif	/* !defined(UNIX_PATH_MAX) */



#if	defined(CONFIG_AX25) || defined(CONFIG_AX25_MODULE)
_PROTOTYPE(static int is_null_axaddr,(ax25_address *a));
_PROTOTYPE(static void print_axaddr,(char *d, ax25_address *a));
#endif	/* defined(CONFIG_AX25) || defined(CONFIG_AX25_MODULE) */


#if	defined(HASUNMINSOCK)
_PROTOTYPE(static void getunixnm,(struct sock *sp, char *bp, char *lp));


/*
 * getunixnm() - get Unix domain socket name from sock structure
 */

static void
getunixnm(sp, bp, lp)
	struct sock *sp;		/* sock structure pointer */
	char *bp;			/* receiving buffer pointer */
	char *lp;			/* receiving buffer limita pointer */
{
	char buf[UNIX_PATH_MAX+1];
	int len;
	struct inode si;

# if	HASUNMINSOCK==1
	struct unix_address ua;
	struct sockaddr_un us;
# endif	/* HASUNMINSOCK==1 */

# if	defined(HASDENTRY)
	struct dentry de;
# endif	/* defined(HASDENTRY) */

/*
 * Get inode number.
 */

# if	defined(HASDENTRY)
	if (sp->protinfo.af_unix.dentry
	&&  !kread((KA_T)sp->protinfo.af_unix.dentry, (char *)&de, sizeof(de))
	&&  de.d_inode
	&&  !kread((KA_T)de.d_inode, (char *)&si, sizeof(si)))
# else	/* !defined(HASDENTRY) */
	if (sp->protinfo.af_unix.inode
	&&  !kread((KA_T)sp->protinfo.af_unix.inode, (char *)&si, sizeof(si)))
# endif	/* defined(HASDENTRY) */

	{
	    Lf->inode = (unsigned long)si.i_ino;
	    Lf->inp_ty = 1;
	}

# if	HASUNMINSOCK==1
/*
 * If the socket has a unix_address member, read the sun_path member of its
 * sockaddr_un structure.
 */
	if (!sp->protinfo.af_unix.addr
	||  kread((KA_T)sp->protinfo.af_unix.addr, (char *)&ua, sizeof(ua)))
	    return;
	if (!(len = ua.len) || len > sizeof(struct sockaddr_un))
	    return;
	if ((len -= offsetof(struct sockaddr_un, sun_path)) <= 0)
	    return;
	if (kread((KA_T)((char *)(sp->protinfo.af_unix.addr) + sizeof(ua)
		  + offsetof(struct sockaddr_un, sun_path)), buf, len))
	    return;
# else	/* HASUNMINSOCK!=1 */
/*
 * If the socket doesn't have a unix_address member, read the name at
 * its name pointer.
 */
	if (!sp->protinfo.af_unix.name
	||  kread((KA_T)sp->protinfo.af_unix.name, buf, UNIX_PATH_MAX))
	    return;
# endif	/* HASUNMINSOCK==1 */

	buf[UNIX_PATH_MAX] = '\0';
	len = strlen(buf);
	if ((bp + len) > lp) {
	    if ((len = lp - bp) < 1)
		return;
	}
	(void) strncpy(bp, buf, len);
	*(bp + len) = '\0';
}
#endif	/* defined(HASUNMINSOCK) */


#if	defined(CONFIG_AX25) || defined(CONFIG_AX25_MODULE)
/*
 * is_null_axaddr() - is AX.25 address a NULL address?
 */

static int
is_null_axaddr(a)
	ax25_address *a;		/* address to test */
{
	int i;

	for (i = 0; i < 6; i++) {
	    if (((a->ax25_call[i] >> 1) && 0x7f) != ' ')
		return(0);
	}
	return(a->ax25_call[6] ? 1 : 0);
}


/*
 * print_axaddr() - print an AX.25 address
 */

static void
print_axaddr(d, a)
	char *d;			/* address destination */
	ax25_address *a;		/* address to print */
{
	char c;
	int i;

	for (i = 0; i < 6; i++) {
	    if ((c = ((a->ax25_call[i] >> 1) & 0x7f)) != ' ')
		*d++ = c;
	}
	(void) sprintf(d, "-%d", ((a->ax25_call[6] >> 1) & 0xf));
}
#endif	/* defined(CONFIG_AX25) || defined(CONFIG_AX25_MODULE) */


/*
 * print_tcptpi() - print TCP/TPI state
 */

void
print_tcptpi(nl)
	int nl;				/* 1 == '\n' required */
{
	char buf[128];
	char *cp = (char *)NULL;
	int ps = 0;
	int s;

	if ((Ftcptpi & TCPTPI_STATE) && Lf->lts.type == 0) {
	    switch ((s = Lf->lts.state.i)) {
	    case TCP_ESTABLISHED:
		cp = "ESTABLISHED";
		break;
	    case TCP_SYN_SENT:
		cp = "SYN_SENT";
		break;
	    case TCP_SYN_RECV:
		cp = "SYN_RECV";
		break;
	    case TCP_FIN_WAIT1:
		cp = "FIN_WAIT1";
		break;
	    case TCP_FIN_WAIT2:
		cp = "FIN_WAIT2";
		break;
	    case TCP_TIME_WAIT:
		cp = "TIME_WAIT";
		break;
	    case TCP_CLOSE:
		cp = "CLOSE";
		break;
	    case TCP_CLOSE_WAIT:
		cp = "CLOSE_WAIT";
		break;
	    case TCP_LAST_ACK:
		cp = "LAST_ACK";
		break;
	    case TCP_LISTEN:
		cp = "LISTEN";
		break;
	    case TCP_CLOSING:
		cp = "CLOSING";
		break;
	    case 0:
		cp = "CLOSED";
		break;
	    default:
		(void) sprintf(buf, "UNKNOWN_TCP_STATE_%d", s);
		cp = buf;
    	    }
	    if (cp) {
		if (Ffield)
		    (void) printf("%cST=%s%c", LSOF_FID_TCPTPI, cp, Terminator);
		else {
		    putchar('(');
		    (void) fputs(cp, stdout);
		}
		ps++;
	    }
	}

# if	defined(HASTCPTPIQ)
	if (Ftcptpi & TCPTPI_QUEUES) {
	    if (Lf->lts.rqs) {
		if (Ffield)
		    putchar(LSOF_FID_TCPTPI);
		else {
		    if (ps)
			putchar(' ');
		    else
			putchar('(');
		}
		(void) printf("QR=%lu", Lf->lts.rq);
		if (Ffield)
		    putchar(Terminator);
		ps++;
	    }
	    if (Lf->lts.sqs) {
		if (Ffield)
		    putchar(LSOF_FID_TCPTPI);
		else {
		    if (ps)
			putchar(' ');
		    else
			putchar('(');
		}
		(void) printf("QS=%lu", Lf->lts.sq);
		if (Ffield)
		    putchar(Terminator);
		ps++;
	    }
	}
# endif	/* defined(HASTCPTPIQ) */

# if	defined(HASTCPTPIW)
	if (Ftcptpi & TCPTPI_WINDOWS) {
	    if (Lf->lts.rws) {
		if (Ffield)
		    putchar(LSOF_FID_TCPTPI);
		else {
		    if (ps)
			putchar(' ');
		    else
			putchar('(');
		}
		(void) printf("WR=%lu", Lf->lts.rw);
		if (Ffield)
		    putchar(Terminator);
		ps++;
	    }
	    if (Lf->lts.wws) {
		if (Ffield)
		    putchar(LSOF_FID_TCPTPI);
		else {
		    if (ps)
			putchar(' ');
		    else
			putchar('(');
		}
		(void) printf("WW=%lu", Lf->lts.ww);
		if (Ffield)
		    putchar(Terminator);
		ps++;
	    }
	}
# endif	/* defined(HASTCPTPIW) */

	if (!Ffield && ps)
	    putchar(')');
	if (nl)
	    putchar('\n');
}


/*
 * process_isock() -- process inode-based socket
 */

void
process_isock(i)
	struct inode *i;		/* socket's inode pointer */
{
	int af, len;
	char *cp;
	char dev_ch[32];
	struct in_addr *fa = (struct in_addr *)NULL;
	int fp, lp;
	struct in_addr *la = (struct in_addr *)NULL;
	unsigned long rq, sq;
	struct socket *sp;
	struct sock sk;
	static unsigned long uxops, uxops_dg, uxops_st;

#if	defined(CONFIG_AX25) || defined(CONFIG_AX25_MODULE)
	struct ax25_cb ax;
	struct device axd;
#endif	/* defined(CONFIG_AX25) || defined(CONFIG_AX25_MODULE) */

#if	LINUXV<1175
	struct socket s;
#endif	/* LINUXV<1175 */

#if	defined(HASUNMINSOCK)
	struct proto_ops pops;
#else	/* !defined(HASUNMINSOCK) */
	static int ft = 1;
	static unsigned long ipops, ipops_dg, ipops_st;
	struct unix_proto_data ud;
#endif	/* defined(HASUNMINSOCK) */


#if	!defined(HASUNMINSOCK)
/*
 * Do first-time only operations.
 */
	if (ft) {
	    if (get_Nl_value("ipops", Drive_Nl, &ipops) < 0)
		ipops = (unsigned long)0;
	    if (get_Nl_value("ipops_dg", Drive_Nl, &ipops_dg) < 0)
		ipops_dg = (unsigned long)0;
	    if (get_Nl_value("ipops_st", Drive_Nl, &ipops_st) < 0)
		ipops_st = (unsigned long)0;
	    if (get_Nl_value("uxops", Drive_Nl, &uxops) < 0)
		uxops = (unsigned long)0;
	    if (get_Nl_value("uxops_dg", Drive_Nl, &uxops_dg) < 0)
		uxops_dg = (unsigned long)0;
	    if (get_Nl_value("uxops_st", Drive_Nl, &uxops_st) < 0)
		uxops_st = (unsigned long)0;
	    ft = 0;
	}
#endif	/* !defined(HASUNMINSOCK) */

/*
 * Identify file type.
 */
	(void) strcpy(Lf->type, "sock");
	Lf->inp_ty = 2;

#if	LINUXV<1175
/*
 * Read the socket.
 */
	if (!i->i_socket
	||  kread((KA_T)i->i_socket, (char *)&s, sizeof(s))) {
	    (void) sprintf(Namech, "can't read socket: %#x", i->i_socket);
	    enter_nm(Namech);
	    return;
	}
	sp = &s;
#else	/* LINUXV>=1175 */
/*
 * Make sure there is a socket structure inside the inode.
 */
	if (!i->i_sock) {
	    (void) strcpy(Namech, "no socket in inode");
	    enter_nm(Namech);
	    return;
	}
	sp = &i->u.socket_i;
#endif	/* LINUXV<1175 */

/*
 * Read the sock and proto_ops structures, if appropriate.
 * Determine the address family.
 */

#if	defined(HASUNMINSOCK)
	if (!sp->SOCKET_SOCK
	||  kread((KA_T)sp->SOCKET_SOCK, (char *)&sk, sizeof(sk))) {
	    (void) sprintf(Namech, "can't read sock: %#x", sp->SOCKET_SOCK);
	    enter_nm(Namech);
	    return;
	}
	if (!sp->ops
	||  kread((KA_T)sp->ops, (char *)&pops, sizeof(pops))) {
	    (void) sprintf(Namech, "can't read proto_ops: %#x", sp->ops);
	    enter_nm(Namech);
	    return;
	}
	af = pops.family;
#else	/* !defined(HASUNMINSOCK) */
	if (ipops && ipops == (unsigned long)sp->ops)
	    af = AF_INET;
	else if (ipops_dg && ipops_dg == (unsigned long)sp->ops)
	    af = AF_INET;
	else if (ipops_st && ipops_st == (unsigned long)sp->ops)
	    af = AF_INET;
	else if (uxops && uxops == (unsigned long)sp->ops)
	    af = AF_UNIX;
	else if (uxops_dg && uxops_dg == (unsigned long)sp->ops)
	    af = AF_UNIX;
	else if (uxops_st && uxops_st == (unsigned long)sp->ops)
	    af = AF_UNIX;
	else {
	    (void) sprintf(Namech, "unsupported address family: %d", af);
	    enter_nm(Namech);
	    return;
	}
#endif	/* defined(HASUNMINSOCK) */

/*
 * Process by address family.
 */
	switch (af) {

#if	defined(CONFIG_AX25) || defined(CONFIG_AX25_MODULE)
	case AF_AX25:
	    (void) strcpy(Lf->type, "ax25");
	    Lf->off_def = 1;
	/*
	 * Read socket and AX.25 control block.
	 */

#if	!defined(HASUNMINSOCK)
	    if (!sp->SOCKET_SOCK
	    ||  kread((KA_T)sp->SOCKET_SOCK, (char *)&sk, sizeof(sk))) {
		(void) sprintf(Namech, "can't read sock: %#x", sp->SOCKET_SOCK);
		enter_nm(Namech);
		return;
	    }
#endif	/* !defined(HASUNMINSOCK) */

	    if (!sk.protinfo.ax25
	    ||  kread((KA_T)sk.protinfo.ax25, (char *)&ax, sizeof(ax))) {
		(void) sprintf(Namech, "can't read AX.25 control block: %#x",
		    sk.protinfo.ax25);
		enter_nm(Namech);
		return;
	    }
	/*
	 * Read AX.25 device structure.  Enter its base address as device
	 * characters and its IRQ as the inode number.
	 */
	    if (ax.ax25_dev
	    &&  kread((KA_T)ax.ax25_dev, (char *)&axd, sizeof(axd)) == 0) {
		(void) sprintf(dev_ch, "0x%08x", axd.base_addr);
		enter_dev_ch(dev_ch);
		Lf->inode = (unsigned long)axd.irq;
		Lf->inp_ty = 1;
	    }
	/*
	 * Enter AX.25 addresses.
	 */
	    if (!is_null_axaddr(&ax.source_addr)) {
		print_axaddr(Namech, &ax.source_addr);
		if (!is_null_axaddr(&ax.dest_addr)) {
		    (void) strcpy(endnm(), " -> ");
		    print_axaddr(endnm(), &ax.dest_addr);
		}
	    }
	    break;
#endif	/* defined(CONFIG_AX25) || defined(CONFIG_AX25_MODULE) */

	case AF_INET:

	/*
	 * Process Internet socket.
	 */
	    if (Fnet)
		Lf->sf |= SELNET;
	    (void) strcpy(Lf->type, "inet");

#if	!defined(HASUNMINSOCK)
	    if (!sp->SOCKET_SOCK
	    ||  kread((KA_T)sp->SOCKET_SOCK, (char *)&sk, sizeof(sk))) {
		(void) sprintf(Namech, "can't read sock: %#x", sp->SOCKET_SOCK);
		enter_nm(Namech);
		return;
	    }
#endif	/* !defined(HASUNMINSOCK) */

	    (void) sprintf(dev_ch, "0x%08x", sp->SOCKET_SOCK);
	    enter_dev_ch(dev_ch);
	    if (sk.protocol == IPPROTO_TCP) {

#if	defined(HAS_ATOMIC_T)
	 	rq = (unsigned long)(sk.tp_pinfo.af_tcp.rcv_nxt -
				     sk.copied_seq);
		sq = (unsigned long)(sk.write_seq - sk.tp_pinfo.af_tcp.snd_una);
#else	/* !defined(HAS_ATOMIC_T) */
		rq = (unsigned long)(sk.acked_seq - sk.copied_seq);
		sq = (unsigned long)(sk.write_seq - sk.rcv_ack_seq);
#endif	/* defined(HAS_ATOMIC_T) */


#if	defined(HASTCPTPIQ)
		Lf->lts.rq = rq;
		Lf->lts.sq = sq;
		Lf->lts.rqs = Lf->lts.sqs = 1;
#endif	/* defined(HASTCPTPIQ) */

		if (Fsize) {
		    if (Lf->access == 'r')
			Lf->sz = rq;
		    else if (Lf->access == 'w')
			Lf->sz = sq;
		    else
			Lf->sz = rq + sq;
		    Lf->sz_def = 1;
		} else {
		    Lf->off = (unsigned long)sk.SOCK_SENT_SEQ;
		    Lf->off_def = 1;
		}
	    } else {
		if (Fsize && (sk.SOCK_WMEM_ALLOC || sk.SOCK_RMEM_ALLOC)) {
		    Lf->sz = (unsigned long)(sk.SOCK_WMEM_ALLOC +
					     sk.SOCK_RMEM_ALLOC);
		    Lf->sz_def = 1;
		} else {
		    Lf->off = (unsigned long)sk.SOCK_SENT_SEQ;
		    Lf->off_def = 1;
		}
	    }
	    printiproto((int)sk.protocol);
	    la = (struct in_addr *)&sk.saddr;
	    lp = (int)ntohs(sk.dummy_th.source);
	    if (sk.daddr != (unsigned long)NULL || sk.dummy_th.dest != 0) {
		fa = (struct in_addr *)&sk.daddr;
		fp = (int)ntohs(sk.dummy_th.dest);
	    }
	    (void) ent_inaddr(la, lp, fa, fp);
	    if (sk.protocol == IPPROTO_TCP) {
		Lf->lts.type = 0;
		Lf->lts.state.i = (int)sk.state;
	    }
	    break;
	case AF_UNIX:

	/*
	 * Process Unix socket.
	 */
	    if (Funix)
		Lf->sf |= SELUNX;
	    (void) strcpy(Lf->type, "unix");

#if	LINUXV>=2000
# if	defined(HASUNMINSOCK) && HASUNMINSOCK==1
	    /*
 	     * The inode number will be supplied by getunixnm().
	     */
# else	/* !defined(HASUNMINSOCK) || HASUNMINSOCK!=1 */
	    if (i->i_ino) {
		Lf->inode = (unsigned long)i->i_ino;
		Lf->inp_ty = 1;
	    }
# endif	/* defined(HASUNMINSOCK) && HASUNMINSOCK==1 */
#endif	/* LINUXV>=2000 */

#if	defined(HASUNMINSOCK)
	    (void) getunixnm(&sk, Namech, &Namech[MAXPATHLEN - 1]);
	    if (sk.pair
	    &&  kread((KA_T)sk.pair, (char *)&sk, sizeof(sk)) == 0) {
		cp = endnm();
		if ((&Namech[MAXPATHLEN - 1] - cp) >= 2) {
		    (void) strcpy(cp, "->");
		    cp += 2;
		}
		(void) getunixnm(&sk, cp, &Namech[MAXPATHLEN - 1]);
	    }
#else	/* !defined(HASUNMINSOCK) */
	    if (!sp->SOCKET_SOCK
	    ||  kread((KA_T)sp->SOCKET_SOCK, (char *)&ud, sizeof(ud))) {
		(void) sprintf(Namech, "can't read unix data: %#x",
		    sp->SOCKET_SOCK);
		enter_nm(Namech);
		return;
	    }
	    if ((len = ud.sockaddr_len)) {
		if (len >= MAXPATHLEN)
		    len = MAXPATHLEN - 1;
		(void) strncpy(Namech, (char *)&ud.sockaddr_un.sun_path, len);
		Namech[len] = '\0';
	    }
	    if (ud.peerupd
	    &&  kread((KA_T)ud.peerupd, (char *)&ud, sizeof(ud)) == 0) {
		cp = endnm();
		len = &Namech[MAXPATHLEN - 1] - cp;
		if (ud.sockaddr_len < len)
		    len = ud.sockaddr_len;
		if (len) {
		    (void) strncpy(cp, (char *)&ud.sockaddr_un.sun_path, len);
		    *(cp + len) = '\0';
		}
	    }
#endif	/* defined(HASUNMINSOCK) */

	    (void) sprintf(dev_ch, "0x%08x", sp->SOCKET_SOCK);
	    enter_dev_ch(dev_ch);
	    Lf->off_def = 1;
	    if (Namech[0] && Sfile) {
		if (is_file_named(Namech, i->i_mode & S_IFMT))
		    Lf->sf |= SELNM;
	    }
	    break;

#if	defined(HASUNMINSOCK)
	default:
	    (void) sprintf(Namech, "unsupported address family: %d", af);
	    enter_nm(Namech);
	    return;
#endif	/* defined(HASUNMINSOCK) */

	}
	if (Namech[0])
	    enter_nm(Namech);
}
