/*
 * dnode.c - Motorola V/88 node functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dnode.c,v 1.14 95/09/19 10:50:46 abe Exp $";
#endif


#include "lsof.h"


#if	V88<40403
_PROTOTYPE(static struct l_dev * finddev,(dev_t *dev));
#else	/* V88>=40403 */
_PROTOTYPE(static struct l_dev * finddev,(dev_t *dev, int stream));
#endif	/* V88<40403 */

#if	V88>=40401
_PROTOTYPE(static int examine_stream,(struct stdata *vs, struct queue *q, char *mn));
_PROTOTYPE(static char isvlocked,(struct vnode *va));


static struct protos {
	char *module;			/* stream module name */
	char *proto;			/* TCP/IP protocol name */
} Protos[] = {

# if	V88<40403
	{ "ipu",	"IP"	},
	{ "icmp",	"ICMP"	},
	{ "tcpu",	"TCP"	},
	{ "udpu",	"UDP"	}
# else	/* V88>=40403 */
	{ "tip",	"TCP"	},
	{ "uip",	"UDP"	}
# endif	/* V88<40403 */

};
#define	NPROTOS	(sizeof(Protos)/sizeof(struct protos))


/*
 * examine_stream() - examine stream
 */

static int
examine_stream(vs, q, mn)
	struct stdata *vs;
	struct queue *q;
	char *mn;
{
	struct module_info mi;
	struct queue *qp;
	struct qinit qi;
	struct stdata sd;
/*
 * Read stream's head.
 */
	if (vs == NULL || readstdata(vs, &sd)) {
		(void) sprintf(Namech, "can't read stream head from %x", vs);
		enter_nm(Namech);
		return(1);
	}
	if (sd.sd_wrq == NULL) {
		enter_nm("no stream write queue");
		return(1);
	}
/*
 * Examine the write queue.
 */
	for (qp = sd.sd_wrq, *mn = '\0'; qp; qp = q->q_next)
	{

	/*
	 * Read stream queue entry.
	 */
		if (kread((KA_T)qp, (char *)q, sizeof(struct queue))) {
			(void) sprintf(Namech,
				"can't read stream queue from %x", qp);
			enter_nm(Namech);
			return(1);
		}
	/*
	 * Read queue's information structure.
	 */
		if (q->q_qinfo == NULL || readstqinit(q->q_qinfo, &qi)) {
			(void) sprintf(Namech,
				"can't read qinit for %x from %x",
				qp, q->q_qinfo);
			enter_nm(Namech);
			return(1);
		}
	/*
	 * Read module information structure.
	 */
		if (qi.qi_minfo == NULL || readstmin(qi.qi_minfo, &mi)) {
			(void) sprintf(Namech,
				"can't read module info for %x from %x",
				q->q_qinfo, qi.qi_minfo);
			enter_nm(Namech);
			return(1);
		}
	/*
	 * Read module name.
	 */
		if (mi.mi_idname == NULL ||
		    kread((KA_T)mi.mi_idname, mn, STRNML-1))
		{
			(void) sprintf(Namech,
				"can't read module name for %x from %x",
				qi.qi_minfo, mi.mi_idname);
			enter_nm(Namech);
			return(1);
		}
		*(mn + STRNML - 1) = '\0';
	}
	return(0);
}
#endif	/* V88>=40401 */


/*
 * finddev() - find device by number
 */

static struct l_dev *

#if	V88<40403
finddev(dev)
#else	/* V88>=40403 */
finddev(dev, stream)
#endif	/* V88<40403 */

	dev_t *dev;			/* device */

#if	V88>=40403
	int stream;			/* 1 if stream node */
#endif	/* V88>=40403 */

{

#if	V88>=40403
	struct clone *c;
#endif	/* V88>=40403 */

	struct l_dev *dp;
/*
 * Search device table for match.
 */
	if ((dp = lkupdev(dev, 0)) != (struct l_dev *)NULL)
		return(dp);

#if	V88>=40403
/*
 * Search for clone.
 */
	if (stream && Clone) {
		for (c = Clone; c; c = c->next) {
			if (get_major(*dev) == get_minor(Devtp[c->dx].rdev))
				return(&Devtp[c->dx]);
		}
	}
#endif	/* V88>=40403 */

	return(NULL);
}


#if	V88>=40401
/*
 * isvlocked() - is a vnode locked
 */

static char
isvlocked(va)
	struct vnode *va;		/* local vnode address */
{
	struct filock f;
	off_t l;

	if (va->v_filocks == NULL)
		return(' ');
	if (kread((KA_T)va->v_filocks, (char *)&f, sizeof(f)))
		return(' ');
	if (f.set.l_whence == 0 && f.set.l_start == 0 && f.set.l_len == 0)
		l = 1;
	else
		l = 0;
	switch (f.set.l_type & (F_RDLCK | F_WRLCK)) {

	case F_RDLCK:
		return((l) ? 'R' : 'r');
	case F_WRLCK:
		return((l) ? 'W' : 'w');
	case (F_RDLCK + F_WRLCK):
		return('u');
	}
	return(' ');
}
#endif	/* V88>=40401 */


#if	V88<40401
/*
 * process_node() - process node
 */

void
process_node(na)
	caddr_t na;			/* inode kernel space address */
{
	char dev_ch[32];
	struct l_dev *dp;
	struct filock fl;
	struct inode i;
	short ity, j, k;
	struct mounts *lm;
	struct mount m;
	struct module_info mi;
	int prdev;
	struct rnode r;
	struct qinit qi;
	struct stdata sd;
	struct queue sh;
	char *tn;
	int type;
/*
 * Read the inode.
 */
	if ( ! na) {
		enter_nm("no inode address");
		return;
	}
	if (readinode(na, &i)) {
		enter_nm(Namech);
		return;
	}
/*
 * Identify the node type.
 */
	ity = i.i_fstyp;
	type = i.i_ftype & IFMT;
	if (ity < 1 || ity >= Nfstyp || Fsinfo[ity].fs_name == NULL) {
		(void) sprintf(Namech, "unknown fstyp (%d) in inode", ity);
		enter_nm(Namech);
		return;
	}
	if (strcasecmp(Fsinfo[ity].fs_name, "NFS") == 0) {

	/*
	 * Get information on NFS file.
	 */
		Ntype = N_NFS;
		if (Fnfs)
			Lf->sf |= SELNFS;
		if (kread((KA_T)i.i_mntdev, (char *)&m, sizeof(m))) {
			(void) sprintf(Namech,
			    "can't read NFS mount structure (%#x)",
			    i.i_mntdev);
			enter_nm(Namech);
			return;
		}
		if (i.i_fsptr == NULL || readrnode((caddr_t)i.i_fsptr, &r)) {
			(void) sprintf(Namech, "can't read rnode (%#x)",
			    i.i_fsptr);
			enter_nm(Namech);
			return;
		}
	} else if (i.i_un2.i_sck_un) {

	/*
	 * Process a socket.
	 */
		process_socket((KA_T)i.i_un2.i_sck_un);
		return;
	} else {

	/*
	 * Determine the node type from the inode file type.
	 */
		switch (type) {

		case IFBLK:
			Ntype = N_BLK;
			break;
		case IFCHR:
			Ntype = N_CHR;
			break;
		case IFIFO:
			Ntype = N_FIFO;
			break;
		case IFMPB:
		case IFMPC:
			Ntype = N_MPC;
			break;
		}
	}
/*
 * Obtain lock information.
 */
	if (i.i_filocks) {
		if ((kread((KA_T)i.i_filocks, (char *)&fl, sizeof(fl)))) {
			(void) sprintf(Namech, "can't read filock (%#x)",
				i.i_filocks);
			enter_nm(Namech);
			return;
		} else {
			if (fl.set.l_type == F_RDLCK)
				Lf->lock = 'R';
			else if (fl.set.l_type == F_WRLCK)
				Lf->lock = 'W';
			else if (fl.set.l_type == (F_RDLCK | F_WRLCK))
				Lf->lock = 'u';
		}
	}
/*
 * Determine the device.
 */
	switch (Ntype) {
	case N_BLK:
	case N_FIFO:
	case N_REGLR:
		Lf->dev = i.i_dev;
		Lf->dev_def = 1;
		break;
	case N_CHR:
		Lf->dev = i.i_rdev;
		Lf->dev_def = 1;
		if (i.i_sptr) {

		/*
		 * Follow the stream structure from its stdata
		 * pointer to the module information structure.
		 * Then read the first STRNML characters of
		 * the module name.
		 */
			(void) strcpy(Namech, "STR:");
			j = strlen(Namech);
			if ((dp = finddev((dev_t *)&i.i_rdev)) != NULL) {
			    if ((j + (k = strlen(dp->name))) < MAXPATHLEN - 1) {
				(void) strcpy(&Namech[j], dp->name);
				j += k;
				if ((j + 2) < MAXPATHLEN - 1) {
				    (void) strcpy(&Namech[j], "->");
				    j += 2;
				}
			    }
			}
			if ((j + STRNML) >= MAXPATHLEN - 1
			||  readstdata((struct stdata *)i.i_sptr, &sd)
			||  readsthead(sd.sd_wrq, &sh)
			||  readstqinit(sh.q_qinfo, &qi)
			||  readstmin(qi.qi_minfo, &mi)
			||  readstidnm(mi.mi_idname, &Namech[j], STRNML - 1)) {
				(void) sprintf(Namech,
					"no module name for stream at %#x",
					i.i_sptr);
			} else {
				Namech[j + STRNML - 1] = '\0';
				Lf->is_stream = 1;
			}
		} else {
			if (strcasecmp(Fsinfo[ity].fs_name, "COM") == 0) {
				Ntype = N_COM;
				Lf->is_com = 1;
			} else
				Lf->is_chr_dev  = 1;
		}
		break;
	case N_NFS:
		Lf->dev = (dev_t)((m.m_dev & 0xffff) | 0xff000000);
		Lf->dev_def = 1;
		break;
	}
/*
 * Determine the inode number.
 */
	switch (Ntype) {
	case N_NFS:
		Lf->inode = (unsigned long)r.r_nfsattr.na_nodeid;
		Lf->inp_ty = 1;
		break;
	case N_BLK:
	case N_CHR:
	case N_COM:
	case N_FIFO:
	case N_REGLR:
		Lf->inode = (unsigned long)i.i_number;
		Lf->inp_ty = 1;
		break;
	}
/*
 * Determine the file size.
 */
	if (Foffset)
		Lf->off_def = 1;
	else {
		switch (Ntype) {
		case N_BLK:
			Lf->sz = i.i_size;
			Lf->sz_def = 1;
			break;
		case N_CHR:
		case N_COM:
			if (!Fsize)
				Lf->off_def = 1;
			break;
		case N_FIFO:
			if (!Fsize)
				Lf->off_def = 1;
			break;
		case N_NFS:
			Lf->sz = r.r_nfsattr.na_size;
			Lf->sz_def = 1;
			break;
		case N_REGLR:
			if (type == IFREG || type == IFDIR)
				Lf->sz = i.i_size;
				Lf->sz_def = 1;
			break;
		}
	}
/*
 * Format the type name.
 */
	switch (type) {
	case IFDIR:
		tn = "DIR";
		break;
	case IFBLK:
		tn = "BLK";
		break;
	case IFCHR:
		tn = "CHR";
		break;
	case IFREG:
		tn = "REG";
		break;
	case IFMPC:
		tn = "MPC";
		break;
	case IFMPB:
		tn = "MPB";
		break;
	case IFIFO:
		tn = "FIFO";
		break;
	case IFLNK:
		tn = "LINK";
		break;
	default:
		if (type > 9999)
			(void) sprintf(Lf->type, "*%03d", type % 1000);
		else
			(void) sprintf(Lf->type, "%4d", type);
		(void) strcpy(Namech, "unknown inode type");
		tn = NULL;
	}
	if (tn)
		(void) strcpy(Lf->type, tn);
/*
 * Save the file system names.
 */
	switch (Ntype) {
	case N_BLK:
	case N_FIFO:
	case N_NFS:
	case N_REGLR:
		if (Lf->dev_def) {
			for (lm = Mtab; lm; lm = lm->next) {
				if (Lf->dev == lm->dev) {
					Lf->fsdir = lm->dir;
					Lf->fsdev = lm->fsname;
					break;
				}
			}
		}
		break;
	}
/*
 * If this is a IFCHR file and it's missing an inode number, try to
 * supply one.
 */
	if (Lf->inp_ty == 0 && type == IFCHR && Lf->dev_def)
		find_ch_ino();
/*
 * Test for specified file.
 */
	if (Sfile && is_file_named(NULL, type))
		Lf->sf |= SELNM;
/*
 * Enter name characters.
 */
	if (Namech[0])
		enter_nm(Namech);
}
#endif	/* V88<40401 */


#if	V88>=40401
/*
 * process_node() - process node
 */

void
process_node(na)
	caddr_t na;			/* vnode kernel space address */
{
	dev_t dev;
	char dev_ch[32];
	struct l_dev *dp;
	struct fifonode f;
	int fx;
	struct inode i;
	int j, k;
	struct vfs kv;
	struct vfs *la = NULL;
	struct module_info mi;
	char mn[STRNML];
	int ni = 0;
	struct pipenode p;
	int px;
	struct queue q;
	struct rnode r;
	struct snode rs;
	struct vnode rv, v;
	struct snode s;
	char *ty;
	enum vtype type;
	struct l_vfs *vfs;

# if	defined(HASPROCFS)
	struct as as;
	struct procfsid *pfi;
	struct pid pids;
	struct prnode prn;
	struct proc proc;
# endif	/* defined(HASPROCFS) */

/*
 * Read the vnode.
 */
	if ( ! na) {
		enter_nm("no vnode address");
		return;
	}
	if (readvnode((caddr_t)na, &v)) {
		enter_nm(Namech);
		return;
	}

# if	defined(HASNCACHE)
	Lf->na = (unsigned long)na;
# endif	/* defined(HASNCACHE) */

/*
 * Determine the vnode type.
 */
	if (v.v_vfsp
	&&  kread((KA_T)v.v_vfsp, (char *)&kv, sizeof(kv)) == 0) {

	/*
	* Check the file system type.
	*/
		fx = kv.vfs_fstype;
		if (fx > 0 && fx <= Fsinfomax) {
			if (strcmp(Fsinfo[fx-1], "fifofs") == 0)
				Ntype = N_FIFO;
			else if (strcmp(Fsinfo[fx-1], "nfs") == 0)
				Ntype = N_NFS;
			else if (strcmp(Fsinfo[fx-1], "pipefs") == 0)
				Ntype = N_PIPE;

# if	defined(HASPROCFS)
			else if (strcmp(Fsinfo[fx-1], HASPROCFS) == 0)
				Ntype = N_PROC;
# endif	/* defined(HASPROCFS) */

		}
	}
	if (Ntype == N_REGLR) {
		if (v.v_type == VFIFO)
			Ntype = N_FIFO;
		else if (v.v_stream) {
			Ntype = N_STREAM;
			Lf->is_stream = 1;
		}
	}
/*
 * Determine the lock state.
 */
	Lf->lock = isvlocked(&v);
/*
 * Establish the local virtual file system structure.
 */
	if (v.v_vfsp == NULL)
		vfs = NULL;
	else {
	    if ((vfs = readvfs(v.v_vfsp, &kv)) == NULL) {
		(void) sprintf(Namech, "bad vfs for %#x at %#x", na, v.v_vfsp);
		enter_nm(Namech);
		return;
	    }
	}
/*
 * Read the fifonode, inode, pipenode, rnode, or snode.
 */
	switch (Ntype) {
	case N_FIFO:
		if (!v.v_data || readfifonode(v.v_data, &f)) {
			(void) sprintf(Namech,
				"vnode at %#x: can't read fifonode (%#x)",
				na, v.v_data);
			enter_nm(Namech);
			return;
		}
		if (f.fn_realvp) {
			if (readvnode((caddr_t)f.fn_realvp, &rv)) {
			    (void) sprintf(Namech,
				"fifonode at %#x: can't read real vnode (%#x)",
				v.v_data, f.fn_realvp);
			    enter_nm(Namech);
			    return;
			}
			if (!rv.v_data
			||  readinode((struct inode *)rv.v_data, &i)) {
				(void) sprintf(Namech,
				    "fifonode at %#x: can't read inode (%#x)",
				    v.v_data, rv.v_data);
				enter_nm(Namech);
				return;
			}
		}
		break;
	case N_NFS:
		if (!v.v_data || readrnode((caddr_t)v.v_data, &r)) {
			(void) sprintf(Namech,
				"vnode at %#x: can't read rnode (%#x)",
				na, v.v_data);
			enter_nm(Namech);
			return;
		}
		break;
	case N_PIPE:
		if (!v.v_data || readpipenode((caddr_t)v.v_data, &p)) {
			(void) sprintf(Namech,
				"vnode at %#x: can't read pipenode (%#x)",
				na, v.v_data);
			enter_nm(Namech);
			return;
		}
		break;

# if	defined(HASPROCFS)
	
	case N_PROC:

	/*
	 * Read the proc file system node and associated information.
	 */
		ni = 1;
		if (v.v_data == NULL
		||  kread((KA_T)v.v_data, (char *)&prn, sizeof(prn))) {
			(void) sprintf(Namech,
				"vnode at %#x: can't read prnode (%#x)",
				na, v.v_data);
			enter_nm(Namech);
			return;
		}
		if (prn.pr_proc == NULL) {
			if (v.v_type == VDIR) {
				(void) sprintf(Namech, "/%s", HASPROCFS);
				enter_nm(Namech);
				Lf->inode = PR_ROOTINO;
				Lf->inp_ty = 1;
			} else {
				(void) sprintf(Namech, "/%s/???", HASPROCFS);
				enter_nm(Namech);
				Lf->inp_ty = 0;
			}
			break;
		}
		if (kread((KA_T)prn.pr_proc, (char *)&proc, sizeof(proc))) {
			(void) sprintf(Namech,
				"prnode at %#x: can't read proc (%#x)",
				v.v_data, prn.pr_proc);
			enter_nm(Namech);
			return;
		}
		if (proc.p_pidp == NULL
		||  kread((KA_T)proc.p_pidp, (char *)&pids, sizeof(pids))) {
			(void) sprintf(Namech,
				"proc struct at %#x: can't read pid (%#x)",
				prn.pr_proc, proc.p_pidp);
			enter_nm(Namech);
			return;
		}
		if (proc.p_as
		&&  kread((KA_T)proc.p_as, (char *)&as, sizeof(as)) == 0) {
			Lf->sz = as.a_size;
			Lf->sz_def = 1;
		}
		(void) sprintf(Namech, "/%s/%0*d", HASPROCFS, PNSIZ,
			 pids.pid_id);
		Lf->inode = (unsigned long)ptoi(pids.pid_id);
		Lf->inp_ty = 1;
		enter_nm(Namech);
		break;
# endif	/* defined(HASPROCFS) */

	case N_STREAM:
		(void) strcpy(Lf->iproto, "STR");
		if (v.v_stream) {

		/*
		 * Get the queue pointer and module name at the
		 * end of the stream.  The module name indentifies
		 * socket streams.
		 */
			if (examine_stream(v.v_stream, &q, mn))
				return;
			for (px = 0; px < NPROTOS; px++) {
				if (strcmp(mn, Protos[px].module) == 0) {
					process_socket(Protos[px].proto, &q);
					return;
				}
			}
			if (mn[0]) {
				(void) strcpy(Namech, "STR:");
				j = strlen(Namech);

#if	V88<40403
				if ((dp = finddev((dev_t *)&v.v_rdev)) != NULL)
#else	/* V88>=40403 */
				if ((dp = finddev((dev_t *)&v.v_rdev,1))!=NULL)
#endif	/* V88<40403 */

				{
				    k = strlen(dp->name);
				    if ((j + k) < MAXPATHLEN - 1) {
					(void) strcpy(&Namech[j], dp->name);
					j += k;
				    }
				    if ((j + 2) < MAXPATHLEN - 1) {
					(void) strcpy(&Namech[j], "->");
					j +=2;
				    }
				}
				if ((j + strlen(mn)) < MAXPATHLEN - 1)
				    (void) strcpy(&Namech[j], mn);
			} else
				(void) strcpy(Namech, "(no module name)");
		}
		break;
	case N_REGLR:
	default:

	/*
	 * Follow a VCHR vnode to its snode, then to its real vnode, finally
	 * to its inode.
	 */
		if (v.v_type == VCHR) {
			if (!v.v_data || readsnode(v.v_data, &s)) {
				(void) sprintf(Namech,
					"vnode at %#x: can't read snode(%#x)",
					na, v.v_data);
				enter_nm(Namech);
				return;
			}
			if (s.s_realvp) {
			    if (readvnode((caddr_t)s.s_realvp, &rv)) {
				(void) sprintf(Namech,
				    "snode at %#x: can't read real vnode (%#x)",
				    v.v_data, s.s_realvp);
				enter_nm(Namech);
				return;
			    }
			    if (!rv.v_data
			    ||  readinode((struct inode *)rv.v_data, &i)) {
				(void) sprintf(Namech,
					"snode at %#x: can't read inode (%#x)",
					v.v_data, rv.v_data);
				enter_nm(Namech);
				return;
			    }
			}
		/*
		 * If there's no real vnode, look for a common vnode and a
		 * common snode.
		 */
			else if (s.s_commonvp) {
			    if (readvnode((caddr_t)s.s_commonvp, &rv)) {
				(void) sprintf(Namech,
				    "snode at %#x: can't read real vnode (%#x)",
				    v.v_data, s.s_commonvp);
				enter_nm(Namech);
				return;
			    }
			    if (!rv.v_data || readsnode(rv.v_data, &rs)) {
				(void) sprintf(Namech,
					"vnode at %#x: can't read snode(%#x)",
					s.s_commonvp, rv.v_data);
				enter_nm(Namech);
				return;
			    }
			    ni = 1;
			} else
				ni = 1;
			break;
		}
		if (!v.v_data || readinode((struct inode *)v.v_data, &i)) {
			(void) sprintf(Namech,
				"vnode at %#x: can't read inode (%#x)",
				na, v.v_data);
			enter_nm(Namech);
			return;
		}
	}
/*
 * Get device and type for printing.
 */
	switch (Ntype) {

# if	defined(HASPROCFS)
	case N_PROC:
# endif	/* defined(HASPROCFS) */

	case N_NFS:
		dev = r.r_attr.va_fsid;
		break;
	case N_PIPE:
		dev = v.v_rdev;
		break;
	default:
		dev = (v.v_type == VCHR || ni) ? v.v_rdev : i.i_dev;
	}
	type = v.v_type;
	if (vfs && vfs->dir == NULL)
		(void) completevfs(vfs, &dev);
/*
 * Obtain the inode number.
 */
	switch (Ntype) {
	case N_NFS:
		Lf->inode = (unsigned long)r.r_attr.va_nodeid;
		Lf->inp_ty = 1;
		break;

#if     defined(HASPROCFS)
	case N_PROC:

	/*
	 * The proc file system inode number is defined when the
	 * prnode is read.
	 */
		break;
#endif	/* defined(HASPROCFS) */

	case N_FIFO:
		if ( ! f.fn_realvp) {
			Lf->inode = (unsigned long)f.fn_ino;
			Lf->inp_ty = 1;
			(void) sprintf(dev_ch, "0x%08x", v.v_data);
			enter_dev_ch(dev_ch);
			if (f.fn_flag & ISPIPE)
				(void) strcpy(Namech, "PIPE");
			if (f.fn_mate)
				(void) sprintf(endnm(), "->0x%08x", f.fn_mate);
			break;
		}
		/* fall through */
	case N_REGLR:
		if (!ni) {
			Lf->inode = (unsigned long)i.i_number;
			Lf->inp_ty = 1;
		}
		break;
	}
/*
 * Obtain the file size.
 */
	if (Foffset)
		Lf->off_def = 1;
	else {
		switch (Ntype) {
		case N_FIFO:
		case N_STREAM:
			if (!Fsize)
				Lf->off_def = 1;
			break;
		case N_NFS:
			Lf->sz = r.r_attr.va_size;
			Lf->sz_def = 1;
			break;
		case N_PIPE:
			Lf->sz = p.fp_count;
			Lf->sz_def = 1;
			break;

#if     defined(HASPROCFS)
		case N_PROC:

		/*
		 * The proc file system size is defined when the
		 * prnode is read.
		 */
			break;
#endif	/* defined(HSPROCFS) */

		case N_REGLR:
			if (type == VREG || type == VDIR) {
				if (!ni) {
					Lf->sz = i.i_size;
					Lf->sz_def = 1;
				}
			} else if (type == VCHR && !Fsize)
				Lf->off_def = 1;
			break;
		}
	}
/*
 * Record an NFS file selection.
 */
	if (Ntype == N_NFS && Fnfs)
		Lf->sf |= SELNFS;
/*
 * Save the file system names.
 */
	if (vfs) {
		Lf->fsdir = vfs->dir;
		Lf->fsdev = vfs->fsname;
	}
/*
 * Format the vnode type, and possibly the device name.
 */
	switch (type) {
	case VNON:
		ty ="VNON";
		break;
	case VREG:
	case VDIR:
		ty = (type == VREG) ? "VREG" : "VDIR";
		Lf->dev_def = 1;
		Lf->dev = dev;
		break;
	case VBLK:
		ty = "VBLK";
		break;
	case VCHR:
		Lf->dev = dev;
		Lf->dev_def = 1;
		ty = "VCHR";
		if (Lf->is_stream == 0 && Lf->is_com == 0)
			Lf->is_chr_dev = 1;
		break;
	case VLNK:
		ty = "VLNK";
		break;

# if	defined(VSOCK)
	case VSOCK:
		ty = "SOCK";
		break;
# endif	/* defined(VSOCK) */

	case VBAD:
		ty = "VBAD";
		break;
	case VFIFO:
		if (!Lf->dev_ch || Lf->dev_ch[0] == '\0') {
			Lf->dev = dev;
			Lf->dev_def = 1;
		}
		ty = "FIFO";
		break;
	default:
		if (type > 9999)
			(void) sprintf(Lf->type, "*%03d", type % 1000);
		else
			(void) sprintf(Lf->type, "%4d", type);
		(void) strcpy(Namech, "unknown type");
		ty = NULL;
	}
	if (ty)
		(void) strcpy(Lf->type, ty);
/*
 * Indicate a pipe.
 */
	if (Ntype == N_PIPE) {
		if (Namech[0])
			(void) strcpy(endnm(), " (PIPE)");
		else if (vfs) {
			(void) sprintf(Namech, "%s %s%s%s (PIPE)",
				(vfs->dir)    ? vfs->dir    : "",
				(vfs->fsname) ? "("         : "",
				(vfs->fsname) ? vfs->fsname : "",
				(vfs->fsname) ? ")"         : "");
		} else
			(void) strcpy(Namech, "(PIPE)");
	}
/*
 * If this is a VCHR file and it's missing an inode number, try to
 * supply one.
 */
	if (Lf->inp_ty == 0 && type == VCHR && Lf->dev_def)
		find_ch_ino();
/*
 * Test for specified file.
 */

#if     defined(HASPROCFS)
	if (Ntype == N_PROC) {
		if (Procsrch)
			Lf->sf |= SELNM;
		else {
			for (pfi = Procfsid; pfi; pfi = pfi->next) {
				if (pfi->pid == pids.pid_id) {
					Lf->sf |= SELNM;
					break;
				}
			}
		}
	} else
#endif  /* defined(HASPROCFS) */

	{
		if (Sfile && is_file_named(NULL, type))
			 Lf->sf |= SELNM;
	}
/*
 * Enter name characters.
 */
	if (Namech[0])
		enter_nm(Namech);
}
#endif	/* V88>=40401 */
