/*
 * dproc.c - Linux process access functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dproc.c,v 1.11 95/09/19 10:15:32 abe Exp $";
#endif

#include "lsof.h"


_PROTOTYPE(static void get_kernel_access,(void));
_PROTOTYPE(static int hash_ksym,(char *nm));
_PROTOTYPE(static char *hasparmhash,(char *nm));
_PROTOTYPE(static int nlist,(char *fnm, struct nlist *nl));
_PROTOTYPE(static void process_mapmem,(struct inode *x, struct vm_area_struct *m));


/*
 * Local static values
 */

static int Coff = 0;			/* kernel loder format is COFF */
static caddr_t *Nc = NULL;		/* node cache */
static MALLOC_S Nn = 0;			/* number of Nc[] entries allocated */


/*
 * Local definitions
 */

#define	KSHASHSZ	256		/* kernel symbol hash table size */


/*
 * gather_proc_info() -- gather process information
 */

void
gather_proc_info()
{
	struct file *f;
	struct file fs;
	struct task_struct *t, ts;
	static struct task_struct **tt = NULL;
	int i, nt;
	MALLOC_S len;
	short pss, sf;
	uid_t uid;
/*
 * Allocate space for task structure pointers and read them.
 */
	len = (MALLOC_S)(NR_TASKS * sizeof(struct task_struct *));
	if (tt == (struct task_struct **)NULL) {
		if ((tt = (struct task_struct **)malloc(len)) == NULL) {
			(void) fprintf(stderr,
				"%s: no space for %d task structure pointers\n",
				Pn, NR_TASKS);
			exit(1);
		}
	}
	if (kread((KA_T)Nl[X_TASK].n_value, (char *)tt, (int)len)) {
		(void) fprintf(stderr,
			"%s: can't read task structure pointers from %#lx\n",
			Pn, Nl[X_TASK].n_value);
		exit(1);
	}
/*
 * Examine task structures and their associated information.
 */
	for (nt = 0, t = &ts; nt < NR_TASKS; nt++) {
		if (tt[nt] == (struct task_struct *)NULL
		||   kread((KA_T)tt[nt], (char *)t, sizeof(ts)))
			continue;
		if (t->state == TASK_ZOMBIE)
			continue;
		if (is_proc_excl((pid_t)t->pid, (pid_t)t->pgrp,
		    (UID_ARG)t->uid, &pss, &sf))
			continue;
	/*
	 * Allocate a local process structure.
	 */
		if (is_cmd_excl(t->comm, &pss, &sf))
			continue;
		alloc_lproc(t->pid, t->pgrp, (UID_ARG)t->uid, t->comm,
			(int)pss, (int)sf);
		Plf = NULL;
	/*
	 * Save current working directory information.
	 */
		if (t->PWD) {
			alloc_lfile(CWD, -1);
			process_inode(t->PWD, (struct file_operations *)NULL);
			if (Lf->sf)
				link_lfile();
		}
	/*
	 * Save root directory information.
	 */
		if (t->ROOT) {
			alloc_lfile(RTD, -1);
			process_inode(t->ROOT, (struct file_operations *)NULL);
			if (Lf->sf)
				link_lfile();
		}
	/*
	 * Save information on the executable file and mapped memory.
	 */

#if	LINUXV<1147
		if (t->executable || t->mmap)
			process_mapmem(t->executable, t->mmap);
#else	/* LINUXV>=1147 */
		if (t->mm->mmap)
			process_mapmem(NULL, t->mm->mmap);
#endif	/* LINUXV<1147 */

	/*
	 * Save information on file descriptors.
	 */

		for (f = &fs, i = 0; i < NR_OPEN; i++) {

#if	LINUXV<1147
			if (t->filp[i] == (struct file *)NULL
			||  kread((KA_T)t->filp[i], (char *)f, sizeof(fs)))
#else	/* LINUXV>=1147 */
			if (t->files->fd[i] == (struct file *)NULL
			||  kread((KA_T)t->files->fd[i], (char *)f, sizeof(fs)))
#endif	/* LINUXV<1147 */

				continue;
			alloc_lfile(NULL, i);
			process_file(f);
			if (Lf->sf)
				link_lfile();
		}
	/*
	 * Examine results.
	 */
		if (examine_lproc())
			return;
	}
}


/*
 * get_kernel_access() - get access to kernel memory
 */

static void
get_kernel_access()
{

#if	defined(WILLDROPGID)
/*
 * If kernel memory isn't coming from KMEM, drop setgid permission
 * before attempting to open the (Memory) file.
 */
	if (Memory)
		(void) dropgid();
#else	/* !defined(WILLDROPGID) */
/*
 * See if the non-KMEM memory file is readable.
 */
	if (Memory && !is_readable(Memory, 1))
		exit(1);
#endif	/* defined(WILLDROPGID) */

/*
 * Open kernel memory access.
 */
	if ((Kmem = open(Memory ? Memory : KMEM, O_RDONLY, 0)) < 0) {
		(void) fprintf(stderr, "%s: can't open %s: %s\n", Pn,
			Memory ? Memory : KMEM, strerror(errno));
		exit(1);
	}

#if	defined(WILLDROPGID)
/*
 * Drop setgid permission, if necessary.
 */
	if (!Memory)
		(void) dropgid();
#else	/* !defined(WILLDROPGID) */
/*
 * See if the name list file is readable.
 */
	if (Nmlst && !is_readable(Nmlst, 1))
		exit(1);
#endif	/* defined(WILLDROPGID) */

/*
 * Access kernel symbols.
 */
	if (nlist(Nmlst ? Nmlst : N_UNIX, Nl) < 0) {
		(void) fprintf(stderr,
			"%s: can't read kernel name list from %s\n",
			Pn, Nmlst ? Nmlst : N_UNIX);
		exit(1);
	}
/*
 * Make sure important kernel symbols are defined.
 */
	if ( ! Nl[X_TASK].n_type) {
		(void) fprintf(stderr,
			"%s: no %stask kernel definition\n",
			Pn, Coff ? "_" : "");
		exit(1);
	}
}


/*
 * hash_ksym() - hash kernel symbol by name
 */

static int
hash_ksym(nm)
	char *nm;
{
	int i, j;

	for (i = j = 0; *nm; nm++) {
		i ^= (int)*nm << j;
		j ^= 1;
	}
	return(i % KSHASHSZ);
}


/*
 * hasparmhash() - does symbol name have parameter hash
 */

static char *
hasparmhash(nm)
	char *nm;			/* pointer to symbol name */
{
	char *cp, *rv;
	int n;

	if ((cp = strrchr(nm, '_')) == NULL || *(cp + 1) != 'R')
		return(NULL);
	rv = cp;
	for (cp += 2, n = 0; n < 8; cp++, n++) {
		if ((*cp >= '0' && *cp <= '9')
		||  (*cp >= 'a' && *cp <= 'f')
		||  (*cp >= 'A' && *cp <= 'F'))
			continue;
		return(NULL);
	}
	return((*cp == '\0') ? rv : NULL);
}


/*
 * initialize() - perform all initialization
 */

void
initialize()
{
	size_t sz;
	unsigned long hi, st;

	get_kernel_access();
	iuidcache(NR_TASKS);
}


/*
 * kread() - read from kernel memory
 *
 * return: 0 = success
 *	   1 = read failed (perhaps some data was read, but not all)
 *	  -1 = seek to kernel memory address failed
 */

int
kread(addr, buf, len)
	KA_T addr;			/* kernel memory address */
	char *buf;			/* buffer to receive data */
	READLEN_T len;			/* length to read */
{
	int br;

	if (lseek(Kmem, (off_t)addr, SEEK_SET) == (off_t)-1)
		return(-1);
	br = read(Kmem, buf, len);
	return((br == (int)len) ? 0 : 1);
}


/*
 * nlist() - get name list
 */

static int
nlist(fnm, nl)
	char *fnm;			/* name of file to search */
	struct nlist *nl;		/* names to locate */
{
	unsigned long addr;
	char buf[128];
	char *cp, *eol, *nm;
	int elf = 0;
	int i, j, mm, nf, nk, nn;
	struct kernel_sym *ks = NULL;
	FILE *nfs;
	struct ksym {
		char *name;
		unsigned long addr;
		struct ksym *next;
	} **kh, *kp, *kpn;
	struct nlist *np;

	if ((nfs = fopen(fnm, "r")) == NULL)
		return(-1);
/*
 * Read the kernel symbols via get_kernel_syms().
 */
	if ((nk = get_kernel_syms(NULL)) < 0) {
	    if (!Fwarn)
		(void) fprintf(stderr,
		    "%s: can't get kernel symbol table size: %s\n",
		    Pn, strerror(errno));
	} else {
	    i = nk * sizeof(struct kernel_sym);
	    if ((ks = (struct kernel_sym *)malloc((MALLOC_S)i)) == NULL) {
		(void) fprintf(stderr, "%s: no space for kernel symbols\n", Pn);
		exit(1);
	    }
	}
	if (nk > 0) {
	    if (get_kernel_syms(ks) < 0) {
		(void) fprintf(stderr, "%s: get_kernel_syms: %s\n",
		    Pn, strerror(errno));
		exit(1);
	    }
	/*
	 * Allocate hash table space.
	 */
	    if ((kh = (struct ksym **)calloc((MALLOC_S)sizeof(struct ksym),
		      KSHASHSZ))
	    == NULL) {
		(void) fprintf(stderr,
		    "%s: no space for %d byte kernel symbol hash table\n",
		    Pn, (KSHASHSZ * sizeof(struct ksym)));
		exit(1);
	    }
	/*
	 * Scan the kernel symbol table and store their addresses, hashed
	 * by their names for later comparison to the same values in
	 * /[z]System.map.
	 *
	 * Look for the symbol "_system_utsname" or "system_utsname" to
	 * determine the loader format of the kernel: it's * COFF if the
	 * symbol has a leading `_'; ELF if it does not.
	 */
	    for (i = 0; i < nk; i++, ks++) {
		if (ks->name[0] == '#')
		    continue;
		if ((kp = (struct ksym *)malloc((MALLOC_S)sizeof(struct ksym)))
		== NULL) {
		    (void) fprintf(stderr,
			"%s: no space for kernel symbol structure: %s\n",
			Pn, nm);
		    exit(1);
		}
		if ((nm = hasparmhash(ks->name)) != NULL)
		    *nm = '\0';
		if ((kp->name = (char *)malloc((MALLOC_S)(strlen(ks->name)+1)))
		== NULL) {
		    (void) fprintf(stderr,
			"%s: no space for kernel symbol name: %s\n", Pn, nm);
		    exit(1);
		}
		j = hash_ksym(ks->name);
		(void) strcpy(kp->name, ks->name);
		if (strcmp(ks->name, "_system_utsname") == 0)
		    Coff = 1;
		else if (strcmp(ks->name, "system_utsname") == 0)
		    elf = 1;
		kp->addr = ks->value;
		kp->next = kh[j];
		kh[j] = kp;
	    }
	    (void) free((FREE_P *)ks);
	}
/*
 * Complain if we don't know the kernel binary's format, COFF or ELF, or if
 * we have conflicting evidence.  In either case, set the format to COFF.
 */
	if ((!Coff && !elf) || (Coff && elf)) {
	    if (!Fwarn) {
		(void) fprintf(stderr, "%s: WARNING: uncertain kernel", Pn);
		(void) fprintf(stderr, "loader format; assuming COFF.\n");
	    }
	    Coff = 1;
	    elf = 0;
	}
/*
 * Read the lines of the name list file.  Look for the symbols defined
 * in Nl[].  Look for any symbols also defined from the get_kernel_syms()
 * call and complain if their addresses don't match.  Count the symbols
 * in Nl[] and quit when addresses have been located for all of them.
 */
	for (nn = 0, np = nl; np->n_name; nn++, np++)
		;
	mm = nf = 0;
	while (nf < nn && fgets(buf, sizeof(buf), nfs) != NULL) {
	    if ((eol = strchr(buf, '\n')) == NULL)
		continue;
	    if ((cp = strchr(buf, ' ')) == NULL)
		continue;
	    nm = cp + 3;
	    *eol = '\0';
	    addr = strtol(buf, &cp, 16);
	    if (kh) {

	    /*
	     * Look for name in the kernel symbol hash table; if it's there,
	     * check the address; complain volubly about the first mismatch;
	     * count mismatches.
	     */
		for (kp = kh[hash_ksym(nm)]; kp; kp = kp->next) {
		    if (strcmp(kp->name, nm) == 0)
			break;
		}
		if (kp && addr != kp->addr) {
		    if (++mm == 1 && !Fwarn) {
			(void) fprintf(stderr,
			    "%s: kernel symbol address mismatch: %s\n",
			    Pn, nm);
			(void) fprintf(stderr,
			    "      get_kernel_syms() value is %#x;", kp->addr);
			(void) fprintf(stderr, " %s value is %#x.\n",
			    fnm, addr);
		    }
		}
	    }
	/*
	 * Search for symbol in Nl[].
	 */
	    for (np = nl; np->n_name; np++) {
		cp = Coff ? np->n_name : (np->n_name + 1);
		if (strcmp(nm, cp) != 0)
		    continue;
		if (np->n_type) {

		/*
		 * Warn about a duplicate.
		 */
		    if (!Fwarn)
			(void) fprintf(stderr,
			    "%s: WARNING: %s: ambiguous symbol: %s\n",
			    Pn, fnm, cp);
		    continue;
		}
	    /*
	     * Save address; set type to 1, signifying an address has been
	     * loccated.  Count the number of addresses located.
	     */
		np->n_value = addr;
		np->n_type = 1;
		nf++;
		break;
	    }
	}
	(void) fclose(nfs);
/*
 * Complete the kernel symbol mismatch message if there were additional
 * mismtaches.  Quit if there were any mismatches.
 */
	if (mm) {
	    if (!Fwarn) {
		if (mm > 1) {
		    (void) fprintf(stderr,
			"      There %s %d additional mismatch%s.\n",
			(mm == 2) ? "was" : "were",
			mm - 1,
			(mm == 2) ? "" : "es");
		 }
		 (void) fprintf(stderr,
			"      %s and the booted kernel may not be a",
			fnm);
		 (void) fprintf(stderr, " matched set.\n");
	    }
	    exit(1);
	}
/*
 * Free the hashed kernel name space.
 */
	for (i = 0; i < KSHASHSZ; i++) {
	    if ((kp = kh[i]) == NULL)
		continue;
	    while (kp) {
		kpn = kp->next;
		if (kp->name)
		    (void) free((FREE_P *)kp->name);
		(void) free((MALLOC_S *)kp);
		kp = kpn;
	    }
	}
	(void) free((FREE_P *)kh);
	return(1);
}


/*
 * process_mapmem() - process mapped memory
 */

static void
process_mapmem(x, m)
	struct inode *x;
	struct vm_area_struct *m;
{
	int f;
	int i, j, lm;
	MALLOC_S ma;
	struct inode *ia;
	char *ty;
	struct vm_area_struct v;

	for (f = 1, i = lm = 0; m; lm++) {

	/*
	 * Avoid infinite loop.
	 */
		if (lm > 1000) {
			if (!Fwarn)
				(void) fprintf(stderr,
					"%s: too many memory maps, PID %d\n",
					Pn, Lp->pid);
				return;
		}
		if (f) {
		
		/*
		 * The first time through the loop, use the executable inode
		 * address.
		 */
			ty = "txt";
			f = 0;
			if ((ia = x) == (struct inode *)NULL)
				continue;
		} else {

		/*
		 * After the first time through the loop, use the inode
		 * addresses in the memory map.
		 */
			if (kread((KA_T)m, (char *)&v, sizeof(v)))
				break;
			m = v.vm_next;
			if ((ia = v.vm_inode) == (struct inode *)NULL)
				continue;
			ty = "mem";
		}
	/*
	 * Skip duplicate inode addresses.
	 */
		for (j = 0; j < i; j++) {
			if (Nc[j] == (caddr_t)ia)
				break;
		}
		if (j < i)
			continue;
	/*
	 * Cache the inode address for duplicate checking.
	 */
		if (i >= Nn) {
			Nn += 10;
			ma = (MALLOC_S)(Nn * sizeof(caddr_t));
			if (Nc)
				Nc = (caddr_t *)realloc(Nc, ma);
			else
				Nc = (caddr_t *)malloc(ma);
			if (!Nc) {
			    (void) fprintf(stderr,
				"%s: no space for memmap pointers, PID %d\n",
				Pn, Lp->pid);
			    exit(1);
			}
		}
		Nc[i++] = (caddr_t)ia;
	/* * Save the inode information.
	 */
		alloc_lfile(ty, -1);
		process_inode(ia, (struct file_operations *)NULL);
		if (Lf->sf)
			link_lfile();
	}
}

