/*
 * dproc.c - FreeBSD process access functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dproc.c,v 1.10 95/07/20 12:37:29 abe Exp $";
#endif

#include "lsof.h"

#if     defined(HASNCACHE)
#include <sys/uio.h>
#include <sys/namei.h>
#endif  /* defined(HASNCACHE) */


_PROTOTYPE(static void enter_vn_text,(KA_T va, int *n));
_PROTOTYPE(static void get_kernel_access,(void));

#if     defined(HASNCACHE)
_PROTOTYPE(static void ncache_load,(void));
#endif  /* defined(HASNCACHE) */

_PROTOTYPE(static void process_text,(KA_T vm));


/*
 * Local static values
 */

static MALLOC_S Nv = 0;			/* allocated Vp[] entries */
static KA_T *Vp = NULL;			/* vnode address cache */


/*
 * enter_vn_text() - enter a vnode text reference
 */

static void
enter_vn_text(va, n)
	KA_T va;			/* vnode address */
	int *n;				/* Vp[] entries in use */
{
	int i;
/*
 * Ignore the request if the vnode has already been entered.
 */
	for (i = 0; i < *n; i++) {
		if (va == Vp[i])
			return;
	}
/*
 * Save the text file information.
 */
	alloc_lfile(" txt", -1);
	process_node((caddr_t)va);
	if (Lf->sf)
		link_lfile();
	if (i >= Nv) {

	/*
	 * Allocate space for remembering the vnode.
	 */
		if (Vp == NULL) {
			if ((Vp = (KA_T *)malloc((MALLOC_S)
				   (sizeof(struct vnode *) * 10)))
			== NULL) {
				(void) fprintf(stderr,
					"%s: no txt ptr space, PID %d\n",
					Pn, Lp->pid);
				exit(1);
			}
			Nv = 10;
		} else {
			Nv += 10;
			if ((Vp = (KA_T *)realloc((MALLOC_P *)Vp,
				   (MALLOC_S)(Nv * sizeof(struct vnode *))))
			== NULL) {
				(void) fprintf(stderr,
					"%s: no more txt ptr space, PID %d\n",
					Pn, Lp->pid);
				exit(1);
			}
		}
	}
/*
 * Remember the vnode.
 */
	Vp[*n] = va;
	(*n)++;
}


/*
 * gather_proc_info() -- gather process information
 */

void
gather_proc_info()
{
	struct filedesc fd;
	int i, nf;
	static int nofb = 0;
	static struct file **ofb = NULL;
	int pgrp, pid;
	short pss, sf;
	int px;
	uid_t uid;

#if	_FREEBSDV<200
	struct proc *p;
	struct pcred pc;
	struct pgrp pg;
#else	/* _FREEBSDV>=200 */
	struct kinfo_proc *p;
#endif	/* _FREEBSDV<200 */

/*
 * Read the process table.
 */

#if	_FREEBSDV<200
	if ((Np = kvm_getprocs(KINFO_PROC_ALL, 0)) < 0)
#else	/* _FREEBSDV>=200 */
	if ((P = kvm_getprocs(Kd, KERN_PROC_ALL, 0, &Np)) == NULL)
#endif	/* _FREEBSDV<200 */

	{
		(void) fprintf(stderr, "%s: can't read process table: %s\n",
			Pn,

#if	_FREEBSDV<200
			kvm_geterr()
#else	/* _FREEBSDV>=200 */
			kvm_geterr(Kd)
#endif	/* _FREEBSDV<200 */

		);
		exit(1);
	}
/*
 * Initialize the UID cache.
 */
	(void) iuidcache(Np);

#if	defined(HASNCACHE)
/*
 * Read kernel name cache.
 */
	ncache_load();
#endif  /* defined(HASNCACHE) */

/*
 * Examine proc structures and their associated information.
 */

#if	_FREEBSDV<200
	for (px = 0; px < Np; px++)
#else	/* _FREEBSDV>=200 */
	for (p = P, px = 0; px < Np; p++, px++)
#endif	/* _FREEBSDV<200 */

	{

#if	_FREEBSDV<200
	/*
	 * Read process information, process group structure (if
	 * necessary),  and User ID (if necessary).
	 */
		if ((p = kvm_nextproc()) == NULL)
			continue;
		pg.pg_id = 0;
		if (Fpgrp && p->p_pgrp) {
			if (kread((KA_T)p->p_pgrp, (char *)&pg, sizeof(pg)))
				continue;
		}
		pgrp = pg.pg_id;
		if (p->p_cred == NULL
		||  kread((KA_T)p->p_cred, (char *)&pc, sizeof(pc)))
			continue;
		uid = pc.p_ruid;
#else	/* _FREEBSDV>=200 */
		pgrp = p->P_PGRP;
		uid = p->kp_eproc.e_ucred.cr_uid;
#endif	/* _FREEBSDV<200 */

	/*
	 * See if process is excluded.
	 *
	 * Read file structure pointers.
	 */
		if (is_proc_excl(p->P_PID, pgrp, (UID_ARG)uid, &pss, &sf))
			continue;
		if (p->P_FD == NULL
		||  kread((KA_T)p->P_FD, (char *)&fd, sizeof(fd)))
			continue;
	/*
	 * Allocate a local process structure.
	 */
		if (is_cmd_excl(p->P_COMM, &pss, &sf))
			continue;
		alloc_lproc(p->P_PID, pgrp, (UID_ARG)uid, p->P_COMM,
			(int)pss, (int)sf);
		Plf = NULL;
	/*
	 * Save current working directory information.
	 */
		if (fd.fd_cdir) {
			alloc_lfile(CWD, -1);
			process_node((caddr_t)fd.fd_cdir);
			if (Lf->sf)
				link_lfile();
		}
	/*
	 * Save root directory information.
	 */
		if (fd.fd_rdir != NULL) {
			alloc_lfile(RTD, -1);
			process_node((caddr_t)fd.fd_rdir);
			if (Lf->sf)
				link_lfile();
		}
	/*
	 * Save information on the text file.
	 */
		if (p->P_VMSPACE)
			process_text((KA_T)p->P_VMSPACE);
	/*
	 * Read open file structure pointers.
	 */
		if (fd.fd_ofiles == NULL || (nf = fd.fd_nfiles) <= 0)
			continue;
		if (ofb == NULL) {
			nofb = nf;
			if ((ofb = (struct file **)malloc((MALLOC_S)
				    (nofb * sizeof(struct file *))))
			== NULL) {
				(void) fprintf(stderr,
					"%s: PID %d, no file * space\n",
					Pn, p->P_PID);
				exit(1);
			}
		} else if (nf > nofb) {
			nofb = fd.fd_nfiles;
			if ((ofb = (struct file **)realloc((MALLOC_P *)ofb,
				    (nofb * sizeof(struct file *))))
			== NULL) {
				(void) fprintf(stderr,
					"%s: PID %d, no realloc file * space\n",
					Pn, p->P_PID);
					exit(1);
			}
		}
		if (kread((KA_T)fd.fd_ofiles, (char *)ofb,
		    nf * sizeof(struct file *)))
			continue;
	/*
	 * Save information on file descriptors.
	 */
		for (i = 0; i < nf; i++) {
			if (ofb[i]) {
				alloc_lfile(NULL, i);
				process_file(ofb[i]);
				if (Lf->sf)
					link_lfile();
			}
		}
	/*
	 * Examine results.
	 */
		if (examine_lproc())
			return;
	}
}


/*
 * get_kernel_access() - get access to kernel memory
 */

static void
get_kernel_access()
{
	char *nunix = N_UNIX;
	int nunix_stat = 0;

#if	_FREEBSDV>=200
/*
 * Set name list file path to booted kernel.
 */
	if (!Nmlst) {
		const char *bf;
		char *bfc;
		MALLOC_S len;

		if ((bf = getbootfile()) != NULL) {
			len = (MALLOC_S)strlen(bf);
			if ((bfc = (char *)malloc(len + 1)) != NULL) {
				(void) strcpy(bfc, bf);
				nunix = bfc;
				nunix_stat = 1;
			}
		}
	}
#endif	/* _FREEBSDV>=200 */

/*
 * Check authorization to access alternate core and name list files.
 */
	if ((Memory != NULL && !is_readable(Memory))
	||  (Nmlst != NULL && !is_readable(Nmlst)))
		exit(1);
/*
 * Access the name list file.
 */
	if (nlist(Nmlst ? Nmlst : nunix, Nl) < 0) {
		(void) fprintf(stderr, "%s: can't read namelist from %s\n",
			Pn, Nmlst ? Nmlst : nunix);
                exit(1);
	}
/*
 * Open kernel memory access.
 */

#if	_FREEBSDV<200
	if (kvm_openfiles(Nmlst, Memory, NULL) == -1)
#else	/* _FREEBSDV>=200 */
	if ((Kd = kvm_open(Nmlst, Memory, NULL, O_RDONLY, NULL)) == NULL)
#endif	/* _FREEBSDV<200 */

	{
		(void) fprintf(stderr,
			"%s: kvm_open%s (namelist=%s, core = %s): %s\n",
			Pn,

#if	_FREEBSDV<200
			"files",
#else	/* _FREEBSDV>=200 */
			"",
#endif	/* _FREEBSDV<200 */

			Nmlst ? Nmlst : "default",
			Memory  ? Memory  : "default",
			strerror(errno));
		exit(1);
	}
/*
 * If space was allocated for the name list file path, free it.
 */
	if (nunix_stat)
		(void) free((FREE_P *)nunix);
}


/*
 * initialize() - perform all initialization
 */

void
initialize()
{
	get_kernel_access();
}


/*
 * kread() - read from kernel memory
 */

int
kread(addr, buf, len)
	KA_T addr;			/* kernel memory address */
	char *buf;			/* buffer to receive data */
	READLEN_T len;			/* length to read */
{
	int br;

#if	_FREEBSDV<200
	br = kvm_read((void *)addr, (void *)buf, len);
#else	/* _FREEBSDV>=200 */
	br = kvm_read(Kd, (u_long)addr, buf, len);
#endif	/* _FREEBSDV<200 */

	return((br == len) ? 0 : 1);
}


/*
 * process_text() - process text information
 */
void
process_text(vm)
	KA_T vm;				/* vm space pointer */
{
	int i, j;
	KA_T ka;
	int n = 0;
	struct pager_struct pg;
	struct vm_map_entry vmme, *e;
	struct vm_object vmo;
	struct vmspace vmsp;
/*
 * Read the vmspace structure for the process.
 */
	if (kread(vm, (char *)&vmsp, sizeof(vmsp)))
		return;
/*
 * Read the vm_map structure.  Search its vm_map_entry structure list.
 */
	if ( ! vmsp.vm_map.is_main_map)
		return;
	for (i = 0; i < vmsp.vm_map.nentries; i++) {

	/*
	 * Read the next vm_map_entry.
	 */
		if (i == 0)
			e = &vmsp.vm_map.header;
		else {
			if ((ka = (KA_T)e->next) == NULL)
				return;
			e = &vmme;
			if (kread(ka, (char *)e, sizeof(vmme)))
				return;
		}
		if (e->is_a_map || e->is_sub_map)
			continue;
	/*
	 * Read the map entry's object and the object's shadow.
	 * Look for a PG_VNODE pager handle.
	 */
		for (j = 0, ka = (KA_T)e->object.vm_object;
		     j < 2 && ka;
		     j++, ka = (KA_T)vmo.shadow)
		{
			if (kread(ka, (char *)&vmo, sizeof(vmo)))
				break;
			if ((ka = (KA_T)vmo.pager) == NULL
			||   kread(ka, (char *)&pg, sizeof(pg)))
				continue;
			if (pg.pg_handle == NULL || pg.pg_type != PG_VNODE)
				continue;
			(void) (enter_vn_text((KA_T)pg.pg_handle, &n));
		}
	}
}


/*
 * The ncache_addr(), ncache_load(), and ncache_lookup() functions are
 * obtained from ../common/rnam.frag.
 */

#if	defined(HASNCACHE)
#define	NCACHE		namecache	/* kernel's structure name */
#define	NCACHE_NM	nc_name		/* name in NCACHE */
#define	NCACHE_NMLEN	nc_nlen		/* name length in NCACHE */
# if	_FREEBSDV<205
#define	NCACHE_NXT	nc_nxt		/* link in NCACHE */
# else	/* _FREEBSDV>=205 */
#define	NCACHE_NXT	nc_lru.tqe_next	/* link in NCACHE */
# endif	/* _FREEBSDV<205 */
#define	NCACHE_NODEADDR	nc_vp		/* node address in NCACHE */
#define	NCACHE_NODEID	nc_vpid		/* node ID in NCACHE */
#define	NCACHE_PARADDR	nc_dvp		/* parent node address in NCACHE */
#define	NCACHE_PARID	nc_dvpid	/* parent node ID in NCACHE */
#endif	/* defined(HASNCACHE) */


