/*
 * dproc.c - Pyramid DC/OSx process access functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * Modified for Pyramid DC/OSx by Anthony Shortland
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dproc.c,v 1.4 95/08/22 13:12:51 abe Exp $";
#endif

#include "lsof.h"


/*
 * Local static values
 */

static struct var Var;			/* kernel variables */



_PROTOTYPE(static void get_kernel_access,(void));
_PROTOTYPE(static void readfsinfo,(void));
_PROTOTYPE(static void process_text,(struct as *pa));


/*
 * gather_proc_info() -- gather process information
 */

void
gather_proc_info()
{
	struct dirent *dp;
	struct file *f;
	int i, j, nf;
	struct proc *p, ps;
	DIR *pdfd;
	int  pfd;
	int pgrp, pid;
	struct pid pids;
	char pn[MAXPATHLEN+1];
	short pss, sf;
	size_t psz;
	static struct user *u = NULL;
	uid_t uid;
/*
 * Allocate space for user area.
 */
	if (!u) {
	    psz = sysconf(_SC_PAGESIZE);
	    if ((u = (struct user *)memalign(psz, (sizeof(struct user) + psz)))
	    == NULL) {
		(void) fprintf(stderr,
		    "%s: can't memalign user structure buffer\n", Pn);
		exit(1);
	    }
	}
/*
 * Examine proc structures and their associated information.
 */
	if ((pdfd = opendir(PROCFS)) == NULL) {
		(void) fprintf(stderr, "%s: can't open %s\n", Pn, PROCFS);
		exit(1);
	}
	(void) rewinddir(pdfd);
	for (p = &ps, pfd = -1; (dp = readdir(pdfd)) != NULL; ) {
		if (pfd >= 0) {
			(void) close(pfd);
			pfd = -1;
		}
		if (dp->d_name[0] == '.')
			continue;
		(void) sprintf(pn, "%s/%s", PROCFS, dp->d_name);
		if ((pfd = open(pn, O_RDONLY, 0)) < 0)
			continue;
		if (ioctl(pfd, PIOCGETPR, p) < 0)
			continue;
		if (p->p_stat == 0 || p->p_stat == SZOMB)
			continue;
	/*
	 * Get Process ID, Process group ID, and User ID.
	 */
		if (p->p_pidp == NULL
		||  kread((KA_T)p->p_pidp, (char *)&pids, sizeof(pids)))
			continue;
		if ((pid = (int)pids.pid_id) != atoi(dp->d_name))
			continue;
		if (p->p_pgidp == NULL
		||  kread((KA_T)p->p_pgidp, (char *)&pids, sizeof(pids)))
			continue;
		pgrp = (int)pids.pid_id;
		uid = (uid_t)p->p_uid;
		if (is_proc_excl(pid, pgrp, (UID_ARG)uid, &pss, &sf))
			continue;
	/*
	 * Get the user area associated with the process.
	 */
		if (ioctl(pfd, PIOCGETU, u) < 0)
			continue;
	/*
	 * Allocate a local process structure.
	 */
		if (is_cmd_excl(p->p_comm, &pss, &sf))
			continue;
		alloc_lproc(pid, pgrp, (UID_ARG)uid, p->p_comm,
			(int)pss, (int)sf);
		Plf = NULL;
	/*
	 * Save current working directory information.
	 */
		if (u->u_cdir) {
			alloc_lfile(CWD, -1);
			process_node((caddr_t)u->u_cdir);
			if (Lf->sf)
				link_lfile();
		}
	/*
	 * Save root directory information.
	 */
		if (u->u_rdir) {
			alloc_lfile(RTD, -1);
			process_node((caddr_t)u->u_rdir);
			if (Lf->sf)
				link_lfile();
		}
	/*
	 * Print information on the text file.
	 */
		if (p->p_as && Nl[X_SEGVN].n_value)
			process_text(p->p_as);
	/*
	 * Save information on file descriptors.
	 */
		for (i = 0, j = 0; i < u->u_nofiles; i++) {
			if (++j > NFPCHUNK) {
			    if ( ! u->u_flist.uf_next)
				break;
			    if (kread((KA_T)u->u_flist.uf_next,
				(char *)&u->u_flist, sizeof(struct ufchunk)))
				    break;
			    j = 1;
			}
			f = u->u_flist.uf_ofile[j-1];
			if (f) {
				alloc_lfile(NULL, i);
				process_file(f);
				if (Lf->sf)
					link_lfile();
			}
		}
	/*
	 * Examine results.
	 */
		if (examine_lproc())
			return;
	}
/*
 * Close proc file system access controls.
 */
	if (pfd >= 0)
		(void) close(pfd);
	(void) closedir(pdfd);
}


/*
 * get_kernel_access() - get access to kernel memory
 */

static void
get_kernel_access()
{
	if ((Memory != NULL && !is_readable(Memory))
	||  (Nmlst != NULL && !is_readable(Nmlst)))
		exit(1);
/*
 * Open kernel memory access.
 */
	if ((Kmem = open(Memory ? Memory : KMEM, O_RDONLY, 0)) < 0) {
		(void) fprintf(stderr, "%s: can't open %s: %s\n", Pn,
			Memory ? Memory : KMEM, strerror(errno));
		exit(1);
	}
/*
 * Access kernel symbols and values.
 */
        if (nlist(Nmlst ? Nmlst : N_UNIX, Nl) < 0) {
		(void) fprintf(stderr,
			"%s: can't read kernel name list from %s\n",
			Pn, Nmlst ? Nmlst : N_UNIX);
		exit(1);
	}
	if (Nl[X_VAR].n_value == NULL
	||  kread((KA_T)Nl[X_VAR].n_value, (char *)&Var, sizeof(Var)))
	{
		(void) fprintf(stderr,
			"%s: can't read system configuration info\n", Pn);
		exit(1);
	}
}


/*
 * initialize() - perform all initialization
 */

void
initialize()
{
	get_kernel_access();
	iuidcache(Var.v_proc);
	readfsinfo();
}


/*
 * kread() - read from kernel memory
 */

int
kread(addr, buf, len)
	KA_T addr;			/* kernel memory address */
	char *buf;			/* buffer to receive data */
	READLEN_T len;			/* length to read */
{
	int br;

	if (lseek(Kmem, (long)addr, L_SET) == (long)-1L)
		return(-1);
	br = read(Kmem, buf, len);
	return((br == len) ? 0 : 1);
}


/*
 * process_text() - process text access information
 */

static void
process_text(pa)
	struct as *pa;			/* address space description pointer */
{
	struct as as;
	int i, j, k, l;
	struct seg s;
	union segdata {
		struct segvn_data vn;
		struct segdev_data dev;
		struct segmap_data map;
	} sd;
	struct vnode *v[MAXSEGS];
/*
 * Get address space description.
 */
	if (kread((KA_T)pa, (char *)&as, sizeof(as))) {
		alloc_lfile(" txt", -1);
		(void) sprintf(Namech, "can't read text segment list (%#x)",
			pa);
		enter_nm(Namech);
		link_lfile();
		return;
	}
/*
 * Loop through the segments.  The loop should stop when the segment
 * pointer returns to its starting point, but just in case, it's stopped
 * when MAXSEGS have been recorded or 2*MAXSEGS have been examined.
 */
	s.s_next = as.a_segs;
	for (i = j = k = 0; i < MAXSEGS && j < 2*MAXSEGS; j++) {
		if (s.s_next == NULL
		||  kread((KA_T)s.s_next, (char *)&s, sizeof(s)))
			break;
		if ((u_long)s.s_ops == Nl[X_SEGVN].n_value
		&&  s.s_data != NULL) {
			if (kread((KA_T)s.s_data, (char *)&sd, sizeof(sd)))
				break;
			if (sd.vn.vp) {
			
			/*
			 * This is a virtual node segment.
			 *
			 * If its vnode pointer has not been seen already,
			 * print its information.
			 */
				for (l = 0; l < k; l++) {
					if (v[l] == sd.vn.vp)
						break;
				}
				if (l >= k) {
					alloc_lfile(" txt", -1);
					process_node((caddr_t)sd.vn.vp);
					if (Lf->sf) {
						link_lfile();
						i++;
					}
					v[k++] = sd.vn.vp;
				}
			}
		}
	/*
	 * Follow the segment link to the starting point in the address
	 * space description.  (The i and j counters place an absolute
	 * limit on the loop.)
	 */
		if (s.s_next == as.a_segs)
			break;
	}
}


/*
 * readfsinfo() - read file system information
 */

static void
readfsinfo()
{
	char buf[FSTYPSZ+1];
	int i;

	if ((Fsinfomax = sysfs(GETNFSTYP)) == -1) {
		(void) fprintf(stderr, "%s: sysfs(GETNFSTYP) error: %s\n",
			Pn, strerror(errno));
		exit(1);
	} 
	if (Fsinfomax == 0)
		return;
	if ((Fsinfo = (char **)malloc((MALLOC_S)(Fsinfomax * sizeof(char *))))
	== NULL) {
		(void) fprintf(stderr, "%s: no space for sysfs info\n", Pn);
		exit(1);
	}
	for (i = 1; i <= Fsinfomax; i++) {
		if (sysfs(GETFSTYP, i, buf) == -1) {
			(void) fprintf(stderr,
				"%s: sysfs(GETFSTYP) error: %s\n",
				Pn, strerror(errno));
			exit(1);
		}
		buf[FSTYPSZ] = '\0';
		if ((Fsinfo[i-1] = (char *)malloc((MALLOC_S)(strlen(buf) + 1)))
		== NULL) {
			(void) fprintf(stderr,
				"%s: no space for file system entry %s\n",
				Pn, buf);
			exit(1);
		}
		(void) strcpy(Fsinfo[i-1], buf);
	}
}
