;;; guided-tour.el --- functions for viewing Phil Sung's Emacs Guided Tour

;; Copyright 2007 Free Software Foundation, Inc.

;; Author: Stephen J. Turnbull <stephen@xemacs.org>
;; Maintainer: XEmacs Development Team
;; Created: 2007-05-02
;; Keywords: doc

;; The permissions notice is contained in the function `guided-tour-about'
;; at the end of this file.  Under the terms granted, you may modify that
;; function, but the appropriate permissions notice must remain somewhere in
;; any file derived from this one.

;;; Synched up with: Not in FSF.

;;; Commentary:

;; Thanks to Jason Spiro and Phil Sung for making substantial effort
;; to make this content as useful to XEmacs as possible.

;;; Code:

;; Variables

(defgroup guided-tour nil "The Guided Tour of Emacs")

;; #### this belongs somewhere more generic
;; #### it would be nice to be able to filter it for platform
;; #### The docstrings don't appear in the Value Menu.  Tried the menu-choice
;; widget and the :tag and :menu-tag keywords in various combinations, no joy.
(defcustom guided-tour-open-utilities
  '((const "open" :doc "Useful on Mac OS X")
    (const "xdg-open" :doc "Freedesktop `open' utility.")
    (const "see" :doc "Open source `open' utility")
    (const "gnome-open" :doc "GNOME `open' utility - some report it doesn't work"))
  "List of generic \"open document\" command-line utilities.
Each utility should be able to determine the right application to use to open
an arbitrary file passed to it on the command line.")

(defcustom guided-tour-odp-viewer "ooimpress"
  "Path to program able to view Open Document Format presentations.
Program is invoked in a shell, so PATH is searched."
  :type (append '(choice string
		  (const "ooimpress"))
		guided-tour-open-utilities) 
  :group 'guided-tour)

(defcustom guided-tour-pdf-viewer "xpdf"
  "Path to program able to view Portable Document Format documents.
Program is invoked in a shell, so PATH is searched."
  :type (append '(choice string
		  (const "xpdf")
		  (const "gv"))
		guided-tour-open-utilities) 
  :group 'guided-tour)

(defconst guided-tour-submenu
  '("A Guided Tour of Emacs"
    ["Guided Tour, Part %_1 (OOo)" guided-tour-odp-1]
    ["Guided Tour, Part %_2 (OOo)" guided-tour-odp-2]
    ["Guided Tour, Part %_3 (OOo)" guided-tour-odp-3]
    ["Guided Tour, Part %_1 (PDF)" guided-tour-pdf-1]
    ["Guided Tour, Part %_2 (PDF)" guided-tour-pdf-2]
    ["Guided Tour, Part %_3 (PDF)" guided-tour-pdf-3]
    ["About COPYING the Tour" guided-tour-about])
  "The submenu for the \"Guided Tour of Emacs\".")

(defvar guided-tour-insinuate-menubar nil
  "If non-nil, inhibits insinuation of the menubar.
Note that if you make this nil and reinsinuate, you are responsible for
removing any existing instances of the submenu.")

;; Functions

;; Helper functions

;; #### this probably should move to help.el or menubar-items.el
(defun guided-tour-find-menubar-help-menu (&optional menubar)
  "Return the Help submenu for MENUBAR if present, else nil."
  (assoc "%_Help" (or menubar default-menubar)))

(defun guided-tour-about-xemacs-index (menu)
  "Return the (zero-based) index of the About XEmacs item in MENU.
Returns nil if not present."
  ;; #### Of course we should actually search for it....
  ;; Does easy-menu provide functions for this?  It should....
  (let ((item (cadr menu)))
    (if (and (vectorp item) (eq 'about-xemacs (aref item 1)))
	0
      nil)))

;; Initialization

;;;###autoload
(defun guided-tour-insinuate-menubar ()
  "Add the Guided Tour of Emacs to the default menubar."
  (unless guided-tour-insinuate-menubar
    (setq guided-tour-insinuate-menubar t)
    (let* ((help (guided-tour-find-menubar-help-menu)))
      (setcdr help (nconc (if (eq 0 (guided-tour-about-xemacs-index help))
			      (list (cadr help) guided-tour-submenu)
			    (list guided-tour-submenu (cadr help)))
			  (cddr help))))))

;; Is this OK?  Don't see how it really hurts.
;;;###autoload
(guided-tour-insinuate-menubar)

;; The Guided Tour

(defun guided-tour (type part)
  "Start the Guided Tour with TYPE viewer, in Part PART."
  (interactive "sWhich format? \nnWhich part? ")
  (let ((viewer (symbol-value (intern (concat "guided-tour-" type "-viewer"))))
	(file (format "emacs-slides-%d.%s" part type)))
    (message "\
`M-x guided-tour-about RET' for FAQ and licensing.")
    (shell-command (format "%s %s" viewer (or (locate-data-file file) file)))))

(defun guided-tour-odp-1 ()
  "Start up the Guided Tour of Emacs, part 1, as an Open Office presentation."
  (interactive)
  (guided-tour "odp" 1))

(defun guided-tour-odp-2 ()
  "Start up the Guided Tour of Emacs, part 2, as an Open Office presentation."
  (interactive)
  (guided-tour "odp" 2))

(defun guided-tour-odp-3 ()
  "Start up the Guided Tour of Emacs, part 3, as an Open Office presentation."
  (interactive)
  (guided-tour "odp" 3))

(defun guided-tour-pdf-1 ()
  "Start up the Guided Tour of Emacs, part 1, in a PDF viewer."
  (interactive)
  (guided-tour "pdf" 1))

(defun guided-tour-pdf-2 ()
  "Start up the Guided Tour of Emacs, part 2, in a PDF viewer."
  (interactive)
  (guided-tour "pdf" 2))

(defun guided-tour-pdf-3 ()
  "Start up the Guided Tour of Emacs, part 3, in a PDF viewer."
  (interactive)
  (guided-tour "pdf" 3))

(defun guided-tour-about ()
  "Document the Guided Tour."
  (interactive)
  (with-displaying-temp-buffer "*About the Guided Tour of Emacs*"
    (princ "\

			A Guided Tour of Emacs
			      Phil Sung

This is the XEmacs package of the Guided Tour of Emacs.
The slides are by Phil Sung.  That's the important part.  Send kudos to Phil.
The XEmacs package is by Stephen Turnbull.  Direct complaints about packaging
to Steve via the XEmacs Developers mailing list <xemacs-beta@xemacs.org>.
The FAQ follows the permissions notice.

Copyright 2007 The Free Software Foundation Inc.

The Guided Tour, is both free software and free content.

You can redistribute it and/or modify it under the terms of the GNU
General Public License as published by the Free Software Foundation;
either version 2, or (at your option) any later version published by
the Free Software Foundation.

You can redistribute it and/or modify it under the terms of GNU Free
Documentation License; either version 1.2, or (at your option) any
later version published by the Free Software Foundation.

A verbatim or modified version can be redistributed under both
licenses simultaneously, which is the authors' preferred method.

The Guided Tour is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License and/or the GNU Free Documentation License for more
details.

FAQ

Section 1: Problems

Q1.1  I select a tour from the menu, but nothing happens!
A1.1  You probably need to configure a viewer.  C-h v guided-tour-pdf-viewer RET
      or C-h v guided-tour-odp-viewer RET.

Section 2: Further Info.

Q2.1  Where can I find the licenses?
A2.1  The GPL is included with XEmacs and Emacs, and the GFDL is included with
      several XEmacs packages and Emacs.  You can also visit the GNU home page
      at http://www.gnu.org/licenses/ where the GPL and GFDL (as well as the
      Library/Lesser GPL) are available for browsing.  Finally, you can write
      to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
      Boston, MA 02111-1307, USA, and request a copy.

Q2.2  Are there updated versions?
A2.2  Perhaps.  See Phil's Emacs page at http://stuff.mit.edu/iap/emacs/ and/or
      http://www.gnu.org/software/emacs/tour/.

Q2.3  Can I see an HTML version?
A2.3  Yes, at Phil's site and at gnu.org (see A2.1).

Q2.4  Will an HTML version be included in the future?
A2.4  Patches welcome!  Ask about how to contribute to the XEmacs package, or
      lobby for inclusion of new/updated content at <xemacs-beta@xemacs.org>.
"
)))

;;; guided-tour.el ends here
