/* IsOn... Copyright 1990-94 NCEMRSoft.  Use at your own risk.
 * This version by Mike Gleason, NCEMRSoft (mgleason@cse.unl.edu).
 * Original version by Phil Dietz, NCEMRSoft (pdietz@cse.unl.edu).
 */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <utmp.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>

#include "config.h"
#include "ison.h"

#if HAVE_STDLIBH
#include <stdlib.h>
#else
extern void *malloc(), *calloc();
#endif

#if HAVE_STDARGH
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#if RPC
#include <sys/socket.h>
#include <rpc/rpc.h>
#include <rpcsvc/rusers.h>
#endif	/* RPC */

#if DBMALLOC
#include <malloc.h>
#endif

#if ansi
int Strncasecmp(register char *a, register char *b, register size_t n)
#else
int Strncasecmp(a, b, n)
	 register char *a, *b;
	 register size_t n;
#endif
{
	register int A, B;
	register int fullcompare;
	
	/* You can supply a 0 to mean just do a regular Strcasecmp. */
	fullcompare = (n == (size_t) 0);
	while ((fullcompare) || (n-- > (size_t) 0)) {
		A = islower(*a) ? tolower((int) *a++) : *a++;
		B = islower(*b) ? tolower((int) *b++) : *b++;
		if (A > B)
			return (A - B);
		if (B > A)
			return (B - A);
		if (A == 0 && B == 0)
			return (0);
	}
	return (0);						   /* equal to n characters if we get
									    * here */
}									   /* Strncasecmp */




#if ansi
char *TimeStr(char *buf, size_t siz, time_t t)
#else
char *TimeStr(buf, siz, t)
	char *buf;
	size_t siz;
	time_t t;
#endif
{
#if HAVE_STRFTIME
	char buf2[128];

	if (t == (time_t)0)
		time(&t);
	(void) strftime(buf2, SZ(127), "%I:%M %p", localtime(&t));
	if (buf2[0] == '0' && buf[1] == '0') {
		buf2[0] = '1';
		buf2[1] = '2';
	}
	(void) strncpy(buf, buf2[0] == '0' ? buf2 + 1 : buf2 , siz);
#else
	if (t == (time_t)0)
		time(&t);
	(void) strncpy(buf, ctime(&t), siz);
	if (t == (time_t)0)
		time(&t);
	buf[strlen(buf) - 1] = '\0';	/* Get rid of the \n. */
#endif
	return (buf);
}	/* TimeStr */




#if ansi && HAVE_STDARGH
static void PrintF(IsonParams *g0, int flags0, char *fmt0, ...)
#else
#if HAVE_STDARGH
static void PrintF(g0, flags0, fmt0, ...)
	IsonParams *g0;
	int flags0;
	char *fmt0;
#else
static void PrintF(va_alist)
	va_dcl
#endif	/* !HAVE_STDARGH*/
#endif	/* !ansi */
{
	va_list ap;
	char tmstr[40];
	IsonParams *g;
	int flags;
	char *fmt;

#if (HAVE_STDARGH == 0)
	va_start(ap);
	g = va_arg(ap, IsonParams *);
	flags = va_arg(ap, int);
	fmt = va_arg(ap, char *);
#else
	va_start(ap, fmt0);
	g = g0; flags = flags0; fmt = fmt0;
#endif

	if (g->outfile == NULL)
		return;
#if DEBUG
	if (((flags & DEBUG_MSG) != 0) && (!g->debug))
		return;
#endif
	if ((flags & FLUSH_MSG) != 0) {
		(void) fflush(g->outfile);
		return;
	}

	if ((flags & NO_HEADER_MSG) == 0) {
		tmstr[0] = '\0';
		if (g->outfile != stderr) {		/* Log file. */
			tmstr[0] = ' ';
			(void) TimeStr(tmstr+1, sizeof(tmstr)-1, (time_t) 0);
		}	
		(void) fprintf(g->outfile, "\r\n%s%s%s IsOn: ",
			(((flags & (NO_BEEP_MSG|DEBUG_MSG)) != 0) ||
				(g->canBeep == 0)) ? "" : "\007",
			((flags & DEBUG_MSG) != 0) ? "#DB#" : "**",
			tmstr
		);
	}
	
	(void) vfprintf(g->outfile, fmt, ap);
	va_end(ap);
}	/* PrintF */


#if ansi
void MakeSureIAmLoggedIn(IsonParams *g)
#else
void MakeSureIAmLoggedIn(g)
	IsonParams *g;
#endif
{
	if (g->autodie) {
#if CHECK_PARENT
		/* Don't kill ourself if stdin was not a tty in the first place,
		 * which means we were probably called from a shell script, and
		 * the shell will exit before we finish.
		 */
		if (g->stdinatty && kill(g->parentPid, 0)) {
			/* we've lost our shell! */
			PrintF(g, 0, "Lost our parent!\n");
			exit(EXIT_NOT_LOGGED_IN);
		}
#endif
#if CHECK_STDERR
		if (!isatty(2)) {
			/* Hmm... wonder where this is going? */
			PrintF(g, 0, "Stderr is not a tty!\n");
			exit(EXIT_NOT_LOGGED_IN);
		}
#endif
	}
}	/* MakeSureIAmLoggedIn */



#if ansi
void Delay(IsonParams *g)
#else
void Delay(g)
	IsonParams *g;
#endif
{
	unsigned int sl;

	/* Kill some time so more important processes can run. */
	g->iter++;
#if DEBUG
#if DBMALLOC
	{
		int inuse = (int) malloc_inuse(NULL);
		if (inuse > g->memInUse) {
			g->memInUse = inuse;
			PrintF(g, DEBUG_MSG, "malloc_inuse: %d\n", inuse);
		}
	}
#endif
#endif
	if ((g->maxIters > 0L) && (g->iter > g->maxIters)) {
		PrintF(g, 0, "Giving up after %ld iterations.\n", g->maxIters);
		exit(EXIT_MAX_ITERATIONS);
	}
	if (g->iter == 1L) {
		if (g->stdinatty && g->pollmsg) {
			/* Print a message if you ran us from an interactive shell. */
			PrintF(g, NO_BEEP_MSG, "Polling.");
			if (g->daemon)
				PrintF(g, NO_HEADER_MSG, "  Type \"kill %d\" to terminate.\n", g->pid);
			else
				PrintF(g, NO_HEADER_MSG, "..\n");
		}
	} else {
		if (g->sleep > 0) {
			/* The user supplied a value to give to sleep. */
			sl = (unsigned int) g->sleep;
		} else if (g->sleep < 0) {
			/* Use one of the defaults.  If you supplied a remote host,
			 * wait a little longer, otherwise use the local delay
			 * which is shorter.
			 */
			sl = (unsigned int) ((g->nRemoteHosts > 0) ? DEFAULT_REMOTE_SLEEP :
				DEFAULT_LOCAL_SLEEP);
		}
#if DEBUG
		PrintF(g, DEBUG_MSG, "Sleeping %3d%s",
			sl,
			(g->daemon == 0) ? ": " : "...\r\n"
		);
		PrintF(g, FLUSH_MSG, "");
#endif
		(void) sleep(sl);
#if DEBUG
		PrintF(g, DEBUG_MSG|NO_HEADER_MSG, "done.\n");
#endif
	}
}	/* Delay */



#if DEBUG
/* If we have debug mode on, we can print a message to the screen telling
 * the user that we are getting ready to do another check.
 */
#if ansi
void IterationDebugMsg(IsonParams *g, char *whichproc, HostPtr hp)
#else
void IterationDebugMsg(g, whichproc, hp)
	IsonParams *g;
	char *whichproc;
	HostPtr hp;
#endif
{
	char tmstr[40];

	PrintF(g, DEBUG_MSG, "Checking %s on %s (try #%ld) at %s\n",
		whichproc, hp->hostname,  g->iter,
		TimeStr(tmstr, sizeof(tmstr), (time_t)0)
	);
}	/* IterationDebugMsg */
#else
#define IterationDebugMsg(a,b,c)
#endif	/* DEBUG */



#if ansi
void RunCommand(HostPtr hp, UserPtr up)
#else
void RunCommand(hp, up)
	HostPtr hp;
	UserPtr up;
#endif
{
	char buf[256];
	char str[64];
	char *catstr;
	char *cp, *dp;
	size_t n;

	if (up->cmd != NULL) {
		for (dp = buf, cp = up->cmd, n = sizeof(buf) - 1; *cp != '\0'; cp++) {
			if (*cp == '%') {
				++cp;
				switch (*cp) {
					case '\0':
						--cp;
						break;
					case 'h':
						catstr = hp->hostname;
						goto cat;
					case 'i':
						if (up->idletime != IS_IDLE) {
							(void) sprintf(str, "%d",
								up->idletime == IS_IDLE ? 0 : up->idletime);
							catstr = str;
						} else catstr = up->idlestr;
						goto cat;
					case 'm':
						if (hp->pollproc == (pollproc_t) Utmp)
							catstr = "utmp";
						else if (hp->pollproc == (pollproc_t) Finger)
							catstr = "finger";
						else
							catstr = "rusers";
						goto cat;
					case 'n':
					case 'u':
						catstr = up->username;
						goto cat;
					case 'N':
					case 'U':
						if (hp->hostname != ISLOCALHOST) {
							(void) sprintf(str, "%s@%s", up->username,
								hp->hostname);
							catstr = str;
						} else
							catstr = up->username;
						goto cat;
					case 't':
						catstr = up->dev;
						goto cat;
					cat:
						for (; (n > 0) && (*catstr != '\0'); --n)
							*dp++ = *catstr++;
						break;
					default:
						if (n > 0) {
							--n;
							*dp++ = *cp;
						}
				}
			} else if (n > 0) {
				--n;
				*dp++ = *cp;
			}
		}
		*dp = 0;

		if (fork() == 0) {
			(void) sleep(1);
			(void) execlp("/bin/sh", "sh", "-c", buf, NULL);
			(void) perror(buf);
			exit(EXIT_FATAL_ERROR);	/* Rarely reached... */
		}
	}
}	/* RunCommand */




#if ansi
char *UserAddress(char *buf, HostPtr hp, UserPtr up)
#else
char *UserAddress(buf, hp, up)
	char *buf;
	HostPtr hp;
	UserPtr up;
#endif
{
	(void) strcpy(buf, up->username);
	if (Strncasecmp(hp->hostname, ISLOCALHOST, SZ(0)) != 0) {
		(void) strcat(buf, "@");
		(void) strcat(buf, hp->hostname);
	}
	return (buf);
}	/* UserAddress */




#if ansi
void UserIsIdle(IsonParams *g, HostPtr hp, UserPtr up)
#else
void UserIsIdle(g, hp, up)
	IsonParams *g;
	HostPtr hp;
	UserPtr up;
#endif
{
	char uabuf[128];

	up->wasIdle = 1;
	if (++up->idlemsg == 1) {
		PrintF(g, 0, "%s is logged in,\r\n         but has been idle ",
			UserAddress(uabuf, hp, up)
		);
		if (up->idletime > 0)
			PrintF(g, NO_HEADER_MSG, "%d minute%s.\n\n",
				up->idletime,
				(up->idletime > 1 ? "s" : "")
			);
		else
			PrintF(g, NO_HEADER_MSG, "%s.\n\n", up->idlestr);
	}
}	/* UserIsIdle */




#if ansi
void DoneWithUser(IsonParams *g, HostPtr hp, UserPtr up)
#else
void DoneWithUser(g, hp, up)
	IsonParams *g;
	HostPtr hp;
	UserPtr up;
#endif
{
	char tmstr[40];
	char useraddr[128];
	char TTY[32];
	char since[10];

	up->isOn = 1;
	if (up->wasOn)	/* Yeah, we know already. */
		return;

	up->logons++;
	up->wasOn = 1;

	(void) UserAddress(useraddr, hp, up);

	if (up->tyme == (time_t)0) {
		since[0] = '\0';
		/* tmstr will contain the time _here_ after the TimeStr() then. */
	} else {
		(void) strcpy(since, " since ");
	}
	(void) TimeStr(tmstr, sizeof(tmstr), up->tyme);

	TTY[0] = '\0';
	if (up->dev[0] != '\0')
		(void) sprintf(TTY, " to %s", up->dev);

	/* We don't want to print the idle message after we print this
	 * next message.
	 */
	++up->idlemsg;
	if (up->idletime > 0) {
		PrintF(g, 0,
			"%s logged on%s%s%s,\r\n         but has been idle %d min.\n",
			useraddr,
			TTY,
			since,
			since[0] == '\0' ? since : tmstr,
			up->idletime
		); 
	} else if (up->idletime == NOT_IDLE) {
		up->wasIdle = 0;
		PrintF(g, 0,
			"%s logged on%s%s%s\r\n         and is not idle.\n",
			useraddr,
			TTY,
			since,
			since[0] == '\0' ? since : tmstr
		); 
	} else if (up->idletime == IDLETIME_UNKNOWN) {
		/* Probably using Finger.  Since not all finger servers spew
		 * their output in the same format, we may not be able to find
		 * out if the user is idle or not.
		 */
		up->wasIdle = 0;
		PrintF(g, 0,
			"Detected login of %s at %s.\n",
			useraddr,
			tmstr
		); 
	} else if (up->idletime == IS_IDLE) {
		/* Again, from Finger.  We don't bother trying to figure out
		 * the exact number of minutes since there are various ways
		 * it prints that figure.  We just know it's there is some
		 * idletime, which is good enough.  Hopefully we won't be
		 * using Finger very much anyway.  We just tell them the
		 * exact same thing Finger told us.
		 */
		PrintF(g, 0,
			"Detected login of %s%s at %s,\r\n         but has been idle %s.\n",
			useraddr,
			TTY,
			tmstr,
			up->idlestr
		); 
	}

	if (!up->detectLogoffs) {
		/* If we are running detect logoffs mode, we want to run forever,
		 * so we don't want to increment the done-users counter.  If we
		 * did that, we would exit once all users had logged on at least
		 * once.  With this mode, we don't want to exit.
		 */
		if (++hp->usersDone == hp->nUsers) {
			++g->hostsDone;
		}
	}

	/* Run a command (script) if the user requested to. */
	RunCommand(hp, up);
}	/* DoneWithUser */




#if ansi
void Utmp(IsonParams *g, HostPtr hp)
#else
void Utmp(g, hp)
	IsonParams *g;
	HostPtr hp;
#endif
{
	struct utmp info;
	int idletime;
	struct stat st;
	char ttypath[128];
	UserPtr up;

	/* Open the utmp file, which is a list of all logged on users. */
	if ((g->utmpfp == NULL) && ((g->utmpfp = fopen(UTMP_FILE, "r")) == NULL)) {
		(void) perror(UTMP_FILE);
		DONEWITHHOST(hp);
	}
	
	/* Reset the utmp file and re-read it. */
	(void) rewind(g->utmpfp);

	IterationDebugMsg(g, "Utmp", hp);

	/* Cycle through all 'users' logged in. */
	while (fread(&info, SZ(sizeof(info)), SZ(1), g->utmpfp) == SZ(1)) {
		/* See if this guy matches any of the users we are looking for. */
		for (up = hp->firstUser; up != NULL; up = up->next) {
			if (Strncasecmp(up->username, info.ut_name, SZ(8)) == 0) {
				/* This user is logged on.  But is the user active? */
				(void) time(&up->tyme);
	
				up->isOn = 1;
				(void) strcat(strcpy(ttypath, "/dev/"), info.ut_line);
				idletime = IDLETIME_UNKNOWN;
				if (stat(ttypath, &st) == 0) {
					idletime = (int) (up->tyme - st.st_mtime) - 0;
					if (idletime < 0) idletime = NOT_IDLE;
					else idletime /= 60;
				}
	
				up->idletime = idletime;
				if (up->pollUntilActive && idletime > 0) {
					UserIsIdle(g, hp, up);
				} else {
					/* The user is not idle;  use the real login time. */
					up->tyme = info.ut_time;
				
					/* Note the tty the user is on. */
					(void) strcpy(up->dev, info.ut_line);

					DoneWithUser(g, hp, up);
				}
			}
		}
	}
}									   /* Utmp */




#if ansi
void Finger(IsonParams *g, HostPtr hp)
#else
void Finger(g, hp)
	IsonParams *g;
	HostPtr hp;
#endif
{
	FILE *in;
	register char *cp;
	int piperesult, pipelines, i;
	char buf[160], pipename[128];
	UserPtr up;
	
	(void) strcat(strcpy(pipename, FINGER), " @");
	(void) strcat(pipename, hp->hostname);

	if ((in = popen(pipename, "r")) == NULL) {
		perror(FINGER);
		exit(EXIT_FATAL_ERROR);
	}

	IterationDebugMsg(g, "Finger", hp);

	/* Cycle through all 'users' logged in. */
	pipelines = 0;
	while (fgets(buf, (int) sizeof(buf), in) != NULL) {
		pipelines++;

#if HAVE_STRSTR
		/* We would like to determine if a user is idle if possible.
		 * Since not all finger daemons format their output the same
		 * way, we may not be able to get the idle time.  We can
		 * find it if the other side prints a column header line.
		 * From my experience almost all of them being with "Login" as
		 * the very first column, and if there is an idle time column,
		 * the word "Idle" will appear on that header line.  If that's
		 * the case, we remember the offset into the line where the
		 * word "Idle" occurred.  Later when we find a user on our list,
		 * we look at the offset into the line.  If a user is idle,
		 * some sort of text will appear usually a number.  If not, it
		 * should be just whitespace.
		 */
		if (Strncasecmp("Login", buf, SZ(5)) == 0) {
			if (hp->idleCol == 0) {
				cp = strstr(buf, "Idle");
				if (cp != NULL)
					hp->idleCol = (int) (cp - buf);
			}

			/* Same thing for the TTY. */
			if (hp->ttyCol == 0) {
				cp = strstr(buf, "TTY");
				if (cp != NULL)
					hp->ttyCol = (int) (cp - buf);
			}
		}
#endif	/* HAVE_STRSTR */

		/* put a \0 in the first space after the username for Strncasecmp */
		cp = buf;
		while (*cp && isspace(*cp) == 0)
			cp++;
		*cp = '\0';
		
		/* See if this guy matches any of the users we are looking for. */
		for (up = hp->firstUser; up != NULL; up = up->next) {
			if (Strncasecmp(up->username, buf, SZ(8)) == 0) {
				up->tyme = (time_t)0;	/* Don't know login time. */
				up->idletime = IDLETIME_UNKNOWN;
				if (hp->idleCol != 0) {
					cp = buf + hp->idleCol;
					cp[4] = 0;
					up->idletime = NOT_IDLE;
					for (i=0; i<4; i++)
						if (!isspace(cp[i])) {
							up->idletime = IS_IDLE;
							/* Well, just save the same string finger
							 * said for later use.
							 */
							while (isspace(*cp) && *cp != '\0')
								cp++;
							(void) strcpy(up->idlestr, cp);
						}
				}
				if (hp->ttyCol != 0) {
					cp = buf + hp->ttyCol;
					cp[3] = 0;
					while (isspace(*cp) && *cp != '\0')
						cp++;
					if (Strncasecmp(cp, "co", SZ(2)) == 0)
						(void) strcpy(up->dev, "console");
					else
						(void) strcat(strcpy(up->dev, "tty"), cp);
				}
				up->isOn = 1;
				if (up->pollUntilActive && up->idletime == IS_IDLE) {
					UserIsIdle(g, hp, up);
				} else {
					DoneWithUser(g, hp, up);
				}
			}
		}
	}

	piperesult = pclose(in);	   /* close pipe */
	if (piperesult) {
		PrintF(g, 0,
			"%sFinger unsuccessful with %s, so no users from it can be polled.\n",
			hp->hostname
		);
		DONEWITHHOST(hp);
	}
	if (pipelines <= 1) {
		/* finger probably puked */
		PrintF(g, 0,
			"%s did not supply any Finger output, so no users from it can be polled.\n",
			hp->hostname
		);
		DONEWITHHOST(hp);
	}
}									   /* Finger */




#if RPC

#if ansi
void RUsers(IsonParams *g, HostPtr hp)
#else
void RUsers(g, hp)
	IsonParams *g;
	HostPtr hp;
#endif
{
	struct utmpidlearr uti;
	UserPtr up;
	int i;

	IterationDebugMsg(g, "RUsers", hp);
	uti.uia_cnt = 0; uti.uia_arr = 0;

	if (rusers(hp->hostname, &uti) != 0) {
		if (g->iter == 1) {
			/* This remote site probably doesn't support RPC at all, so
			 * try finger instead.
			 */
#if DEBUG
			PrintF(g, DEBUG_MSG, "RPC failed with %s, will try Finger next time.\n",
				hp->hostname);
#endif
		} else {
			/* We were able to use RPC at least once, but it isn't working
			 * anymore.  We can still try using finger, and if that doesn't
			 * work, we'll have to give up on this host.
			 */
			PrintF(g, 0, "RPC not reliable with %s, falling back to Finger for this host.\n",
				hp->hostname);
		}
		hp->pollproc = (pollproc_t) Finger;
		Finger(g, hp);
		return;
	}

/* I prefer to use my own temp variable, but the internal structure names
 * differ across different versions of RPCSVC :-(
 */
#define R_UTMP(a) ((*uti.uia_arr[i]).ui_utmp)

	/* Cycle through each entry in the remote utmp list, and see if any
	 * of the entries match the users we are looking for.
	 */
	for (i=0; i<uti.uia_cnt; i++) {
		for (up = hp->firstUser; up != NULL; up = up->next) {
			if (Strncasecmp(up->username, R_UTMP(uti).ut_name,
				SZ(8)) == 0)
			{
				/* We have found one of our users. */
				
				/* We can print exact login time, unlike Finger. */
				up->tyme = R_UTMP(uti).ut_time;

				/* We can also get the exact idletime for this user,
				 * without any hassle.
				 */
				up->idletime = (int) uti.uia_arr[i]->ui_idle;

				up->isOn = 1;
				if (up->pollUntilActive && up->idletime > 0) {
					UserIsIdle(g, hp, up);
				} else {
					/* Note the tty the user is on. */
					(void) strcpy(up->dev, R_UTMP(uti).ut_line);
					DoneWithUser(g, hp, up);
				}
			}
		}
	}

	/* We're done with this chunk, so get rid of it and get a new one
	 * at the next iteration.
	 */
	xdr_free(xdr_utmpidlearr, &uti);
}	/* RUsers */
#endif	/* RPC */




#if ansi
int AddUser(IsonParams *g, char *username, int F, int noIdle, int mon, char *cmd)
#else
int AddUser(g, username, F, noIdle, mon, cmd)
	IsonParams *g;
	char *username, *cmd;
	int F, mon, noIdle;
#endif
{
	HostPtr hp;
	UserPtr up;
	int isLocal;
	char *cp;
	char *hostname;
	char userandhost[128];

	(void) strcpy(userandhost, username);	/* Don't modify original. */
	
	/*
	 * Check the username for an @, which would suggest that it is a
	 * domain-style address.
	 */
	if ((cp = INDEX(userandhost, '@')) != NULL) {
		*cp = 0;				/* now will contain only the username. */
		hostname = cp + 1;		/* points to the part after the @. */
		isLocal = 0;
	} else {
		isLocal = 1;
		hostname = ISLOCALHOST;	/* Give it an arbitrary name, so we can
								 * group all the local users together.
								 */
	}

	for (hp=g->firstHost; hp != NULL; hp = hp->next) {
		if (Strncasecmp(hostname, hp->hostname, 0) == 0) {
			/* We already have a host in the list by this name. */
			break;
		}
	}
	
	if (hp == NULL) {
		/* If we didn't have the hostname in question in our host list,
		 * add a new one.
		 */
		hp = (HostPtr) CALLOC1(sizeof(Host));
		if (hp == NULL) goto memerr;
		hp->hostname = (char *) malloc(strlen(hostname) + 1);
		if (hp->hostname == NULL) goto memerr;
		(void) strcpy(hp->hostname, hostname);

		/* Attach hp to the host list. */
		if (g->firstHost == NULL)
			g->firstHost = g->lastHost = hp;
		else {
			g->lastHost->next = hp;
			g->lastHost = hp;
		}

		hp->nUsers = 0;
		hp->firstUser = hp->lastUser = NULL;
		g->nHosts++;
	}

	if (isLocal)
		hp->pollproc = (pollproc_t) Utmp;
	else {
		g->nRemoteHosts++;
#if RPC
		/* Try RPC first, unless you said not to. */
		hp->pollproc = (pollproc_t) (F ? Finger : RUsers);
#else
		hp->pollproc = (pollproc_t) Finger;
#endif
	}
	
	up = (UserPtr) CALLOC1(sizeof(User));
	if (up == NULL) goto memerr;
	up->next = NULL;
	(void) strncpy(up->username, userandhost, sizeof(up->username) - 1);
	up->cmd = NULL;
	if (cmd != NULL) {
		up->cmd = (char *) malloc(strlen(cmd) + 1);
		if (up->cmd == NULL) goto memerr;
		(void) strcpy(up->cmd, cmd);
	}
	up->pollUntilActive = noIdle;
	up->detectLogoffs = mon;
	
	if (hp->lastUser == NULL)
		hp->firstUser = hp->lastUser = up;
	else {
		hp->lastUser->next = up;
		hp->lastUser = up;
	}
	hp->nUsers++;

	return (0);
	
memerr:
	return (-1);
}	/* AddUser */




#if ansi
void Poll(IsonParams *g)
#else
void Poll(g)
	IsonParams *g;
#endif
{
	HostPtr hp;
	UserPtr up;
	char uabuf[128], tmstr[40];

	do {
		Delay(g); /* Delay a little so we won't hog the CPU */
		MakeSureIAmLoggedIn(g);

#if DEBUG
		if (g->debug) {
			PrintF(g, NO_HEADER_MSG,
				"\r\nIsOn's PID: %d;  Parent: %d.\n", g->pid, g->parentPid);

			for (hp=g->firstHost; hp != NULL; hp = hp->next) {
				PrintF(g, NO_HEADER_MSG, "\r\nHost: %-40s  Mode: %c\n",
					hp->hostname,
					hp->pollproc == (pollproc_t) Finger ? 'F'
					: (hp->pollproc == (pollproc_t) Utmp ? 'U' : 'R')
				);		
				for (up = hp->firstUser; up != NULL; up = up->next) {
					PrintF(g, NO_HEADER_MSG,
						"\r    %-8s  %c  %-3s  logons=%-2d  idle=%-2d",
						up->username,
						up->pollUntilActive ? 'I' : ' ',
						up->isOn ? "ON" : "off",
						up->logons,
						up->idletime
					);
					if (up->cmd != NULL)
						PrintF(g, NO_HEADER_MSG, "  cmd='%s'", up->cmd);
					PrintF(g, NO_HEADER_MSG, "\n");
				}
			}
		}
#endif	/* DEBUG */

		for (hp=g->firstHost; hp != NULL; hp = hp->next) {
			if (hp->nUsers > hp->usersDone) {
				for (up = hp->firstUser; up != NULL; up = up->next)
					up->isOn = 0;
				(*hp->pollproc)(g, hp);

				/* Now go through and see if any users were logged on
				 * the last time, but were no longer detected on.
				 * If not, we can print a msg saying that this user
				 * logged out.
				 *
				 * Note that we always do this, no matter what
				 * up->detectLogoffs is set to, because we want to
				 * report the rare instance where a user has been
				 * idle, but sneaks on and logs off during ison's
				 * sleep period.
				 *
				 * Basically we print the message if the user was
				 * already on but idle then logs off before we
				 * noticed that she wasn't idle, OR you said to
				 * report logoffs.
				 */
				for (up = hp->firstUser; up != NULL; up = up->next) {
					if ((up->wasOn || up->wasIdle) && (!up->isOn) &&
						(up->detectLogoffs || up->logons == 0)) {
						PrintF(g, 0,
							"Detected log off of %s at %s.\n",
							UserAddress(uabuf, hp, up),
							TimeStr(tmstr, sizeof(tmstr), (time_t)0)
						);
						up->wasOn = up->wasIdle = 0;
						up->idletime = 0;
					}
				}
			}
		}
	} while (g->nHosts > g->hostsDone);

	if (g->utmpfp != NULL)
		(void) fclose(g->utmpfp);
	PrintF(g, NO_BEEP_MSG, "Done!\n");
}	/* Poll */




#if ansi
void ReadDataFile(IsonParams *g, char *fname)
#else
void ReadDataFile(g, fname)
	IsonParams *g;
	char *fname;
#endif
{
	FILE *fp;
	int usersAdded, linenum;
	int fingerOnly, pollUntilActive, detectLogoffs;
	char buf[256], username[64], *cp, *dp, *cmd;

	if ((fp = fopen(fname, "r")) == NULL) {
		perror(fname);
		exit(EXIT_FATAL_ERROR);
	}

	usersAdded = linenum = 0;
	while (fgets(buf, sizeof(buf) - 1, fp) != NULL) {
		++linenum;
		for (cp = buf; ; cp++) {			/* Skip whitespace. */
			if (*cp == '\0') goto skip;		/* Blank line? */
			if (!isspace(*cp)) break;
		}
		if (*cp == COMMENT_CHAR) goto skip;
		for (dp = cp; ; dp++) {
			if (*dp == '\0') goto syntax;
			if (isspace(*dp)) break;
		}
		*dp++ = 0;
		(void) strcpy(username, cp);

		for (; ; dp++) {					/* Skip whitespace. */
			if (*dp == '\0') goto addUsr;	/* No flags or cmd. */
			if (!isspace(*dp)) break;
		}

		fingerOnly = pollUntilActive = detectLogoffs = 0;
		cmd = NULL;

		if (*dp	== '-') {
			/* Collect options for this user. */
			for (++dp; ; dp++) {
				if (*dp == '\0') goto addUsr;	/* no EOLN? */
				if (isspace(*dp)) break;
				
				/* Check options. */
				if (*dp == 'I')
					pollUntilActive = 1;
				else if (*dp == 'F')
					fingerOnly = 1;
				else if (*dp == 'm')
					detectLogoffs = 1;
				else {
					PrintF(g, 0, "Illegal option '%c' on line %d.\n",
						(int) *dp, linenum);
					goto skip;
				}
			}
		}

		for (; ; dp++) {					/* Skip whitespace. */
			if (*dp == '\0') goto addUsr;	/* No command given. */
			if (!isspace(*dp)) break;
		}
		
		/* The command is what's left, if any. */
		cmd = dp;
		
		/* But let's strip off the EOLN. */
		cp = dp + strlen(dp) - 1;
		if (isspace(*cp))
			*cp-- = '\0';
		if (isspace(*cp))
			*cp = '\0';

addUsr:	
		if (AddUser(g, username, fingerOnly, pollUntilActive,
			detectLogoffs, cmd) < 0) {
			PrintF(g, 0, "Too many users, out of memory!\n");
			break;
		}
		++usersAdded;
		continue;
syntax:
		PrintF(g, 0, "Syntax error on line %d.\n", linenum);
skip:
		continue;
	}
	if (usersAdded == 0) {
		PrintF(g, 0, "No users in data file.\n");
		exit(EXIT_USAGE);
	}
}	/* ReadDataFile */





#if ansi
void Usage(IsonParams *g)
#else
void Usage(g)
	IsonParams *g;
#endif
{
	(void) fprintf(stderr, "\n\
IsOn Usage:\n\
    %s [-bq%s%sILmP] [-p N] [-i N] [-o outfile] [-c cmd] [-f userfile] [users]%s\n\
Usernames:\n\
    They can be in the form \"user@remote.host\" to poll remote hosts, or\n\
    just \"user\" for the local host.  You can also use the -f option to\n\
    supply a data file with many users (see the manual for the format).\n",
				   g->progname,
#if RPC
					"F",
#else
					"",
#endif
#if DEBUG
					"d",
#else
					"",
#endif
#if DAEMON
				   ""
#else
				   " &"
#endif
	);

	(void) fprintf(stderr, "\
Flags:\n\
   -I     : Poll until users are both logged on and not idle.\n\
   -L     : Live (don't exit) when you log off.  Usually used with -c.\n%s%s\
   -q     : Don't print any output at all.\n\
   -P     : Don't print 'Polling...' message.\n\
   -b     : %seep when IsOn prints an important message.\n\
   -j     : %sly.\n",
#if RPC
				"   -F     : Use finger right away, don't bother with RPC.\n",
#else
				"",
#endif
#if DEBUG
				"   -d     : Print debugging information while running.\n",
#else
				"",
#endif
#if BEEP
				"Don't b",
#else
				"B",
#endif
#if DAEMON
				"Don't go into the background automatical"
#else
				"Go into the background immediate"
#endif
	);

	(void) fprintf(stderr, "\
   -m     : Report logons and logoffs of specified users forever.\n\
   -p N   : Seconds between iterations (defaults: local=%d, remote=%d).\n\
   -i N   : Give up after 'N' iterations (default is ",
				   DEFAULT_LOCAL_SLEEP,
				   DEFAULT_REMOTE_SLEEP
	);
#if (MAXITER == NO_LIMIT)
	(void) fprintf(stderr, "infinity");
#else
	(void) fprintf(stderr, "%ld", MAXITER);
#endif

	(void) fprintf(stderr, ").\n\
   -o fil : Send output to 'fil' instead of the screen.\n\
   -c cmd : Command to run for each user found (e.g. \"write %%u %%t <msg\").\
\n%s by Phil Dietz & Mike Gleason, NCEMRSoft.\n",
				   VERSION_STR);

	exit(EXIT_USAGE);
}									   /* Usage */



#if ansi
void main(int argc, char **argv)
#else
main(argc, argv)
	 int argc;
	 char **argv;
#endif
{
	IsonParams g;
	int flag;
	char *cp, promptedName[64];
	char validOpts[32];
	char *cmd = COMMAND;
	int fingerOnly = 0;
	int pollUntilActive = 0;
	int dataFileUsed = 0;
	int detectLogoffs = 0;
	int i;

	g.sleep = -1;
	g.progname = argv[0];
	if ((cp = RINDEX(g.progname, '/')) != NULL)
		g.progname = cp + 1;
	g.daemon = DAEMON;
	g.debug = 0;
	g.nRemoteHosts = 0;
	g.memInUse = 0;
	g.autodie = 1;
	g.maxIters = MAXITER;
	g.iter = 0;
	g.pollmsg = 1;
	g.utmpfp = NULL;
	g.parentPid = getppid();
	g.stdinatty = isatty(0);
	g.nHosts = g.nRemoteHosts = g.hostsDone = 0;
	g.firstHost = g.lastHost = NULL;
	g.canBeep = BEEP;
	g.outfile = stderr;

	(void) strcpy(validOpts, "bIjqPmLc:o:p:f:i:");
#if RPC
	(void) strcat(validOpts, "F");
#endif
#if DEBUG
	(void) strcat(validOpts, "d");
#endif

	while ((flag = getopt(argc, argv, validOpts)) != EOF) {
		switch (flag) {
			case 'b':
				g.canBeep = !g.canBeep;
				break;
			case 'f':
				ReadDataFile(&g, optarg);
				dataFileUsed = 1;
				break;
			case 'F':
				fingerOnly++;
				break;
			case 'd':
				g.debug++;
				break;
			case 'j':
				g.daemon = !g.daemon;
				break;
			case 'm':
				detectLogoffs = 1;
				break;
			case 'o':
				if ((g.outfile = fopen(optarg, "a")) == NULL) {
					perror(optarg);
					exit(EXIT_FATAL_ERROR);
				}
				break;
			case 'q':
				g.outfile = NULL;
				break;
			case 'L':
				g.autodie = 0;
				break;
			case 'c':
				cmd = optarg;
				break;
			case 'P':
				g.pollmsg = !g.pollmsg;
				break;
			case 'p':
				g.sleep = atoi(optarg);
				break;
			case 'i':
				g.maxIters = atol(optarg);
				break;
			case 'I':
				pollUntilActive = 1;
				break;
			default:
				Usage(&g);
		}
	}

	if ((argv[optind] == NULL) && (dataFileUsed == 0)) {
		/* No users supplied on the command line. */
		if (g.stdinatty) {
			(void) fprintf(stderr, "User to poll: ");
			(void) fgets(promptedName, sizeof(promptedName), stdin);
			promptedName[strlen(promptedName) - 1] = '\0';
			if (promptedName[0] == '\0')
				Usage(&g);	/* They just hit return. */
			(void) AddUser(&g, promptedName, fingerOnly, pollUntilActive,
				detectLogoffs, cmd);
		} else
			Usage(&g);	/* Can't prompt from a shell script, etc. */
	} else {
		for (i=optind; i<argc; i++) {
			if (AddUser(&g, argv[i], fingerOnly, pollUntilActive,
				detectLogoffs, cmd) < 0) {
				PrintF(&g, 0, "Too many users, out of memory!\n");
				break;
			}
		}
	}


	/* Print a warning if the user chose a scenario where we know we
	 * will never exit!
	 */
	if (detectLogoffs && !g.autodie) {
		PrintF(&g, 0,
"NOTE: since you enabled continuous log on/off detection and turned off\n\
self-termination when you logoff, ison will run forever!  You will have to\n\
use 'ps -u yourname' or some such to kill it later.");
	}

	/* Don't leave ^G's in actual files. */
	if (g.outfile != stderr)
		g.canBeep = 0;

#if NICE
	/* lower our process' priority (nice) */
	(void) nice(20);
#endif

	if (g.daemon) {
		if (fork())					   /* automatically puts this task in
									    * background! */
			exit(EXIT_PARENT);

		(void) signal(SIGINT, SIG_IGN);
		(void) signal(SIGQUIT, SIG_IGN);
	}

	(void) sleep(1);	/* wait for your shell prompt to catch up. */
	(void) signal(SIGHUP, SIG_DFL);

	g.pid = getpid();

	Poll(&g);	/* Finally, get down to business. */

	exit(EXIT_SUCCESSFUL);
}									   /* main */

/* eof ison.c */
