/* ncftpput.c
 *
 * A simple, non-interactive utility to send files to a remote FTP server.
 * Very useful in shell scripts!
 */

#ifdef HAVE_CONFIG_H
#	include <Config.h>
#endif

#include <unistd.h>
#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
#include <setjmp.h>

#include <ncftp.h>				/* Library header. */
#include <Strn.h>				/* Library header. */
#include "gpshare.h"

jmp_buf gJmp;
int gGotSig = 0;
int gCanJmp = 0;

extern char *optarg;
extern int optind;

static void
Usage(void)
{
	FILE *fp;

	fp = popen("more", "w");
	if (fp == NULL)
		fp = stderr;

	fprintf(fp, "Usages:\n");
	fprintf(fp, "  ncftpput [flags] remote-host remote-dir local-files...   (mode 1)\n");
	fprintf(fp, "  ncftpput -f login.cfg [flags] remote-dir local-files...  (mode 2)\n");
	fprintf(fp, "\nFlags:\n\
  -u XX  Use username XX instead of anonymous.\n\
  -p XX  Use password XX with the username.\n\
  -P XX  Use port number XX instead of the default FTP service port (21).\n\
  -d XX  Use the file XX for debug logging.\n\
  -e XX  Use the file XX for error logging.\n\
  -U XX  Use value XX for the umask.\n\
  -t XX  Timeout after XX seconds.\n");
	fprintf(fp, "\
  -a     Use ASCII transfer type instead of binary.\n\
  -m     Attempt to mkdir the dstdir before copying.\n\
  -v/-V  Do (do not) use progress meters.\n\
  -f XX  Read the file XX for host, user, and password information.\n\
  -A     Append to remote files instead of overwriting them.\n\
  -T XX  Upload into temporary files prefixed by XX.\n\
  -S XX  Upload into temporary files suffixed by XX.\n\
  -R     Recursive mode; copy whole directory trees.\n");
	fprintf(fp, "\nExamples:\n\
  ncftpput -u gleason -p my.password Elwood.probe.net /home/gleason stuff.txt\n\
  ncftpput -u gleason Elwood.probe.net /home/gleason a.txt (prompt for pass)\n\
  ncftpput -a -u gleason -p my.password -m -U 007 Bozo.probe.net /tmp/tmpdir a.txt\n");
	fprintf(fp, "\nLibrary version: %s.\n", gLibNcFTPVersion + 5);
#ifdef UNAME
	fprintf(fp, "System: %s.\n", UNAME);
#endif
	fprintf(fp, "\nThis is a freeware program by Mike Gleason (mgleason@probe.net).\n");

	if (fp != stderr)
		(void) pclose(fp);
	exit(kExitUsage);
}	/* Usage */



static void
Abort(int sigNum)
{
	if (gCanJmp == 0) {
		gCanJmp = 0;
		gGotSig = sigNum;
		longjmp(gJmp, 1);
	}
}	/* Abort */




static int
Copy(FTPCIPtr cip, char *dstdir, char **files, int rflag, int xtype, int appendflag, const char *tmppfx, const char *tmpsfx)
{
	int i;
	int result;
	char *file;
	int rc = 0;

	for (i=0; ; i++) {
		file = files[i];
		if (file == NULL)
			break;
		result = FTPPutFiles2(cip, file, dstdir, rflag, kGlobNo, xtype, appendflag, tmppfx, tmpsfx);
		if (result != 0) {
			fprintf(stderr, "ncftpput: file send error: %s.\n", FTPStrError(result));
			rc = result;
		}
	}
	return (rc);
}	/* Copy */



void
main(int argc, char **argv)
{
	int result, c;
	volatile int rflag = 0;
	volatile int appendflag = 0;
	const char *volatile tmppfx = "";
	const char *volatile tmpsfx = "";
	volatile int xtype = kTypeBinary;
	volatile ExitStatus es;
	FTPLibraryInfo li;
	FTPConnectionInfo fi;
	volatile int wantMkdir = 0;
	char *volatile Umask = NULL;
	char *dstdir;
	const char * volatile errstr;
	int progmeters;
#ifdef HAVE_GETPASS
	char *password;
#endif

	result = FTPInitLibrary(&li);
	if (result < 0) {
		fprintf(stderr, "ncftpget: init library error %d (%s).\n", result, FTPStrError(result));
		exit(kExitInitLibraryFailed);
	}
	result = FTPInitConnectionInfo(&li, &fi, kDefaultFTPBufSize);
	if (result < 0) {
		fprintf(stderr, "ncftpget: init connection info error %d (%s).\n", result, FTPStrError(result));
		exit(kExitInitConnInfoFailed);
	}

	fi.debugLog = NULL;
	fi.errLog = stderr;
	STRNCPY(fi.user, "anonymous");
	fi.netTimeout = (unsigned int) 180;
	progmeters = ((isatty(2) != 0) && (getppid() > 1)) ? 1 : 0;

	while ((c = getopt(argc, argv, "P:u:p:e:d:U:t:marRvVf:AT:S:")) > 0) switch(c) {
		case 'P':
			fi.port = atoi(optarg);	
			break;
		case 'u':
			STRNCPY(fi.user, optarg);
			break;
		case 'p':
			STRNCPY(fi.pass, optarg);	/* Don't recommend doing this! */
			break;
		case 'e':
			if (strcmp(optarg, "stdout") == 0)
				fi.errLog = stdout;
			else if (optarg[0] == '-')
				fi.errLog = stdout;
			else if (strcmp(optarg, "stderr") == 0)
				fi.errLog = stderr;
			else
				fi.errLog = fopen(optarg, "a");
			break;
		case 'd':
			if (strcmp(optarg, "stdout") == 0)
				fi.debugLog = stdout;
			else if (optarg[0] == '-')
				fi.debugLog = stdout;
			else if (strcmp(optarg, "stderr") == 0)
				fi.debugLog = stderr;
			else
				fi.debugLog = fopen(optarg, "a");
			break;
		case 'U':
			Umask = optarg;
			break;
		case 't':
			fi.netTimeout = (unsigned int) atoi(optarg);	
			break;
		case 'm':
			wantMkdir = 1;
			break;
		case 'a':
			xtype = 0;	/* Use ascii. */
			break;
		case 'R':
		case 'r':
			rflag = 1;
			break;
		case 'v':
			progmeters = 1;
			break;
		case 'V':
			progmeters = 0;
			break;
		case 'f':
			ReadConfigFile(optarg, &fi);
			break;
		case 'A':
			appendflag = 1;
			break;
		case 'T':
			tmppfx = optarg;
			break;
		case 'S':
			tmpsfx = optarg;
			break;
		default:
			Usage();
	}
	if (optind > argc - 3)
		Usage();

#ifdef HAVE_GETPASS
	if (strcmp(fi.user, "anonymous") && strcmp(fi.user, "ftp")) {
		if ((fi.pass[0] == '\0') && (isatty(2) != 0)) {
			password = getpass("Password: ");		
			if (password != NULL) {
				STRNCPY(fi.pass, password);
				/* Don't leave cleartext password in memory. */
				memset(password, 0, strlen(fi.pass));
			}
		}
	}
#endif

	if (progmeters != 0)
		fi.progress = PrStatBar;

	STRNCPY(fi.host, argv[optind]);
	if (setjmp(gJmp) == 0) {
		signal(SIGINT, Abort);
		signal(SIGTERM, Abort);
		signal(SIGALRM, Abort);
		errstr = "could not open remote host";
		es = kExitOpenTimedOut;
		gCanJmp = 1;
		if ((result = FTPOpenHost(&fi)) < 0) {
			fprintf(stderr, "ncftpput: cannot open %s: %s.\n", fi.host, FTPStrError(result));
			es = kExitOpenFailed;
			exit(es);
		}
		dstdir = argv[optind + 1];
		if (Umask != NULL) {
			errstr = "could not set umask on remote host";
			result = FTPUmask(&fi, Umask);
			if (result != 0)
				fprintf(stderr, "ncftpput: umask failed: %s.\n", FTPStrError(result));
		}
		if (wantMkdir != 0) {
			errstr = "could not mkdir on remote host";
			result = FTPMkdir(&fi, dstdir, kRecursiveYes);
			if (result != 0)
				fprintf(stderr, "ncftpput: mkdir failed: %s.\n", FTPStrError(result));
		}
		if (result >= 0) {
			errstr = "could not write to file on remote host";
			es = kExitXferTimedOut;
			if (Copy(&fi, dstdir, argv + optind + 2, rflag, xtype, appendflag, (const char *) tmppfx, (const char *) tmpsfx) < 0)
				es = kExitXferFailed;
			else
				es = kExitSuccess;
		}
	} else {
		signal(SIGALRM, SIG_IGN);
		signal(SIGINT, SIG_IGN);
		signal(SIGTERM, SIG_IGN);
		if (gGotSig == SIGALRM) {
			fprintf(stderr, "ncftpput: %s: timed-out.\n", errstr);
		} else {
			fprintf(stderr, "\nncftpput: caught signal, cleaning up...\n");
		}
	}

	if (setjmp(gJmp) == 0) {
		signal(SIGINT, Abort);
		signal(SIGTERM, Abort);
		signal(SIGALRM, Abort);
		gCanJmp = 1;
		FTPCloseHost(&fi);
	} else {
		signal(SIGALRM, SIG_IGN);
		signal(SIGINT, SIG_IGN);
		signal(SIGTERM, SIG_IGN);
		if (gGotSig == SIGALRM) {
			fprintf(stderr, "ncftpput: %s: timed-out.\n", errstr);
		} else {
			fprintf(stderr, "\nncftpput: caught signal, cleaning up...\n");
		}
	}
	exit(es);
}	/* main */
