/* minincftp.c
 *
 * Example of a command-line shell interface to FTP using the library.
 */

#ifdef HAVE_CONFIG_H
#	include <Config.h>
#endif

#include <unistd.h>
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>		/* They have string.h... */
#include <sys/time.h>
#include <time.h>

#ifdef HAVE_GETPASS
#	include <pwd.h>
#endif
#include <signal.h>
#include <setjmp.h>

#include <Strn.h>				/* Library header. */
#include <ncftp.h>				/* Library header. */

jmp_buf gJmp;

extern char *optarg;
extern int optind;

static void
Usage(void)
{
	fprintf(stderr, "MiniNcFTP, copyright 1995 by Mike Gleason, NCEMRSoft.\n");
	fprintf(stderr, "Usage: minincftp [-P port] [-u user] [-p password] hostname\n");
	fprintf(stderr, "Library version: %s.\n", gLibNcFTPVersion + 5);
#ifdef UNAME
	fprintf(stderr, "System: %s.\n", UNAME);
#endif
	exit(2);
}	/* Usage */




static int
Ls(const FTPCIPtr cip, char *cmdstr, char *lsflag)
{
	int cmd;
	int result;

	cmd = (cmdstr[0] == 'd') ? 1 : 0;
	fflush(stdout);
	result = FTPList(cip, STDOUT_FILENO, cmd, lsflag);
	return result;
}	/* Ls */



#if 0
static void
NotYet(FTPCIPtr cip, ...)
{
	printf("Command not implemented in MiniNcFTP yet.\n");
}	/* NotYet */
#endif



static void
Shell(const FTPCIPtr cip)
{
	char line[256];
	char argv[10][80], *cp, *token;
	char s1[256];
	int i, c1, c2, c3, code;
	int argc;
	time_t mdtm;
	size_t size;

	for (;;) {
		code = kNoErr;
		printf("MiniNcFTP> ");
		if (fgets(line, sizeof(line), stdin) == NULL)
			break;
		cp = line;
		for (argc=0; argc<10; argc++) {
			argv[argc][0] = '\0';
			token = strtok(cp, " \n\t\r");
			if (token == NULL)
				break;
			STRNCPY(argv[argc], token);
			cp = NULL;
		}
		for (i=argc; i<10; i++)
			argv[i][0] = '\0';
		printf("\n");
		c1 = argv[0][0];
		c2 = argv[0][1];
		c3 = argv[0][2];
		cip->errno = 0;

		switch (c1) {
			case 'c':
				if ((c2 == 'd') || (c3 == 'd')) {
					/* chdir, cd */
					code = FTPChdir(cip, argv[1]);
				} else if (c3 == 'm') {
					code = FTPChmod(cip, argv[2], argv[1], kGlobYes);
				} else goto bad;
				break;
			case 'd':
				if (c2 == 'i') {
					code = Ls(cip, argv[0], argv[1]); /* dir */
				} else if (c2 == 'e') {
					code = FTPDelete(cip, argv[1], kRecursiveNo, kGlobYes);
				} else goto bad;
				break;
			case 'g':
				if (c3 == 'x')
					code = FTPGetOneFile(cip, argv[1], argv[2]);
				else
					code = FTPGetFiles(cip, argv[1], argv[2], kRecursiveNo, kGlobYes);
				if (cip->bytesTransferred > 0L) {
					printf("%lu bytes transferred, %.2f kB/sec.\n", (unsigned long) cip->bytesTransferred, cip->kBytesPerSec);
				}
				break;
			case 'l':
				code = Ls(cip, argv[0], argv[1]);
				if (cip->bytesTransferred > 0L) {
					printf("%lu bytes transferred, %.2f kB/sec.\n", (unsigned long) cip->bytesTransferred, cip->kBytesPerSec);
				}
				break;
			case 'm':
				if (c2 == 'k') {
					code = FTPMkdir(cip, argv[1], kRecursiveYes);
				} else if (c2 == 'o') {
					code = FTPFileModificationTime(cip, argv[1], &mdtm);
					printf("%s modtime = %lu.\n",
						argv[1], (unsigned long) mdtm);
				} else goto bad;
				break;
			case 'p':
				if (c2 == 'w') {
					code = FTPGetCWD(cip, s1, sizeof(s1));	/* pwd */
					printf("cwd = [%s]\n", s1);
				} else if (c2 == 'u') {
					if (c3 == 'x')
						code = FTPPutOneFile(cip, argv[1], argv[2]);
					else
						code = FTPPutFiles(cip, argv[1], argv[2], kRecursiveNo, kGlobYes);
					if (cip->bytesTransferred > 0L) {
						printf("%lu bytes transferred, %.2f kB/sec.\n", (unsigned long) cip->bytesTransferred, cip->kBytesPerSec);
					}
				} else goto bad;
				break;
			case 'q':
			case 'Q':
				if (c3 == 'o') {
					/* quote */
					STRNCPY(s1, argv[1]);
					for (i=2; i<10; i++) {
						if (argv[i][0] == '\0') break;
						STRNCAT(s1, " ");
						STRNCAT(s1, argv[i]);
					}
					FTPCmd(cip, "%s", s1);
					break;
				}
				return;
			case 'r':
				if (c2 == 'e') {
					code = FTPRename(cip, argv[1], argv[2]);
				} else if (c2 == 'm') {
					code = FTPRmdir(cip, argv[1], kRecursiveNo, kGlobYes);
				} else goto bad;
				break;
			case 's':
				code = FTPFileSize(cip, argv[1], &size, 'I');
				printf("%s size = %lu.\n", argv[1], (unsigned long) size);
				break;
			case 'u':
				code = FTPUmask(cip, argv[1]);
				break;
			case 'X':
			case 'x':
				return;
			case '\0':
				break;
			default:
			bad:
				printf("Commands are:  cd      dir    mkdir   pwd     rename umask\n");
				printf("               chmod   get    modtime quit    rmdir  \n");
				printf("               delete  ls     put     quote   size   \n");
		}
		if (code != kNoErr) {
			fprintf(stderr, "Return code for %s was %d (%s).\n", argv[0], code, FTPStrError(code));
		}
	}
}	/* Shell */



static void
Abort(int sigNum)
{
	static int onceOnly = 0;

	if (onceOnly == 0) {
		onceOnly = sigNum + 1;
		longjmp(gJmp, 1);
	}
}	/* Abort */



void
main(int argc, char **argv)
{
	int result, c;
	FTPLibraryInfo li;
	FTPConnectionInfo fi;
#ifdef HAVE_GETPASS
	char *password;
#endif

	result = FTPInitLibrary(&li);
	if (result < 0) {
		fprintf(stderr, "Init library error %d.\n", result);
		exit(1);
	}
	result = FTPInitConnectionInfo(&li, &fi, kDefaultFTPBufSize);
	if (result < 0) {
		fprintf(stderr, "Init connection info error %d.\n", result);
		exit(1);
	}

	fi.debugLog = stdout;
	fi.errLog = stderr;
	STRNCPY(fi.user, "anonymous");

	while ((c = getopt(argc, argv, "d:e:P:p:u:")) > 0) switch(c) {
		case 'P':
			fi.port = atoi(optarg);	
			break;
		case 'u':
			STRNCPY(fi.user, optarg);
			break;
		case 'p':
			STRNCPY(fi.pass, optarg);	/* Don't recommend doing this! */
			break;
		case 'e':
			fi.errLog = fopen(optarg, "a");
			break;
		case 'd':
			fi.debugLog = fopen(optarg, "a");
			break;
		default:
			Usage();
	}
	if (optind > argc - 1)
		Usage();

#ifdef HAVE_GETPASS
	if (strcmp(fi.user, "anonymous") && strcmp(fi.user, "ftp")) {
		if (fi.pass[0] == '\0') {
			password = getpass("Password: ");		
			if (password != NULL) {
				STRNCPY(fi.pass, password);
				/* Don't leave cleartext password in memory. */
				memset(password, 0, strlen(fi.pass));
			}
		}
	}
#endif

	STRNCPY(fi.host, argv[optind]);

	if (setjmp(gJmp) == 0) {
		signal(SIGINT, Abort);
		signal(SIGTERM, Abort);
		if ((result = FTPOpenHost(&fi)) < 0) {
			fprintf(stderr, "Cannot open host: %s.\n", FTPStrError(result));
			exit(1);
		}
		Shell(&fi);
	} else {
		signal(SIGINT, SIG_IGN);
		signal(SIGTERM, SIG_IGN);
		fprintf(stderr, "Caught signal, cleaning up...\n");
	}

	if (setjmp(gJmp) == 0) {
		signal(SIGINT, Abort);
		signal(SIGTERM, Abort);
		FTPCloseHost(&fi);
	} else {
		signal(SIGINT, SIG_IGN);
		signal(SIGTERM, SIG_IGN);
		fprintf(stderr, "Caught signal, exiting.\n");
	}
	exit(0);
}	/* main */
