/* signal.c: a Hugh-approved signal handler. */

#include <signal.h>
#include <setjmp.h>
#include "rc.h"
#include "sigmsgs.h"
#include "jbwrap.h"

/*
   (to quote Hugh)
   This code is almost safe.  

   - ANSI doesn't promise that sq_head or sq_tail can be safely fetched.
     I figure that they probably can.

   - It can lose signals that happen while it is running.

   - It can be crashed by the same signal happening twice quickly
     (exercise: change to "safe signals")

   - sq can overflow.

   - On BSD systems, I/O to "slow" devices will be resumed!  The fix is
     to have the signal handler do a longjmp if slow I/O is being
     performed.  Ugh.
*/

#define SQ_ROOF 256

Jbwrap slowbuf;
volatile SIG_ATOMIC_T slow, interrupt_happened;
void (*sighandlers[NUMOFSIGNALS])(int);

static volatile SIG_ATOMIC_T sq_head, sq_tail, sq[SQ_ROOF];

extern void catcher(int s) {
	int t = sq_tail;
	int nt = (t+1) % SQ_ROOF;
	if (forked)
		exit(1); /* exit unconditionally on a signal in a child process */
	if (nt == sq_head)
		return;	/* queue full: go home */
	sq[t] = s;
	sq_tail = nt;
	signal(s, catcher); /* System V needs this reset */
	interrupt_happened = TRUE;
#ifndef SVSIGS
	if (slow)
		longjmp(slowbuf.j, 1);
#endif
}

extern void sigchk() {
	void (*h)(int);
	int s;
	if (sq_head == sq_tail)
		return; /* ho hum; life as usual */
	if (forked)
		exit(1); /* exit unconditionally on a signal in a child process */
	h = sighandlers[s = sq[sq_head++]];
	if (h == SIG_DFL)
		panic("caught signal set to SIG_DFL");
	if (h == SIG_IGN)
		panic("caught signal set to SIG_IGN");
	(*h)(s);
}

extern void (*rc_signal(int s, void (*h)(int)))(int) {
	void (*old)(int);
	SIGCHK;
	old = sighandlers[s];
	if (h == SIG_DFL || h == SIG_IGN) {
		signal(s, h);
		sighandlers[s] = h;
	} else {
		sighandlers[s] = h;
		signal(s, catcher);
	}
	return old;
}

extern void initsignal() {
	void (*h)(int);
	int i;
	for (i = 1; i < NUMOFSIGNALS; i++) {
		if ((h = signal(i, SIG_DFL)) != SIG_DFL)
			signal(i, h);
		sighandlers[i] = h;
	}
}
