/*
 * Command Input Shell
 * Dave Clemans
 * 12/88-1/89
 *
 * "spiritually" based on Bourne, Korn shells
 *
 * Wild card expansion
 *
 * $Id: wild.c,v 1.4 89/02/20 20:03:40 dclemans Exp $
 *
 * $Log:	wild.c,v $
 * Revision 1.4  89/02/20  20:03:40  dclemans
 * Add RCS identifiers
 * 
 */
#include <stdio.h>
#include "shell.h"
#ifdef  GEMDOS
#include "ndir.h"
#else
#include <sys/types.h>
#ifndef USG
#include <sys/dir.h>
#else
#include <dirent.h>
#endif  /* USG */
#endif  /* GEMDOS */

#define WILDCARDS       "*?["

/**** Start of "glob" package ****/
/* This software is copyright (c) 1986 by Stichting Mathematisch Centrum.
 * You may use, modify and copy it, provided this notice is present in all
 * copies, modified or unmodified, and provided you don't make money for it.
 *
 * Written 86-jun-28 by Guido van Rossum, CWI, Amsterdam <guido@mcvax.uucp>
 * Stripped down for local use, more portability by Dave Clemans
 */

/* Pattern matching function for filenames */
/* Each occurrence of the * pattern causes a recursion level */

#define EOS '\0'

#define BOOL int
#define NO 0
#define YES 1

#define M_ALL '\001'        /* * */
#define M_ONE '\002'        /* ? */
#define M_SET '\003'        /* [ */
#define M_CMP '\004'        /* ^ */
#define M_RNG '\005'        /* - */
#define M_END '\006'        /* ] */

static BOOL match(name, pat)
    char *name;
    char *pat;
{
    register char c, k;
    BOOL ok,cmp;

    for (c = *pat++; c != EOS; c = *pat++) {
        switch (c) {

        case M_ONE:
            if (*name++ == EOS)
                return NO;
            break;

        case M_ALL:
            if (*pat == EOS)
                return YES;
            for (; *name != EOS; ++name) {
                if (match(name, pat))
                    return YES;
            }
            return NO;

        case M_SET:
            cmp= NO;
            ok= NO;
            k= *name++;
            while ((c= *pat++) != M_END) {
                if (c == M_CMP)
                    cmp= YES;
                else if (*pat == M_RNG) {
                    if (c <= k && k <= pat[1])
                        ok= YES;
                    pat += 2;
                }
                else if (c == k)
                    ok= YES;
            }
            if (!cmp && !ok)
                return NO;
            if (cmp && ok)
                return NO;
            break;

        default:
            if (*name++ != c)
                return NO;
            break;

        }
    }
    return *name == EOS;
}
/**** End of "glob" package ****/

char *wild_compile(pattern,upcase)
char *pattern;
int upcase;
{
    register char *compiled;
    register char *in,*out;

    compiled = new_string(strlen(pattern)+1);
    if (compiled == (char *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("wild_compile"));
        return (char *)NULL;
    }
    out = compiled;
    for (in = pattern; *in; in++)
    {   /* copy, "compiling" the pattern */
        switch (*in)
        {   /* a "special" character? */
            case '\'':
                in++;
                while (*in && *in != '\'')
                    *out++ = *in++;
                if (*in != '\'')
                    break;
                break;
            case '"':
                in++;
                while (*in && *in != '"')
                {   /* scan the string */
                    if (*in == ESCAPE_CHAR)
                    {   /* embedded escape char? */
                        in++;
                        if (*in == '\0')
                            break;
                    }
                    *out++ = *in++;
                }
                if (*in != '"')
                    break;
                break;
            case ESCAPE_CHAR:
                in++;
                if (*in == '\0')
                    break;
                *out++ = *in;
                break;
            case '*':
                *out++ = M_ALL;
                break;
            case '?':
                *out++ = M_ONE;
                break;
            case '[':
                *out++ = M_SET;
                in++;
                while (*in && *in != ']')
                {   /* scan the set */
                    if (*in == '-' &&
                       (in[-1] == '[' || (in[-1] == '^' && in[-2] == '[')))
                        *out++ = '-';
                    else if (*in == '-')
                        *out++ = M_RNG;
                    else if (*in == '^' && in[-1] == '[')
                        *out++ = M_CMP;
                    else if (*in == '^')
                        *out++ = '^';
                    else if (*in == ESCAPE_CHAR && in[1] != '\0')
                        *out++ = *++in;
                    else if (*in == ESCAPE_CHAR)
                        *out++ = ESCAPE_CHAR;
                    else *out++ = *in;
                    in++;
                }
                if (*in != ']')
                {   /* if error found */
                    errmsg(0,LOC("wild_compile"),"incomplete character class: missing ']'");
                    free(compiled);
                    return (char *)NULL;
                }
                *out++ = M_END;
                break;
            default:
                *out++ = *in;
                break;
        }
        if (*in == '\0')
            break;
    }
#ifdef  GEMDOS
    *out = '\0';
    for (out = compiled; *out; out++)
    {   /* fixups for mono-case matching */
        if (!upcase)
            continue;
        if (islower(*out))
            *out = _toupper(*out);
    }
#endif  /* GEMDOS */
    *out = '\0';

    /* finally done, return compiled string */
    return compiled;
}   /* end of wild_compile */

int wild_match(string,pattern)
char *string;
char *pattern;
{
    register char *compiled;
    int rc;

    compiled = wild_compile(pattern,0);
    if (compiled == (char *)NULL)
    {   /* any errors? message already printed */
        return 0;
    }
    rc = match(string,compiled);
    free(compiled);
    return rc;
}   /* end of wild_match */

void wild_save(path,tokens,exp)
char *path;
struct token **tokens;
int exp;
{
    register struct token *new,*tp,*ltp;
    int rc;
    char buffer[6],*dir,*npath;
#ifdef  GEMDOS
    char *p;
#endif  /* GEMDOS */

    dir = (char *)NULL;
    if (path[0] == '~')
    {   /* need tilde expansion? */
        sprintf(buffer,"~%c*",DIR_SEPARATOR);
        if (wild_match(path,buffer) || strcmp(path,"~") == 0)
        {   /* paste in home directory? */
            dir = strcopy(base_env.homedir);
            npath = &path[1];
        }
        sprintf(buffer,"~+%c*",DIR_SEPARATOR);
        if (wild_match(path,buffer) || strcmp(path,"~+") == 0)
        {   /* paste in current directory? */
            dir = strcopy(base_env.dir->current);
            npath = &path[2];
        }
        sprintf(buffer,"~-%c*",DIR_SEPARATOR);
        if (wild_match(path,buffer) || strcmp(path,"~-") == 0)
        {   /* paste in old current directory */
            dir = var_normal("$OLDPWD");
            npath = &path[2];
        }
        if (dir != (char *)NULL)
        {   /* if something to paste in */
            path = new_string(strlen(dir)+strlen(npath)+1);
            if (path == (char *)NULL)
            {   /* enough memory? */
                errmsg(SHERR_NOMEM,LOC("wild_save"));
                return;
            }
            sprintf(path,"%s%s",dir,npath);
        }
    }
    new = new_token(strlen(path));
    if (new == (struct token *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("wild_save"));
        return;
    }
    new->type = SYM_WORD;
    strcpy(new->name,path);
    if (dir != (char *)NULL)
        free(path);
#ifdef  GEMDOS
    if (exp)
    {   /* mono-case support */
        for (p = new->name; *p; p++)
            if (isupper(*p))
                *p = _tolower(*p);
    }
#endif  /* GEMDOS */

    ltp = (struct token *)NULL;
    for (tp = *tokens; tp != (struct token *)NULL; tp = tp->next)
    {   /* look for place to put name */
        rc = strcmp(new->name,tp->name);
        if (rc < 0)
            break;
        ltp = tp;
    }
    if (*tokens == (struct token *)NULL)
        *tokens = new;
    else if (ltp == (struct token *)NULL)
    {   /* head of list? */
        new->next = *tokens;
        *tokens = new;
    }
    else
    {   /* middle of list */
        new->next = tp;
        ltp->next = new;
    }
}   /* end of wild_save */

struct token *wild_search(path,tokens)
char *path;
struct token **tokens;
{
    register char *leadin,*pattern,*trailer;
    char *p,*pp;
    int rc,wild,sep,mask;
    char *compiled;
    DIR *source;
#ifndef USG
    struct direct *name;
#else
    struct dirent *name;
#endif  /* USG */
    char lead[2];
    struct token *expansion;

    /* break into 3 pieces: leadin, pattern, everything else */
    wild = sep = 0;
    leadin = pattern = path;
    trailer = (char *)NULL;
    for (p = path; *p; p++)
    {   /* scan, looking for special characters */
        switch (*p)
        {   /* an interesting char? */
            case ESCAPE_CHAR:
                p++;
                break;
            case '\'':
                p++;
                while (*p && *p != '\'')
                    p++;
                break;
            case '"':
                while (*p && *p != '"')
                    p++;
                break;
#ifdef  GEMDOS
            case ':':
#endif  /* GEMDOS */
            case DIR_SEPARATOR:
                sep++;
                if (wild)
                {   /* if wild cards in last segment */
                    if (leadin == pattern)
                        leadin = ".";
                    trailer = p;
                    break;
                }
                pattern = p+1;
                break;
            case '*':
                wild++;
                break;
            case '?':
                wild++;
                break;
            case ']':
                wild++;
                while (*p && *p != ']')
                    p++;
                break;
            default:
                break;
        }
        if (*p == '\0')
            break;
        if (trailer != (char *)NULL)
            break;
    }
    if (tokens == (struct token **)NULL)
        tokens = &expansion;
    expansion = (struct token *)NULL;
    if (wild == 0)
    {   /* then no expansion (except for possible tildes) */
        if (path[0] != '~')
            return (struct token *)NULL;
        wild_save(path,tokens,0);
        return *tokens;
    }
    if (trailer == (char *)NULL)
    {   /* if pieces not set up yet */
        if (sep == 0)
        {   /* just a pattern... */
            leadin = ".";
            pattern = path;
        }
        else
        {   /* leadin - pattern */
            p = strrchr(path,DIR_SEPARATOR);
            if (p == (char *)NULL)
                p = strrchr(path,':');
            leadin = path;
            pattern = p+1;
        }
        for (trailer = pattern; *trailer; trailer++)
            /* do nothing */;
    }
    lead[0] = lead[1] = '\0';
    if (strcmp(leadin,".") != 0)
    {   /* make leadin string usable */
        /* ASSERT that leadin == path */
        leadin = new_string((int)(pattern-path)+1);
        if (leadin == (char *)NULL)
        {   /* enough memory? */
            errmsg(SHERR_NOMEM,LOC("wild_search"));
            return (struct token *)NULL;
        }
        strncpy(leadin,path,(int)(pattern-path));
        leadin[(int)(pattern-path)] = '\0';
        for (p = leadin; *p; p++)
            /* do nothing */;
        p--;
        if (*p == DIR_SEPARATOR && p != leadin
#ifdef  GEMDOS
                                               && p[-1] != ':'
#endif  /* GEMDOS */
                                                              )
        {   /* remove trailing sep */
            *p = '\0';
            lead[0] = DIR_SEPARATOR;
        }
        stripquotes(leadin);
        if (strcmp(leadin,"~") == 0)
        {   /* paste in home directory */
            free(leadin);
            leadin = strcopy(base_env.homedir);
            if (leadin == (char *)NULL)
            {   /* enough memory? */
                /* message already printed */
                return (struct token *)NULL;
            }
        }
        else if (strcmp(leadin,"~+") == 0)
        {   /* paste in current directory */
            free(leadin);
            leadin = strcopy(base_env.dir->current);
            if (leadin == (char *)NULL)
            {   /* enough memory? */
                /* message already printed */
                return (struct token *)NULL;
            }
        }
        else if (strcmp(leadin,"~-") == 0)
        {   /* paste in old current directory */
            free(leadin);
            leadin = var_normal("$OLDPWD");
            if (leadin == (char *)NULL)
                leadin = strcopy(base_env.dir->current);
            if (leadin == (char *)NULL)
            {   /* enough memory? */
                /* message already printed */
                return (struct token *)NULL;
            }
        }
    }
    else sep = 0;
    pp = new_string((int)(trailer-pattern)+1);
    if (pp == (char *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("wild_search"));
        if (sep)
            free(leadin);
        return (struct token *)NULL;
    }
    strncpy(pp,pattern,(int)(trailer-pattern));
    pp[(int)(trailer-pattern)] = '\0';
    pattern = pp;
    compiled = wild_compile(pattern,1);
    if (compiled == (char *)NULL)
    {   /* any errors? message already printed */
        free(pattern);
        if (sep)
            free(leadin);
        return (struct token *)NULL;
    }

#ifdef  GEMDOS
    if (*trailer == ':')
    {   /* wild-carded drive name? */
        lead[1] = '\0';
        mask = Drvmap();
        for (rc = 0; rc < 16; rc++)
        {   /* for each possible drive */
            if (mask & (1<<rc))
            {   /* if drive present */
                lead[0] = 'A' + rc;
                if (match(lead,compiled))
                {   /* if this drive is wanted */
                    pp = new_string(strlen(trailer)+2);
                    if (pp == (char *)NULL)
                    {   /* enough memory? */
                        errmsg(SHERR_NOMEM,LOC("wild_search"));
                        break;
                    }
                    sprintf(pp,"%s%s",lead,trailer);
                    wild_search(pp,tokens);
                    free(pp);
                }
            }
        }
        free(pattern);
        free(compiled);
        return *tokens;
    }
#endif  /* GEMDOS */

    source = opendir(leadin);
    if (source == (DIR *)NULL)
    {   /* did it work? */
        errmsg(0,LOC("wild_search"),"can't open directory %s",leadin);
        free(compiled);
        free(pattern);
        if (sep)
            free(leadin);
        return (struct token *)NULL;
    }
    while ((name = readdir(source)) != NULL)
    {   /* while there are names... */
        if (strcmp(name->d_name,".") == 0 || strcmp(name->d_name,"..") == 0)
            continue;
        rc = match(name->d_name,compiled);
        if (rc)
        {   /* if a match was found */
            if (*trailer == '\0' && sep)
            {   /* save the path */
                pp = new_string(strlen(leadin)+strlen(lead)+strlen(name->d_name)+1);
                if (pp == (char *)NULL)
                {   /* enough memory? */
                    errmsg(SHERR_NOMEM,LOC("wild_search"));
                    break;
                }
                sprintf(pp,"%s%s%s",leadin,lead,name->d_name);
                wild_save(pp,tokens,1);
                free(pp);
            }
            else if (*trailer == '\0')
                wild_save(name->d_name,tokens,1);
            else
            {   /* need to recurse to handle trailer */
                pp = new_string(strlen(leadin)+strlen(name->d_name)+strlen(trailer)+2);
                if (pp == (char *)NULL)
                {   /* enough memory */
                    errmsg(SHERR_NOMEM,LOC("wild_search"));
                    break;
                }
                sprintf(pp,"%s%s%s%s",leadin,lead,name->d_name,trailer);
                wild_search(pp,tokens);
                free(pp);
            }
        }
    }
    closedir(source);
    free(compiled);
    free(pattern);
    if (sep)
        free(leadin);
    return *tokens;
}   /* end of wild_search */
