/*
 * vi		- an "vi" style single line editor for reader
 *
 * Dave Clemans, 4/84 (initial version); 1/89 (more generality)
 *
 * Commands supported:
 *    Input mode:
 *	alphabetic chars are inserted
 *	normal backspace, kill supported
 *	kill word	^W
 *	eof		^D
 *    Editing mode:
 *	forward		l
 *	forward word	e
 *	fwd word nws	E
 *	back word	b
 *	back word nws	B
 *	next word	w
 *	next word nws	W
 *	backward	h
 *	start line	0
 *	start line nws	^
 *	end line	$
 *	prev line	k
 *	next line	j
 *	search prev	/
 *	append		a
 *	append eol	A
 *	kill eol&append	C
 *	kill end line	D
 *	insert		i
 *	insert, 1st nws	I
 *	repeat prev	.
 *	delete char	x
 *	retype		^L
 *	erase		^H
 *	replace char	r
 *	repeat search	n
 *	find char	f
 *	repeat find	;
 *	change chars	c
 *	delete chars	d
 *	pop kill, ins	P
 *	pop kill, app	p
 *	undo change	u
 *	undo all	U
 *	goto		G
 *	arguments	digits
 *
 * If we return 0, we did not handle the input character; it should be
 * handled by the standard editor.
 * If we return <0, signal end-of-file.
 * Otherwise the character has been
 * handled and the next character can be read.
 *
 * $Id: vi.c,v 1.2 89/02/20 20:20:30 dclemans Exp $
 *
 * $Log:	vi.c,v $
 * Revision 1.2  89/02/20  20:20:30  dclemans
 * Add RCS identifiers
 * 
 */
#include <ctype.h>
#ifndef GEMDOS
#include <sgtty.h>
#endif  GEMDOS

#include "history.h"

/*
 * If true, in "vi" editing mode; else just normal tty style editing
 */
static int editingMode;

/*
 * Saved copies of things for undo's, etc.
 */
static char editingChar;
static char searchChar;
static struct historyLine saveChange;
static struct historyLine saveAll;
static int argument = 1;

/*
 * Check for "vi" style commands
 */
int _doVi(c)
register char c;
{
	register char cc;
	register int counter;
	register int cursor;

	if (!editingMode)
	{	/* if not doing real editing; i.e. just text insertion */
		if (c == '\033')
		{	/* switch to editing mode */
			strncpy(saveAll.contents,History.currentLine->contents,History.currentLine->size);
			saveAll.size = History.currentLine->size;
			saveAll.cursor = History.currentLine->cursor;
			saveAll.ttycursor = History.currentLine->ttycursor;
			strncpy(saveChange.contents,History.currentLine->contents,History.currentLine->size);
			saveChange.size = History.currentLine->size;
			saveChange.cursor = History.currentLine->cursor;
			saveChange.ttycursor = History.currentLine->ttycursor;
			editingMode = 1;
			return(1);
		}
		if (c == '\004')
		{	/* end of file */
			editingMode = 0;
			return(-1);
		}
		return(0);
	}

	/*
	 * Check for vi style editing commands
	 */
	if (c == '.')
		cc = editingChar;
	else	cc = c;
	switch (cc)
	{	/* if going to make a change, save current state */
	    case 'x':
	    case 'a':
	    case 'A':
	    case 'i':
	    case 'I':
	    case 'c':
	    case 'C':
	    case 'd':
	    case 'D':
	    case '\010':
	    case 'p':
	    case 'P':
	    case 'G':
		strncpy(saveChange.contents,History.currentLine->contents,History.currentLine->size);
		saveChange.size = History.currentLine->size;
		saveChange.cursor = History.currentLine->cursor;
		saveChange.ttycursor = History.currentLine->ttycursor;
		break;
	    default:	/* no need to do anything here */
		break;
	}
	switch (c)
	{	/* see which command ... */
	    case 'l':	/* forward one character */
		_doStandardSpace(c);
		break;
	    case 'h':	/* backward one character */
		_doStandardBackspace(c);
		break;
	    case 'e':	/* forward one word */
		_spaceNextStdWord();
		break;
	    case 'b':	/* backward one word */
		_spacePrevStdWord();
		break;
	    case 'w':	/* to next word */
		_skipNextStdWord();
		break;
	    case 'W':	/* to next word; non-white space */
		_skipNextWord(" \t");
		break;
	    case '0':	/* beginning of line */
		_doStandardStartLine(c);
		break;
	    case '$':	/* end of line */
		_doStandardEndLine(c);
		break;
	    case 'k':	/* previous line */
		_doStandardPrevLine(c);
		break;
	    case 'j':	/* next line */
		_doStandardNextLine(c);
		break;
	    case '/':	/* search lines */
		_doStandardSrchPrevLine(c,0);
		break;
	    case 'n':	/* repeat last search */
		_doStandardSrchPrevLine(c,1);
		break;
	    case 'a':	/* append chars */
		_doStandardSpace(c);
		editingMode = 0;
		break;
	    case 'A':	/* append to end of line */
		_doStandardEndLine(c);
		editingMode = 0;
		break;
	    case 'C':	/* delete to end of line; start appending */
		_doStandardDelEndLine(c);
		editingMode = 0;
		break;
	    case 'D':	/* delete to end of line */
		_doStandardDelEndLine(c);
		break;
	    case 'i':	/* start inserting */
		editingMode = 0;
		break;
	    case 'x':	/* delete current character */
		_doStandardDelete(c);
		break;
	    case '\014':	/* retype line */
		_doStandardRetype(c);
		break;
	    case '^':	/* first non-white space in line */
		_doStandardStartLine(c);
		_skipNextWord(" \t");
		break;
	    case 'I':	/* goto to first non-white space char in line, start inserting */
		_doStandardStartLine(c);
		while ((History.currentLine->cursor < History.currentLine->size) &&
		    !isspace(History.currentLine->contents[History.currentLine->cursor]))
			_doStandardSpace(c);
		editingMode = 0;
		break;
	    case '\010':/* do character erase */
		_doStandardErase(c);
		break;
	    case '.':	/* repeat previous editing command */
		_savechar(editingChar);
		break;
	    case 'r':	/* replace char */
		if (_savedState.stream < 0)
		{	/* nothing to read from? */
			_writechar('\007');
			return(1);
		}
		cc = _readchar(_savedState.stream);
		_doStandardDelete(c);
		_doStandardCharacter(cc);
		break;
	    case '\030':/* expand command or filename */
		_doStandardExpand(c);
		break;
	    case 'E':	/* next word defined by white space */
		_spaceNextWord(" \t");
		break;
	    case 'B':	/* prev word defined by white space */
		_spacePrevWord(" \t");
		break;
	    case 'f':	/* find char */
		if (_savedState.stream < 0)
		{	/* nothing to read from? */
			_writechar('\007');
			return(1);
		}
		searchChar = _readchar(_savedState.stream);
		/* FALL THROUGH!!! */
	    case ';':	/* repeat last find char */
		for (counter = History.currentLine->cursor+1; counter < History.currentLine->size; counter++)
			if (History.currentLine->contents[counter] == searchChar)
				break;
		if ((counter >= History.currentLine->size) ||
		    (History.currentLine->contents[counter] != searchChar))
		{	/* not found, at end of line, etc. */
			_writechar('\007');
			return(1);
		}
		while (History.currentLine->cursor < counter)
			_doStandardSpace('\000');
		break;
	    case 'c':	/* change chars */
	    case 'd':	/* delete chars */
		if (_savedState.stream < 0)
		{	/* nothing to read from? */
			_writechar('\007');
			return(1);
		}
		cc = _readchar(_savedState.stream);
		counter = History.currentLine->cursor;
		_doVi(cc);
		editingChar = c;
		if (counter <= History.currentLine->cursor)
		{	/* set start and stop points for range delete */
			cursor = counter;
			counter = History.currentLine->cursor;
		}
		else
		{	/* moved other direction */
			cursor = History.currentLine->cursor;
		}
		_deleteTo(cursor,counter);
		if (c == 'c')
			editingMode = 0;
		break;
	    case 'P':	/* pop from kill ring, inserting before cursor */
		_savechar('i');
		_popOffKillRing();
		_savechar('\033');
		break;
	    case 'p':	/* pop from kill ring, appending to cursor */
		_savechar('a');
		_popOffKillRing();
		_savechar('\033');
		break;
	    case 'U':	/* undo all changes */
		_doStandardKill('\000');
		strncpy(History.currentLine->contents,saveAll.contents,saveAll.size);
		History.currentLine->size = saveAll.size;
		History.currentLine->cursor = saveAll.cursor;
		History.currentLine->ttycursor = saveAll.ttycursor;
		_doStandardRetype('\000');
		break;
	    case 'u':	/* undo last changes */
		_doStandardKill('\000');
		strncpy(History.currentLine->contents,saveChange.contents,saveChange.size);
		History.currentLine->size = saveChange.size;
		History.currentLine->cursor = saveChange.cursor;
		History.currentLine->ttycursor = saveChange.ttycursor;
		_doStandardRetype('\000');
		break;
	    case 'G':	/* goto a command */
		/* so far, support only for default case; goto oldest command */
		_doStandardKill('\000');
		strncpy(History.currentLine->contents,History.listBottom->contents,History.listBottom->size);
		History.currentLine->size = History.listBottom->size;
		History.currentLine->cursor = History.listBottom->cursor;
		History.currentLine->ttycursor = History.listBottom->ttycursor;
		_doStandardRetype('\000');
		_doStandardEndLine('\000');
		break;
	    case '\004':/* eof; use default handling */
	    case '\r':	/* eol; use default handling */
	    case '\n':
		editingMode = 0;
		return(0);
	    case '1':	/* for setting numerical argument */
	    case '2':
	    case '3':
	    case '4':
	    case '5':
	    case '6':
	    case '7':
	    case '8':
	    case '9':
		argument = 0;
		if (_savedState.stream < 0)
		{	/* nothing to read from? */
			_writechar('\007');
			return(1);
		}
		argument = 0;
		for (; isdigit(c); c = _readchar(_savedState.stream))
		{	/* build value */
			argument *= 10;
			argument += (int)((int)c - (int)'0');
		}
		if (c == EOF)
			return(-1);
		_savechar(c);
		return(1);
	    default:	/* we don't know about it */
		_writechar('\007');
		break;
	}
	editingChar = c;
	argument = 1;
	return(1);
};	/* end of _doVi */
