/*
 * Command Input Shell
 * Dave Clemans
 * 12/88-1/89
 *
 * "spiritually" based on Bourne, Korn shells
 *
 * Built-in Commands (part 3)
 *
 * $Id: cmd3.c,v 1.8 89/02/25 17:40:00 dclemans Exp $
 *
 * $Log:	cmd3.c,v $
 * Revision 1.8  89/02/25  17:40:00  dclemans
 * miscellaneous bug fixes/speedups
 * 
 * Revision 1.7  89/02/22  16:27:08  dclemans
 * Implement [[, ]] brackets
 * 
 * Revision 1.6  89/02/22  08:16:43  dclemans
 * implement left-justified, right-justified, etc. parameter attributes
 * 
 * Revision 1.5  89/02/20  20:14:22  dclemans
 * Add RCS identifiers
 * 
 */
#include <stdio.h>
#include "shell.h"

#ifdef  GEMDOS
#include <types.h>
#include <stat.h>
#ifdef  MWC
#include <basepage.h>
#endif  /* MWC */
#else
#include <sys/types.h>
#include <sys/stat.h>
#endif  /* GEMDOS */

static int test_parse(tp)
struct token *tp;
{
    register char *s1,*s2,*arg;
    int n1,n2;
    int rc;
    struct stat statb;
    long date1;

    if (tp == (struct token *)NULL)
        return 0;
    if (strcmp(tp->name,"]") == 0)
        return 0;
    if (strcmp(tp->name,"]]") == 0)
        return 0;
    if (strcmp(tp->name,"!") == 0)
        return test_parse(tp->next);
    if (strcmp(tp->name,"(") == 0)
        return !test_parse(tp->next);
    if (strcmp(tp->name,")") == 0)
        tp = tp->next;
    rc = 0;

    arg = tp->name;
    s1 = (char *)NULL;
    tp = tp->next;
    if (tp != (struct token *)NULL)
    {   /* get file */
        s1 = tp->name;
        tp = tp->next;
    }
    if (arg != (char *)NULL)
        stripquotes(arg);
    if (s1 != (char *)NULL)
        stripquotes(s1);
    if (strcmp(arg,"-b") == 0)
    {   /* a block file? */
#ifdef  GEMDOS
        if (s1 == (char *)NULL)
            rc = 1;
        else
        {   /* is it a drive name? */
            if (islower(*s1))
                *s1 = _toupper(*s1);
            if (strcmp(&s1[1],":") == 0 && *s1 >= 'A' && *s1 <= 'P')
                rc = 0;
            else    rc = 1;
        }
#else
        rc = stat(s1,&statb);
        if (rc >= 0)
        {   /* if something to look at */
            if ((statb.st_mode & S_IFMT) == S_IFBLK)
                rc = 0;
            else rc = 1;
        }
#endif  /* GEMDOS */
    }
    else if (strcmp(arg,"-c") == 0)
    {   /* a character file? */
#ifdef  GEMDOS
        if (s1 == (char *)NULL)
            rc = 1;
        else
        {   /* check for char device names */
            for (s2 = s1; *s2; s2++)
                if (islower(*s2))
                    *s2 = _toupper(*s2);
            if (strcmp(s1,"CON:") == 0 ||
                strcmp(s1,"TTY:") == 0 ||
                strcmp(s1,"AUX:") == 0 ||
                strcmp(s1,"PRN:") == 0 ||
                strcmp(s1,"PRT:") == 0)
                rc = 0;
            else    rc = 1;
        }
#else
        rc = stat(s1,&statb);
        if (rc >= 0)
        {   /* if something to look at */
            if ((statb.st_mode & S_IFMT) == S_IFCHR)
                rc = 0;
            else rc = 1;
        }
#endif  /* GEMDOS */
    }
    else if (strcmp(arg,"-d") == 0)
    {   /* is a directory? */
        if (s1 == (char *)NULL || stat(s1,&statb) < 0)
            rc = 1;
        else
        {   /* check the modes */
#ifdef  GEMDOS
            if (statb.st_mode & S_IJDIR)
                rc = 0;
            else    rc = 1;
#else
        rc = stat(s1,&statb);
        if (rc >= 0)
        {   /* if something to look at */
            if ((statb.st_mode & S_IFMT) == S_IFDIR)
                rc = 0;
            else rc = 1;
        }
#endif  /* GEMDOS */
        }
    }
    else if (strcmp(arg,"-f") == 0)
    {   /* is a regular file? */
        if (s1 == (char *)NULL || stat(s1,&statb) < 0)
            rc = 1;
        else
        {   /* check the modes */
#ifdef  GEMDOS
            if (statb.st_mode & S_IJVOL)
                rc = 1;
            else    rc = 0;
#else
        rc = stat(s1,&statb);
        if (rc >= 0)
        {   /* if something to look at */
            if ((statb.st_mode & S_IFMT) == S_IFREG)
                rc = 0;
            else rc = 1;
        }
#endif  /* GEMDOS */
        }
    }
    else if (strcmp(arg,"-g") == 0)
    {   /* setgid set? */
#ifdef  GEMDOS
        rc = -1;
#else
        rc = stat(s1,&statb);
        if (rc >= 0)
        {   /* if something to look at */
            if (statb.st_mode & S_ISGID)
                rc = 0;
            else rc = 1;
        }
#endif  /* GEMDOS */
    }
    else if (strcmp(arg,"-k") == 0)
    {   /* sticky set? */
#ifdef  GEMDOS
        rc = -1;
#else
        rc = stat(s1,&statb);
        if (rc >= 0)
        {   /* if something to look at */
            if (statb.st_mode & S_ISVTX)
                rc = 0;
            else rc = 1;
        }
#endif  /* GEMDOS */
    }
    else if (strcmp(arg,"-l") == 0)
    {   /* is a link? */
#ifdef  GEMDOS
        rc = -1;
#else
        rc = stat(s1,&statb);
        if (rc >= 0)
        {   /* if something to look at */
#ifndef USG
            if ((statb.st_mode & S_IFMT) == S_IFLNK)
                rc = 0;
            else rc = 1;
#else
            rc = -1;
#endif  /* USG */
        }
#endif  /* GEMDOS */
    }
    else if (strcmp(arg,"-n") == 0)
    {   /* is string length zero? */
        if (s1 == (char *)NULL || strlen(s1) == 0)
            rc = 0;
        else    rc = 1;
    }
    else if (strcmp(arg,"-p") == 0)
    {   /* is a pipe? */
#ifdef  GEMDOS
        rc = -1;
#else
        rc = -1;
#endif  /* GEMDOS */
    }
    else if (strcmp(arg,"-r") == 0)
    {   /* is readable? */
        if (s1 == (char *)NULL)
            rc = 1;
        else rc = !isread(s1);
    }
    else if (strcmp(arg,"-t") == 0)
    {   /* is a terminal? */
#ifdef  GEMDOS
        rc = -1;
#else
        rc = !isatty(base_env.io->input);
#endif  /* GEMDOS */
    }
    else if (strcmp(arg,"-u") == 0)
    {   /* setuid set? */
#ifdef  GEMDOS
        rc = -1;
#else
        rc = stat(s1,&statb);
        if (rc >= 0)
        {   /* if something to look at */
            if (statb.st_mode & S_ISUID)
                rc = 0;
            else rc = 1;
        }
#endif  /* GEMDOS */
    }
    else if (strcmp(arg,"-w") == 0)
    {   /* is writable? */
        if (s1 == (char *)NULL || stat(s1,&statb) < 0)
            rc = 1;
        else
        {   /* check the modes */
#ifdef  GEMDOS
            if (statb.st_mode & S_IJRON)
                rc = 1;
            else    rc = 0;
#else
            rc = -1;
#endif  /* GEMDOS */
        }
    }
    else if (strcmp(arg,"-x") == 0)
    {   /* is executable? */
        if (s1 == (char *)NULL)
            rc = 1;
        else rc = !isexec(s1);
    }

    s2 = s1;
    s1 = arg;
    arg = s2;
    if (tp != (struct token *)NULL)
        s2 = tp->name;
    else    s2 = (char *)NULL;
    if (s2 != (char *)NULL)
        stripquotes(s2);
    if (s1 != (char *)NULL)
        n1 = atoi(s1);
    else    n1 = -1;
    if (s2 != (char *)NULL)
        n2 = atoi(s2);
    else    n2 = -1;
    if (tp != (struct token *)NULL)
    {   /* if possible dyadic? */
        if (strcmp(arg,"=") == 0)
        {   /* equivalent strings? */
            rc = strcmp(s1,s2);             
            tp = tp->next;
        }
        else if (strcmp(arg,"!=") == 0)
        {   /* not equivalent strings */
            rc = !strcmp(s1,s2);
            tp = tp->next;
        }
        else if (strcmp(arg,"-eq") == 0)
        {   /* equal numbers? */
            if (n1 == n2)
                rc = 0;
            else    rc = 1;
            tp = tp->next;
        }
        else if (strcmp(arg,"-gt") == 0)
        {   /* greater than? */
            if (n1 > n2)
                rc = 0;
            else    rc = 1;
            tp = tp->next;
        }
        else if (strcmp(arg,"-ge") == 0)
        {   /* greater than or equal to? */
            if (n1 >= n2)
                rc = 0;
            else    rc = 1;
            tp = tp->next;
        }
        else if (strcmp(arg,"-lt") == 0)
        {   /* less than? */
            if (n1 < n2)
                rc = 0;
            else    rc = 1;
            tp = tp->next;
        }
        else if (strcmp(arg,"-le") == 0)
        {   /* less than or equal to? */
            if (n1 <= n2)
                rc = 0;
            else    rc = 1;
            tp = tp->next;
        }
        else if (strcmp(arg,"-ne") == 0)
        {   /* not equal to? */
            if (n1 != n2)
                rc = 0;
            else    rc = 1;
            tp = tp->next;
        }
        else if (strcmp(arg,"-nt") == 0)
        {   /* file newer? */
            if (stat(s1,&statb) < 0)
                date1 = -1;
            else    date1 = statb.st_mtime;
            if (stat(s2,&statb) < 0)
                rc = 1;
            else
            {   /* check dates */
                if (date1 > statb.st_mtime)
                    rc = 0;
                else    rc = 1;
            }
            tp = tp->next;
        }
        else if (strcmp(arg,"-ot") == 0)
        {   /* file older? */
            if (stat(s1,&statb) < 0)
                date1 = -1;
            else    date1 = statb.st_mtime;
            if (stat(s2,&statb) < 0)
                rc = 1;
            else
            {   /* check dates */
                if (date1 < statb.st_mtime)
                    rc = 0;
                else    rc = 1;
            }
            tp = tp->next;
        }
    }

    if (s1 != (char *)NULL && arg == (char *)NULL && s2 == (char *)NULL)
    {   /* just a single string? */
        if (strlen(s1) != 0)
            rc = 0;
        else    rc = 1;
    }

    if (tp != (struct token *)NULL)
    {   /* check for continued expressions? */
        if (strcmp(tp->name,"-a") == 0)
        {   /* anded exprs? */
            if (rc != 0)
                return rc;
            return test_parse(tp->next);
        }
        else if (strcmp(tp->name,"-o") == 0)
        {   /* ored exprs? */
            if (rc == 0)
                return rc;
            return test_parse(tp->next);
        }
        else if (strcmp(tp->name,"]") == 0)
            return rc;
        else if (strcmp(tp->name,"]]") == 0)
            return rc;
    }

    if (tp != (struct token *)NULL)
        errmsg(0,LOC("test_parse"),"expression syntax error at token %s",tp->name);
    return rc;
}   /* end of test_parse */

int cmd_test(pp)
struct phrase *pp;
{
    return test_parse(pp->body->next);
}   /* end of cmd_test */

int cmd_version()
{
    char buffer[BUFSIZ];
#ifdef  GEMDOS
    unsigned version;
    long oldssp;
    int *sysbase;
    int romvers;
#endif  /* GEMDOS */

    strcpy(buffer,shell_version);
    strcat(buffer,"\n");
    io_writestring(0,buffer);
#ifdef  LINED
    sprintf(buffer,"    emacs & vi line editing code installed.\n");
    io_writestring(0,buffer);
#endif  /* LINED */
#ifdef  GEMDOS
    sprintf(buffer,"    compiled for Atari ST systems.\n");
    io_writestring(0,buffer);
#endif  /* GEMDOS */
#ifdef  unix
#ifndef USG
    sprintf(buffer,"    compiled for BSD systems.\n");
#else
    sprintf(buffer,"    compiled for SYSV systems.\n");
#endif  /* USG */
    io_writestring(0,buffer);
#endif  /* unix */

#ifdef  GEMDOS
    oldssp = Super(0L);
    sysbase = *(int **)0x4f2;
    romvers = *(sysbase+1);
    Super(oldssp);
    version = Sversion();
    sprintf(buffer,"    running on Atari ST; TOS %d.%d; GEM %d.%d\n",(romvers >> 8) & 0xFF,romvers & 0xFF,version & 0xFF,(version >> 8) & 0xFF);
    io_writestring(0,buffer);
    sprintf(buffer,"\nAuthor:\n");
    io_writestring(0,buffer);
    sprintf(buffer,"    Dave Clemans\n");
    io_writestring(0,buffer);
    sprintf(buffer,"    c/o ST Enthusiasts Of Portland\n");
    io_writestring(0,buffer);
    sprintf(buffer,"    4470 SW Hall Blvd., Suite 325\n");
    io_writestring(0,buffer);
    sprintf(buffer,"    Beaverton, OR 97005\n");
    io_writestring(0,buffer);
#endif  /* GEMDOS */

    return 0;
}   /* end of cmd_version */

static char *kbytes(value)
long value;
{
    static char buf[16];
    register long first;
    register int last;

    first = value / 1024L;
    last = ((int)((value % 1024L) * 10)) / 1024;
    sprintf(buf,"%ld.%dK",first,last);
    return buf;
}   /* end of kbytes */

#ifdef  MYMALLOC
int cmd_memory()
{
    char buffer[BUFSIZ];
    register char *p;
    extern long poolSize,mallocTotal,mallocHighWater;
    long memblock;

#ifdef  GEMDOS
#ifdef  MWC
    long totalMemory;
    extern long _stksize;
    int percent;

    sprintf(buffer,"%s: text base,length=0x%lx,%s bytes\n",var_arg0,
        BP->p_tbase,kbytes(BP->p_tlen));
    totalMemory = BP->p_tlen;
    io_writestring(0,buffer);
    sprintf(buffer,"    data base,length=0x%lx,%s bytes\n",
        BP->p_dbase,kbytes(BP->p_dlen));
    totalMemory += BP->p_dlen;
    io_writestring(0,buffer);
    sprintf(buffer,"    bss base,length=0x%lx,%s bytes\n",
        BP->p_bbase,kbytes(BP->p_blen));
    totalMemory += BP->p_blen;
    io_writestring(0,buffer);
    p = (char *)_stksize;
    for (p = p+1; *p == '\0' && p < (char *)(_stksize+DEFSTACK); p++)
        /* do nothing */;
    percent = (int)(((DEFSTACK-(long)((long)p-_stksize))*100L)/DEFSTACK);
    sprintf(buffer,"  Stack: %s bytes; estimated max usage=%d%%\n",kbytes(DEFSTACK),percent);
    io_writestring(0,buffer);
    percent = (int)(((DEFSTACK-(long)((long)buffer-_stksize))*100L)/DEFSTACK);
    sprintf(buffer,"         estimated current usage=%d%%\n",percent);
    totalMemory += DEFSTACK;
    io_writestring(0,buffer);
#else
    sprintf(buffer,"%s:\n",var_arg0);
    io_writestring(0,buffer);
#endif  /* MWC */
#else
    sprintf(buffer,"%s:\n",var_arg0);
    io_writestring(0,buffer);
#endif  /* GEMDOS */

    sprintf(buffer,"  Dynamic memory: pool=%s bytes, allocated=%ld bytes\n",
        kbytes(poolSize),mallocTotal);
    io_writestring(0,buffer);
    sprintf(buffer,"                  max allocated=%ld bytes\n",mallocHighWater);
    io_writestring(0,buffer);
#ifdef  GEMDOS
#ifdef  MWC
    totalMemory += poolSize;
    sprintf(buffer,"  Total memory used: %s bytes\n",kbytes(totalMemory));
    io_writestring(0,buffer);
#endif  /* MWC */
    memblock = Malloc(-1L);
    sprintf(buffer,"\n  Largest free memory block in system: %s\n",kbytes(memblock));
    io_writestring(0,buffer);
#endif  /* GEMDOS */

    return 0;
}   /* end of cmd_memory */
#endif  /* MYMALLOC */

void func_dump(fp,flag)
register struct function *fp;
register int flag;
{
    if (fp == (struct function *)NULL)
        return;
    if (fp->left != (struct function *)NULL)
        func_dump(fp->left,flag);
    phrase_dump(fp->code,flag,0);
    if (fp->right != (struct function *)NULL)
        func_dump(fp->right,flag);
}   /* end of func_dump */

int cmd_typeset(pp)
struct phrase *pp;
{
    register char *p;
    char *cp;
    int type,mask,misc,functions;
    register struct token *tp;
    char buffer[BUFSIZ];

    functions = type = 0;
    mask = ~0;
    misc = 0;
    tp = pp->body->next;
    if (tp != (struct token *)NULL)
        stripquotes(tp->name);
    if (tp != (struct token *)NULL && (tp->name[0] == '-' || tp->name[0] == '+'))
    {   /* pick up options */
        for (p = &tp->name[1]; *p; p++)
        {   /* what options? */
            switch (*p)
            {   /* select which one */
                case 'f':
                    if (tp->name[0] == '-')
                        functions = 1;
                    else functions = 2;
                    break;
                case 'r':
                    if (tp->name[0] == '-')
                        type |= TYPE_READONLY;
                    else mask &= ~TYPE_READONLY;
                    break;
                case 'x':
                    if (tp->name[0] == '-')
                        type |= TYPE_EXPORTED;
                    else mask &= ~TYPE_EXPORTED;
                    break;
                case 'i':
                    if (tp->name[0] == '-')
                        type |= TYPE_INTEGER;
                    else mask &= ~TYPE_INTEGER;
                    break;
                case 'u':
                    if (tp->name[0] == '-')
                        type |= TYPE_UPPERCASE;
                    else mask &= ~TYPE_UPPERCASE;
                    break;
                case 'l':
                    if (tp->name[0] == '-')
                        type |= TYPE_LOWERCASE;
                    else mask &= ~TYPE_LOWERCASE;
                    break;
                case 't':
                    if (tp->name[0] == '-')
                        type |= TYPE_TAGGED;
                    else mask &= ~TYPE_TAGGED;
                    break;
                case 'R':
                    if (tp->name[0] == '-')
                        type |= TYPE_RIGHTJUST;
                    else mask &= ~TYPE_RIGHTJUST;
                    if (p[1] == 'Z')
                    {   /* add in zero stuff? */
                        cp = &p[2];
                        if (tp->name[0] == '-')
                            type |= TYPE_ZEROS;
                        else mask &= ~TYPE_ZEROS;
                    }
                    else cp = &p[1];
                    misc = atoi(cp);
                    p = (char *)NULL;
                    mask &= ~TYPE_LEFTJUST;
                    break;
                case 'L':
                    if (tp->name[0] == '-')
                        type |= TYPE_LEFTJUST;
                    else mask &= ~TYPE_LEFTJUST;
                    if (p[1] == 'Z')
                    {   /* add in zero stuff? */
                        cp = &p[2];
                        if (tp->name[0] == '-')
                            type |= TYPE_ZEROS;
                        else mask &= ~TYPE_ZEROS;
                    }
                    else cp = &p[1];
                    misc = atoi(cp);
                    p = (char *)NULL;
                    mask &= ~TYPE_RIGHTJUST;
                    break;
                case 'Z':
                    if (tp->name[0] == '-')
                        type |= (TYPE_RIGHTJUST|TYPE_ZEROS);
                    else mask &= ~(TYPE_RIGHTJUST|TYPE_ZEROS);
                    misc = atoi(&p[1]);
                    p = (char *)NULL;
                    mask &= ~TYPE_LEFTJUST;
                    break;
                case 'H':
                    if (tp->name[0] == '-')
                        type |= TYPE_HOSTMAP;
                    else mask &= ~TYPE_HOSTMAP;
                    break;
                default:
                    errmsg(0,LOC("cmd_typeset"),"unknown typeset option in: %s",tp->name);
                    return 1;
            }
            if (p == (char *)NULL)
                break;
        }
        tp = tp->next;		/* skip past options */
    }
    if (tp == (struct token *)NULL)
    {   /* just dump out definitions */
        if (functions)
            func_dump(base_env.func_table,functions == 1);
        else
        {   /* dump all or part of variable table */
            if (mask == ~0)
                var_dump(type,1);
            else var_dump(~mask,0);
        }
        return 0;
    }
    for (; tp != (struct token *)NULL; tp = tp->next)
    {   /* for the rest of the args */
        stripquotes(tp->name);
        p = strchr(tp->name,'=');
        if (p != (char *)NULL)
        {   /* set type and value? */
            strncpy(buffer,tp->name,(int)(p-tp->name));
            buffer[(int)(p-tp->name)] = '\0';
        }
        else strcpy(buffer,tp->name);
        if (p != (char *)NULL)
            var_define0(buffer,&p[1],1);
        var_setmisc(buffer,misc);
        var_settype(buffer,type,mask);
    }
    return 0;
}   /* end of cmd_typeset */

int cmd_let(pp)
struct phrase *pp;
{
    int result;
    register struct token *tp;

    result = 0;
    for (tp = pp->body->next; tp != (struct token *)NULL; tp = tp->next)
    {   /* eval each argument */
        stripquotes(tp->name);
        result = parse_c_expression(tp->name,1);
    }
    if (result != 0)
        return 0;
    else return 1;
}   /* end of cmd_let */

int cmd_dparen(pp)
struct phrase *pp;
{
    register struct token *tp;
    int length,result;
    char *expression;

    length = 0;
    for (tp = pp->body->next; tp != (struct token *)NULL; tp = tp->next)
    {   /* get length of args */
        stripquotes(tp->name);
        length += strlen(tp->name);
        if (tp->next != (struct token *)NULL)
            length++;
    }
    expression = new_string(length+1);
    if (expression == (char *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("cmd_dparen"));
        return 1;
    }
    expression[0] = '\0';
    for (tp = pp->body->next; tp != (struct token *)NULL; tp = tp->next)
    {   /* get length of args */
        if (tp->type == SYM_DRPAREN)
            continue;
        strcat(expression,tp->name);
    }
    result = parse_c_expression(expression,1);
    free(expression);
    if (result != 0)
        return 0;
    return 1;
}   /* end of cmd_dparen */

int cmd_whence(pp)
struct phrase *pp;
{
    register struct token *tp,*ftp;
    int verbose;
    struct aliases *ap;
    char *ptr;
    char buffer[BUFSIZ];

    verbose = 0;
    tp = pp->body->next;
    if (tp != (struct token *)NULL && strcmp(tp->name,"-v") == 0)
    {   /* verbose flag set? */
        verbose = 1;
        tp = tp->next;
    }
    for (; tp != (struct token *)NULL; tp = tp->next)
    {   /* for each arg */
        stripquotes(tp->name);
        if (verbose)
        {   /* check reserved words, etc. */
            buffer[0] = '\0';
            if (reserved_word(tp->name))
            {   /* is it this? */
                sprintf(buffer,"%s is a reserved word",tp->name);
            }
            else if ((ap = alias_get(tp->name)) != (struct aliases *)NULL)
            {   /* is it this? */
                if (ap->type & TYPE_TRACKED)
                    sprintf(buffer,"%s is a tracked alias for ",tp->name);
                else sprintf(buffer,"%s is an alias for ",tp->name);
                for (ftp = ap->tp; ftp != (struct token *)NULL; ftp = ftp->next)
                {   /* dump each token */
                    strcat(buffer,ftp->name);
                    if (ftp->next != (struct token *)NULL)
                        strcat(buffer," ");
                }
            }
            else if (func_get(tp->name) != (struct function *)NULL)
            {   /* is it this? */
                sprintf(buffer,"%s is a shell function",tp->name);
            }
            else if (builtin_word(tp->name))
            {   /* is it this? */
                sprintf(buffer,"%s is a shell built-in",tp->name);
            }
            if (buffer[0] != '\0')
            {   /* generated a message? */
                io_writestring(0,buffer);
                io_writestring(0,"\n");
                continue;
            }
        }
        if ((ap = alias_get(tp->name)) != (struct aliases *)NULL)
        {   /* if alias or tracked alias */
            if (ap->type & TYPE_TRACKED)
            {   /* give full name of tracked alias */
                buffer[0] = '\0';
                for (ftp = ap->tp; ftp != (struct token *)NULL; ftp = ftp->next)
                {   /* dump each token */
                    strcat(buffer,ftp->name);
                    if (ftp->next != (struct token *)NULL)
                        strcat(buffer," ");
                }
                io_writestring(0,buffer);
            }
            else io_writestring(0,tp->name);
        }
        else if (reserved_word(tp->name) ||
            func_get(tp->name) != (struct function *)NULL ||
            builtin_word(tp->name))
        {   /* is it this? */
            io_writestring(0,tp->name);
        }
        else
        {   /* normal path search for program */
            ptr = exec_pathsearch(tp->name);
            if (ptr == (char *)NULL)
            {   /* not there */
                if (!verbose)
                    sprintf(buffer,"%s",tp->name);
                else sprintf(buffer,"%s not found",tp->name);
            }
            else
            {   /* actually found the program */
                if (!verbose)
                    sprintf(buffer,"%s",ptr);
		else sprintf(buffer,"%s is %s",tp->name,ptr);
                free(ptr);
            }
            io_writestring(0,buffer);
        }
        io_writestring(0,"\n");
    }

    return 0;
}   /* end of cmd_whence */
