/*
 * Command Input Shell
 * Dave Clemans
 * 12/88-1/89
 *
 * "spiritually" based on Bourne, Korn shells
 *
 * Built-in Commands (part 1)
 *
 * $Id: cmd1.c,v 1.8 89/02/25 17:39:38 dclemans Exp $
 *
 * $Log:	cmd1.c,v $
 * Revision 1.8  89/02/25  17:39:38  dclemans
 * miscellaneous bug fixes/speedups
 * 
 * Revision 1.7  89/02/22  21:31:37  dclemans
 * Implement simple background job monitoring facility
 * 
 * Revision 1.6  89/02/22  16:27:05  dclemans
 * Implement [[, ]] brackets
 * 
 * Revision 1.5  89/02/20  22:29:08  dclemans
 * Add new builtin commands "trap" and "kill" to command switch table.
 * 
 * Revision 1.4  89/02/20  20:14:15  dclemans
 * Add RCS identifiers
 * 
 */
#include <stdio.h>
#include "shell.h"
#ifdef  LINED
#include "reader\history.h"
#endif  /* LINED */

int cmd_lastrc = 0;
int cmd_forceexit = 0;
int cmd_returnexit = 0;
int cmd_count = 0;

extern  int cmd_for();      /* from part 2 */
extern  int cmd_select();
extern  int cmd_while();
extern  int cmd_function();
extern  int cmd_if();
extern  int cmd_case();
extern  int cmd_break();
extern  int cmd_continue();
extern  int cmd_return();
extern  int cmd_fc();
extern  int cmd_eval();
extern  int cmd_exec();
extern  int cmd_time();

extern  int cmd_test();     /* from part 3 */
extern  int cmd_version();
#ifdef  MYMALLOC
extern  int cmd_memory();
#endif  /* MYMALLOC */
extern  int cmd_typeset();
extern  int cmd_let();
extern  int cmd_dparen();
extern  int cmd_whence();

extern  int cmd_trap();     /* from trap */
#ifndef GEMDOS
extern  int cmd_kill();
extern  int cmd_jobs();
#endif  /* GEMDOS */

int cmd_null()
{
    return 0;
}   /* end of cmd_null */

int cmd_print(pp)
struct phrase *pp;
{
    int flag,noesc;
    register struct token *tp;
    register char *p,*b;
    char buffer[BUFSIZ];

    flag = 1;
    noesc = 0;
    b = buffer;
    tp = pp->body->next;
    while (tp != (struct token *)NULL && tp->name[0] == '-')
    {   /* check for special flags */
        if (tp->name[1] == '\0')
        {   /* forced end of args? */
            tp = tp->next;
            break;
        }
        for (p = &tp->name[1]; *p; p++)
        {   /* for each possibility */
            switch (*p)
            {   /* select what it does */
                case 'n':
                    flag = 0;
                    break;
                case 'R':
                case 'r':
                    noesc = 1;
                    break;
                default:
                    errmsg(0,LOC("cmd_print"),"unknown print arg in: %s",tp->name);
                    return 1;
            }
        }
        tp = tp->next;
    }
    for (; tp != (struct token *)NULL; tp = tp->next)
    {   /* for each argument, echo its characters */
        for (p = tp->name; *p; p++)
        {   /* for each char */
            switch (*p)
            {   /* what char? */
                case '\'':
                    p++;
                    while (*p && *p != '\'')
                    {   /* dump the string */
                        *b++ = *p;
                        p++;
                    }
                    break;
                case '"':
                    p++;
                    while (*p && *p != '"')
                    {   /* dump the string */
                        if (!noesc && *p == ESCAPE_CHAR)
                        {   /* process the escape char */
                            p++;
                            switch (*p)
                            {   /* what was escaped */
                                case 'a':
                                    *b++ = '\007';
                                    break;
                                case 'b':
                                    *b++ = '\b';
                                    break;
                                case 'c':
                                    flag = 0;
                                    break;
                                case 'f':
                                    *b++ = '\f';
                                    break;
                                case 'n':
                                    *b++ = '\n';
                                    break;
                                case 'r':
                                    *b++ = '\r';
                                    break;
                                case 't':
                                    *b++ = '\t';
                                    break;
                                default:
                                    *b++ = *p;
                                    break;
                            }
                        }
                        else    *b++ = *p;
                        p++;
                    }
                    break;
                case ESCAPE_CHAR:
                    if (!noesc)
                    {   /* if just ordinary chars... */
                        p++;
                        switch (*p)
                        {   /* what was escaped */
                            case 'a':
                                *b++ = '\007';
                                break;
                            case 'b':
                                *b++ = '\b';
                                break;
                            case 'c':
                                flag = 0;
                                break;
                            case 'f':
                                *b++ = '\f';
                                break;
                            case 'n':
                                *b++ = '\n';
                                break;
                            case 'r':
                                *b++ = '\r';
                                break;
                            case 't':
                                *b++ = '\t';
                                break;
                            default:
                                *b++ = *p;
                                break;
                        }
                    } else *b++ = *p;
                    break;
                default:
                    *b++ = *p;
                    break;
            }
        }
        if (tp->next != (struct token *)NULL)
            *b++ = ' ';
    }
    if (flag)
    {   /* add in an eol */
        *b++ = '\n';
    }
    *b = '\0';
    io_writestring(0,buffer);
    return 0;
}   /* end of cmd_print */

int cmd_alias(pp)
struct phrase *pp;
{
    register struct token *tp,*args;
    register char *p;
    int type;

    type = 0;
    args = pp->body->next;
    while (args != (struct token *)NULL && args->name[0] == '-')
    {   /* if there are some arguments */
        for (p = &args->name[1]; *p; p++)
        {   /* what kind of alias? */
            switch (*p)
            {   /* what switch? */
                case 't':
                    type = TYPE_TRACKED;
                    break;
                default:
                    errmsg(0,LOC("cmd_alias"),"bad switch: %s",args->name);
                    break;
            }
        }
        args = args->next;
    }

    if (args == (struct token *)NULL)
        alias_dump(base_env.alias_table,type);
    else
    {   /* should be a definition */
        tp = args;
        p = strchr(tp->name,'=');
        if (p == (char *)NULL && tp->next == (struct token *)NULL && type == 0)
        {   /* print definition of alias */
            alias_print(tp->name);
            return 0;
        }
        if (p != (char *)NULL && tp->next == (struct token *)NULL)
        {   /* ksh style alias */
            *p++ = '\0';
            alias_sdefine(tp->name,p,type);
            *--p = '=';
        }
        else
        {   /* csh style alias */
            alias_define(tp->name,tp->next,type);
        }
    }
    return 0;
}   /* end of cmd_alias */

int cmd_unalias(pp)
struct phrase *pp;
{
    register struct token *tp;

    for (tp = pp->body->next; tp != (struct token *)NULL; tp = tp->next)
        alias_define(tp->name,(struct token *)NULL,0);
    return 0;
}   /* end of cmd_unalias */

static struct token *proc_opts(tp,value)
register struct token *tp;
int value;
{
    char buffer[BUFSIZ];

    buffer[0] = '\0';
    if (tp->next == (struct token *)NULL)
    {   /* just print current options state */
        if (flag_allexport)
            strcat(buffer,"allexport ");
        if (flag_noglob)
            strcat(buffer,"noglob ");
        if (flag_cmdhash)
            strcat(buffer,"trackall ");
        if (flag_interactive)
            strcat(buffer,"interactive ");
        if (flag_keywords)
            strcat(buffer,"keyword ");
        if (flag_monitor)
            strcat(buffer,"monitor ");
        if (flag_noexec)
            strcat(buffer,"noexec ");
        if (flag_varerr)
            strcat(buffer,"nounset ");
        if (flag_echoinput)
            strcat(buffer,"verbose ");
        if (flag_echoexec)
            strcat(buffer,"xtrace ");
#ifdef  LINED
        if (_savedState.isEmacs)
            strcat(buffer,"emacs ");
        if (_savedState.isVi)
            strcat(buffer,"vi ");
#endif  /* LINED */
        strcat(buffer,"\n");
        io_writestring(0,buffer);
        return tp;
    }
    if (strcmp(tp->next->name,"allexport") == 0)
    {   /* export all? */
        flag_allexport = value;
        return tp->next;
    }
    if (strcmp(tp->next->name,"keyword") == 0)
    {   /* flag keywords? */
        flag_keywords = value;
        return tp->next;
    }
    if (strcmp(tp->next->name,"monitor") == 0)
    {   /* monitor background jobs */
        flag_monitor = value;
        return tp->next;
    }
    if (strcmp(tp->next->name,"noexec") == 0)
    {   /* no execution? */
        flag_noexec = value;
        return tp->next;
    }
    if (strcmp(tp->next->name,"noglob") == 0)
    {   /* file expansion? */
        flag_noglob = value;
        return tp->next;
    }
    if (strcmp(tp->next->name,"nounset") == 0)
    {   /* unknown variables? */
        flag_varerr = value;
        return tp->next;
    }
    if (strcmp(tp->next->name,"trackall") == 0)
    {   /* command hashing? */
        flag_cmdhash = value;
        return tp->next;
    }
    if (strcmp(tp->next->name,"verbose") == 0)
    {   /* input echoing */
        flag_echoinput = value;
        return tp->next;
    }
    if (strcmp(tp->next->name,"xtrace") == 0)
    {   /* exec echoing */
        flag_echoexec = value;
        return tp->next;
    }
#ifdef  LINED
    if (strcmp(tp->next->name,"emacs") == 0)
    {   /* emacs mode? */
        _savedState.isEmacs = -1;
        _savedState.isVi = 0;
        return tp->next;
    }
    if (strcmp(tp->next->name,"vi") == 0)
    {   /* vi mode? */
        _savedState.isEmacs = 0;
        _savedState.isVi = -1;
    }
#endif  /* LINED */
    errmsg(0,LOC("proc_opts"),"unknown 'set -o' option: %s",tp->next->name);
    return tp;
}   /* end of proc_opts */

int cmd_set(pp)
struct phrase *pp;
{
    register struct token *tp;
    register char *p;
    int rc,value;

    rc = 0;
    if (pp->body == pp->body_end)
    {   /* print out defined variables */
        var_dump(0,1);
    }
    else
    {   /* set flags and/or positional args */
        for (tp = pp->body->next; tp != (struct token *)NULL; tp = tp->next)
        {   /* look for flags */
            if (tp->name[0] != '-' && tp->name[0] != '+')
                break;
            else if (strcmp(tp->name,"-") == 0)
            {   /* forced end of flags */
                flag_echoinput = 0;
                flag_echoexec = 0;
                tp = tp->next;
                break;
            }
            else if (strcmp(tp->name,"--") == 0)
            {   /* forced end of flags */
                tp = tp->next;
                break;
            }
            if (tp->name[0] == '-')
                value = 1;
            else value = 0;
            for (p = &tp->name[1]; *p; p++)
            {   /* set switches as appropriate */
                switch (*p)
                {   /* what switch? */
                    case 'o':   /* options? */
                        tp = proc_opts(tp,value);
                        break;
                    case 'a':   /* export all? */
                        flag_allexport = value;
                        break;
                    case 'f':   /* no file expansion? */
                        flag_noglob = value;
                        break;
                    case 'm':   /* monitor jobs? */
                        flag_monitor = value;
                        break;
                    case 'n':   /* no execution? */
                        flag_noexec = value;
                        break;
                    case 'u':   /* flag var errors */
                        flag_varerr = value;
                        break;
                    case 'v':   /* echo input? */
                        flag_echoinput = value;
                        break;
                    case 'x':   /* echo execution? */
                        flag_echoexec = value;
                        break;
                    case 'k':   /* all keywords */
                        flag_keywords = value;
                        break;
                    case 'h':   /* command hashing? */
                        flag_cmdhash = value;
                        break;
                    case 'i':   /* interactive? */
                        flag_interactive = value;
                        break;
                    default:
                        errmsg(0,LOC("cmd_set"),"bad switch: %s",tp->name);
                        rc = 1;
                        break;
                }
            }
            if (tp == (struct token *)NULL)
                break;
        }
        if (tp != (struct token *)NULL)
        {   /* any positional args to reset? */
            var_resetargs(tp);
        }
    }
    return rc;
}   /* end of cmd_set */

int cmd_unset(pp)
struct phrase *pp;
{
    register struct token *tp;

    for (tp = pp->body->next; tp != (struct token *)NULL; tp = tp->next)
        var_settype(tp->name,TYPE_DELETED,~TYPE_DELETED);
    return 0;
}   /* end of cmd_unset */

int cmd_export(pp)
struct phrase *pp;
{
    register struct token *tp;

    if (pp->body->next == (struct token *)NULL)
    {   /* just list currently exported things? */
        var_dump(TYPE_EXPORTED,1);
    }
    else
    {   /* actually export some things */
        for (tp = pp->body->next; tp != (struct token *)NULL; tp = tp->next)
            var_settype(tp->name,TYPE_EXPORTED,~TYPE_EXPORTED);
    }
    return 0;
}   /* end of cmd_export */

int cmd_unexport(pp)
struct phrase *pp;
{
    register struct token *tp;

    for (tp = pp->body->next; tp != (struct token *)NULL; tp = tp->next)
        var_settype(tp->name,0,~TYPE_EXPORTED);
    return 0;
}   /* end of cmd_unexport */

int cmd_readonly(pp)
struct phrase *pp;
{
    register struct token *tp;

    if (pp->body->next == (struct token *)NULL)
    {   /* just list currently readonly things? */
        var_dump(TYPE_READONLY,1);
    }
    else
    {   /* actually make some things readonly */
        for (tp = pp->body->next; tp != (struct token *)NULL; tp = tp->next)
            var_settype(tp->name,TYPE_READONLY,~TYPE_READONLY);
    }
    return 0;
}   /* end of cmd_readonly */

int cmd_shift(pp)
struct phrase *pp;
{
    if (pp->body->next == (struct token *)NULL)
        var_shiftargs(1);
    else if (pp->body->next->type == SYM_NUMBER)
        var_shiftargs(atoi(pp->body->next->name));
    else    errmsg(0,LOC("cmd_shift"),"bad argument: %s",pp->body->next->name);
    return 0;
}   /* end of cmd_shift */

int cmd_exit(pp)
struct phrase *pp;
{
    cmd_forceexit = 1;
    if (pp->body->next == (struct token *)NULL)
        return cmd_lastrc;
    else    return atoi(pp->body->next->name);
}   /* end of cmd_exit */

int cmd_source(pp)
struct phrase *pp;
{
    if (pp->body->next == (struct token *)NULL)
    {   /* any file to source? */
        errmsg(0,LOC("cmd_source"),"no file given");
        return 1;
    }
    if (io_pushfile(pp->body->next->name,0,0,0))
        return 1;
    return 0;
}   /* end of cmd_source */

int cmd_chdir(pp)
struct phrase *pp;
{
    struct token *tp;
    char *dir,*newdir;
    int rc;

    tp = pp->body->next;
    if (tp == (struct token *)NULL)
    {   /* goto home directory */
        dir = base_env.homedir;
    }
    else
    {   /* goto specified directory */
        dir = new_string(strlen(tp->name)+1);
        if (dir == (char *)NULL)
        {   /* enough memory */
            errmsg(SHERR_NOMEM,LOC("cmd_chdir"));
            return 1;
        }
        strcpy(dir,tp->name);
        stripquotes(dir);
        while (dir[0] == '\'' || dir[0] == '"')
            stripquotes(dir);
        if (strlen(dir) == 0)
        {   /* if nothing left */
            free(dir);
            dir = base_env.homedir;
            tp = (struct token *)NULL;
        }
    }
    newdir = (char *)NULL;
    rc = lchdir(dir,&newdir);
    if (rc != 0)
    {   /* if we didn't get there */
        errmsg(0,LOC("cmd_chdir"),"couldn't change to %s",dir);
        if (tp != (struct token *)NULL)
            free(dir);
        return rc;
    }
    if (tp != (struct token *)NULL)
        free(dir);
    var_define0("OLDPWD",base_env.dir->current,0);
    var_define0("PWD",newdir,0);
    free(newdir);
    return 0;
}   /* end of cmd_chdir */

int cmd_read(pp)
struct phrase *pp;
{
    char buffer[BUFSIZ+1];
    register char *first,*last;
    register struct token *tp;
    int rc,ctr;

    tp = pp->body->next;
    buffer[0] = '\0';
    for (ctr = 0; ; ctr++)
    {   /* get a line */
        rc = read(base_env.io->input,&buffer[ctr],1);
        if (rc <= 0)
            return 1;
        if (buffer[ctr] == '\n')
            break;
    }
    buffer[++ctr] = '\0';
    for (first = buffer; *first; first++)
        if (*first == '\n')
            *first = '\0';
    for (last = buffer; *last; )
    {   /* break up into words, assign to variables */
        while (*last && strchr(base_env.separators,*last) != (char *)NULL)
            last++;
        first = last;
        if (tp == (struct token *)NULL)
        {   /* just assign everything to REPLY? */
            var_define0("REPLY",first,0);
            break;
        }
        else if (tp->next == (struct token *)NULL)
        {   /* just assign everything to one variable */
            var_define0(tp->name,first,0);
            break;
        }
        /* else break up into words, keep going */
        while (*last && strchr(base_env.separators,*last) == (char *)NULL)
            last++;
        *last++ = '\0';
        var_define0(tp->name,first,0);
        tp = tp->next;
    }
    return 0;
}   /* end of cmd_read */

int cmd_pwd()
{
    char buffer[BUFSIZ];

    sprintf(buffer,"%s\n",base_env.dir->current);
    io_writestring(0,buffer);
    return 0;
}   /* end of cmd_pwd */

int cmd_sleep(pp)
struct phrase *pp;
{
#ifdef  GEMDOS
    long tstart,tend;
#endif  /* GEMDOS */
    int tdiff;

    if (pp->body->next == (struct token *)NULL)
        return 0;
    if (pp->body->next->type != SYM_NUMBER)
    {   /* an amount of time to sleep? */
        errmsg(0,LOC("cmd_sleep"),"bad argument to sleep: %s",pp->body->next->name);
        return 1;
    }
    tdiff = atoi(pp->body->next->name);
    if (tdiff < 0)
        return 0;
#ifdef  GEMDOS
    tstart = time(0L);
    do
    {   /* while the waiting period is in progress */
        tend = time(0L);
    } while ((tstart + (long)tdiff) > tend);
#else
    sleep(tdiff);
#endif  /* GEMDOS */
    return 0;
}   /* end of cmd_sleep */

struct commands cmd_builtin[] =
{   /* keep this list sorted... */
    { "((",         cmd_dparen },
    { ".",          cmd_source },
    { ":",          cmd_null },
    { "[",          cmd_test },
    { "[[",         cmd_test },
    { "alias",      cmd_alias },
    { "break",      cmd_break },
    { "case",       cmd_case },
    { "cd",         cmd_chdir },
    { "chdir",      cmd_chdir },
    { "continue",   cmd_continue },
    { "do",         cmd_null },
    { "done",       cmd_null },
    { "elif",       cmd_null },
    { "else",       cmd_null },
    { "esac",       cmd_null },
    { "eval",       cmd_eval },
    { "exec",       cmd_exec },
    { "exit",       cmd_exit },
    { "export",     cmd_export },
    { "fc",         cmd_fc },
    { "fi",         cmd_null },
    { "for",        cmd_for },
    { "function",   cmd_function },
    { "if",         cmd_if },
#ifndef GEMDOS
    { "jobs",       cmd_jobs },
    { "kill",       cmd_kill },
#endif  /* GEMDOS */
    { "let",        cmd_let },
#ifdef  MYMALLOC
    { "memory",     cmd_memory },
#endif  /* MYMALLOC */
    { "print",      cmd_print },
    { "pwd",        cmd_pwd },
    { "read",       cmd_read },
    { "readonly",   cmd_readonly },
    { "return",     cmd_return },
    { "select",     cmd_select },
    { "set",        cmd_set },
    { "shift",      cmd_shift },
    { "sleep",      cmd_sleep },
    { "test",       cmd_test },
    { "then",       cmd_null },
    { "time",       cmd_time },
    { "trap",       cmd_trap },
    { "typeset",    cmd_typeset },
    { "unalias",    cmd_unalias },
    { "unexport",   cmd_unexport },
    { "unset",      cmd_unset },
    { "until",      cmd_while },
    { "version",    cmd_version },
    { "whence",     cmd_whence },
    { "while",      cmd_while },
    { "{",          cmd_null },
    { "}",          cmd_null },
    { (char *)NULL }
};

int builtin_word(word)
char *word;
{
    register int i;

    for (i = 0; cmd_builtin[i].name != (char *)NULL; i++)
    {   /* is this a builtin command? */
        if (strcmp(cmd_builtin[i].name,word) == 0)
            return 1;
    }
    return 0;
}   /* end of builtin_word */
