/* $Header: out.c,v 3.0 88/04/13 16:06:33 jos Locked $ */
/*
 *  This file is part of the Amsterdam SGML Parser.
 *
 *  Copyright: Faculteit Wiskunde en Informatica
 *             Department of Mathematics and Computer Science
 *             Vrije Universiteit Amsterdam
 *             The Netherlands
 *
 *  Authors:   Sylvia van Egmond
 *             Jos Warmer
 */

#include "types.h"
#include "att_par.h"
#include "group.h"
#include "out.h"
#include "replace.h"
#include "rep_pars.h"
#include "report.h"
#include "tags.h"

/*
 *  `new_line' == TRUE means that the last character written on the
 *  output was the newline character.
 *  Otherwise `new_line' == FALSE.
 *
 *  `must_newline' == TRUE if the next character on the output must
 *  be a newline character.
 *  If the next character that is going to be written is not the
 *  newline character, then the newline character is inserted.
 */

static Bool  new_line     = TRUE;
static Bool  must_newline = FALSE;
static Bool  replace      = FALSE;

/*
 *  Return the last character of `s'
 *  Returns '\0' if `s' is empty.
 */

char lastchar(s)
String s;
{
    int i;

    i = strlen(s);
    return ( i == 0 ? '\0' : s[i-1] );
}

/*
 * Write `ch' to output
 */
void   out_char   (ch)
char    ch;
{
    if (must_newline) {
	must_newline = FALSE;
	if (ch != '\n') putchar('\n');
    }

    putchar(ch);

    new_line = (ch == '\n');
}

/*
 * Write `string' to output
 */
void   out_string(string)
String   string;
{
    char ch;

    if ((ch = lastchar(string)) == '\0') return;

    if (must_newline) {
	must_newline = FALSE;
	if (string[0] != '\n') putchar('\n');
    }

    printf("%s", string);
    new_line = (ch == '\n');
}

void   out_newline()
{
    if (new_line) return;
    else must_newline = TRUE;
}


static String empty_string = "";

void out_rep_tag(info, rep)
Parserinfo  info;
P_Replace   rep;
{
    P_Iterator    it;
    P_Chunk       chunk;
    P_Attdeflist  attlist;
    P_Attdef      att;
    String        att_value;

    if (rep == 0) return;

    if (replace_start_newline(rep) && !new_line) { out_char('\n'); }

    it      = group_iterator( replace_chunk_group(rep) );
    attlist = info_attlist(info);
    while (chunk = next_chunk(it)) {
	switch (chunk_type(chunk)) {
	    case CHUNK_STRING:
		out_string(chunk_string(chunk));
	        break;
	    case CHUNK_ATTNAME:
		att = attdef_name_lookup(attlist, chunk_attname(chunk)) ;
		/*
		 *  Put the attribute value in 'att_value'
		 */
		if( not (att_value = att_current_value(att)) ){
		    if( not (att_value = att_default_value(att)) ){
			att_value = empty_string;
		    }
		}
		out_string(att_value);
	        break;
	    default:
		report(NO_LABEL, FATAL, 0, 0, chunk_type(chunk), "out_rep_tag");
	        break;
	}
    }
    if (replace_end_newline(rep)) { out_newline(); }
}

Bool dubble_quotes(value)
String value;
{
	int i = 0;

	while (value[i] and (value[i] != '"')) i++;
	return(value[i] == '"');
}

void out_standard_starttag(info)
Parserinfo info;
{
	P_Att_iter it;
	P_Attdef   attdef;
	String     att_val;
	char       att_pr[LITLEN + NAMELEN + 6 + 1];

	out_char('<');
	out_string(info_name(info));
	it = attdef_iterator(info_attlist(info));
	while (attdef = att_next(it)) {
	    if ((att_val = att_current_value(attdef)) or
		(att_val = att_default_value(attdef)))
	    {
		if (dubble_quotes(att_val)) {
		    sprintf(att_pr, " %s = \'%s\'", att_name(attdef), att_val);
		} else {
		    sprintf(att_pr, " %s = \"%s\"", att_name(attdef), att_val);
		}
		out_string(att_pr);
	    }
	}
	out_char('>');
}

void out_standard_endtag(info)
Parserinfo info;
{
	out_string("</");
	out_string(info_name(info));
	out_char('>');
}

void out_tag(tag)
int tag;
{
	Parserinfo info;

	if ((info = starttag_info(tag)) != (Parserinfo) 0) {
	    if (replace) out_rep_tag(info, info_start_rep(info));
	    else out_standard_starttag(info);
	}
	else if ((info = endtag_info(tag)) != (Parserinfo) 0) {
	    if (replace) out_rep_tag(info, info_end_rep(info));
	    else out_standard_endtag(info);
	}
	else report(TAG_UNKNOWN, FATAL, 0, 0, "start/end", "unknown");
}

void out_pi(instruction)
String instruction;
{
	if (replace) {
	    out_string(instruction);
	} else {
	    out_string("<?");
	    out_string(instruction);
	    out_char('>');
	}
}

void init_out(filename)
String filename;
{
	if (filename) {
	    replace = TRUE;
	    init_rep_parser(filename);
	}
	fatal_report();
}
