/* $Header: node.c,v 3.0 88/04/13 16:04:57 jos Locked $ */
/*
 *  This file is part of the Amsterdam SGML Parser.
 *
 *  Copyright: Faculteit Wiskunde en Informatica
 *             Department of Mathematics and Computer Science
 *             Vrije Universiteit Amsterdam
 *             The Netherlands
 *
 *  Authors:   Sylvia van Egmond
 *             Jos Warmer
 */

#define INDENT       4 		/* for printing the tree. */
static int in = 0;              /* for printing the tree. */

#include "types.h"
#include "ambigu.h"
#include "keywords.h"
#include "node.h"
#include "symtable.h"

typedef struct node_struct {
    TYPE              type;    /* GROUP.SEQ,AND,OR,GI,GI-START/END/NAME,KEY*/
    int               occ;     /* The occurrence indicator of this node */
    int               status;
    int               flagin;
    int               flagout;
    int               nullable;  /* Yes, No, Undecided */
    P_Set             firstsym;  /* set of first  symbols */
    P_Set             followsym; /* set of follow symbols */
    Bool              busy;
    Bool              done;
    P_Node            parent;  /* The parent node to which this node belongs  */
    union {
        P_Group       group;		/* only for SEQ, AND, OR, GI */
        String        string;	       	/* only for GI_NAME/END/START*/
        int           key;              /* only for KEY              */
    } info;
} Node;

static P_Node node_copy(PAR  P_Node  RAP);
static P_Node node_assign(PAR  P_Node  RAP);

/*  creates a node structure and fills in the common fields.
 */
static P_Node create_node(type, occ)
int  type;
int  occ;
{
    P_Node n;

    n           = (P_Node)MALLOC( (unsigned)sizeof(struct node_struct) );
    n->type     = type;
    n->occ      = occ;
    n->status   = UNDEFINED;
    n->nullable = Undecided;
    n->parent   = (P_Node)0;
    n->busy     = FALSE;
    n->done     = FALSE;
    n->firstsym = 0;
    n->followsym= 0;
    return n;
}

P_Node new_node_group()
{
    P_Node  n;

    n             = create_node(GROUP, NO_OCC);
    n->info.group = group_create();
    return n;
}

P_Node new_node_gi(str)
String  str;
{
    P_Node  m;
    P_Node  n;
    
    n = create_node(GI, NO_OCC);
    n->info.group = group_create();
    m = create_node(GI_START, NO_OCC);
    m->info.string = str;
    node_add(n, m);
    m = create_node(GI_NAME, NO_OCC);
    m->info.string = str;
    node_add(n, m);
    m = create_node(GI_END, NO_OCC);
    m->info.string = str;
    node_add(n, m);

    return n;
}

P_Node new_node_key(key) /* for PCDATA, ANY, CDATA, ...*/
int   key;
{
    P_Node  n;

    n           = create_node(KEY, NO_OCC);
    n->info.key = key;
    return n;
}

static void dummy(n) P_Node n; { }

void delete_node(node)
P_Node  node;
{
    switch( node_type(node) ){
	case GI      : delete_group(node->info.group, delete_node);
		       break;
	case CONN_AND:
	case CONN_SEQ:
	case CONN_OR :
	case GROUP   : delete_group(node->info.group, dummy);
		       break;
	default : break;
    }
    FREE(node, sizeof(Node));
}
static void set_converted_parents(node)
P_Node  node;
{
    P_Iterator it;
    P_Node     child;

    if( is_connector(node_type(node)) or (node_type(node)==GI) ){
	it = group_iterator( node_group(node) );
	while( child = next_node(it) ){
	    child->parent = node;
	}
    }
}
/*
 *  occurrence indicator OCC_PLUS(elem) becomes CONN_SEQ(elem, elem*)
 */
void node_convert_plus(node)
P_Node  node;
{
    P_Node n1;
    P_Node n2;
    assert( node_occ(node)==OCC_PLUS );

    n2 = node_copy(node);
    n1 = node_assign(node);

    node->type = CONN_SEQ;
    node->info.group = group_create();
    node->occ = NO_OCC;

    node_set_occ(n1, NO_OCC);
    node_set_occ(n2, OCC_REP);
    node_add(node, n1);
    node_add(node, n2);

    set_converted_parents(n1);
    set_converted_parents(n2);

    node_set_nullable(n2, Yes);
}

void node_set_group_type(node, type)
P_Node  node;
int     type;
{
    node->type = type;
}

void node_add(node, a)
P_Node  node;
P_Node  a;
{
    group_add(node->info.group, a);
    a->parent = node;
}

P_Group node_group(node)
P_Node   node;
{
    assert( is_connector(node->type) or (node->type==GI) );
    return ( node->info.group );
}

String        node_gi   (node)
P_Node   node;
{
    int t;
    P_Iterator iterator;

    t = node_type(node);
    assert( (t==GI_NAME) or (t==GI_END) or (t==GI_START) or (t==GI) );
    if( t==GI_NAME or t==GI_START or t==GI_END ){
	return node->info.string;
    } else {
	iterator = group_iterator(node_group(node));
	node     = (P_Node)iterator_next(iterator);
        iterator_delete(iterator);
	return node->info.string;
    }
}

int  node_key(node)
P_Node  node;
{
    assert( node->type == KEY );
    return node->info.key;
}

static void indent(file)
FILE *file;
{
    int i;

    fprintf(file, "\n");
    for(i=0; i<in; i++){
	fprintf(file, " ");
    }
}

static P_Node node_copy(node)
P_Node node;
{
    P_Iterator it;
    P_Iterator nt;
    P_Node     n;
    P_Node     ch1, ch2;

    switch( node_type(node) ){
	case GROUP :
	case CONN_SEQ :
	case CONN_AND :
	case CONN_OR :
	    n = new_node_group();
	    node_set_group_type(n, node_type(node));

	    it = group_iterator(node_group(node));
	    while( ch1 = (P_Node)iterator_next(it) ){
		node_add(n, node_copy(ch1));
	    };
	    break;
	case GI :
	    n = new_node_gi(strsave(node_gi(node)));

	    it = group_iterator(node_group(node));
	    nt = group_iterator(node_group(n));
	    while( ch1 = next_node(it) ){
		ch2 = next_node(nt);
		node_set_occ      (ch2, node_occ (ch1));
		node_set_firstsym (ch2, node_firstsym(ch1));
		node_set_followsym(ch2, node_firstsym(ch1));
		node_set_flagin   (ch2, node_flagin(ch1));
		node_set_flagout  (ch2, node_flagout(ch1));
		node_set_nullable (ch2, node_nullable(ch1));
		node_set_done     (ch2, node_done(ch1));
		node_set_busy     (ch2, node_busy(ch1));
		node_set_status   (ch2, node_status(ch1));
	    };
	    break;
	case KEY :
	    n = new_node_key( node_key(node) );
	    break;
	default :
	    report(NO_LABEL, FATAL, 0, 0, node_type(node), "node_copy");
	    break;
    }
    node_set_occ      (n, node_occ (node));
    node_set_firstsym (n, node_firstsym(node));
    node_set_followsym(n, node_firstsym(node));
    node_set_flagin   (n, node_flagin(node));
    node_set_flagout  (n, node_flagout(node));
    node_set_nullable (n, node_nullable(node));
    node_set_done     (n, node_done(node));
    node_set_busy     (n, node_busy(node));
    node_set_status   (n, node_status(node));

    return n;
}

static P_Node node_assign(src)
P_Node  src;
{
    P_Node  dest;

    dest = create_node(node_type(src), node_occ(src));
    (*dest) = (*src);
    node_set_firstsym (dest, node_firstsym(src));
    node_set_followsym(dest, node_firstsym(src));
    node_set_flagin   (dest, node_flagin(src));
    node_set_flagout  (dest, node_flagout(src));
    node_set_nullable (dest, node_nullable(src));
    node_set_done     (dest, node_done(src));
    node_set_busy     (dest, node_busy(src));
    node_set_status   (dest, node_status(src));
    return dest;
}

void node_set_key(node, key)
P_Node   node;
int      key ;
{
    assert(node->type == KEY);
    node->info.key = key;
}

/*
 *  The following functions simply access one of the fields of the node
 *  structure. They can easily be redefined as macros.
 */
TYPE     node_type (node) P_Node node; { return node->type; }
P_Node  node_parent(node) P_Node node; { return node->parent; }

void node_set_occ(node, occ) P_Node node; int occ; { node->occ = occ;  }
int  node_occ    (node     ) P_Node node;          { return node->occ; }

void node_set_status(node, s) P_Node node; int s; { node->status = s; }
int  node_status    (node   ) P_Node node;        { return node->status; }

int  node_flagin     (node  ) P_Node node;       { return node->flagin;}
void node_set_flagin (node,f) P_Node node; int f;{ node->flagin = f;  }

int  node_flagout    (node   ) P_Node node;        { return node->flagout;}
void node_set_flagout(node, f) P_Node node; int f; { node->flagout = f;}


P_Set node_firstsym (node) P_Node  node; { return (node->firstsym  ); }
P_Set node_followsym(node) P_Node  node; { return (node->followsym ); }

void node_set_firstsym (node, s) P_Node node; P_Set s; { node->firstsym  = s; }
void node_set_followsym(node, s) P_Node node; P_Set s; { node->followsym = s; }

void node_set_busy(node, b) P_Node node; Bool b; { node->busy = b; }
Bool node_busy    (node   ) P_Node node;         { return node->busy; }

void node_set_done(node, b) P_Node node; Bool b; { node->done = b; }
Bool node_done    (node   ) P_Node node;         { return node->done; }

void node_set_nullable(node, null) P_Node node; int null; { node->nullable = null;}
int  node_nullable    (node      ) P_Node node;           { return node->nullable;}

/*
 *  Functions for travelling the tree.
 *  Depth first traversal starting in 'node'.
 *  At every node the function 'action' is called with the current node
 *  as parameter.
 *  In 'traverse_post' the function is called after the children are visited.
 *  In 'traverse_pre' the function is called before the children are visited.
 */
void node_traverse_post(node, action, elem)
P_Node      node;
P_Node_func action;
P_Element   elem;
{
    P_Iterator  it;
    P_Node      n;
    int         t = node_type(node);

    if( (t==CONN_SEQ) or (t==CONN_OR) or (t==CONN_AND) or (t==GI) ){
        it = group_iterator(node_group(node));
        while( n = next_node(it) ){
            node_traverse_post(n, action, elem);
        }
    }
    (*action)(node, elem);
}

void node_traverse_pre(node, action, elem)
P_Node      node;
P_Node_func action;
P_Element   elem;
{
    P_Iterator  it;
    P_Node      n;
    int         t;

    (*action)(node, elem);

    t = node_type(node);
    if( (t==CONN_SEQ) or (t==CONN_OR) or (t==CONN_AND) or (t==GI) ){
        it = group_iterator(node_group(node));
        while( n = next_node(it) ){
            node_traverse_pre(n, action, elem);
        }
    }
}

int combine_occ(occ1, occ2)
int occ1, occ2;
{
    if( occ1==NO_OCC ){
	return occ2;
    }
    if( occ2==NO_OCC ){
	return occ1;
    }
    switch( occ1 ){
	case OCC_REP : return OCC_REP;
	case OCC_OPT : if( occ2 == OCC_OPT ){
			   return OCC_OPT;
			} else {
			    return OCC_REP;
			}
	case OCC_PLUS: if( occ2 == OCC_PLUS ){
			   return OCC_PLUS;
		       } else {
			   return OCC_REP;
		       }
	default: report(NO_LABEL, FATAL, 0, 0, occ1, "combine_occ");
		 break;
    }
}

#ifdef DEBUG
/*
 *  Functions for printing the tree in different ways.
 */
String  type_to_string(t)
int t;
{
    String answer;
    static SymbolTable type_strings = { 
			    { "And",      CONN_AND },
			    { "Or",       CONN_OR  },
			    { "Seq",      CONN_SEQ },
			    { "Group",    GROUP  },
			    { "GiStart",  GI_START},
			    { "GiEnd",    GI_END },
			    { "GiName",   GI_NAME },
			    { "Gi",       GI },
			    { "Key",      KEY },
			    { ILL_STRING, ILL_SYMBOL }
			  };

    answer = symbol_string(type_strings, t);
    return answer;
}

void node_print(file, n)
FILE     *file;
P_Node    n;
{
    P_Iterator     iterator;
    String         empty;
    String         occ;
    int            t = node_type(n);

    fprintf(file,"......... First: ");
    sym_set_print(file, node_firstsym(n));
    indent(file);

    switch( node_nullable(n) ) {
	case Undecided  : empty =  "Un ";
		          break;
	case Yes        : empty =  "Yes";
		          break;
	case No         : empty =  "No ";
    		          break;
        default         : empty =  "   ";
	                  break;
    }

    switch( node_occ(n) ) {
	case OCC_OPT    : occ =  "?";
		          break;
	case OCC_REP    : occ =  "*";
		          break;
	case OCC_PLUS   : occ =  "+";
    		          break;
        default         : occ =  " ";
	                  break;
    }

    fprintf(file,"......... follow : ");
    sym_set_print(file, node_followsym(n));
    indent(file);

    switch( node_type(n) ) {
        case KEY    : fprintf(file, "Key%s(%s)(%d %s)", occ,
			      key_str(node_key(n)), node_status(n), empty);
		      break;
	case GI     : if( FALSE ){
			  fprintf(file, "%s%s(%d %s)", node_gi(n),
				  occ, node_status(n), empty);
			  break;
		      }
        case GROUP  : 
        case CONN_AND    :
	case CONN_SEQ    :
	case CONN_OR     :
		      in += INDENT;
		      fprintf(file, "%s%s (%d %s)",
			      type_to_string(node_type(n)), occ,
			      node_status(n), empty);
		      iterator = group_iterator(node_group(n));
		      while( iterator_next(iterator) ){
			  indent(file);
			  node_print(file, (P_Node)iterator_elem(iterator) );
		      }
		      in -= INDENT;
		      break;
	case GI_START    :
		      fprintf(file, "ST_%s%s(%d %s)", node_gi(n),
			      occ, node_status(n), empty);
		      break;
	case GI_END      :
		      fprintf(file, "END_%s%s(%d %s)", node_gi(n),
			      occ, node_status(n), empty);
		      break;
	case GI_NAME     :
		      fprintf(file, "%s%s(%d %s)", node_gi(n),
			      occ, node_status(n), empty);
		      break;
	default : report(NO_LABEL, FATAL, 0, 0, node_type(n), "print_node");
		  break;
    }
}

void node_write(file, n)
FILE     *file;
P_Node    n;
{
    P_Iterator     iterator;
    String         nul;
    char           ch = ' ';
    char           c = ' ';

    switch( node_nullable(n) ) {
	case Undecided  : nul = "Un ";
		          break;
	case Yes        : nul = "Yes";
		          break;
	case No         : nul = "No ";
    		          break;
	default : report(NO_LABEL, FATAL, 0, 0, node_nullable(n), "node_write");
		  break;
    }

    switch( node_occ(n) ) {
	case OCC_OPT    : ch = '?';
		          break;
	case OCC_REP    : ch = '*';
		          break;
	case OCC_PLUS   : ch = '+';
    		          break;
	case NO_OCC     : ch = ' ';   break;
	default : report(NO_LABEL, FATAL, 0, 0, node_occ(n), "node_write");
		  break;
    }
    if( is_connector( node_type(n) ) ){
	switch( node_type(n) ) {
	    case CONN_SEQ    : c = ',';
			      break;
	    case CONN_OR    : c = '|';
			      break;
	    case CONN_AND   : c = '&';
			      break;
	}
    }

    switch( node_type(n) ) {
        case KEY    : fprintf(file, "%s%c ", key_str(node_key(n)),
			      ch);
		      break;
	case GI     : if( TRUE ){
			  fprintf(file, "%s%c ", node_gi(n),
				  ch);
			  break;
		      }
        case GROUP  : 
        case CONN_AND    :
	case CONN_SEQ    :
	case CONN_OR     :
	      /*fprintf(file, "%s[", type_to_string(node_type(n)), nul );*/
	      fprintf(file, "[");
	      iterator = group_iterator(node_group(n));
	      while( iterator_next(iterator) ){
		  node_write(file, (P_Node)iterator_elem(iterator) );
		  fprintf(file, "%c", c);
	      }
	      fprintf(file, "]%c ", ch);
	      break;
	case GI_START    :
		      fprintf(file, "ST_%s(%s)%c ", node_gi(n),
			      nul, ch);
		      break;
	case GI_END      :
		      fprintf(file, "END_%s(%s)%c ", node_gi(n),
			      nul, ch);
		      break;
	case GI_NAME     :
		      fprintf(file, "%s(%s)%c ", node_gi(n),
			      nul, ch);
		      break;
	default : report(NO_LABEL, FATAL, 0, 0, node_type(n), "node_write");
		  break;
    }
}

void node_put(file, n)
FILE     *file;
P_Node    n;
{
    P_Iterator     iterator;
    char           ch = ' ';
    char           c = ' ';
    Bool           first;

    switch( node_occ(n) ) {
	case OCC_OPT    : ch = '?';
		          break;
	case OCC_REP    : ch = '*';
		          break;
	case OCC_PLUS   : ch = '+';
    		          break;
	case NO_OCC     : ch = ' ';   break;
	default : report(NO_LABEL, FATAL, 0, 0, node_occ(n), "print_node");
		  break;
    }
    if( is_connector( node_type(n) ) ){
	switch( node_type(n) ) {
	    case CONN_SEQ    : c = ',';
			      break;
	    case CONN_OR    : c = '|';
			      break;
	    case CONN_AND   : c = '&';
			      break;
	}
    }

    switch( node_type(n) ) {
        case KEY    : fprintf(file, "%s%c ", key_str(node_key(n)), ch);
		      break;
	case GI     : if( TRUE ){
			  fprintf(file, "%s%c ", node_gi(n), ch);
			  break;
		      }
        case GROUP  : 
        case CONN_AND    :
	case CONN_SEQ    :
	case CONN_OR     :
	    fprintf(file, "(");
	    iterator = group_iterator(node_group(n));
	    first = TRUE;
	    while( iterator_next(iterator) ){
		if( not first ){
		    fprintf(file, "%c", c);
		} else {
		    first = FALSE;
		}
		node_put(file, (P_Node)iterator_elem(iterator) );
	    }
	    fprintf(file, ")%c ", ch);
	    break;
	case GI_START    :
		      fprintf(file, "ST_%s%c", node_gi(n), ch);
		      break;
	case GI_END      :
		      fprintf(file, "END_%s%c", node_gi(n), ch);
		      break;
	case GI_NAME     :
		      fprintf(file, "NAME_%s%c ", node_gi(n), ch);
		      break;
	default : report(NO_LABEL, FATAL, 0, 0, node_type(n),
			 "print_node");
		  break;
    }
}
#endif
