/* $Header: ambigu.c,v 3.0 88/04/13 15:37:03 jos Exp $ */
/*
 *  This file is part of the Amsterdam SGML Parser.
 *
 *  Copyright: Faculteit Wiskunde en Informatica
 *             Department of Mathematics and Computer Science
 *             Vrije Universiteit Amsterdam
 *             The Netherlands
 *
 *  Authors:   Sylvia van Egmond
 *             Jos Warmer
 */

#include "types.h"
#include "ambigu.h"
#include "element.h"
#include "keywords.h"
#include "node.h"


static P_Element  current_elem;
static int        nr_elems = 0;    /* total number of elements */
static int        set_size = 0;    /* size of the set of follow symbols */

/* forward declaration of recursive functions */
static void check_loop_node       (PAR  P_Node           RAP);
static void Celement_first_symbols(PAR  P_Element        RAP);
static Bool test_propagate        (PAR  P_Element, P_Set RAP);
static void add_propagate         (PAR  P_Element, P_Set RAP);
static void clash                 (PAR  P_Node, P_Node, P_Set, P_Element RAP);


/****************************** S T A T I S T I C S **************************/
#define  NR_KEYS  6

#ifdef DEBUG
static Bool       debug = FALSE;
static Bool       omit_flag = FALSE;

static int nr_start = 0;    /* number of GI_START nodes  */
static int nr_end   = 0;    /* number of GI_END   nodes  */
static int nr_name  = 0;    /* number of GI_NAME  nodes  */
static int nr_gi    = 0;    /* number of GI       nodes  */
static int nr_key   = 0;    /* number of KEY      nodes  */
static int nr_or    = 0;    /* number of CONN_OR  nodes  */
static int nr_and   = 0;    /* number of CONN_AND nodes  */
static int nr_seq   = 0;    /* number of CONN_SEQ nodes  */
static int nr_other = 0;    /* number of other    nodes, should remain 0 */

static int nr_nodes = 0;    /* total number of nodes  */

static void statistics(node)
P_Node    node;
{
    switch( node_type(node) ){
	case GI_START  : nr_start++;  break;
	case GI_END    : nr_end++;    break;
        case GI_NAME   : nr_name++;   break;
	case GI        : nr_gi++;     break;
	case CONN_OR   : nr_or++;     break;
	case CONN_AND  : nr_and++;    break;
	case CONN_SEQ  : nr_seq++;    break;
	case KEY       : nr_key++;    break;
	default        : nr_other++;  break;
    }
}

static void get_info()
{
    P_Iterator iter;
    P_Element  elem;
    FILE      *file;

    nr_nodes = nr_start + nr_end + nr_gi + nr_or + nr_and + nr_seq + nr_other
	       + nr_name + nr_key;

    file = fopen("statistics", "w");
    if (file == 0) {
	report(FILE_OPEN, FATAL, 0, 0, "statistics");
    }
    fprintf(file, " nr_start : %d\n", nr_start );
    fprintf(file, " nr_end   : %d\n", nr_end   );
    fprintf(file, " nr_name  : %d\n", nr_name  );
    fprintf(file, " nr_gi    : %d\n", nr_gi    );
    fprintf(file, " nr_key   : %d\n", nr_key   );
    fprintf(file, " nr_or    : %d\n", nr_or    );
    fprintf(file, " nr_and   : %d\n", nr_and   );
    fprintf(file, " nr_seq   : %d\n", nr_seq   );
    fprintf(file, " nr_other : %d\n", nr_other );
    fprintf(file, " nr_nodes : %d\n", nr_nodes );
    fprintf(file, " nr_elems : %d\n", nr_elems );
    fclose(file);
    file = fopen("grammar", "w");
    if (file == 0) {
	report(FILE_OPEN, FATAL, 0, 0, "grammar");
    }
    iter = element_iterator();
    while( elem = next_element(iter) ){
	group_print(file, name_group(elem));
	fprintf(file, ": ");
	node_write(file, content(elem));
	fprintf(file, "\n");
    }
    fclose(file);
}
#endif DEBUG


static int key_to_nr(key)
int key;
{
    switch( key ){
	case  CDATA  : return 1;
	case RCDATA  : return 2;
	case PCDATA  : return 3;
	case EMPTY   : return 4;
	case ANY     : return 5;
	case XPCDATA : return 6;
	default      : report(NO_LABEL, FATAL, 0, 0, key, "key_to_nr");
    }
}

static int nr_to_key(nr)
int nr;
{
    int base = nr_elems + nr_elems;

    if( nr == base+1 ){  return CDATA  ;}
    if( nr == base+2 ){  return RCDATA  ;}
    if( nr == base+3 ){  return PCDATA  ;}
    if( nr == base+4 ){  return EMPTY  ;}
    if( nr == base+5 ){  return ANY    ;}
    if( nr == base+6 ){  return XPCDATA  ;}
    report(NO_LABEL, FATAL, 0, 0, nr, "nr_to_key");
}

/********************* initialize leaf numbers and node sets ******************/

/*  Initializes the firstsymbols and followsymbols sets
 *  for `node'.
 */
static void init_node(node, elem)
P_Node    node;
P_Element elem;
{
    int t = node_type(node);

    if( t == GI_NAME ){
	elem_reference(node_gi(node), node, elem);
    } else { /* t != GI_NAME */
	node_set_firstsym (node, new_set(set_size  ));
    }
    node_set_followsym(node, new_set(set_size  ));

#ifdef DEBUG
    if (debug) statistics(node);
#endif
}

/*
 *  Initializes each element.
 */
static void initialize()
{
    nr_elems = number_of_elements();
    set_size = nr_elems + nr_elems + NR_KEYS + 1;

    elem_traverse_post(init_node);
}

/************************ F I R S T   S E T S *********************************/

static void Cnode_first_symbols(node, dummy)
P_Node     node;
P_Element  dummy;
{
    P_Element  elem;
    P_Iterator iter;
    P_Node     n;
    Bool       empty;

    if( node_done(node) ){ return; }

    switch( node_type(node) ){
        case GI_START :
	    DEB1("add START %s\n", node_gi(node));
	    set_add(node_firstsym(node), element_number(node_gi(node)) );
	    break;
	case GI_END:
	    DEB1("add END %s\n", node_gi(node));
	    set_add(node_firstsym(node),
		    nr_elems + element_number( node_gi(node) ) );
	    break;
	case KEY:
	    DEB1("add KEY %s\n",key_str(node_key(node)));
	    set_add(node_firstsym(node),
		    nr_elems + nr_elems + key_to_nr( node_key(node) ) );
	    break;
	case GI_NAME:
	    elem = element(node_gi(node));
	    Celement_first_symbols(elem);
	    node_set_firstsym(node, node_firstsym(content(elem)));
	    break;
        case CONN_AND:
        case CONN_OR :
	    iter = group_iterator(node_group(node));
	    while( n = next_node(iter) ){
		Cnode_first_symbols(n, dummy);
	        set_union( node_firstsym(node), node_firstsym(n) );
	    }
	    break;
	case GI:
        case CONN_SEQ :
            iter = group_iterator(node_group(node));
            empty = TRUE;
            while( (n = next_node(iter)) and empty ){
		Cnode_first_symbols(n, dummy);
                set_union( node_firstsym(node), node_firstsym(n) );
                empty = (node_nullable(n) == Yes);
            }
	    if( n ){ iterator_delete(iter); }
	    break;
        default :
	    report(NO_LABEL, FATAL, 0, 0, node_type(node),
		   "Cnode_first_symbols");
	    break;
    }
    node_set_done(node, TRUE);
}

static void Celement_first_symbols(elem)
P_Element  elem;
{
    if( element_done(elem) ){ return; }

    if( element_busy(elem) ){
	report(AMB_LOOP, FATAL, name_group(elem), 0);
    } else {
        element_set_busy(elem, TRUE);
        Cnode_first_symbols(content(elem), elem);
        element_set_busy(elem, FALSE);
        element_set_done(elem, TRUE );
    }
}

static void calc_element_first_symbols()
{
    P_Iterator iter;
    P_Element  elem;

    iter = element_iterator();
    while( elem = next_element(iter) ){
        Celement_first_symbols(elem);
    }
}

static void calc_all_first_symbols()
{
    elem_traverse_post(Cnode_first_symbols);
}

/*********************************************************************/

static void check_alternate_node(node, elem)
P_Node     node;
P_Element  elem;
{
    P_Iterator  it1;
    P_Iterator  it2;
    P_Node      n;
    P_Node      m;
    P_Set       s;
    int         t = node_type(node);

    if( (t==CONN_OR) or (t==CONN_AND) ){
	s = new_set(set_size);
	it1 = group_iterator(node_group(node));
	while( n = next_node(it1) ){
	    it2 = group_iterator(node_group(node));
	    while( (m = next_node(it2)) != n );
	    while( m = next_node(it2) ){
		set_assign(s, node_firstsym(n));
		set_intersect(s, node_firstsym(m));
		if( not set_empty(s) ){
		    report(AMB_ALTER, FATAL, name_group(current_elem), s);
		}
	    }
	}
	delete_set(s);
    }
}


static void check_alternates()
{
    P_Iterator iter;
    P_Element  elem;

    iter = element_iterator();
    while( elem = next_element(iter) ){
#ifdef DEBUG
	if( debug ){
	    fprintf(fpdg, "********* CHECK ALTERNATES ELEMENTS ");
	    group_print(fpdg, name_group(elem));
	    fprintf(fpdg, "\n");
	}
#endif
	current_elem = elem;
        node_traverse_post(content(elem), check_alternate_node, elem);
    }
}

/*********************************************************************/

static void check_follows_node(node, elem)
P_Node     node;
P_Element  elem;
{
    P_Set       s;

    if( (node_nullable(node) == Yes) or (node_occ(node) == OCC_PLUS) ){
	s = new_set(set_size);
	set_assign(s, node_firstsym(node));
	set_intersect(s, node_followsym(node));
	if( not set_empty(s) ){
	    report(AMB_FOLLOWS, FATAL, name_group(current_elem), s);
	}
	delete_set(s);
    }
}

static void check_follows()
{
    P_Iterator iter;
    P_Element  elem;

    iter = element_iterator();
    while( elem = next_element(iter) ){
#ifdef DEBUG
	if( debug ){
	    fprintf(fpdg, "********* CHECK FOLLOWS ELEMENTS ");
	    group_print(fpdg, name_group(elem));
	    fprintf(fpdg, "\n");
	}
#endif
	current_elem = elem;
        node_traverse_post(content(elem), check_follows_node, elem );
    }
}

/*********************************************************************/

static void followsyms(node, elem)
P_Node     node;
P_Element  elem;
{
    P_Node       n;
    P_Node       m;
    P_Iterator   iter;
    P_Iterator   it2;
    Bool         empty;

    switch( node_type(node) ){
        case GI :
        case CONN_SEQ :
	    iter = group_iterator(node_group(node));
            while( n = next_node(iter) ){
		it2 = group_iterator(node_group(node));
		while( (m = next_node(it2)) != n );
		empty = TRUE;
                while( empty and (m = next_node(it2)) ){
		    set_union(node_followsym(n), node_firstsym(m));
                    empty = (node_nullable(m) == Yes);
                }
		if( m ){ iterator_delete(it2); }
		if( empty ){
		    if( node_parent(n) ){  /* if not, it's the root */
			set_union(node_followsym(n), node_followsym(node_parent(n)));
		    }
		}
            }
            break;
        case CONN_AND :
        case CONN_OR  :
	    iter = group_iterator(node_group(node));
            while( n = next_node(iter) ){
		set_assign(node_followsym(n), node_followsym(node));
	    }
            break;
	case GI_START :
	case GI_END   :
	case GI_NAME  :
	case KEY      : break;
        default       : report(NO_LABEL, FATAL, 0, 0, node_type(node),
		        "followsyms");
    }
}

static void calc_all_follow_symbols()
{
    elem_traverse_pre(followsyms);
}

/*********************************************************************/

static FILE *print_file ;
static void print_tag(t)
int t;
{
    String name;

    if( t <= nr_elems ){
	name = number_to_name(t);
	fprintf(print_file, "'%s' ", name);
    } else if( t <= (nr_elems + nr_elems) ){
	name = number_to_name(t-nr_elems);
	fprintf(print_file, "'%s' ", name);
    } else {
	fprintf(print_file, "'#%s' ", key_str(nr_to_key(t)));
    }
}

void sym_set_print(file, set)
FILE  *file;
P_Set  set;
{
    print_file = file;
    set_generator(set, print_tag);
}

void sym_print(file, sym)
FILE  *file;
int    sym;
{
    print_file = file;
    print_tag(sym);
}

static void check_loop_element(elem)
P_Element elem;
{
    if( element_done(elem) ){ return; }
    if( element_busy(elem) ){
	report(AMB_LOOP, FATAL, name_group(elem), 0);
	return;
    }

    element_set_busy(elem, TRUE);
    check_loop_node(content(elem));
    element_set_done(elem, TRUE);
    element_set_busy(elem, FALSE);
}

static void check_loop_node(node)
P_Node  node;
{
    P_Iterator iter;
    P_Node     child;
    int        t = node_type(node);

    if( node_nullable(node) == No ){
        if( is_connector(t) or (t==GI) ){
	    iter = group_iterator(node_group(node));
	    while( child = next_node(iter) ){
		check_loop_node(child);
	    }
	}
	if( t == GI_NAME ){
	    check_loop_element(element(node_gi(node)) );
	}
    }
}

static void check_loops()
{
    P_Iterator iter;
    P_Element  elem;

    iter = element_iterator();
    while( elem = next_element(iter) ){
        check_loop_element(elem);
    }

    iter = element_iterator();
    while( elem = next_element(iter) ){
	element_set_done(elem, FALSE);
    }
}

static void check_repetition(home, node, add_followsyms, elem)
P_Node      home;
P_Node      node;
P_Set       add_followsyms;
P_Element   elem;
{
    P_Iterator it;
    Bool       empty;
    P_Node     child;

    switch( node_type(node) ){
    case GI:
    case CONN_SEQ:
	it = group_iterator(node_group(node));
	empty = TRUE;
	while( empty and (child = previous_node(it)) ){
	    clash(home, child, add_followsyms, elem);
	    check_repetition(home, child, add_followsyms, elem);
	    empty = (node_nullable(child)==Yes) or (node_occ(child)==OCC_PLUS);
	}
	break;
    case CONN_AND:
    case CONN_OR:
	it = group_iterator(node_group(node));
	while( child = next_node(it) ){
	    clash(home, child, add_followsyms, elem);
	    check_repetition(home, child, add_followsyms, elem);
	}
	break;
    }
}

static void repetitions(node, elem)
P_Node    node;
P_Element elem;
{
    P_Set    add_followsyms;

    if( (node_occ(node)==OCC_REP) or (node_occ(node)==OCC_PLUS) ){
#ifdef DEBUG
	if( debug ){
	    fprintf(fpdg, "node : ");
	    node_put(fpdg, node);
	    fprintf(fpdg, "  REP/PLUS\n");
	    fflush(fpdg);
	}
#endif
	add_followsyms = new_set(set_size);
	set_assign(add_followsyms, node_firstsym(node) );
	check_repetition(node, node, add_followsyms, elem);
	delete_set(add_followsyms);
    } 
}

static void find_repetitions()
{
    elem_traverse_pre(repetitions);
}

static Bool is_clash(home, node, set_elem)
P_Node      home;
P_Node      node;
int         set_elem;
{
    P_Iterator it;
    P_Node     child;

#ifdef DEBUG
    if( debug ){
	fprintf(fpdg, "is_clash: home: ");
	node_put(fpdg, home); fprintf(fpdg, "\n");
	fprintf(fpdg, "          node: ");
	node_put(fpdg, node); fprintf(fpdg, "\n");
    }
#endif

    switch( node_type(home) ){
    case CONN_SEQ:
	it = group_iterator( node_group(home) );
	while( child = next_node(it) ){
	    if( set_member(node_firstsym(child), set_elem) ){
		if( child==node ){
		    DEB("child==node\n");
		    return FALSE;
		} else if( is_connector(node_type(child)) ){
		    return is_clash(child, node, set_elem);
		} else {
		    DEB("child!=node\n");
		    return TRUE;
		}
	    }
	}
	break;
    case CONN_AND:
    case CONN_OR:
	it = group_iterator( node_group(home) );
	while( child = next_node(it) ){
	    if( set_member(node_firstsym(child), set_elem) ){
		if( child==node ){
		    DEB("child==node\n");
		    return FALSE;
		} else if( is_connector(node_type(child)) ){
		    return is_clash(child, node, set_elem);
		} else {
		    DEB("child!=node\n");
		    return TRUE;
		}
	    }
	}
	break;
    default: 
	report(NO_LABEL, FATAL, 0, 0, node_type(home), "is_clash");
	return FALSE;
    }
}

static void clash(home, node, add_followsyms, elem)
P_Node      home;
P_Node      node;
P_Set       add_followsyms;
P_Element   elem;
{
    P_Set           s;
    P_Set_iterator  setit;
    int             set_elem;

#ifdef DEBUG
    if( debug ){
	fprintf(fpdg, "clash: home: ");node_put(fpdg,home);fprintf(fpdg,"\n");
	fprintf(fpdg, "       node: ");node_put(fpdg,node);fprintf(fpdg,"\n");
	fprintf(fpdg, "first: ");
	sym_set_print(fpdg, node_firstsym(node)); fprintf(fpdg, "\n");
	fflush(fpdg);
    }
#endif

    if( (node_nullable(node)==Yes) or (node_occ(node)==OCC_PLUS) ){
	s = new_set(set_size);
	set_assign(s, add_followsyms);
	set_intersect(s, node_firstsym(node));
	if( not set_empty(s) ){
	    /* maybe a clash */
	    setit = set_iterator(s);
	    while( set_elem = set_next(setit) ){
		if( is_clash(home, node, set_elem) ){
		    report(AMB_CLASH, FATAL, name_group(elem), 0, set_elem);
		} 
#ifdef DEBUG
		else {
		    if (debug) {
		        fprintf(fpdg, "No clash for element ");
                        sym_print(fpdg, set_elem);
		        fprintf(fpdg, "\n");
		    }
		}
#endif
	    }
	}
	delete_set(s);
    }
    set_union(node_followsym(node), add_followsyms);
}

/**********************************************************************
 *
 *  Main procedure
 *
 **********************************************************************/
void check_ambigu()
{
    initialize();
#ifdef DEBUG
    if (debug) get_info();
#endif
    check_loops();

    calc_element_first_symbols();
    calc_all_first_symbols();
    calc_all_follow_symbols();

    check_alternates();
    check_follows();
    find_repetitions();

    fatal_report();
}

static Bool test_node_firstsyms(node,  add_firstsyms, elem)
P_Node     node;
P_Set      add_firstsyms;
P_Element  elem;
{
    P_Node      parent;
    P_Set       test_set;
    P_Node      child;
    P_Iterator  it;
    Bool        ok;

    if( node_busy(node) ){ return TRUE; }  /* this node is done, avoid loops */
    node_set_busy(node, TRUE);

    parent = node_parent(node);
    if( not parent ){
	return test_propagate(elem, add_firstsyms);
    }
#ifdef DEBUG
if( omit_flag ){
    fprintf(fpdg,"test_node_firstsyms set is: "); sym_set_print(fpdg,add_firstsyms);
    fprintf(fpdg,"\nparent : ");          node_put(fpdg, parent);
    fprintf(fpdg,"\nnode: ");             node_put(fpdg, node);
    fprintf(fpdg,"\n");
}
#endif


    switch( node_type(parent) ){
	case GI: DEB("test_firstsyms: GI\n");
	case GI_NAME:
	case CONN_SEQ:
	    it = group_iterator( node_group(parent) );
	    while( (child = previous_node(it)) != node );

	    while( child = previous_node(it) ){
		set_minus(add_firstsyms, node_followsym(child));
		if( set_empty(add_firstsyms) ){		/* no changes anymore */
		    node_set_busy(node, FALSE);
		    return TRUE;
		}

	        if( node_nullable(child)==Yes ){
		    test_set = set_copy(node_firstsym(child));
		    set_intersect(test_set, add_firstsyms);
		    if( not set_empty(test_set) ){
#ifdef DEBUG
			if( omit_flag ){
			    fprintf(fpdg,"AMBIGUITY: first == follow: ");
			    sym_set_print(fpdg, test_set);
			    fprintf(fpdg,"\n");
			}
#endif
			delete_set(test_set);
			node_set_busy(node, FALSE);
			return FALSE;
		    } else {
			delete_set(test_set);
		    }
		} else {
		    node_set_busy(node, FALSE);
		    return TRUE;
		}
	    }
	    ok = test_node_firstsyms(parent,add_firstsyms,elem);
	    node_set_busy(node, FALSE);
	    return ok;
	case CONN_OR:
	case CONN_AND:
	    test_set = set_copy(add_firstsyms);
	    set_intersect(test_set, node_firstsym(parent));
	    if( not set_empty(test_set) ){
#ifdef DEBUG
		if (debug) {
		    fprintf(fpdg,"AMBIGUITY in OR/AND : ");
		    sym_set_print(fpdg, add_firstsyms);
		    fprintf(fpdg,"\n");
		}
#endif
		delete_set(test_set);
		node_set_busy(node, FALSE);
		return FALSE;
	    } else {
		delete_set(test_set);
		ok = test_node_firstsyms(parent,add_firstsyms,elem);
		node_set_busy(node, FALSE);
		return ok;
            }
	    break;
	default:
	    report(NO_LABEL, FATAL, 0, 0, node_type(parent),
		   "test_node_firstsyms");
    }
}

static void add_node_firstsyms(node,  add_firstsyms, elem)
P_Node     node;
P_Set      add_firstsyms;
P_Element  elem;
{
    P_Node      parent;
    P_Node      child;
    P_Iterator  it;

    if( node_busy(node) ){ return; }     /* this node is done, avoid loops */
    node_set_busy(node, TRUE);

    parent = node_parent(node);
    if( not parent ){
	add_propagate(elem, add_firstsyms);
    }

    switch( node_type(parent) ){
	case GI: 
	case GI_NAME:
	case CONN_SEQ:
	    it = group_iterator( node_group(parent) );
	    while( (child = previous_node(it)) != node );

	    while( child = previous_node(it) ){
		set_minus(add_firstsyms, node_followsym(child));
		set_union(node_followsym(child), add_firstsyms);
		if( set_empty(add_firstsyms) ){		/* no changes anymore */
		    node_set_busy(node, FALSE);
		    return ;
		}

	        if( node_nullable(child)==Yes ){
		    set_union(node_firstsym(child), add_firstsyms);
		} else {
		    node_set_busy(node, FALSE);
		    return ;
		}
	    }
	    add_node_firstsyms(parent,add_firstsyms,elem);
	    node_set_busy(node, FALSE);
	    return ;
	case CONN_OR:
	case CONN_AND:
	    set_minus(add_firstsyms, node_firstsym(parent));
	    if( not set_empty(add_firstsyms) ){
		set_union(node_firstsym(parent), add_firstsyms);
		add_node_firstsyms(parent, add_firstsyms, elem);
            }
	    break;
	default:
	    report(NO_LABEL, FATAL, 0, 0, node_type(parent),
		   "add_node_firstsyms");
    }
}

/*  Check whether it is ok to omit the starttag of 'node'.
 *  'elem' is the element which contains 'node.
 */
Bool omit_ok(node, elem)
P_Node     node;
P_Element  elem;
{
    P_Set       add_firstsyms;
    P_Set       set;
    int         ok;
    assert( node_type(node) == GI );

#ifdef DEBUG
    if( omit_flag ){
	fprintf(fpdg,"omit_ok: "); group_print(fpdg, name_group(elem) );
	fprintf(fpdg,"\nnode : "); node_put   (fpdg, node);
	fprintf(fpdg,"\n");
    }
#endif

    add_firstsyms = set_copy(node_firstsym(content(element(node_gi(node)))));

    /*  Test for recursive element 
     */
    set = set_copy( node_firstsym(node) );
    set_intersect(set, add_firstsyms);
    if( not set_empty(set) ){
	DEB("omit_ok FALSE, recursive element\n");
	delete_set(add_firstsyms);
	delete_set(set);
	return FALSE;
    }
    delete_set(set);

    /*  Make node nullable for the moment and
     *  check first == follow for this node.
     *  This is a choice, depending on the interpretation of
     *  the term 'ambiguous' in the Standard.
     *  We choose to make it not ambigu.
     */
    /*
    if( (node_nullable(content(element( node_gi(node) )))==Yes) ){
	DEB("contents may be empty, nullable set to Yes\n");
	node_set_nullable(node, Yes); 
        set = set_copy(node_firstsym(node));
	set_union(set, add_firstsyms);
	set_intersect(set, node_followsym(node));
	if( not set_empty(set) ){
	    DEB("omit_ok FALSE, element first == follow\n");
	    delete_set(add_firstsyms);
	    delete_set(set);
	    return FALSE;
	}
	delete_set(set);
    }
    */
    /*  Check if it is ok to omit the starttag
     *  Goes through the grammar tree  as far as needed.
     */
    ok = test_node_firstsyms(node, add_firstsyms, elem);

    delete_set(add_firstsyms);

    /*  Now make changes according to the omitted tag
     */
    if( ok ){
	add_firstsyms= set_copy(node_firstsym(content(element(node_gi(node)))));
	add_node_firstsyms(node, add_firstsyms, elem);
	delete_set(add_firstsyms);
    } else {
	/* node_set_nullable(node, No); */     /* Just in case it was changed */
    }

    return ok;
}

static Bool test_propagate(elem, set)
P_Element  elem;
P_Set      set;
{
    P_Iterator    it;
    P_Reference   ref;
    Bool          ok = TRUE;

#ifdef DEBUG
    if( omit_flag ){
	fprintf(fpdg,"propagate to element: ");
	group_print(fpdg, name_group(elem) );
	fprintf(fpdg,"\n");
    }
#endif
    it = group_iterator( elem_references(elem) );
    while( ref = (P_Reference)iterator_next(it) ){
	DEB("next reference");
	ok = test_node_firstsyms( reference_node(ref), set,
			   reference_elem(ref))
	     and ok;
	if( not ok ){ /* ambiguity found, so don't look further */
	    return ok;
	}
    }
    return ok;
}

static void add_propagate(elem, set)
P_Element  elem;
P_Set      set;
{
    P_Iterator    it;
    P_Reference   ref;

    it = group_iterator( elem_references(elem) );
    while( ref = (P_Reference)iterator_next(it) ){
	add_node_firstsyms( reference_node(ref), set,
			    reference_elem(ref));
    }
}

#ifdef DEBUG
void debug_ambigu(b)
Bool b;
{
    omit_flag = b;
    debug     = b;
}
#endif
