;;; genscmint.scm -- Generate an SCM interface to foreign functions.
;;;
;;; This reads a file containing forms that define foreign functions and
;;; generates an output file containing C source code that can be compiled and
;;; linked with SCM to provide Scheme functions that convert parameters to the
;;; form expected by the foreign functions, call the foreign functions, and
;;; convert any results to SCM's internal form and return them in a vector.
;;; The generated C source code includes the file SCMINT.H and calls functions
;;; defined in SCMINT.C.
;;;
;;; The define-foreign forms each describe one foreign function.  Each
;;; function returns a vector that is at least one element long.  The
;;; first element is the status code returned by the routine.  Any
;;; other elements are the values of the out parameters after the
;;; call.
;;;
;;; Optional parameters may be replaced by a boolean, in which case
;;; they are ignored.
;;;
;;; Out parameters are returned in the result vector.
;;;
;;; In-out parameters appear in both the parameter list and the result
;;; vector.
;;;
;;;
;;; I use this to generate a SMG interface for SCM from a file generated from
;;; a .SDI file, so it has lots of VMSisms.
;;;
;;; Usage: 
;;;        (driver "in.file" "module-name" "out.file")
;;;
;;; Note: the generated code uses BIGNUMS!
;;;
;;; Here is the definition of the define-foreign form:
;;;
;;; (define-foreign <internal-name>
;;;   (<external-name> <return-type>
;;;                    (<param-name> <param-type> <access> <pass-mechanism>
;;;                                  [<modifiers> ...])
;;;                    ...
;;;                    ))
;;; Where:
;;;  param-type is one of  longword, byte, word, character, quadword
;;;             (character is a string)
;;; access is one of in, out, in-out
;;; pass-mechanism is one of ref, value
;;; modifiers is zero or more of optional, descriptor.
;;;
;;;
;;; Todo:  Modify to produce additional files that contain wrapper Scheme
;;;        routines that return multiple results or take success and failure
;;;        continuations (as per suggestions from comp.lang.scheme).

(load "def-mac.scm")
(load "def-rec.scm")
;(require 'pretty-print)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Data Structures

(define-record foreign (internal-name external-name result-type parameters))

(define-record parameter (name type access mechanism modifiers))

(define-record function-entry (name-var int-fun))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Support routines

(define (tell . objs)
  (for-each display objs)
  (newline))

(define (count-trues pred? ls)
  (let loop ((n 0)
	     (ls ls))
    (if (null? ls)
	n
	(loop (if (pred? (car ls)) (+ 1 n) n) (cdr ls)))))


(define (in-parameter? param)
  (or (eq? (parameter->access param) 'in)
      (eq? (parameter->access param) 'in-out)))


(define (out-parameter? param)
  (or (eq? (parameter->access param) 'out)
      (eq? (parameter->access param) 'in-out)))


(define (for-each-with-count fun ls)	;returns number of items processed
  (let loop ((idx 0)
	     (ls ls))
    (cond ((null? ls)
	   idx)
	  (else
	   (fun (car ls) idx)
	   (loop (+ 1 idx) (cdr ls))))))
	


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Parsing

(define (parse-foreign obj)
  (let ((ext (caddr obj)))
    (make-foreign (cadr obj)
		  (car ext)
		  (cadr ext)
		  (parse-parameters (cddr ext)))))

(define (parse-parameters param-list)
  (let loop ((param-list param-list)
	     (out-list '()))
    (cond
     ((null? param-list)
      (reverse out-list))
     (else
      (let ((param (car param-list)))
	(loop (cdr param-list)
	      (cons (make-parameter (car param)
				    (cadr param)
				    (caddr param)
				    (cadddr param)
				    (cddddr param))
		    out-list)))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Processing

(define (external-parameter->string param)
  (case (parameter->type param)
    ((ADDRESS)
     (if (eq? (parameter->mechanism param) 'ref)
	 "/*address*/long *"
	 "/*address*/long"))
    ((ANY)
     (if (eq? (parameter->mechanism param) 'ref)
	 "/*any*/long *"
	 "/*any*/long"))
    ((BYTE)
     (if (eq? (parameter->mechanism param) 'ref)
	 "/*byte*/unsigned char*"
	 "/*byte*/unsigned char"))
    
    ((CHARACTER)			;string, actually
     (if (eq? (parameter->mechanism param) 'ref)
	 "/*character*/struct dsc$descriptor_s *"
	 "/*character*/struct dsc$descriptor_s"))
    ((LONGWORD)
     (if (eq? (parameter->mechanism param) 'ref)
	 "/*longword*/long *"
	 "/*longword*/long"))
    ((WORD)
     (if (eq? (parameter->mechanism param) 'ref)
	 "/*word*/unsigned short *"
	 "/*word*/unsigned short"))
    ((QUADWORD)
     (if (eq? (parameter->mechanism param) 'ref)
	 "/*quadword*/struct sit_quadword *"
	 "/*quadword*/struct sit_quadword"))
    (else
     (tell "internal: unhandled type: " (parameter->type param))
     "/*oops*/unknown")))


(define (write-external-parameters param-list)
  (let loop ((param-list param-list)
	     (first #t))
    (cond
     ((not (null? param-list))
      (if (not first)
	  (display-out ", "))
      (display-out (external-parameter->string (car param-list)))
      (loop (cdr param-list) #f)))))


(define (type->string type)
  (case type
    ((ADDRESS) "long")
    ((BYTE) "unsigned char")
    ((CHARACTER) "struct dsc$descriptor_s")
    ((BOOLEAN) "long")
    ((DECIMAL) "long")
    ((DFLOAT) "double")
    ((FFLOAT) "float")
    ((GFLOAT) "/*oops*/gfloat")
    ((HFLOAT) "/*oops*/hfloat")
    ((LONGWORD) "long")
    ((OCTAWORD) "/*oops*/octaword")
    ((QUADWORD) "/*oops*/struct sit_quadword")
    ((BITFIELD) "/*oops*/bitfield")
    ((WORD) "unsigned short")
    ((STRUCTURE) "/*oops*/structureX")
    ((UNION) "/*oops*/unionX")
    ((ANY) "long")
    ((ENTRY) "long")
    ((DLFOAT_COMPLEX) "/*oops*/dfloat_complex")
    ((FLFOAT_COMPLEX) "/*oops*/ffloat_complex")
    ((GLFOAT_COMPLEX) "/*oops*/gfloat_complex")
    ((HLFOAT_COMPLEX) "/*oops*/hfloat_complex")
    (else
     (tell "internal: unknown type: " type)
     "/*oops*/unknown")))


(define external-name->string symbol->string)


(define (write-external f)
  (display-out "extern "
	       (type->string (foreign->result-type f))
	       " " 
	       (external-name->string (foreign->external-name f))
	       " (")
  (write-external-parameters (foreign->parameters f))
  (display-out ");" nl))


;;; sis = Scm Interface String
(define (external-name->name-var ext-name)
  (string-append "sis_" (external-name->string ext-name)))

;;; sif = Scm Interface Function
(define (external-name->int-fun ext-name)
  (string-append "sif_" (external-name->string ext-name)))


(define (write-internal-name-string f)
  (display-out "static char "
	       (external-name->name-var (foreign->external-name f))
	       "[] = \""
	       (foreign->internal-name f)
	       "\";"
	       nl))


(define (write-internal-defn f)
  (display-out "SCM" nl
	       (external-name->int-fun (foreign->external-name f))
	       " (SCM l)" nl))


(define (declare-inargs param-list)
  (for-each-with-count
   (lambda (param idx)
     (cond ((in-parameter? param)
	    (display-out "  SCM "
			 "inarg_" (number->string (+ 1 idx)) ";" nl) )))
   param-list))


(define (for-each-with-count&true-count fun ls pred?)
  (let loop ((idx 0)
	     (in-idx 0)
	     (ls ls))
    (cond ((null? ls)
	   idx)
	  (else
	   (fun (car ls) idx in-idx)
	   (loop (+ 1 idx)
		 (if (pred? (car ls))
			       (+ 1 in-idx)
			       in-idx)	;was idx, why?
		 (cdr ls))))))


(define (declare-outres param-list)
  (for-each-with-count
   (lambda (param idx)
     (cond ((out-parameter? param)
	    (display-out "  SCM "
			 "outres_" (number->string (+ 1 idx)) ";" nl) )))
   param-list))


(define (declare-params param-list)
  (for-each-with-count
   (lambda (param idx)
     (display-out "  " (type->string (parameter->type param)) " "
		  (parameter->name param))
     (if (eq? 'CHARACTER (parameter->type param))
	 (display-out " = {0, DSC$K_DTYPE_T, DSC$K_CLASS_D, 0}"))
     (display-out ";" nl))
   param-list))


(define (write-call f)
  (display-out "  extres = " (foreign->external-name f) nl
	       "    (")
  (for-each-with-count&true-count
   (lambda (param idx in-idx)
     (if (> idx 0) (display-out ", " nl "     "))
     (cond
      ((and (in-parameter? param)
	    (member 'optional (parameter->modifiers param)))
       (display-out "((num_args > " (number->string in-idx)
		    ") && !si_booleanp (inarg_" (number->string (+ idx 1))
		    ") ? " (if (eq? (parameter->mechanism param) 'ref) "&" "")
		    (parameter->name param) " : 0)"))
      (else
       (if (eq? (parameter->mechanism param) 'ref) (display-out "&"))
       (display-out (parameter->name param)))))
   (foreign->parameters f)
   in-parameter?)
  (display-out ");" nl))

;;; couldn't this be replaced with (count-trues in-parameter? param-list) ???
(define (number-of-in-parameters param-list)
  (let loop ((n 0)
	     (param-list param-list))
    (if (null? param-list)
	n
	(loop (if (in-parameter? (car param-list))
		  (+ 1 n)
		  n)
	      (cdr param-list)))))

;;; couldn't this be replaced with (count-trues out-parameter? param-list) ???
(define (last-required-in-arg param-list)
  (let loop ((lra 0)
	     (idx 0)
	     (param-list param-list))
    (if (null? param-list)
	lra
	(let ((param (car param-list)))
	  (loop (if (and (in-parameter? param)
			 (not (member 'optional
				      (parameter->modifiers param))))
		    (+ 1 idx)
		    lra)
		(if (in-parameter? param)
		    (+ 1 idx)
		    idx)
		(cdr param-list))))))


(define (check-number-of-args f)
  (let ((param-list (foreign->parameters f))
	(name-var (external-name->name-var (foreign->external-name f))))
    (cond
     ((null? param-list)
      (display-out "  ASSERT (num_args == 0, l, SI_WNA, " name-var ");" nl))
     (else
      (display-out "  ASSERT ((num_args >= " (last-required-in-arg param-list)
		   ") && (num_args <= " (number-of-in-parameters param-list)
		   "), l, WNA, " name-var ");" nl)))))


(define (type->type-test type)
  (case type
    ((ADDRESS) "si_longwordp")
    ((BYTE) "si_unsigned_charp")
    ((CHARACTER) "si_stringp")
    ((BOOLEAN) "si_longwordp")
    ((DECIMAL) "si_longwordp")
    ((DFLOAT) "si_doublep")
    ((FFLOAT) "si_floatp")
    ((GFLOAT) "/*oops*/si_gfloatp")
    ((HFLOAT) "/*oops*/si_hfloatp")
    ((LONGWORD) "si_longwordp")
    ((OCTAWORD) "/*oops*/si_octawordp")
    ((QUADWORD) "/*oops*/si_quadwordp")
    ((BITFIELD) "/*oops*/si_bitfieldp")
    ((WORD) "si_unsigned_shortp")
    ((STRUCTURE) "/*oops*/si_structureXp")
    ((UNION) "/*oops*/si_unionXp")
    ((ANY) "si_longwordp")
    ((ENTRY) "si_longwordp")
    ((DLFOAT_COMPLEX) "/*oops*/si_dfloat_complexp")
    ((FLFOAT_COMPLEX) "/*oops*/si_ffloat_complexp")
    ((GLFOAT_COMPLEX) "/*oops*/si_gfloat_complexp")
    ((HLFOAT_COMPLEX) "/*oops*/si_hfloat_complexp")
    (else
     (tell "internal: unknown type: " type)
     "/*oops*/si_unknownp")))


(define (type->from-scm-proc type)
  (case type
    ((ADDRESS) "si_to_longword")
    ((BYTE) "si_to_unsigned_char")
    ((CHARACTER) "si_to_string")
    ((BOOLEAN) "si_to_longword")
    ((DECIMAL) "si_to_longword")
    ((DFLOAT) "si_to_double")
    ((FFLOAT) "si_to_float")
    ((GFLOAT) "/*oops*/si_to_gfloat")
    ((HFLOAT) "/*oops*/si_to_hfloat")
    ((LONGWORD) "si_to_longword")
    ((OCTAWORD) "/*oops*/si_to_octaword")
    ((QUADWORD) "/*oops*/si_to_quadword")
    ((BITFIELD) "/*oops*/si_to_bitfield")
    ((WORD) "si_to_unsigned_short")
    ((STRUCTURE) "/*oops*/si_to_structureX")
    ((UNION) "/*oops*/si_to_unionX")
    ((ANY) "si_to_longword")
    ((ENTRY) "si_to_longword")
    ((DLFOAT_COMPLEX) "/*oops*/si_to_dfloat_complex")
    ((FLFOAT_COMPLEX) "/*oops*/si_to_ffloat_complex")
    ((GLFOAT_COMPLEX) "/*oops*/si_to_gfloat_complex")
    ((HLFOAT_COMPLEX) "/*oops*/si_to_hfloat_complex")
    (else
     (tell "internal: unknown type: " type)
     "/*oops*/si_to_unknown")))


(define (type->to-scm-proc type)
  (case type
    ((ADDRESS) "si_from_longword")
    ((BYTE) "si_from_unsigned_char")
    ((CHARACTER) "si_from_string")
    ((BOOLEAN) "si_from_longword")
    ((DECIMAL) "si_from_longword")
    ((DFLOAT) "si_from_double")
    ((FFLOAT) "si_from_float")
    ((GFLOAT) "/*oops*/si_from_gfloat")
    ((HFLOAT) "/*oops*/si_from_hfloat")
    ((LONGWORD) "si_from_longword")
    ((OCTAWORD) "/*oops*/si_from_octaword")
    ((QUADWORD) "/*oops*/si_from_quadword")
    ((BITFIELD) "/*oops*/si_from_bitfield")
    ((WORD) "si_from_unsigned_short")
    ((STRUCTURE) "/*oops*/si_from_structureX")
    ((UNION) "/*oops*/si_from_unionX")
    ((ANY) "si_from_longword")
    ((ENTRY) "si_from_longword")
    ((DLFOAT_COMPLEX) "/*oops*/si_from_dfloat_complex")
    ((FLFOAT_COMPLEX) "/*oops*/si_from_ffloat_complex")
    ((GLFOAT_COMPLEX) "/*oops*/si_from_gfloat_complex")
    ((HLFOAT_COMPLEX) "/*oops*/si_from_hfloat_complex")
    (else
     (tell "internal: unknown type: " type)
     "/*oops*/si_from_unknown")))


(define (initialize-inargs f)
  (let ((name-var (external-name->name-var (foreign->external-name f)))
	(param-list (foreign->parameters f)))
    (for-each-with-count&true-count
     (lambda (param idx in-idx)
       (if (in-parameter? param)
	   (let ((arg (string-append "inarg_" (number->string (+ idx 1)))))
	     (display-out "  if (num_args > " in-idx ")" nl
			  "    {" nl
			  "      " arg " = CAR (l); l = CDR (l);" nl
			  "      ASSERT (" (type->type-test
					    (parameter->type param))
			  " (" arg ")")
	     (if (member 'optional (parameter->modifiers param))
		 (display-out " || si_booleanp (" arg ")"))
	     (display-out ", " arg ", ARG" (number->string (+ in-idx 1)) ", "
			  name-var ");" nl)
	     (if (member 'optional (parameter->modifiers param))
		 (display-out "      if (!si_booleanp (" arg "))"))	;???
	     (display-out "      " (parameter->name param) " = "
			  (type->from-scm-proc (parameter->type param))
			  " (" arg ");" nl)
	     (display-out "    }" nl))))
	   param-list
	   in-parameter?)))


(define (initialize-params f)
  #f)


(define (build-result f)
  (let* ((param-list (foreign->parameters f))
	 (num-out-params (count-trues out-parameter? param-list)))
    (display-out "  result = make_vector (MAKINUM (" num-out-params
		 "+1), UNSPECIFIED);" nl
		 "  data = VELTS (result); " nl)
    (display-out "  *data++ = " (type->to-scm-proc (foreign->result-type f))
		 " (extres);" nl)
    (for-each-with-count&true-count
     (lambda (param idx out-idx)
       (if (out-parameter? param)
	   (display-out "  *data++ = " (type->to-scm-proc
					(parameter->type param))
			" (" (parameter->name param) ");" nl))
       (if (eq? (parameter->type param) 'CHARACTER)
	   (display-out "  lib$sfree1_dd (&" (parameter->name param) ");" nl)))
     param-list
     out-parameter?)))


(define (write-internal-body f)
  (let* ((param-list (foreign->parameters f))
	 (num-in-params (count-trues in-parameter? param-list))
	 (num-out-params (count-trues out-parameter? param-list))
	 (num-results (+ 1 num-out-params)))
    (display-out "{" nl)
    ;; interesting stuff goes here
    (display-out "  int num_args = ilength (l);" nl
		 "  SCM result;" nl
		 "  " (type->string (foreign->result-type f)) " extres;" nl
		 "  SCM *data;" nl
		 )
    (declare-inargs param-list)
;    (declare-outres param-list);; not needed, stuffed in array
    (declare-params param-list)
    (check-number-of-args f)
    (initialize-inargs f)
    (initialize-params f)
    (write-call f)
    (build-result f)
    (display-out "  return result;" nl)
    (display-out "} /* " (external-name->int-fun (foreign->external-name f))
		 " */" nl)))


(define (save-function-entry f)
  (let ((ext-name (foreign->external-name f)))
    (set! function-entries
	  (cons (make-function-entry (external-name->name-var ext-name)
				     (external-name->int-fun ext-name))
		function-entries))))


(define (write-internal f)
  (write-internal-name-string f)
  (write-internal-defn f)
  (write-internal-body f)
  (save-function-entry f))


(define (process-foreign f)
  (write-external f)
  (write-internal f)
  (display-out nl nl nl))


(define (write-function-entries entry-list)
  (display-out "static iproc si_lsubrs[] = {" nl)
  (for-each
   (lambda (entry)
     (display-out "  { "
		  (function-entry->name-var entry)
		  ", "
		  (function-entry->int-fun entry)
		  " },"
		  nl))
   (reverse entry-list))
  (display-out "  { 0, 0 }," nl
	       "};" nl))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Simpleminded documentation

(define (write-doc f)
  (doc-out (foreign->external-name f) nl "    Inputs:" nl)
  (for-each-with-count&true-count
   (lambda (param idx tidx)
     (cond ((in-parameter? param)
	    (doc-out "        " (parameter->name param) " (" (+ idx 1) ") "
		     (parameter->type param))
	    (if (member 'optional (parameter->modifiers param))
		(doc-out " optional"))
	    (doc-out nl))))
   (foreign->parameters f)
   in-parameter?)
  (doc-out "    Results:" nl)
  (doc-out "        status (return value) longword" nl)
  (for-each-with-count&true-count
   (lambda (param idx tidx)
     (cond ((out-parameter? param)
	    (doc-out "        " (parameter->name param) " (" (+ idx 1) ") "
		     (parameter->type param) nl))))
   (foreign->parameters f)
   out-parameter?)
  (doc-out nl nl nl)
  #f)



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Read from file

(define write-out 'forward)		;write on output file
(define display-out 'forward)		;display on output file
;(define pretty-print-out 'forward)	;pretty-print on output file
(define doc-out 'forward)		;display on documentation file

(define nl #\newline)

(define function-entries '())

(define prog-name "genscmint")
(define prog-version "0.4a")

(define (driver in-name module-name . rest)
  (let ((in-file (open-input-file in-name))
	(out-file (if (null? rest)
		      (current-output-port)
		      (open-output-file (car rest))))
	(doc-file (if (= (length rest) 2)
		      (open-output-file (cadr rest))
		      #f)))
    (set! write-out
	  (lambda objs (for-each (lambda (obj) (write obj out-file)) objs)))
    (set! display-out
	  (lambda objs (for-each (lambda (obj) (display obj out-file)) objs)))
    (if doc-file
	(set! doc-out
	      (lambda objs (for-each (lambda (obj)
				       (display obj doc-file)) objs)))
	(set! doc-out (lambda objs #f)))
;    (set! pretty-print-out (lambda (obj)
;			     (pretty-print obj out-file)))
    (set! function-entries '())
    (display-out "/* -*- C -*-    Module " module-name " generated by "
		 prog-name " "
		 prog-version " */" nl nl)
    (display-out "#include \"scmint.h\"" nl nl nl)
    (let loop ((obj (read in-file))
	       (num 1))
      (cond ((eof-object? obj)
	     (tell "items processed: " num nl))
	    (else
	     (if (and (pair? obj)
		      (eq? (car obj) 'define-foreign))
		 (let ((f (parse-foreign obj)))
		   (process-foreign f)
		   (if doc-file (write-doc f)))
		 (tell "unhandled object: " obj))
	     (loop (read in-file) (+ num 1)))))
    (write-function-entries function-entries)
    (display-out "void " nl
		 "init_" module-name "()" nl
		 "{" nl
		 "  init_iprocs (si_lsubrs, tc7_lsubr);" nl
		 "}" nl)
    (close-input-port in-file)
    (if doc-file (close-output-port doc-file))
    (or (null? rest) (close-output-port out-file))))
	
;;; EOF
