; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SCHEME-INTERNAL; -*-
; File eval.lisp / Copyright (c) 1989 Jonathan Rees / See file COPYING

;;;; Pseudoscheme runtime system

(lisp:in-package "SCHEME-INTERNAL")

(export '(
	  scheme-eval
	  scheme-compile
	  scheme-compile-file
	  translate-file
	  ;; Contexts
	  set-rep-context!
	  scheme-user-context
	  usual-context
	  ;; Utilitical
	  scheme-error
	  pp
	  ;; REP loop
	  scheme
	  quit
	  ))

(defvar *scheme-file-type* (preferred-case "SCM"))
(defvar *translated-file-type* (preferred-case "PSO"))

(defvar scheme-user-env)
(defvar scheme-user-context)
(defvar usual-context)

(defvar *default-load-context*)
(defvar *current-rep-context*)
(defvar *translator-exists?* nil)

; Initialization

(defun access-translator (name)
  (environment-ref scheme-translator-env name))

(defun initialize-for-evaluation ()
  ;; Translator needs an ERROR procedure.
  (environment-define! scheme-translator-env
		       'scheme::error
		       #'scheme-error)

  (let ((usual-macrologies
	 (access-translator 'scheme::usual-macrologies))
	(usual-integrations
	 (access-translator 'scheme::usual-integrations))
	(almost-no-integrations
	 (access-translator 'scheme::almost-no-integrations)))

    (if revised^3-scheme-context
	;; Unnecessary but convenient
	(initialize-context! revised^3-scheme-context
			     usual-macrologies
			     usual-integrations))

    (if scheme-translator-context
	;; Unnecessary but convenient
	(initialize-context! scheme-translator-context
			     usual-macrologies
			     usual-integrations))

    ;; Is this the right place for this?
    (setq scheme-user-env
	  (make-environment 'scheme::scheme-user
			    scheme-package))

    (setq scheme-user-context
	  (make-context 'scheme::scheme-user
			scheme-user-env
			usual-macrologies
			almost-no-integrations
			scheme-package))

    (setq usual-context
	  (make-context 'scheme::usual
			scheme-user-env
			usual-macrologies
			usual-integrations
			scheme-package))

    ;; Get integrations by default.
    (set-rep-context! usual-context)

    (setq *translator-exists?* t)

    'done))

; Copy all exported bindings from Revised^3 package to SCHEME package

(defun initialize-scheme-user-env ()
  (flet ((set (name val)
	   (environment-define! scheme-user-env name val)))
    (mapc #'(lambda (name)
	      (set name
		   (environment-ref revised^3-scheme-env name)))
	  (funcall (access-translator 'scheme::signature-vars)
		   (access-translator 'scheme::revised^3-scheme-sig)))

    ;; Nonstandard things, generally innocent and useful
    (set 'scheme::quit  #'quit)
    (set 'scheme::pp    #'pp)
    (set 'scheme::compile        #'scheme-compile)
    (set 'scheme::compile-file   #'scheme-compile-file)
    (set 'scheme::translate-file #'translate-file)

    ;; Not quite so innocent, but also useful
    (set 'scheme::error #'scheme-error)
    'done))

; EVAL itself

(defun scheme-eval (form context)
  (eval (translate-in-context form context)))

(defun translate-in-context (form context)
  (funcall (access-translator 'scheme::translate)
	   form
	   (context-macrologies context)
	   (context-meta-env context)
	   (context-package context)))

; COMPILE -- compile a single procedure

(defun scheme-compile (name &optional (source nil source-supplied?))
  (let* ((env (context-env *current-rep-context*))
	 (CL-sym (if name
		     (find-symbol-renaming-perhaps
		        (symbol-name name)
			(environment-package env))
		     nil)))
    (prog1 (if source-supplied?
	       ;;+++ This loses due to the (locally (declare ...) ...)
	       (let ((exp (translate-in-context source
						*current-rep-context*)))
		 (if (and (consp exp) (eq (car exp) 'function))
		     (compile CL-sym (cadr exp))
		     (error "Bad argument to COMPILE -- ~S" source)))
	       (compile CL-sym))
      (when name
	(environment-set! env
			  name
			  (symbol-function CL-sym))))))

; "Roadblock" readtable.  Behaves exactly like a regular Common Lisp
; read table, except when the SCHEME package (or a package associated
; with the current Scheme context) is current, in which case it reads
; a form using the Scheme readtable and package, then wraps (BEGIN
; ...) around it so that the translator will kick in and translate the
; form.

(defparameter roadblock-readtable (copy-readtable scheme-readtable))

#+Symbolics
(pushnew roadblock-readtable si:*valid-readtables*)

(defun roadblock-read-macro (stream ch)
  (unread-char ch stream)
  (if (and *translator-exists?*
	   (or (eq *package* scheme-package)
	       (eq *package* (context-package *target-context*))
	       (eq *package* (context-package *current-rep-context*))))
      (let ((*package* scheme-package)
	    (*readtable* scheme-readtable))
	(multiple-value-call
	  #'(lambda (&optional (thing nil thing-p))
	      (if thing-p
		  `(scheme-form ,thing)
		  (values)))
	  (read stream nil 0 t)))
      (let ((*readtable* scheme-hacks:*non-scheme-readtable*))
	(read stream nil 0 t))))

(let ((*readtable* roadblock-readtable))
  (mapc #'(lambda (s)
	    (map nil
		 #'(lambda (c)
		     (set-macro-character c #'roadblock-read-macro nil))
		 s))
	;; Intentionally absent: right parenthesis, semicolon, whitespace
	'(
	  ;; Non-constituents
	  "\"#'(,`"
	  ;; Constituents (more or less)
	  ;;
	  ;; Actually we don't want to hack these, since otherwise the
	  ;; printer (which we can't hook, in general) will be
	  ;; printing all symbols as |FOO|.  This will only matter for
	  ;; symbol evaluation at an unhooked REP or debugging loop,
	  ;; where evaluation is supposed to be in some environment
	  ;; other than that initial one.
	  ;;
	  ;; On the other hand, if in some implementation we CAN
	  ;; reliably hook the printer, or else sufficiently restrict
	  ;; the use of the roadblock readtable (e.g. by passing it
	  ;; explicitly to LOAD and COMPILE-FILE), then we SHOULD
	  ;; block the constituent characters.  Thus I have left them
	  ;; here in this comment.
	  ;;
	  ;; "!$%&*+-./0123456789:<=>?"
	  ;; "@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_"
	  ;; "abcdefghijklmnopqrstuvwxyz{|}~"
	  )))

(defmacro scheme-form (&whole whole form)
  (let* ((new-form (translate-scheme-form form))
	 (new-form (if (consp new-form) new-form `(progn ,new-form))))
    ;; The following tries to compensate for some versions of LOAD and
    ;; COMPILE-FILE that imagine that macroexpansion is cheap.
    (setf (car whole) (car new-form))
    (setf (cdr whole) (cdr new-form))
    new-form))

; Use ROADBLOCK-EVAL to evaluate a form that was known to have been
; read by the roadblock readtable.

(defun roadblock-eval (form context)
  (cond ((and (consp form) (eq (car form) 'scheme-form))
	 (scheme-eval (cadr form) context))
	((symbolp form)
	 (scheme-eval form context))
	(t
	 (eval form))))

; LOAD

(defun scheme-load (filespec &optional context
			     &rest keys)
  (when (keywordp context) (push context keys) (setq context nil))
  (using-file-context filespec context "Loading"
    #'(lambda (context)
	(apply #'scheme-hacks:clever-load filespec
	       :source-type (or (getf keys :source-type)
				*scheme-file-type*)
	       :message (format nil "into ~S" context)
	       #+LispM :package #+LispM (context-package context)
	       keys))))

; COMPILE-FILE

(defun scheme-compile-file (filespec &optional context
				     &rest keys)
  (when (keywordp context) (push context keys) (setq context nil))
  (using-file-context filespec context "Compiling"
    #'(lambda (context)
	(let ((path
	       (merge-pathnames filespec
				(make-pathname :type *scheme-file-type*))))
	  (format t "~&Compiling ~A using ~S~%" (namestring path) context)
	  (apply #'compile-file
		 path
		 #+LispM :package #+LispM (context-package context)
		 keys)))))

; Axuiliary for the above

(defun using-file-context (filespec context activity fun)
  (let* ((context (get-file-context filespec context activity))
	 (*readtable* roadblock-readtable)
	 (*package* (context-package context))
	 (*target-context* context))
    (funcall fun context)))

; TRANSLATE-FILE

(defun translate-file (filespec &optional context)
  (let* ((context (get-file-context filespec context "Translating"))
	 (path (merge-pathnames (if (symbolp filespec)
				    (symbol-name filespec)
				    filespec))))
    (funcall (access-translator 'scheme::really-translate-file)
	     (if (member (pathname-type path) '(nil :unspecific))
		 (make-pathname :type *scheme-file-type*
				:defaults path)
		 path)
	     (lisp:make-pathname :type *translated-file-type*
				 :defaults path)
	     (context-macrologies context)
	     (context-meta-env	  context)
	     (context-package	  context))))

; Auxiliary routine called when reading from Gnu Emacs using LEDIT package

(defun ledit-eval (filename form)
  (if (eq *package* scheme-package)
      (scheme-eval form
		   (if filename
		       (get-file-context filename nil "Evaluating LEDIT form")
		       *current-rep-context*))
      (eval form)))

(locally (declare (special user::*ledit-eval*))
  (setq user::*ledit-eval* #'ledit-eval))

;

(defun set-rep-context! (context)
  (check-type context context "a context")
  (setq *current-rep-context* context)
  (setq *target-context* context)
  *current-rep-context*)

; Data base of file -> context associations, mostly for benefit of
; the LEDIT package.

(defvar file-context-table (make-hash-table :test 'equal))

(defun filespec-key (filespec)
  (let ((path (merge-pathnames filespec)))
    (cons (pathname-directory path)
	  (pathname-name      path))))

(defun maybe-get-file-context (filespec)
  (gethash (filespec-key filespec) file-context-table))

(defun get-file-context (filespec context activity)
  (if filespec
      (let ((context
	      (or context
		  (let ((probe (maybe-get-file-context filespec)))
		    (if probe
			(progn
			  (unless (eq probe *current-rep-context*)
			    (format t
				    "~&[~A using ~S.]~%"
				    activity
				    probe))
			  probe)
			*current-rep-context*)))))
	(set-file-context! filespec context)
	context)
      (or context *current-rep-context*)))

(defun forget (filespec)  ;forget-file-context
  (shiftf (gethash (filespec-key filespec) file-context-table)
	  nil))

(defun set-file-context! (filespec context)
  (let* ((name (filespec-key filespec))
	 (other-context (gethash name file-context-table)))
    (when (not (eq context other-context))
      (when other-context
	(format t "~&Changing context for file ~A from ~S to ~S.~%"
		name other-context context))
      (setf (gethash name file-context-table)
	    context))))

; These things don't really belong here, but what the heck.

; ERROR (nonstandard)

(defun scheme-error (message &rest irritants)
  (if (and (stringp message)
	   (find #\~ message))
      (apply #'error message irritants)
      (apply #'error
	     (apply #'concatenate
		    'string
		    (if (stringp message) "~a" "~s")
		    (mapcar #'(lambda (irritant)
				(declare (ignore irritant))
				"~%  ~s")
			    irritants))
	     message
	     irritants)))

#+LispM
(setf (get 'scheme-error :error-reporter) t)  ;Thanks to KMP

; PP (nonstandard)

(defun pp (obj &optional (port *standard-input*))
  (let ((*print-pretty* t)
	(*print-length* nil)
	(*print-level* nil))
    (format port "~&")
    (print obj port)
    (values)))

; Set up "trampolines" to allow evaluation of Scheme forms directly by
; the Common Lisp evaluator.  Alsp, give some help to the pretty-printer
; by way of indicating where &bodies are.

(defun translate-scheme-form (form)
  (translate-in-context form *target-context*))

(defmacro scheme::case (key &body clauses)
  (translate-scheme-form `(scheme::case ,key ,@clauses)))

(defmacro scheme::define (pat &body body)
  (translate-scheme-form `(scheme::define ,pat ,@body)))

(defmacro scheme::do (specs end &body body)
  (translate-scheme-form `(scheme::do ,specs ,end ,@body)))

(defmacro scheme::lambda (bvl &body body)
  (translate-scheme-form `(scheme::lambda ,bvl ,@body)))

(defmacro scheme::let (specs &body body)
  (translate-scheme-form `(scheme::let ,specs ,@body)))

(defmacro scheme::let* (specs &body body)
  (translate-scheme-form `(scheme::let* ,specs ,@body)))

(defmacro scheme::letrec (specs &body body)
  (translate-scheme-form `(scheme::letrec ,specs ,@body)))

; Other trampolines...

(defmacro translate-me (&whole form &rest rest)
  (declare (ignore rest))
  (translate-scheme-form form))

(mapc #'(lambda (scheme-sym)
	  ;; Allow (LISP:EVAL '(SCHEME::AND ...))
	  (setf (macro-function scheme-sym)
		(macro-function 'translate-me)))
      '(scheme::and
	scheme::begin
	scheme::cond
	scheme::delay
	scheme::if
	scheme::or
	scheme::quasiquote
	scheme::quote
	scheme::set!))

; Read-eval-print loop

(defvar *rep-state-vars* '())

(defun enter-scheme ()
  (set-scheme-value '*package* scheme-package)
  (set-scheme-value '*print-array* t)	     ;for #(...)
  (set-scheme-value '*readtable* roadblock-readtable)
  (setq scheme-hacks:*non-scheme-readtable*
	(get '*readtable* 'non-scheme-value))
  (format t "~&This is ~A.~&"
	  (funcall (access-translator 'scheme::translator-version)))
  (values))

(defun exit-scheme ()
  (format t "~&Leaving Pseudoscheme.~&")
  (mapc #'(lambda (var)
	    (let ((probe (get var 'non-scheme-value 'no-such-property)))
	      (unless (eq probe 'no-such-property)
		(set-standard-value var probe))))
	*rep-state-vars*)
  (values))

(defun set-scheme-value (var value)
  (pushnew var *rep-state-vars*)
  (let ((old-value (symbol-value var)))
    (unless (eq value old-value)
      (setf (get var 'non-scheme-value) old-value))
    (set-standard-value var value)))

(defun set-standard-value (var value)
  #-Symbolics
  (setf (symbol-value var) value)
  #+Symbolics
  (if (member var '(*package* *print-array* *readtable*))
      (setf (sys:standard-value var :setq-p t)
	    value)
      (setf (symbol-value var) value)))

;;; EVAL and PRINT functions to be used by the REP loop:

(defun scheme-rep-eval (exp)
  (roadblock-eval exp *current-rep-context*))

(defvar *result-display-style* :normalize)  ;or :eval

(defun write-result (result &optional (stream *standard-output*))
  (if (and (eq *result-display-style* :normalize)
	   (not (or (eq result t)  ;self-evaluating-p
		    ;; Don't include nil in this list, since it
		    ;; prints as (), which ISN'T generally self-evaluating.
		    (numberp result)
		    (characterp result)
		    (stringp result)
		    (scheme-hacks:photon-p result))))
      (write-char #\' stream))
  (funcall (environment-ref revised^3-scheme-env 'scheme::write)
	   result
	   stream))

; (SCHEME) and (QUIT) are system-specific REP loop entry and exit
; routines.

#-(or :DEC Symbolics) (progn
(defun scheme ()
  (enter-scheme))

(defun quit () 
  (exit-scheme))
) ;end (progn ...)

#+:DEC (progn
(defun scheme ()
  (unwind-protect
      (progn
	(enter-scheme)
	(system::read-eval-print-loop
	   "Scheme> "
	   :eval 'scheme-rep-eval
	   :print #'(lambda (vals stream)
		      (format stream "~&")
		      (do ((v vals (cdr v)))
			  ((null v) (values))
			(write-result (car v) stream)
			(if (not (null (cdr v)))
			    (format stream " ;~%")))))
	(values))
    (exit-scheme)))

(defun quit ()
  (vax-lisp:continue))
) ;end #+:DEC (progn ...)

#+Symbolics (progn 'compile

(defun scheme ()
  "Initialize for execution of Scheme programs."
  (enter-scheme)
  (set-scheme-value 'si:*command-loop-eval-function*
		    'scheme-rep-eval)
  (set-scheme-value 'si:*command-loop-print-function*
		    #'(lambda (values)
			(mapc #'(lambda (value)
				  (zl:send zl:standard-output :fresh-line)
				  (write-result value))	;?
			      values)))
  (values))

(defun quit ()
  (exit-scheme))
) ;end #+Symbolics (progn ...)
