/************************************************************************
*									*
* The SB-Prolog System							*
* Copyright SUNY at Stony Brook, 1986; University of Arizona, 1987	*
*									*
************************************************************************/

/*-----------------------------------------------------------------
SB-Prolog is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the SB-Prolog General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
SB-Prolog, but only under the conditions described in the
SB-Prolog General Public License.   A copy of this license is
supposed to have been given to you along with SB-Prolog so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies. 
------------------------------------------------------------------ */
/* $funrel.P */
/* the predicates in this file are called from $getclauses: they
   traverse a clause and convert functional syntax in arithmetic
   expressions to relational expressions.  For example,
   
   	X is 1 + sqrt(Y)

   is converted to

	square(Z, Y), X is 1 + Z

   The set of "evaluable functors", and the predicates they are
   converted to, are defined in the table $fun_to_rel/3 at the
   end of this file.
*/

$funrel_export([$fun_rel/2,$expand_body/2]).
$funrel_use($bmeta,[_,_,_,_,$structure/1,_,_,_,_,_,_,_,_]).
$funrel_use($meta,[$functor/3,-,_]).


$fun_rel((H :- B), (H :- B1)) :- !, $expand_body(B, B1).
$fun_rel(T, T).

$expand_body((A1,B1), (A2,B2)) :-
     !, $expand_body(A1,A2), $expand_body(B1,B2).
$expand_body((A1;B1), (A2;B2)) :-
     !, $expand_body(A1,A2), $expand_body(B1,B2).
$expand_body((A1 -> B1), (A2 -> B2)) :-
     !, $expand_body(A1,A2), $expand_body(B1,B2).
$expand_body('=:='(A1,B1), G) :-
     !, $funrel_proc_exp(A1,A2,G1), $funrel_proc_exp(B1,B2,G2),
     $chk_goals(G1,G2,'=:='(A2,B2), G).
$expand_body('=/='(A1,B1), G) :-
     !, $funrel_proc_exp(A1,A2,G1), $funrel_proc_exp(B1,B2,G2),
     $chk_goals(G1,G2,'=/='(A2,B2), G).
$expand_body('>='(A1,B1), G) :-
     !, $funrel_proc_exp(A1,A2,G1), $funrel_proc_exp(B1,B2,G2),
     $chk_goals(G1,G2,'>='(A2,B2), G).
$expand_body('>'(A1,B1), G) :-
     !, $funrel_proc_exp(A1,A2,G1), $funrel_proc_exp(B1,B2,G2),
     $chk_goals(G1,G2,'>'(A2,B2), G).
$expand_body('=<'(A1,B1), G) :-
     !, $funrel_proc_exp(A1,A2,G1), $funrel_proc_exp(B1,B2,G2),
     $chk_goals(G1,G2,'=<'(A2,B2), G).
$expand_body('<'(A1,B1), G) :-
     !, $funrel_proc_exp(A1,A2,G1), $funrel_proc_exp(B1,B2,G2),
     $chk_goals(G1,G2,'<'(A2,B2), G).
$expand_body(not(A1), not(A2)) :-
     !, $expand_body(A1,A2).
$expand_body(is(A,Exp1), B) :- 
     !, $funrel_proc_exp(Exp1,Exp2,G,Conv),
     (Conv =:= 1 ->
          (var(Exp2) ->
               (A = Exp2, B = G) ;
	       B = (G,is(A,Exp2))
	  ) ;
	  B = is(A,Exp1)
     ).
$expand_body(T,T).

$funrel_proc_exp(E1,E2,G) :- $funrel_proc_exp(E1,E2,G,_).

$funrel_proc_exp(E1,E3,G,C) :-
     $structure(E1) ->
          ($functor(E1,F,N), $functor(E2,F,N),
           $funrel_proc_subexps(N,E1,E2,true,G1,0,C0),
           ($fun_to_rel(E2,G2,E3) ->
	       (C = 1,
                (G1 = true -> G = G2 ; G = (G1,G2))
	       ) ;
	       (E3 = E2, G = G1, C = C0)
           )
          ) ;
	  (E3 = E1, G = true, C = 0).

$funrel_proc_subexps(N,Ein,Eout,Gin,Gout,Cin,Cout) :-
     N =:= 0 ->
          (Cin = Cout, Gin = Gout) ;
	  (arg(N,Ein,T1), arg(N,Eout,T2),
	   $funrel_proc_exp(T1,T2,G,C0),
	   N1 is N-1,
	   (Gin = true -> Gmid = G ; Gmid = (Gin,G)),
	   (Cin =:= 1 -> C1 = 1 ; C1 = C0),
	   $funrel_proc_subexps(N1,Ein,Eout,Gmid,Gout,C1,Cout)
	  ).

$chk_goals(A,B,C,D) :-
     A ?= true ->
          (B ?= true ->
	       D = C ;
	       D = (B,C)
          ) ;
	  (B ?= true ->
	       D = (A,C) ;
	       D = (A,B,C)
          ).

% $fun_to_rel(T1, T2, Z) is true iff T2 represents the "relational syntax"
% version of the "functional syntax" expression "Z = T1".

$fun_to_rel(sqrt(X), square(Y,X), Y).
$fun_to_rel(square(X), square(X,Y), Y).
$fun_to_rel(integer(X), floor(X,Y), Y).
$fun_to_rel(float(X), floor(Y,X), Y).
$fun_to_rel(exp(X), exp(X,Y), Y).
$fun_to_rel(ln(X), exp(Y,X), Y).
$fun_to_rel(sin(X), sin(X,Y), Y).
$fun_to_rel(arcsin(X), sin(Y,X), Y).

