(*  Title: 	queue
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   Cambridge University 1988

Applicative queues represented by (heads, reversed tails).
Can add to front or back, but only remove from front (deq).
A deq can take a long time but amortized time per operation is constant.

F.W. Burton, ``An efficient functional implementation of FIFO queues''
  Information Processing Letters 14 (1982) 205-206.
*)

signature QUEUE = 
sig
  type 'a q
  exception QUEUE of string
  val deq: 'a q -> 'a q
  val empty: 'a q
  val enq: 'a q * 'a -> 'a q
  val head: 'a q -> 'a
  val null: 'a q -> bool
  val push: 'a * 'a q -> 'a q
end;

functor QueueFun() : QUEUE = 
struct
datatype 'a q = Queue of ('a list * 'a list);

exception QUEUE of string;

(*The empty queue*)
val empty = Queue([],[]);

(*Normalized queue, if nonempty, has nonempty heads list*)
fun norm (Queue([],tails)) = Queue(rev tails, [])
  | norm q = q;

(*Add to end: result is updated queue*)
fun enq(Queue(heads,tails), x) = norm(Queue(heads, x::tails));

(*Add to front: result is updated queue*)
fun push(x, Queue(heads,tails)) = Queue(x::heads, tails);

(*Test for empty queue*)
fun null(Queue([],[])) = true
  | null _ = false;

(*Remove from front: result is updated queue*)
fun deq(Queue([],_)) = raise QUEUE "deq"
  | deq(Queue(x::heads,tails)) = norm(Queue(heads,tails));

(*Return front element of queue*)
fun head(Queue([],_)) = raise QUEUE "head"
  | head(Queue(x::_,_)) = x;

end;
