(*  Title: 	pretty
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   Cambridge University 1988

Pretty printing method based on
  D. C. Oppen, "Pretty Printing",
  ACM Transactions on Programming Languages and Systems (1980), 465-483. 

*)

signature PRETTY = 
sig
  val bg: int -> unit
  val bgc: int -> unit
  val bgs: string -> unit   
  val brk: int * int -> unit
  val brks: string -> int -> unit   
  val en: unit -> unit
  val ens: string -> unit   
  val init: unit -> unit
  val int: int -> unit
  val list: string * string * ('a -> unit) -> 'a list -> unit   
  val margin: int ref
  val nl: unit -> unit
  val st: string -> unit
end;

functor PrettyFun(Queue: QUEUE) : PRETTY = 
struct

datatype break = Consist | Inconsist | Fits;


(*Printing items: strings, breaks, and begin/end of phrases*)
datatype token = 
    String of string
  | Break of (*width*)int * (*offset*)int
  | Begin of (*indent*)int * break
  | End;

type qitem = int ref * token * int;

val infinity = 9999;


(*Queued printing items*)
val tok_queue = ref (Queue.empty: qitem Queue.q);

val scan_stack = ref ([]: (int * qitem) list);

val margin = ref 80
and space = ref 0
and left_total = ref 0
and right_total = ref 0
and end_count = ref 0
and pstack = ref ([]: (break*int)list);


(*Push break type and indentation*)
fun push_pstack (br:break, offset:int) =
    pstack := (br,offset) :: !pstack;

(*Output n blanks and reduce space accordingly*)
fun blanks n =
  if n<0 then error"blanks"  else
    let fun pb n = if n<>0 then  (prs" "; pb(n-1))  else  ()
    in  pb n;  space := !space - n  end;

(*Output newline and reset space*)
fun new_line () = (prs"\n";  space := !margin);


(*Compute tab column.  If far across then try to squeeze in flush right
	 (up to 90% of margin) or else reduce indentation*)
fun tab_amount (nbls,size) =
  if nbls > !margin*2 div 3  
	then  if size <= !margin*9 div 10  then  min[!margin-size, nbls]
	else  nbls div 4
  else nbls;


(*Inconsistent breaks.  Squeeze if necessary;
  if really far across then reduce indentation;
  suppress break if it only adds 10% of the margin. *)
fun print_break (nbls,size,wid) =
  if size <= !space then  blanks wid
  else
    let val tabcol = tab_amount (nbls,size)
    in  if tabcol + !margin div 10 > !margin - !space + wid  then  blanks wid
	else  (new_line();  blanks tabcol)
    end;


(*Output a token: inserts "break" before "begin" if space is scare. *)
fun print_token (String s, size) = (prs s; space := !space-size)
  | print_token (Begin(ind,br), size) = 
      if size <= !space then push_pstack(Fits, !space-ind)
      else (if !space-ind <= !margin div 3
            then print_break (!margin-(!space-ind), size, 0)
            else ();   push_pstack(br, !space-ind))
  | print_token (End, size) = pstack := tl (!pstack)
	(*COULD record whether block was broken to force a break afterwards,
	  as in   [ [long list get broken], [x] ]  -- should break before [x] *)
  | print_token (Break(wid,off), size) = 
      case hd (!pstack) of
	  (Consist, bgspace) => (new_line();  blanks(!margin-bgspace+off))
	| (Inconsist, bgspace) => print_break (!margin-bgspace+off, size, wid)
	| (Fits, bgspace) => blanks wid;


(*Set everything to empty*)
fun clear_queue() =
    (left_total := 1;  right_total := 1;  end_count := 0;
     tok_queue := Queue.empty);


(*Enter token into queue, return the entry*)
fun enqueue (size:int, tt:token, len:int) =
  let val new = (ref size, tt, len)
  in  tok_queue := Queue.enq(!tok_queue, new);
      right_total := !right_total + len;
      new
  end;


(*Print if token size is known or printing is lagging, i.e.
	if the queued text cannot fit on current line.
  Negative size means size is unknown. *)
fun advance_left () = if Queue.null (!tok_queue) then ()
    else
    let val (ref size, tt, len) = Queue.head (!tok_queue)
    in  if size<0  andalso 
           !right_total - !left_total <= !margin  then ()  
	   (*was !space, not !margin
	     current version may try to squeeze it in*)
	else
	(print_token (tt, if size<0 then infinity else size);
	   left_total := !left_total + len;
	   tok_queue := Queue.deq (!tok_queue);
	   advance_left())
    end;


(*Push new queued element onto scan_stack*)
fun scan_push new = scan_stack := (!right_total, new) :: !scan_stack;


(*If top of scan stack points to a queue element accepted by the 
	disc-riminator, pop it and set size of the queue element.
  Clear stack if top element is out-of-date *)
fun set_size disc =
  case !scan_stack of
      [] => ()
    | (left, (rsize,tt,_)) :: stl  => 
	  if left < !left_total then scan_stack := []
	  else if disc tt then
	      (scan_stack := stl;  rsize := !rsize + !right_total)
	  else ();


(*Discriminators for tokens*)

fun is_break (Break _ ) = true
  | is_break _ = false;

fun is_begin (Begin _ ) = true
  | is_begin _ = false;


(*Execute pending "end" operations*)
fun do_ends() =
  while !end_count>0 do
	(enqueue(0, End, 0);  set_size is_break;  set_size is_begin;
	 end_count := !end_count - 1);

(*Output a string*)
fun st s =
  let val len = size s in
      (enqueue (len, String s, len); advance_left())
  end;

(*Output an integer*)
fun int n = st (string_of_int n);

(*Begin a consistent/inconsistent block*)
fun begin_block brk ind =
  (do_ends();  scan_push(enqueue (~(!right_total), Begin(ind,brk), 0)));

val bgc = begin_block Consist
and bg = begin_block Inconsist;

(*End a block*)
fun en() = end_count := !end_count + 1;


(*Insert a break (width of black space, indentation offset) *)
fun brk (wid, offset) =
  (do_ends();  
   let val new = enqueue ( ~(!right_total), Break(wid,offset), wid)
   in  set_size is_break;  scan_push new  end);


(*Initialize and begin the outer block*)
fun init () =
  (clear_queue(); 
   scan_stack := [];
   space := !margin;
   bgc 0;
   pstack := []);


(*Output a newline (preceded by all queued text). *)
fun nl () =
  (en();  do_ends();  right_total := infinity;  advance_left();
   new_line();  init());

(*Begin a block by printing the string*)
fun bgs a = (bg(size a);  st a);

(*End a block after printing the string*)
fun ens a = (st a;  en());

(*Print string and break*)
fun brks a wid = (st a;  brk(wid,0));


(*Print a list bracketed by lpar and rpar with comma (,) as separator*)
fun list (lpar, rpar, pre) [] = st (lpar ^ rpar)
  | list (lpar, rpar, pre) (x::xs) = 
	(bgs lpar;  pre x;  seq (fn x => (brks "," 1;  pre x)) xs;
	 ens rpar);

end;
