(*  Title: 	logic
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   Cambridge University 1988
*)

infix occs;


signature LOGIC = 
sig
  val add_term_frees: term * term list -> term list
  val add_term_vars: term * term list -> term list
  val assum_pairs: term -> (term * term) list
  val atless: term * term -> bool
  val close_form: term -> term   
  val delete_asm: int -> term -> term
  val dest_equals: term -> term * term
  val incr_indexes: typ list * int -> term -> term
  val insert_aterm: term * term list -> term list
  val lift_fns: term * int -> (term -> term) * (term -> term)
  val list_equals: (term * term) list * term -> term
  val list_implies: term list * term -> term
  val mk_equals: term * term -> term
  val occs: term * term -> bool
  val rename_params: string * term -> term  
  val rule_of: (term * term) list * term list * term -> term
  val strip_assums: term -> (term * int) list * (string * typ) list * term
  val strip_assums_concl: term -> term
  val strip_assums_hyp: term -> term list
  val strip_equals: term -> (term*term) list * term
  val strip_horn: term -> (term*term) list * term list * term
  val strip_imp_concl: term -> term
  val strip_imp_prems: term -> term list
  val strip_prems: int * term list * term -> term list * term
  val varify: term -> term  
  end;

functor LogicFun (Unify: UNIFY) : LOGIC  = 
struct

(*Make an equality.  DOES NOT CHECK TYPE OF u*)
fun mk_equals(t,u) = equals(type_of t) $ t $ u;

fun dest_equals (Const("==",_) $ t $ u)  =  (t,u)
  | dest_equals t = raise TERM_ERROR("dest_equals", [t]);

(* [A1,...,An], B  goes to  A1==>...An==>B  *)
fun list_implies ([], B) = B : term
  | list_implies (A::AS, B) = implies $ A $ list_implies(AS,B);


(* A1==>...An==>B  goes to  [A1,...,An], where B is not an implication *)
fun strip_imp_prems (Const("==>", _) $ A $ B) = A :: strip_imp_prems B
  | strip_imp_prems _ = [];


(* A1==>...An==>B  goes to B, where B is not an implication *)
fun strip_imp_concl (Const("==>", _) $ A $ B) = strip_imp_concl B
  | strip_imp_concl A = A : term;


(*make equality antecedents: ( [(a1,b1),...,(an,bn)] , C )
    goes to (a1==b1) ==>...(an==bn)==>C *)
fun list_equals ([], A) = A
  | list_equals ((t,u)::pairs, A) =
	implies $ (mk_equals(t,u)) $ list_equals(pairs,A);


fun strip_equals_aux (pairs, Const("==>", _) $ (Const("==",_)$t$u) $ C) =
        strip_equals_aux ((t,u)::pairs, C)
  | strip_equals_aux X = X;

(*strip equality antecedents: (a1==b1)==>...(an==bn)==>C
    goes to   ( [(a1,b1),...,(an,bn)] , C )	 *)
fun strip_equals A = apfst rev (strip_equals_aux([],A));


(*strip a proof state (Horn clause): 
   (a1==b1)==>...(am==bm)==>B1==>...Bn==>C
    goes to   ( [(a1,b1),...,(am,bm)] , [B1,...,Bn] , C)    *)
fun strip_horn A =
  let val (tpairs,horn) = strip_equals A 
  in  (tpairs, strip_imp_prems horn, strip_imp_concl horn)   end;



(*strip premises: (i, [], A1==>...Ai==>B)
    goes to   ([Ai, A(i-1),...,A1] , B) 	(REVERSED) 
  if  i<0 or else i too big then raises  TERM_ERROR*)
fun strip_prems (0, As, B) = (As, B) 
  | strip_prems (i, As, Const("==>", _) $ A $ B) = 
	strip_prems (i-1, A::As, B)
  | strip_prems (_, As, A) = raise TERM_ERROR("strip_prems", A::As);


(*Make the object-rule tpairs==>As==>B   *)
fun rule_of (tpairs, As, B) = list_equals(tpairs, list_implies(As, B));



(*Does t occur in u?  Or is alpha-convertible to u?
  The term t must contain no loose bound variables*)
fun t occs u = (t aconv u) orelse 
      (case u of
          Abs(_,_,body) => t occs body
	| f$t' => t occs f  orelse  t occs t'
	| _ => false);
  

(*a partial ordering (not reflexive) for atomic terms*)
fun atless (Const (a,_), Const (b,_))  =  a<b
  | atless (Free (a,_), Free (b,_)) =  a<b
  | atless (Var(v,_), Var(w,_))  =  xless(v,w)
  | atless (Bound i, Bound j)  =   i<j
  | atless _  =  false;


(*insert atomic term into partially sorted list, trying to suppress duplicates*)
fun insert_aterm (t,us) =
  let fun inserta [] = [t]
        | inserta (us as u::us') = 
	      if atless(t,u) then t::us
	      else if t=u then us (*duplicate*)
	      else u :: inserta(us')
  in  inserta us  end;


(*Accumulates the Vars in the term, suppressing duplicates*)
fun add_term_vars (t, vars: term list) = case t of
    Var   _ => insert_aterm(t,vars)
  | Abs (_,_,body) => add_term_vars(body,vars)
  | f$t =>  add_term_vars (f, add_term_vars(t, vars))
  | _ => vars;


(*Accumulates the Frees in the term, suppressing duplicates*)
fun add_term_frees (t, frees: term list) = case t of
    Free   _ => insert_aterm(t,frees)
  | Abs (_,_,body) => add_term_frees(body,frees)
  | f$t =>  add_term_frees (f, add_term_frees(t, frees))
  | _ => frees;


(*Close up a formula over all free variables by quantification*)
fun close_form A =
    let val sfrees = sort atless (add_term_frees (A,[]))
    in  list_all_free (map dest_Free sfrees,   A)  end;


(*For all variables in the term, increment indexnames and lift over the Us
    result is ?Gidx(B.(lev+n-1),...,B.lev) where lev is abstraction level *)
fun incr_indexes (Us: typ list, inc:int) t = 
  let fun incr (Var ((a,i), T), lev) = 
		Unify.combound (Var((a, i+inc), Us--->T),    lev,   
				length Us)
	| incr (Abs (a,T,body), lev) = Abs (a, T, incr(body,lev+1))
	| incr (f$t, lev) = incr(f,lev) $ incr(t,lev)
	| incr (t,lev) = t
  in  incr(t,0)  end;


(*Make lifting functions from subgoal and increment.*)
fun lift_fns (B,inc) =
  let fun lift_abs (Us, Const("==>", _) $ _ $ B) u = lift_abs (Us,B) u
	| lift_abs (Us, Const("all",_)$Abs(a,T,t)) u =
	      Abs(a, T, lift_abs (T::Us, t) u)
	| lift_abs (Us, _) u = incr_indexes(rev Us, inc) u
      fun lift_all (Us, Const("==>", _) $ A $ B) u =
	      implies $ A $ lift_all (Us,B) u
	| lift_all (Us, Const("all",_)$Abs(a,T,t)) u = 
	      all T $ Abs(a, T, lift_all (T::Us,t) u)
	| lift_all (Us, _) u = incr_indexes(rev Us, inc) u;
  in  (lift_abs([],B), lift_all([],B))  end;


(*Strips assumptions in goal, result is
    ( [ (Hn,kn),...,(H1,k1)], [xm,...,x1], B )
  where ki is the number of parameters enclosing Hi
  and x1...xm are all parameters encountered.   *)
fun strip_assums_aux (Hks, params, Const("==>", _) $ H $ B) = 
	strip_assums_aux ((H, length params) :: Hks, params, B)
  | strip_assums_aux (Hks, params, Const("all",_)$Abs(a,T,t)) =
	strip_assums_aux (Hks, (a,T)::params, t)
  | strip_assums_aux (Hks, params, B) = (Hks, params, B);

fun strip_assums A = strip_assums_aux ([],[],A);


(*Strips assumptions in goal, yielding list of hypotheses.   *)
fun strip_assums_hyp (Const("==>", _) $ H $ B) = H :: strip_assums_hyp B
  | strip_assums_hyp (Const("all",_)$Abs(a,T,t)) = strip_assums_hyp t
  | strip_assums_hyp B = [];

(*Strips assumptions in goal, yielding conclusion.   *)
fun strip_assums_concl (Const("==>", _) $ H $ B) = strip_assums_concl B
  | strip_assums_concl (Const("all",_)$Abs(a,T,t)) = strip_assums_concl t
  | strip_assums_concl B = B;



(*Produces disagreement pairs, one for each assumption proof, in order.
  A is the first premise of the lifted rule, and thus has the form
    H1 ==> ... Hk ==> B   and the pairs are (H1,B),...,(Hk,B)
  [This explanation ignores parameters, which complicate matters] *)
fun assum_pairs A =
  let val (Hks, params, B) = strip_assums A;
      val nparams = length params
      and D = Unify.rlist_abs(params, B);
      fun pair(H,k) = (Unify.rlist_abs(params, incr_boundvars (nparams-k) H), D)
  in  map pair (rev Hks)  end;


(*Delete the assumption number n from the formula, returning the remainder.
  Raises TERM_ERROR if n too big or formula ill-formed.
  Used in bicompose. *)
fun delete_asm n (Const("==>", _) $ H $ B) =
	if n=1 then B  else  implies $ H $ delete_asm (n-1) B
  | delete_asm n (Const("all",_)$Abs(a,T,t)) = 
	all T $ Abs(a, T, delete_asm n t)
  | delete_asm n A = raise TERM_ERROR("delete_asm", [A]);


(*Makes parameters in a goal have distinctive names (not guaranteed unique!)
  A name clash could cause the printer to rename bound vars;
    then lift_inst_tac would not work properly.*)
fun rename_params (a, Const("==>", _) $ A $ B) =
      implies $ A $ rename_params (a, B)
  | rename_params (a, Const("all",_)$Abs(_,T,t)) = 
      all T $ Abs(a, T, rename_params (bump_string a, t))
  | rename_params (a, B) = B;


(*Converts Frees to Vars so that axioms can be written without (?) everywhere*)
fun varify (Free(a,T)) = Var((a,0), T)
  | varify (Abs (a,T,body)) = Abs (a, T, varify body)
  | varify (f$t) = varify f $ varify t
  | varify t = t;


end;


