(*  Title: 	LK/syntax
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

Sequent Calculus for classical first-order logic.
    Adapted from Philippe de Groote's work.
*)

signature LK_SYNTAX =
sig
  structure Syntax : PURE_SYNTAX
  val Aterm: typ
  val Aform: typ
  val Asequ: typ
  val Asobj: typ
  val const_decs: (string list * typ) list
  val prin: term -> unit
  val read: string -> term
  val syn: Syntax.syntax
end;

functor LK_SyntaxFun (Syntax: PURE_SYNTAX) : LK_SYNTAX = 
struct
structure Syntax = Syntax;
local open Syntax 
in

(*Types of expressions, objects (?), hypotheses, formulae*)
val Aterm  = Ground "term";
val Asobj = Ground "sobj";
val Asequ = Asobj-->Asobj;
val Aform = Ground "form";

val True = "True";
val Seqof = "Seqof";
val The = "The";
val Forall = "Forall";
val Exists = "Exists";

val STrue = " True";
val MtSeq = " MtSeq";
val NmtSeq = " NmtSeq";
val MtSeqCont = " MtSeqCont";
val seqcont = " seqcont";
val SeqId = " SeqId";
val SeqVar = " SeqVar";
val Ssequence = Ground "$sequence";
val Sseqcont = Ground "$seqcont";
val Sseqobj = Ground "$sobj";

val SThe = " The";
val SForall = " Forall";
val SExists = " Exists";

(*Abstract over "sobj" -- representation of a sequence of formulae *)
fun abs_sobj t = Abs("sobj", Asobj, t);

(*Representation of empty sequence*)
val Sempty =  abs_sobj (Bound 0);

fun seq_obj_tr(Const(" SeqId",_)$id) = id |
    seq_obj_tr(Const(" SeqVar",_)$id) = id |
    seq_obj_tr(fm) = Const(Seqof,Adummy)$fm;

fun seq_tr(_$obj$seq) = seq_obj_tr(obj)$seq_tr(seq) |
    seq_tr(_) = Bound 0;

fun seq_tr1(Const(" MtSeq",_)) = Sempty |
    seq_tr1(seq) = abs_sobj(seq_tr seq);

fun true_tr[s1,s2] = Const(True,Adummy)$seq_tr1 s1$seq_tr1 s2;

fun seq_obj_tr'(Const("Seqof",_)$fm) = fm |
    seq_obj_tr'(id) = Const(SeqId,Adummy)$id;

fun seq_tr'(obj$sq,C) =
      let val sq' = case sq of
            Bound 0 => Const(MtSeqCont,Adummy) |
            _ => seq_tr'(sq,Const(seqcont,Adummy))
      in C $ seq_obj_tr' obj $ sq' end;

fun seq_tr1'(Bound 0) = Const(MtSeq,Adummy) |
    seq_tr1' s = seq_tr'(s,Const(NmtSeq,Adummy));

fun true_tr'(_$Abs(_,_,s1)$Abs(_,_,s2)) =
      Const(STrue,Adummy)$seq_tr1' s1$seq_tr1' s2;

fun quant_tr q [idl,P] = abs_list_tr(Const(q,Adummy), idl, P);

fun quant_tr'(Q,q) (tm as _ $ Abs(_,_,_)) =
    abs_list_tr'(Const(q,Adummy), strip_qnt_vars Q tm, strip_qnt_body Q tm);

val mixfix =
 [ (*Representation of sequents*)
  Mixfix("((_)/ |- (_))", [Ssequence,Ssequence]--->Aprop, STrue, [6,6], 5),
  Delimfix("", Ssequence, MtSeq),
  Delimfix("__", [Sseqobj,Sseqcont]--->Ssequence, NmtSeq),
  Delimfix("", Sseqcont, MtSeqCont),
  Delimfix(",/ __", [Sseqobj,Sseqcont]--->Sseqcont, seqcont),
  
  Delimfix("_", Aform --> Sseqobj, ""),
  Delimfix("$_", SId --> Sseqobj, SeqId),
  Delimfix("$_", SVar --> Sseqobj, SeqVar),
  
  Infixl("=", [Aterm,Aterm]--->Aform, 50),
  Mixfix("~_", Aform --> Aform, "not", [40], 40),
  Infixr("&", [Aform,Aform]--->Aform, 35),
  Infixr("|", [Aform,Aform]--->Aform, 30),
  Infixr("-->", [Aform,Aform]--->Aform, 25),
  Infixr("<->", [Aform,Aform]--->Aform, 25),
  
  Mixfix("(3THE _./ _)", [SId, Aform]--->Aterm, SThe, [], 10),
  Mixfix("(3ALL _./ _)", [id_list, Aform]--->Aform, SForall, [], 10),
  Mixfix("(3EX _./ _)", [id_list, Aform]--->Aform, SExists, [], 10) ];

val const_decs = constants mixfix @
[([True], [Asequ,Asequ]--->Aprop),
 ([Seqof], Aform-->Asequ),
 ([The], (Aterm-->Aform) --> Aterm),
 ([Forall,Exists], (Aterm-->Aform) --> Aform)];

val ext = {logical_types=[Aterm,Aform], 
  mixfix=mixfix,
  parse_translation=
    [(STrue,true_tr), 
     (SThe,quant_tr The),
     (SForall,quant_tr Forall),
     (SExists,quant_tr Exists)],
  print_translation=
    [(True,true_tr'), 
     (The,quant_tr'(The,SThe)),
     (Forall, quant_tr'(Forall,SForall)), 
     (Exists, quant_tr'(Exists,SExists))]};

val syn = Syntax.extend pure ext;

fun read a = Syntax.read syn Any a;
fun prin t = Syntax.print_top_level syn t;

end;
end;
