(*  Title: 	LK/set/syntax
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

Zermelo-Fraenkel Set Theory
*)

signature SET_SYNTAX =
sig
  structure Syntax : PURE_SYNTAX
  val const_decs: (string list * typ) list
  val prin: term -> unit
  val read: string -> term
  val syn: Syntax.syntax
end;

functor Set_SyntaxFun (LK_Syntax: LK_SYNTAX) : SET_SYNTAX = 
struct
structure Syntax = LK_Syntax.Syntax;
structure Symtab = Syntax.Symtab;
local open Syntax LK_Syntax
in

(*** finite sets ***)

val Elist = Ground" Elist";

val mtset = Const("0",Aterm);
val setcons = Const("::",[Aterm,Aterm]--->Aterm);

(* enumeration of finite set elements *)
fun make_finset (Const(" Sing",_)$e) = setcons$e$mtset
  | make_finset (Const(" List",_)$e$l) = setcons $ e $ make_finset l;

fun enum_tr[elts] = make_finset elts;

val sing = Const(" Sing", Aterm-->Elist)
and list = Const(" List", [Aterm,Elist]--->Elist)
and enum = Const(" Enum", Elist-->Aterm);

(*Return the elements of a finite set, raise exception if ill-formed.*)
fun dest_finset (Const("::",_) $ e $ Const("0",_)) = sing $ e
  | dest_finset (Const("::",_) $ e $ l) = list $ e $ dest_finset l
  | dest_finset (Const("0",_)) = mtset
  | dest_finset _ = raise Match;

fun enum_tr' x = enum $ dest_finset x;

(*Replacement [ b[x] || x:A ]       = Replace(%(x)b[x], A)
  Collection  [  x   || x:A, P[x] ] = Collect(A, %(x)P[x])
  both	      [ b[x] || x:A, P[x] ] = Replace(%(x)b[x], Collect(A, %(x)P[x]))
*)

val Replace = Const("Replace",Adummy);
val Collect = Const("Collect",Adummy);

fun rep_tr[bx,Free(x,T),A] = Replace$absfree(x,T,bx)$A;

fun repcoll_tr[bx,x as Free(y,T),A,Px] =
    let val coll = Collect$A$absfree(y,T,Px)
    in if bx=x then coll else Replace$absfree(y,T,bx)$coll end;

val RepColl = Const(" RepColl",[Aterm, SId, Aterm, Aform]--->Aterm);
val Rep = Const(" Replace",[Aterm, SId, Aterm, Aform]--->Aterm);

fun coll_tr'(_$A$Abs(x,T,P)) =
    let val (y,Q) = variant_abs(x,T,P)
    in RepColl$Free(y,T)$Free(y,T)$A$Q end;

fun rep_tr'(_$Abs(x,T,bx)$(Const("Collect",_)$A$Abs(_,_,P))) =
      let val frees = add_term_names(bx, add_term_names(P,[]));
	  val y = variant frees x;
	  val by = subst_bounds ([Free(y,T)], bx);
	  val Q = subst_bounds ([Free(y,T)], P)
      in RepColl$by$Free(y,T)$A$Q end
  | rep_tr'(_$Abs(x,T,bx)$A) =
    let val (y,by) = variant_abs(x,T,bx) in  Rep$by$Free(y,T)$A end;

val mixfix =
 [Delimfix("0", Aterm, "0"),
  Infixl("`", [Aterm,Aterm]--->Aterm, 65),
  Infixr("Int", [Aterm,Aterm]--->Aterm, 60),
  Infixr("Un", [Aterm,Aterm]--->Aterm, 55),
  Infixr("-", [Aterm,Aterm]--->Aterm, 55),
  Infixr("::", [Aterm,Aterm]--->Aterm, 55),
  Infixr("<=", [Aterm,Aterm]--->Aform, 50),
  Infixr(":", [Aterm,Aterm]--->Aform, 50),
  Delimfix("(1<_,/_>)", [Aterm,Aterm]--->Aterm, "Pair"),
  Delimfix("{}", Aterm, "0"),
  Delimfix("{_}", Elist-->Aterm, " Enum"),
  Delimfix("_", Aterm-->Elist, " Sing"),
  Delimfix("_, _", [Aterm,Elist]--->Elist, " List"),
  Delimfix("(2[ _ ||/ _: _])", [Aterm,SId,Aterm]--->Aterm, " Replace"),
  Delimfix("(2[ _ ||/ _: _, _])", [Aterm,SId,Aterm,Aform]--->Aterm, " RepColl")
 ];

val ext = {logical_types=[],
  mixfix=mixfix,
  parse_translation=
    [(" Enum", enum_tr),
     (" Replace", rep_tr),
     (" RepColl", repcoll_tr) ],
  print_translation=
    [("Replace",rep_tr'),
     ("Collect",coll_tr'),
     ("::",enum_tr') ]};


(*Constants and their types*)
val const_decs = constants mixfix @
  [ (["Pow","Choose","Union","Inter","succ"],	Aterm-->Aterm),
    (["Collect"],	[Aterm, Aterm-->Aform] ---> Aterm ),
    (["Replace"],	[Aterm-->Aterm, Aterm] ---> Aterm ),
    (["INF"],	Aterm ) ];

val syn = Syntax.extend LK_Syntax.syn ext;

fun read a = Syntax.read syn Any a;
fun prin t = Syntax.print_top_level syn t;

end;
end;
