(*  Title: 	Printer
    Author: 	Tobias Nipkow
*)

signature PRINTER =
sig
  structure XGram: XGRAM
  type PrintTab
  val mk_print_tab: string XGram.Prod list * (term -> term) XGram.Symtab.table
			-> PrintTab
  val printer: PrintTab -> term -> unit
  val printer_nl: PrintTab -> term -> unit
end;

functor PRINTER_FUN(structure Lexicon:LEXICON and Pure_Ext:PURE_EXT
		    and XGram:XGRAM and Pretty:PRETTY) : PRINTER =
struct

structure XGram = XGram;

local open XGram in

datatype Symb = NT of int | T of  Lexicon.Token | S of string |
		Bg' of int | En' | Brk' of int;
datatype PrInfo = Prnt of Symb list * int | Trnslt of term -> term
		| TorP of (term -> term) * Symb list * int
type PrintTab = PrInfo Symtab.table;

fun noargs(NT(_)::l) = noargs(l)+1
  | noargs(T(_)::l) = noargs(l)+1
  | noargs(_::l) = noargs(l)
  | noargs(_) = 0;

fun merge(s,S(s')::l) = S(s^s')::l
  | merge(s,l) = S(s)::l;

fun add_pr(Terminal(s),l) = merge(s,l)
  | add_pr(Space(sp),l) = merge(sp,l)
  | add_pr(Nonterminal(s,p),l) = let val tk = Lexicon.predef_term s
	in (if tk = Lexicon.end_token then NT(p) else T(tk))::l end
  | add_pr(Bg(i),l) = Bg'(i)::l
  | add_pr(Brk(i),l) = Brk'(i)::l
  | add_pr(En,l) = En'::l;

fun syn2pr(sy) = itlist_right add_pr (sy,[]);

fun add_prod(Prod(_,_,"",_),prt) = prt
  | add_prod(Prod(_,sy,opn,p),prt) = (case Symtab.lookup(prt,opn) of
	None => Symtab.update((opn,Prnt(syn2pr sy,p)),prt)
      | Some(Prnt _) => prt
      | Some(Trnslt f) => Symtab.update((opn,TorP(f,syn2pr sy,p)),prt)
      | Some(TorP _) => prt);

fun add_tr(prt,(opn,f)) = Symtab.update((opn,Trnslt f),prt);

fun mk_print_tab(prodl,trtab: (term->term)Symtab.table) : PrintTab =
    let val prt0 = itlist_left add_tr (Symtab.null,Symtab.alist_of trtab)
    in itlist_right add_prod (prodl,prt0) end;

fun string_of_vname (a,idx) = if is_digit(hd(rev(explode a)))
	then a ^ "." ^ string_of_int idx
	else if idx = 0 then a else a ^ string_of_int idx

fun printer (prtab:PrintTab) tm =
    let fun print_syn([],_) = () |
            print_syn(symb::rest,args) =
              case symb of
                T(tk) => (printm(hd args,0); print_syn(rest,tl args))
              | S(s) => (Pretty.st s; print_syn(rest,args))
              | NT(p) => (printm(hd args,p); print_syn(rest,tl args))
	      | Bg'(i) => (Pretty.bg i; print_syn(rest,args))
	      | En' => (Pretty.en(); print_syn(rest,args))
	      | Brk'(i) => (Pretty.brk(i,0); print_syn(rest,args))

        and print_par(pr, args, p, p':int) = if p > p'
              then (Pretty.bgs"("; print_syn(pr,args); Pretty.ens")")
              else print_syn(pr,args)

        and trans_print(opn,tm,p) =
              let val Some(Trnslt f) = Symtab.lookup(prtab,opn)
              in printm(f tm,p) end

	and print_prefix(tm,opn,noargs,p) = if noargs=0 then Pretty.st opn
		else trans_print(Pure_Ext.appl_const,tm,p)

        and print_comb(opn,args,tm,p) =
              let val la = length args
              in case Symtab.lookup(prtab,opn) of
                   None => print_prefix(tm,opn,la,p)
                 | Some(Prnt(pr,p')) =>
                     if la = noargs pr then print_par(pr,args,p,p')
                     else print_prefix(tm,opn,la,p)
                 | Some(Trnslt f) => (printm(f tm,p) handle Match =>
					print_prefix(tm,opn,la,p))
                 | Some(TorP(f,pr,p')) => (printm(f tm,p) handle Match =>
			if la = noargs pr then print_par(pr,args,p,p')
			else print_prefix(tm,opn,la,p))
              end

        and print_appl(tm,p) =
              case strip_comb tm of
                (Const(opn,_),args) => print_comb(opn,args,tm,p) |
                (_,_) => trans_print(Pure_Ext.appl_const,tm,p)

        and printm(Free(name,_),_) = Pretty.st name |
            printm(Var(uname,_),_) = Pretty.st("?" ^ string_of_vname uname) |
            printm(tm as Const(opn,_),p) = print_comb(opn,[],tm,p) |
            printm(tm as _$_,p) = print_appl(tm,p) |
            printm(tm as Abs _,p) = trans_print(Pure_Ext.abs_const,tm,p) |
            printm(Bound i,_) = Pretty.st("B." ^ string_of_int i)

    in printm(tm,0) end;

fun printer_nl (prtab:PrintTab) tm =
	(Pretty.init(); printer prtab tm;  Pretty.nl());

end;
end;
