signature OutStream =
sig

(* OUTPUT STREAMS

   Created by:  Dave Berry, LFCS, University of Edinburgh
                db@lfcs.ed.ac.uk
   Date:        12 Nov 1989

   Maintenance: Author


   DESCRIPTION

   Functions on output streams.


   SEE ALSO

   InStream, StreamPair


   NOTES

   All counterparts of functions in the initial basis have been given both
   the original names and names that fit the library conventions.

   The third argument of the General.Io exception is a system-specific
   message, excpet in the case of the output and output' functions.
*)

  val version: real
   (* version = 0.1 *)


(* TYPES *)

  type OutStream
  type outstream
    sharing type outstream = OutStream


(* MANIPULATORS *)

  val stdOut: OutStream
   (* stdOut; the standard output stream, by default connected to the
      terminal. *)

  val std_out: OutStream
   (* std_out = stdOut *)

  val openOut: string -> OutStream
   (* openOut s; create a new file s for writing to, overwriting or renaming
      any existing file called s.  Raises (General.Io ("openOut", s, "")) if
      the new file can't be created. *)

  val open_out: string -> OutStream
   (* open_out = openOut *)

  val closeOut: OutStream -> unit
   (* closeOut os; terminates os, flushing any buffered characters. *)

  val close_out: OutStream -> unit
   (* close_out = closeOut *)

  val output: OutStream * string -> unit
   (* output (os, s); writes the characters of s to os.
      Raises (General.Io ("output", filename, s)) if os has been closed. *)

  val output': OutStream -> string -> unit
   (* output' os s = curry output *)


(* SYSTEM *)

  exception NotImplemented of string
   (* NotImplemented fn; raised if fn is not provided. *)

  val openAppend: string -> OutStream
   (* openAppend file; returns a new outstream whose consumer is the file s.
      Output to this stream is appended to s.  If s doesn't exist, it is
      created, initially empty. *)

  val flush: OutStream -> unit
   (* flush os; ensures that all characters output on os have been or can
      be received by the associated consumer without being delayed by the
      implementation.  For example, characters output to a stream connected
      to a terminal will appear on that screen, even if they are normally
      buffered. *)

  val eof: OutStream -> unit
   (* eof os: signal an end of stream on os without closing os. *)

  val interactive: OutStream -> bool
   (* interactive os; returns true if os is associated with a terminal. *)
end

