signature MonoVector =
sig

(* CONSTANT VECTORS OF A NAMED TYPE

   Created by:	Dave Berry, LFCS, University of Edinburgh
		db@lfcs.ed.ac.uk
   Date:	31 Oct 1989

   Maintenance:	Author


   DESCRIPTION

   An MonoVector is a single-dimensional constant array of objects.  The first
   element has the index 0.  MonoVectors are equal if they contain the same
   elements and these elements admit equality.

   To create a MonoVector of MonoVectors, use the "generate" functions.

   Example MonoVectors are ByteVectors and BoolVectors, which will often
   be implemented specially.  For example, a BoolVector can use one bit
   per element.


   SEE ALSO

   Vector, MonoRefVector.
*)

  val version: real
   (* version = 0.1 *)


(* TYPES *)

  type Object
  type MonoVector


(* CREATORS *)

  val create: Nat -> Object -> MonoVector
   (* create n i; create a MonoVector of n locations, each containing i.
      Raise (Nat ("create", n)) if n < 0. *)

  val generate: Nat -> (int -> Object) -> MonoVector
   (* generate n f; create a MonoVector v of n locations, (v sub 0) to
      (v sub (n-1)) with (v sub i) initialised to (f i).
      Raise (Nat ("generate", n)) if n < 0. *)

  val generate': Nat -> ('b -> Object * 'b) -> 'b -> MonoVector
   (* generate' n f base; create a MonoVector of n locations, (v sub 0) to
      (v sub (n-1)) with (v sub 0) initialised to (# 1 (f base)) and
      (v sub i (i > 0)) initialised to (# 1 (f (# 2 f_i))), where f_i is
      the result of the i^th application of f.
      Raise (Nat ("generate'", n)) if n < 0. *)


(* CONVERTERS *)

  val stringSep: string -> string -> string -> MonoVector -> string

  val string: MonoVector -> string

  val parseSepN': string -> string -> string -> Nat -> string list ->
                  (MonoVector * string list,
		   MonoVector Option * string list) Result

  val parseSep': string -> string -> string -> string list ->
                 (MonoVector * string list,
		  MonoVector Option * string list) Result

  val parseN': Nat -> string list ->
               (MonoVector * string list,
		MonoVector Option * string list) Result

  val parse': string list ->
	      (MonoVector * string list,
	       MonoVector Option * string list) Result

  val parseSepN: string -> string -> string ->
                 Nat -> string -> (MonoVector, MonoVector Option) Result

  val parseSep: string -> string -> string ->
                string -> (MonoVector, MonoVector Option) Result

  val parseN: Nat -> string -> (MonoVector, MonoVector Option) Result

  val parse: string -> (MonoVector, MonoVector Option) Result

  val readSep: string -> string -> string -> InStream ->
               (MonoVector, MonoVector Option) Result

  val readSepN: string -> string -> string -> Nat -> InStream ->
               (MonoVector, MonoVector Option) Result

  val readN: Nat -> InStream -> (MonoVector, MonoVector Option) Result

  val read: InStream -> (MonoVector, MonoVector Option) Result

  val list: MonoVector -> Object list
   (* list v; make a list containing (only) the elements of v, in
      the same order. *)

  val fromList: Object list -> MonoVector
   (* from_list l; make a Vector containing (only) the elements of l, in
      the same order. *)

  val fromFile: string -> MonoVector
   (* fromFile name; read the contents of the file called name into an
      MonoVector.  Stops reading the file as soon as a read of an Object returns
      Fail. Raises General.Io if something goes wrong. *)

  val file: string -> MonoVector -> unit
   (* file name v; write the contents of v to the new file called name.
      Raises General.Io if something goes wrong. *)


(* OBSERVERS *)

  val empty: MonoVector -> bool
   (* empty v; returns true if v is empty. *)

  val size: MonoVector -> Nat
   (* size v; return the number of elements in v. *)

  val eq: MonoVector -> MonoVector -> bool
   (* eq x y; returns true if (size x = size y) and for all i,
         0 <= i <= size x, (Object.eq (x sub i) (y sub i)). *)

  val ne: MonoVector -> MonoVector -> bool
   (* ne x y; returns true if (size x = size y) and for all i,
         0 <= i <= size x, (Object.ne (x sub i) (y sub i)). *)

  val lt: MonoVector -> MonoVector -> bool
   (* lt x y; returns true if (Object.lt (x sub i) (y sub i)) for
         0 <= i <= size x or if x is empty and y is not empty. *)

  val le: MonoVector -> MonoVector -> bool
   (* le x y; returns true if (Object.le (x sub i) (y sub i)) for
         0 <= i <= size x or if x is empty. *)

  val gt: MonoVector -> MonoVector -> bool
   (* gt x y; returns true if (Object.gt (x sub i) (y sub i)) for
         0 <= i <= size y or if y is empty and x is not empty. *)

  val ge: MonoVector -> MonoVector -> bool
   (* ge x y; returns true if (Object.ge (x sub i) (y sub i)) for
         0 <= i <= size y or if y is empty. *)


(* SELECTORS *)

  exception Sub of string * int

  (* infix 9 sub *)
  val sub: MonoVector * int -> Object
   (* v sub n; return the n+1'th element of v.
      Raise (Sub ("sub", n)) if not (0 <= n <= size v). *)

  val nth: int -> MonoVector -> Object
   (* nth n v; return the n+1'th element of v.
      Raise (Sub ("nth", n)) if not (0 <= n <= size v). *)

  exception Extract of int * int
  val extract: int -> int -> MonoVector -> MonoVector
   (* extract start finish v; returns the sub-vector of v starting with
      (v sub start) and ending with (v sub (finish - 1)).
      Returns the empty vector if (start = finish).
      Raise (Extract (start, finish)) if not (0 <= start,finish <= size v). *)


(* MANIPULATORS *)

  val rev: MonoVector -> MonoVector
   (* rev v; builds a new MonoVector containing the elements of v in
      reverse order. *)

  (* infix 6 ^ *)
  val ^ : MonoVector * MonoVector -> MonoVector
   (* v ^ v'; builds a new MonoVector containing the elements of v' appended
      to those of v. *)


(* REDUCERS *)

  val foldR: (Object -> 'b -> 'b) -> 'b -> MonoVector -> 'b
   (* foldR f base v; folds using f associating to the right over the
      base element.
      foldR f [a1,a2,...,an] base = f(a1,f(a2,...,f(an,base)...)). *)

  val foldL: (Object -> 'b -> 'b) -> 'b -> MonoVector -> 'b
   (* foldL f v base; folds using f associating to the left over the
      base element.
      foldL f [a1,a2,...,an] base = f(an,...,f(a2,f(a1,base))...). *)

  exception Empty of string

  val foldR': (Object -> Object -> Object) -> MonoVector -> Object
   (* foldR' f v; folds using f associating to the right over the
      last element of v.  Raises (Empty "foldR'") if v is empty. *)

  val foldL': (Object -> Object -> Object) -> MonoVector -> Object
   (* foldL' f v; folds using f associating to the right over the
      last element of v.  Raises (Empty "foldL'") if v is empty. *)

  val pairwise: (Object -> Object -> bool) -> MonoVector -> bool
   (* pairwise f v; true if (f (v sub i) (v sub (i + 1))) is true for all
      0 <= i < size v, or if v is empty. *)


(* ITERATORS *)

  val map: (Object -> Object) -> MonoVector -> MonoVector
   (* map f v; builds a new vector by applying f to each element of v. *)

  val apply: (Object -> unit) -> MonoVector -> unit
   (* apply f v; applies f to each element of v. *)

  val iterate: (Object * int -> Object) -> MonoVector -> MonoVector
   (* iterate f v; builds a new vector by applying f to each element of v
      paired with its index. *)

  val iterateApply: (Object * int -> unit) -> MonoVector -> unit
   (* iterate f v; applies f to each element of v paired with its index. *)
end
