(*$StreamPair : InStreamType OutStreamType GeneralTypes Int *)

loadSig "StreamPair";

structure StreamPair: StreamPair =

(* PAIRS OF ONE INPUT STREAM AND ONE OUTPUT STREAM

   Created by:  Dave Berry, LFCS, University of Edinburgh
                db@lfcs.ed.ac.uk
   Date:        12 Nov 1989

   Maintenance: Author


   SEE ALSO

   InStream, OutStream
*)

struct

  val version = 0.1


(* TYPES *)

  datatype StreamPair = Pair of InStream * OutStream


(* CONSTANTS *)

  val std = Pair (InStream.stdIn, OutStream.stdOut)


(* CREATORS *)

  val create = Pair


(* CONVERTORS *)

  fun streams (Pair x) = x


(* SYSTEM *)

  fun openTemporary () = raise General.NotImplemented "openTemporary"
  fun execute s = raise General.NotImplemented "execute"


(* MANIPULATORS *)

  fun openPair s =
	OK (Pair (InStream.openIn s, OutStream.openOut s))
	handle General.Io s => Fail s

  fun closePair (Pair (i, os)) = (InStream.closeIn i; OutStream.closeOut os)

  fun output (io as Pair (_, os), s) =
	OutStream.output (os, s)
     
  val output' = General.curry output

  fun input (Pair (i, _), n) = InStream.input (i, n)

  fun input1 io = input (io, 1)

  fun input' (Pair (i, _)) n = InStream.input' i n

  fun lookahead (Pair (i, _)) = InStream.lookahead i

  fun eof (Pair (i, _)) = InStream.eof i

  fun canInput (Pair (i, _)) n =
	InStream.canInput i n

  fun flush (Pair (_, os)) =
	OutStream.flush os

  fun reset (Pair (i, _)) =
	InStream.reset i

  fun interactive (Pair (i, os)) =
	InStream.interactive i andalso OutStream.interactive os

  fun readString (Pair (i, _)) s =
	InStream.readString i s

  fun skip p (Pair (i, _)) =
	InStream.skip p i

  fun inputLine (Pair (i, _)) =
	InStream.inputLine i

  fun prompt io s =
	(output' io (s ^ " ");
	 inputLine io)

  fun ask (io as Pair (i, os)) s p =
	(OutStream.output' os (s ^ "\n");
	 case p i of
	   Fail _ =>
	     ( InStream.inputLine i;
	       output' io "Invalid input\n";
	       ask io s p
	     )
	 | OK x => x)

  fun confirm io s =
	(output' io (s ^ "\n");
	 case input1 io of
	   "y" => (inputLine io; true)
	 | "n" => (inputLine io; false)
	 |  _  => (inputLine io; confirm io s))

  fun menu io t l =
        let
          fun readInt i =
            case Int.read i of
	      OK n =>
		if n > 0 andalso n <= List.size l then OK n
		else Fail None
	    | Fail x => Fail x

	  fun outputEntry (x, n) =
	    let val num = String.padL " " 2 (Int.string (n+1))
	    in output' io (num ^ ". " ^ x ^ "\n")
	    end
	in
	  output' io (t ^ "\n\n");
	  List.iterate outputEntry l;
	  ask io "Select Entry: " readInt
	end
end
