(*$MonoRefVector : Object RefVector GeneralTypes InStreamType OutStreamType *)

loadSig "MonoRefVector";

functor MonoRefVector (

(* VARIABLE VECTORS OF A GIVEN TYPE

   Created by:	Dave Berry, LFCS, University of Edinburgh
		db@lfcs.ed.ac.uk
   Date:	12 Feb 1990

   Maintenance: Author


   DESCRIPTION

   A MonoRefVector is a unique sequence of objects that can be updated in
   place.  There is only one empty Vector.

   This is a reference implementation.  Most systems will provide most of
   the following as built-in functions.  Any sensible implementation will
   provide constant time access to each element.


   SEE ALSO

   MonoVector, RefVector.
*)

  structure Object: Object
): MonoRefVector =
struct

  val version = 0.1


(* TYPES *)

  type Object = Object.T

  type MonoRefVector = Object RefVector.RefVector


(* CREATORS *)

  val create = RefVector.create

  val generate = RefVector.generate

  val generate' = RefVector.generate'


(* ITERATORS *)

  val map = RefVector.map

  val apply = RefVector.apply

  val iterate = RefVector.iterate

  val iterateApply = RefVector.iterateApply


(* OBSERVERS *)

  val size = RefVector.size

  fun empty v = (size v = 0)

  val same = RefVector.same

  val different = RefVector.different

  fun eq v v' = RefVector.eq Object.eq v v'

  fun ne v v' = RefVector.ne Object.ne v v'

  fun lt v v' = RefVector.lt Object.lt v v'

  fun le v v' = RefVector.le Object.le v v'

  fun gt v v' = RefVector.gt Object.gt v v'

  fun ge v v' = RefVector.ge Object.ge v v'


(* CONVERTORS *)

  val list = RefVector.list

  val fromList = RefVector.fromList

  fun stringSep start finish sep v =
	RefVector.stringSep start finish sep Object.string v

  fun string v =
        if Object.fixedWidth then
          stringSep "" "" "" v
        else
          stringSep "" "" " " v

  exception Sep of string * string * string * string

  (* The parse, parse' and read functions assume that entries have a fixed
     width or are separated by formatting characters. *)

  fun parseSepN' start finish sep n l =
        RefVector.parseSepN' start finish sep Object.parse' n l
        handle RefVector.Sep x => raise Sep x

  fun parseSep' start finish sep l =
        RefVector.parseSep' start finish sep Object.parse' l
        handle RefVector.Sep x => raise Sep x

  fun parseN' n l =
        if n < 0 then raise General.Nat ("parseN'", n)
        else if Object.fixedWidth
	then parseSepN' "" "" "" n l
	else parseSepN' "" "" " " n l

  fun parse' l =
        if Object.fixedWidth
	then parseSep' "" "" "" l
	else parseSep' "" "" " " l

  fun parseSepN start finish sep n s =
        if n < 0 then raise General.Nat ("parseSepN", n)
        else
          case parseSepN' start finish sep n (explode s) of
            OK (v, _) => OK v
          | Fail (x, _) => Fail x

  fun parseSep start finish sep s =
        case parseSep' start finish sep (explode s) of
          OK (v, _) => OK v
        | Fail (x, _) => Fail x

  fun parseN n s =
        if n < 0 then raise General.Nat ("parseN", n)
        else if Object.fixedWidth
	then parseSepN "" "" "" n s
	else parseSepN "" "" " " n s

  fun parse s =
        if Object.fixedWidth
	then parseSep "" "" "" s
	else parseSep "" "" " " s

  fun readSepN start finish sep n i =
        RefVector.readSepN start finish sep Object.read n i
        handle RefVector.Sep x => raise Sep x

  fun readSep start finish sep i =
        RefVector.readSep start finish sep Object.read i
        handle RefVector.Sep x => raise Sep x

  fun readN n i =
        if n < 0 then raise General.Nat ("readN", n)
        else if Object.fixedWidth
	then readSepN "" "" "" n i
	else readSepN "" "" " " n i

  fun read i =
        if Object.fixedWidth
	then readSep "" "" "" i
	else readSep "" "" " " i

  fun fromFile name =
	let fun readList i =
		  case Object.read i
		  of Fail None => (InStream.closeIn i; [])
		  |  Fail (Some x) => (InStream.closeIn i; [x])
		  |  OK x => x :: readList i
	in fromList (readList (InStream.openIn name))
	end

  fun file name v =
	let val os = OutStream.openOut name
	    fun out s = OutStream.output' os s
	in RefVector.apply (out o Object.string) v;
	   OutStream.closeOut os
	end


(* SELECTORS *)

  exception Sub of string * int
  fun sub (v, i) =
	RefVector.sub (v, i)
	handle RefVector.Sub x => raise Sub x
  infix 9 sub;

  fun nth n v = v sub n
		handle Sub _ => raise Sub ("nth", n)

  exception Extract of int * int
  fun extract start finish v =
	RefVector.extract start finish v
	handle RefVector.Extract x => raise Extract x


(* MANIPULATORS *)

  val rev = RefVector.rev

  infix 6 ^
  val op ^ = RefVector.^

  exception Update of int
  fun update i x v =
        RefVector.update i x v
        handle RefVector.Update _ => raise Update i

  exception Copy of int * int * int
  fun copy start finish v start' v' =
  	RefVector.copy start finish v start' v'
	handle RefVector.Copy x => raise Copy x

  exception UpdateRange of int * int
  fun updateRange start finish i v =
  	RefVector.updateRange start finish i v
        handle RefVector.UpdateRange x => raise UpdateRange x

  fun sort v = RefVector.sort Object.lt v


(* REDUCERS *)

  exception Empty of string

  val foldL = RefVector.foldL

  fun foldL' f v =
	if empty v then raise Empty "foldL'"
        else RefVector.foldL' f v

  val foldR = RefVector.foldR

  fun foldR' f v =
	if empty v then raise Empty "foldR'"
        else RefVector.foldR' f v

  val pairwise = RefVector.pairwise
end
