(*$String : Object StringType GeneralTypes OutStreamType List' Int Pair *)

loadSig "String";
loadStr "StringParse";
loadStr "StringObject";
loadStr "StringObject'";

structure String: String =

(* ASCII STRINGS

   Created by:  Dave Berry, LFCS, University of Edinburgh
                db@lfcs.ed.ac.uk
   Date:        4 Oct 1989

   Maintenance: Author

   DESCRIPTION

   Standard functions on the built-in type "string".

   Functions such as  search  and  index  take an integer
   offset.  This is because it's more efficient to index into
   a string than to take a substring.

*)

struct

  open StringType


(* PERVASIVES *)

  exception Chr = Chr
  and Ord = Ord

  val size = size
  val ord = ord
  val chr = chr
  val explode = explode
  val implode = implode
  val op ^ = op ^


(* LOCAL *)

  fun pairApply f (x, y) = (f x, f y)


(* CREATORS *)

  fun create n s =
	if n < 0 then raise General.Nat ("create", n)
	else General.iterate n (fn x => s ^ x) ""


(* CONVERTORS *)

  fun fromFile name =
        let fun fromFile' i =
    	          let val s = InStream.input (i, 256)
	          in if size s < 256 then (InStream.closeIn i; s)
	             else s ^ fromFile' i
	          end
        in fromFile' (InStream.openIn name)
        end

  fun file name s =
	let val os = OutStream.openOut name
	in OutStream.output' os s;
	   OutStream.closeOut os
	end


(* ITERATORS *)

  fun map f s = implode (List'.map f (explode s))

  fun apply f s = List'.apply f (explode s)

  fun mapAll p f s = implode (List'.mapAll p f (explode s))

  fun applyAll p f s = (List'.applyAll p f (explode s))


(* SELECTORS *)

  exception Sub of string * int

  exception Extract of int * int

  fun extract start finish s =
	OldString.substring (s, start, finish - start)
	handle OldString.Substring => raise Extract (start, finish)


  fun forAll p s = List'.forAll p (explode s)

  fun exists p s = List'.exists p (explode s)

  fun prefixes s1 s2 n =
	List'.prefixes (explode s1) (explode (extract n (size s2 - 1) s2))
	handle Extract _ => raise Sub ("prefixes", n)

  fun postfixes s1 s2 n =
	List'.prefixes (rev (explode s1)) (rev (explode (extract 0 n s2)))
	handle Extract _ => raise Sub ("postfixes", n)



(* SOME MANIPULATORS *)

  fun upper s =
        if isLower s
        then chr (ord s + ord "A" - ord "a") ^ extract 1 (size s - 1) s
        else s
        handle Empty _ => raise Empty "upper"

  fun lower s =
        if isUpper s
        then chr (ord s + ord "a" - ord "A") ^ extract 1 (size s - 1) s
        else s
        handle Empty _ => raise Empty "lower"

  fun ascii s =
        if isAscii s then s
        else chr (ord s - 128) ^ extract 1 (size s - 1) s
        handle Empty _ => raise Empty "ascii"

  fun control s =
        let val s' = chr (ord s - 64)
        in if isControl s'
           then s' ^ extract 1 (size s - 1) s
           else s
        end
        handle Chr => s
        |      Empty _ => raise Empty "control"


(* OBJECT *)

  structure Object = StringObject
  structure Object' = StringObject'
  open Object

  val le' = Object'.le
  val lt' = Object'.lt
  val ge' = Object'.ge
  val gt' = Object'.gt
  val eq' = Object'.eq
  val ne' = Object'.ne


(* SEARCHING AND INDEXING *)

  local
    fun search' _ nil _ = Fail ()
    |   search' s (s' as _::t') n =
      if List'.prefixes s s' then OK n
      else search' s t' (n+1)	(* Boyer and Moore?  Never heard of them! *)
  in
    fun search s' s n =
	  if n < 0 orelse n >= size s then raise Sub ("search", n)
	  else if s' = "" then OK 0
	  else search' (explode (extract n (size s - 1) s)) (explode s') 0
    fun revSearch s' s n =
	  if n < 0 orelse n >= size s then raise Sub ("revSearch", n)
	  else if s' = "" then OK (size s - 1)
	  else case search' (List'.rev (explode (extract n (size s - 1) s)))
		            (List'.rev (explode s')) 0
	       of (OK i) => OK (size s - i - size s')
	       |  Fail () => Fail ()
  end

  fun occurs s' s n =
	case search s' s n 
	of OK _ => true
	|  Fail () => false
	handle Sub _ => raise Sub ("occurs", n)

  fun revOccurs s' s n =
	case revSearch s' s n 
	of OK _ => true
	|  Fail () => false
	handle Sub _ => raise Sub ("revOccurs", n)

  fun index p s n =
	List'.index p (explode (extract n (size s - 1) s))
	handle Extract _ => raise Sub ("index", n)

  fun revIndex p s n =
	let val (OK i) = List'.index p (rev (explode (extract 0 n s)))
	in OK (size s - i - 1)
	end
	handle Extract _ => raise Sub ("revIndex", n)


(* MANIPULATING THE NTH ELEMENT *)

  infix 9 sub
  fun s sub n = List'.sub (explode s, n)
		handle List'.Sub x => raise Sub x

  fun nth n s = s sub n
		handle Sub _ => raise Sub ("nth", n)

  exception Char of string * string

  fun updateSub n s' s =
	if size s' <> 1 then raise Char ("updateSub", s')
	else implode (List'.updateSub n s' (explode s))
	handle List'.Sub p => raise Sub p

  fun changeSub n f s =
	implode (List'.changeSub n f (explode s))
	handle List'.Sub p => raise Sub p

  fun insertSub n s' s =
	implode (List'.insertSub n (explode s') (explode s))
	handle List'.Sub p => raise Sub p

  fun appendSub n s' s =
	implode (List'.appendSub n (explode s') (explode s))
	handle List'.Sub p => raise Sub p

  fun spliceSub n s' s =
	implode (List'.spliceSub n (explode s') (explode s))
	handle List'.Sub p => raise Sub p


(* MANIPULATING THE FIRST ELEMENT THAT SATISFIES A PREDICATE *)

  exception First of string

  fun first p s =
	List'.first p (explode s)
	handle List'.First p => raise First p

  fun dropFirst p s =
	implode (List'.dropFirst p (explode s))
	handle List'.First p => raise First p

  fun splitFirst p s =
	pairApply implode (List'.splitFirst p (explode s))
	handle List'.First p => raise First p

  fun updateFirst p s' s =
	if size s' <> 1 then raise Char ("updateFirst", s')
	else implode (List'.updateFirst p s' (explode s))
	handle List'.First p => raise First p

  fun changeFirst p f s =
	implode (List'.changeFirst p f (explode s))
	handle List'.First p => raise First p

  fun insertFirst p s' s =
	implode (List'.insertFirst p (explode s') (explode s))
	handle List'.First p => raise First p

  fun appendFirst p s' s =
	implode (List'.appendFirst p (explode s') (explode s))
	handle List'.First p => raise First p

  fun spliceFirst p s' s =
	implode (List'.spliceFirst p (explode s') (explode s))
	handle List'.First p => raise First p


(* TAKING A PREFIX OF ELEMENTS THAT SATISFY A PREDICATE *)

  fun prefix p s = implode (List'.prefix p (explode s))

  fun dropPrefix p s = implode (List'.dropPrefix p (explode s))

  fun splitPrefix p s = pairApply implode (List'.splitPrefix p (explode s))


(* MANIPULATING ALL ELEMENTS THAT SATISFY A PREDICATE *)

  fun all p s = implode (List'.all p (explode s))

  fun dropAll p s = implode (List'.dropAll p (explode s))

  fun splitAll p s = pairApply implode (List'.splitAll p (explode s))

  fun updateAll p s' s =
	if size s' <> 1 then raise Char ("updateAll", s')
	else implode (List'.updateAll p s' (explode s))

  fun changeAll p f s = implode (List'.changeAll p f (explode s))

  fun insertAll p s' s = implode (List'.insertAll p (explode s') (explode s))

  fun appendAll p s' s = implode (List'.appendAll p (explode s') (explode s))

  fun spliceAll p s' s = implode (List'.spliceAll p (explode s') (explode s))


(* OTHER MANIPULATORS *)

  fun subst c s' s =
	if size c <> 1 then raise Char ("subst", c)
	else implode (List'.updateAll (eq c) s' (explode s))

  fun subst' c s' s =
	if size c <> 1 then raise Char ("subst'", c)
	else implode (List'.updateAll (eq' c) s' (explode s))

  fun dropRepeats s = implode (List'.dropRepeats eq (explode s))

  fun dropRepeats' s = implode (List'.dropRepeats eq' (explode s))

  fun rev s = implode (List'.rev (explode s))

  fun padL c w s =
	if size c <> 1 then raise Char ("padL", c)
	else if size s >= w then s
	else (create (w - size s) c) ^ s

  fun padR c w s =
	if size c <> 1 then raise Char ("padL", c)
	else if size s >= w then s
	else s ^ (create (w - size s) c)

  fun padC c w s =
	if size c <> 1 then raise Char ("padL", c)
	else if size s >= w then s
	else
	  let val n = w - size s
	      val l = n div 2
	      val r = if n mod 2 = 0 then n div 2 else n div 2 + 1
	  in (create l c) ^ s ^ (create r c)
	  end

  fun truncL w s =
	if size s <= w then s
	else extract (size s - w) (size s) s

  fun truncR w s =
	if size s <= w then s
	else extract 0 w s

  fun truncC w s =
	if size s <= w then s
	else
	  let val n = size s - w
	      val r = n div 2
	      val l = if n mod 2 = 0 then n div 2 else n div 2 + 1
	  in extract l r s
	  end

  fun dropL c s =
	if size c <> 1 then raise Char ("dropL", c)
	else dropPrefix (fn x => (x = c)) s

  fun dropR c s =
	if size c <> 1 then raise Char ("dropR", c)
	else rev (dropPrefix (fn x => (x = c)) (rev s))

  local
    fun show' h =
      if isVisible h then [h]
      else
        case h
	of "\n"   => ["\\n"]
	|  "\t"   => ["\\t"]
	|  " "    => [" "]
	|  "\127" => ["\\127"]
	|  _ =>
	    if isControl h then
	      ["\\^", chr (ord h + ord "@")]
	    else
	      ["\\", padL "0" 3 (Int.string (ord h))]
	
    fun show nil = nil
    |   show (h::t) = show' h @ show t
  in
    fun showAscii s = implode (show (explode s))
  end

  fun words' sep singles s =
	let val sep' = explode sep
	    val singles' = explode singles
	    fun takeWord [] = ("", [])
	    |   takeWord (h::t) =
		  if List'.member h sep' then ("", t)
		  else if List'.member h singles' then (h, t)
		  else 
		    let val (s, l) = takeWord t
		    in (h^s, l)
		    end
	    fun words'' l =
		 case
		   takeWord l
		 of (w, []) => [w]
		 |  (w, t)  => w :: words'' t
	in words'' (explode s)
	end

  fun words sep s = words' sep "" s

end
