(*$RefVector: GeneralTypes List InStreamType OutStreamType *)

loadSig "RefVector";

structure RefVector: RefVector =

(* VARIABLE VECTORS

   Created by:	Dave Berry, LFCS, University of Edinburgh
   Date:	30 Oct 1989

   This is a reference implementation.  Most systems will provide most of
   the following as built-in functions.

   Maintenance:	Author


   SEE ALSO

   Vector.
*)

struct

  val version = 0.1


(* TYPES *)

  type 'a RefVector = 'a Array.array

  type 'a T = 'a RefVector


(* SUB *)

  exception Sub of string * int
  fun sub (v, n) =
  	Array.sub (v, n)
	handle Array.Subscript => raise Sub ("sub", n)
  infix 9 sub;

  fun nth n v = v sub n
		handle Sub _ => raise Sub ("nth", n)

(* LIST CONVERTORS *)

  fun list v =
    let val n = Array.length v
	fun list' i =
    	  if i = n then []
    	  else v sub i :: list' (i+1)
    in list' 0
    end

  val fromList = Array.arrayoflist;


(* EXTRACT *)

  exception Extract of int * int
  fun extract start finish v =
        fromList (List.extract start finish (list v))
        handle List.Extract r => raise Extract r


(* CREATORS *)

  fun create size init =
        if size < 0 then raise General.Nat ("create", size)
        else fromList (List.create size init)

  fun generate size f =
        if size < 0 then raise General.Nat ("generate", size)
        else fromList (List.generate size f)

  fun generate' size f base =
        if size < 0 then raise General.Nat ("generate'", size)
	else fromList (List.generate' size f base)


(* ITERATORS *)

  fun map f v = fromList (List.map f (list v))

  fun apply f v = List.apply f (list v)

  fun iterate f v = fromList (List.iterate f (list v))

  fun iterateApply f v = List.iterateApply f (list v)


(* CONVERTERS *)

  fun stringSep start finish sep p v =
        List.stringSep start finish sep p (list v)
	
  fun string p v = stringSep "" "" " " p v
	
  exception Sep of string * string * string * string

  (* The parse, parse' and read functions assume that entries
     are separated by formatting characters. *)

  fun parseSepN' start finish sep p n l =
	( case List.parseSepN' start finish sep p n l of
	    OK (l', s) => OK (fromList l', s)
	  | Fail (Some l', s) => Fail (Some (fromList l'), s)
	  | Fail (None, s) => Fail (None, s)
	)
	handle List.Sep x => raise Sep x

  fun parseSep' start finish sep p l =
        ( case List.parseSep' start finish sep p l of
            OK (l', s) => OK (fromList l', s)
          | Fail (Some l', s) => Fail (Some (fromList l'), s)
          | Fail (None, s) => Fail (None, s)
        )
        handle List.Sep x => raise Sep x

  fun parseN' p n l =
        if n < 0 then raise General.Nat ("parseN'", n)
        else parseSepN' "" "" "" p n l

  fun parse' p l = parseSep' "" "" "" p l

  fun parseSepN start finish sep p n s =
        if n < 0 then raise General.Nat ("parseSepN", n)
        else
          case parseSepN' start finish sep p n (explode s) of
            OK (v, _) => OK v
          | Fail (x, _) => Fail x

  fun parseSep start finish sep p s =
        case parseSep' start finish sep p (explode s) of
          OK (v, _) => OK v
        | Fail (x, _) => Fail x

  fun parseN p n s =
        if n < 0 then raise General.Nat ("parseN", n)
        else parseSepN "" "" "" p n s

  fun parse p s = parseSep "" "" "" p s

  fun readSepN start finish sep p n i =
        ( case List.readSepN start finish sep p n i of
            OK l  => OK (fromList l)
          | Fail (Some l) => Fail (Some (fromList l))
          | Fail None => Fail None
        )
        handle List.Sep x => raise Sep x

  fun readSep start finish sep p i =
        case List.readSep start finish sep p i of
          OK l  => OK (fromList l)
        | Fail (Some l) => Fail (Some (fromList l))
        | Fail None => Fail None
        handle List.Sep x => raise Sep x

  fun readN p n i =
        if n < 0 then raise General.Nat ("readN", n)
        else readSepN "" "" "" p n i

  fun read p i = readSep "" "" "" p i

  fun fromFile p name =
        let fun readList i =
                  case p i
                  of Fail _ => (InStream.closeIn i; [])
                  |  OK x => x :: readList i
        in fromList (readList (InStream.openIn name))
        end

  fun file p v name =
        let val os = OutStream.openOut name
	    fun out s = OutStream.output' os s
        in apply (out o p) v;
           OutStream.closeOut os
        end


(* OBSERVERS *)

  val size = Array.length

  fun empty v = (size v = 0)

  fun same v v' = (v = v')

  fun different v v' = v <> v'

  fun eq p v v' =
        List.eq p (list v) (list v')

  fun ne p v v' =
        List.ne p (list v) (list v')

  fun lt p v v' =
        List.lt p (list v) (list v')

  fun le p v v' =
        List.le p (list v) (list v')

  fun gt p v v' =
        List.gt p (list v) (list v')

  fun ge p v v' =
        List.ge p (list v) (list v')


(* MANIPULATORS *)

  fun rev v = fromList (List.rev (list v))

  infix 6 ^
  fun op ^ (v, v') = fromList (list v @ list v')

  exception Update of int
  fun update i x v =
	Array.update (v, i, x)
        handle Array.Subscript => raise Update i

  exception Copy of int * int * int
  local
    fun copy' start finish v start' v' =
	  if start = finish then ()
	  else (update start' (v sub start) v';
	        copy' (start + 1) finish v (start' + 1) v')
  in
    fun copy start finish v start' v' =
          if finish < start orelse start < 0 orelse finish > size v orelse
	     start' < 0 orelse start' + finish - start > size v'
	  then raise Copy (start, finish, start')
	  else copy' start finish v start' v'
  end
           
  exception UpdateRange of int * int
  local
    fun update' start finish i v =
	  if start = finish then ()
	  else (update start i v;
	        update' (start + 1) finish i v)
  in
    fun updateRange start finish i v =
	  if finish < start orelse start < 0 orelse finish > size v
	  then raise UpdateRange (start, finish)
	  else update' start finish i v
  end

  fun sort p v =
        let val v' = fromList (List.sort p (list v))
        in copy 0 (size v') v' 0 v
        end


(* REDUCERS *)

  exception Empty of string

  fun foldL f base v =
	List.foldL f base (list v)

  fun foldL' f v =
	if size v = 0 then raise Empty "foldL'"
	else List.foldL' f (list v)

  fun foldR f base v =
	List.foldR f base (list v)

  fun foldR' f v =
	if size v = 0 then raise Empty "foldR'"
	else List.foldR' f (list v)

  fun pairwise f v =
	List.pairwise f (list v)
end
