(*************************************************************************)
(*                                                                       *)
(*                     Projet      Formel                                *)
(*                                                                       *)
(*                    CAML: users' library                               *)
(*                                                                       *)
(*************************************************************************)
(*                                                                       *)
(*                            LIENS                                      *)
(*                        45 rue d'Ulm                                   *)
(*                         75005 PARIS                                   *)
(*                            France                                     *)
(*                                                                       *)
(*************************************************************************)

(* pictures.ml                                                           *)
(*              Emmanuel Chailloux & Guy Cousineau                       *)

module pictures
using
type point = {xc:float;yc:float};
type geom_element;
type frame = {xmin:float;xmax:float;ymin:float;ymax:float};
type extension = All_ext | Horiz_ext | Vertic_ext | Left_ext
               | Right_ext | Top_ext | Bottom_ext;
type transformation;
type path;
type sketch = {path:path;frame:frame;size:int};
type font;
type bitmap = {b_width:int;
               b_height:int;
               b_depth:int;
               b_bits:string vect};
type text;
type color;
type linecap;
type linejoin;
type linestyle={linewidth:float;linecap:linecap;
                linejoin:linejoin;dashpattern:int list};
type fillstyle;
type clipstyle;
type graphic_state;

value transform_point : transformation -> point -> point;
value handle_transform : point * point -> point * point -> transformation;
value frame_sketch : frame -> sketch;
value frame_center : frame -> point;
value compose_frames : frame -> frame -> frame;
value transform_frame : transformation -> frame -> frame;
value frame_to_frame_transform : frame -> frame -> transformation;
value extend_frame : extension -> float -> frame -> frame;
value bitmap_frame : bitmap -> frame;
value make_text    : string -> font -> text;
value make_default_text :  string -> text;
value text_frame : text -> frame;
value translation : float * float -> transformation;
value scaling: float * float -> transformation;
value rotation: point -> float -> transformation;
value vsymetry : float -> transformation;
value hsymetry : float -> transformation;
value CT: transformation * transformation -> transformation;
value default_color : unit -> color;
value default_linestyle : unit -> linestyle;
value default_fillstyle : unit -> fillstyle;;

#pragma infix "JPICT";;
#pragma infix "BPICT";;
#pragma infix "OPICT";;
#pragma infix "BPICT'";;
#pragma infix "OPICT'";;
#pragma infix "APICT";;
#pragma infix "CT";;

#arith int;;
#arith float;;

type interface = No_port | One_port of point * point
                    | Ports of (string * interface) list;;

type pict =  Draw of path * linestyle * color * int   (* the int here is the *)
          |  Fill of path * fillstyle * color * int   (* path length which is*)
          |  Clip of clipstyle * path * pict * int    (* subject to display  *)
          |  Bitmap of bitmap                         (* limitations         *)
          |  Text of text * color
          |  Tpict of transformation * pict
          |  Cpict of pict * pict;;

type picture = {pict : pict ; frame : frame;
               input_interface:interface; output_interface:interface};;


(* Operations to access picture components                *)

let picture_frame (pict:picture) = pict.frame;;

let picture_center (pict:picture) = frame_center (pict.frame);;

let picture_input_interface p = p.input_interface;;

let picture_output_interface p = p.output_interface;;


(* Operations that modify characteristics of a picture     *)
(* in a non destructive way                                *)

let change_color_picture c {pict=pict; frame=f;
                            input_interface=ip; output_interface=op}
= match pict with
    (Draw(p,lsty,_,n)) -> {pict=Draw(p,lsty,c,n); frame=f;
                            input_interface=ip; output_interface=op}
  | (Fill(p,fsty,_,n)) -> {pict=Fill(p,fsty,c,n); frame=f;
                            input_interface=ip; output_interface=op}
  |  (Text(t,_))        -> {pict=Text(t,c); frame=f;
                            input_interface=ip; output_interface=op}
  |  _  -> 
   failwith "color change operation apply only to draw or fill picture";;

let change_linestyle_picture lsty {pict=pict; frame=f;
                            input_interface=ip; output_interface=op}
= match pict with
    (Draw(p,_,c,n))
      -> {pict=Draw(p,lsty,c,n);
           frame=f;
           input_interface=ip; 
           output_interface=op}
    |  _  -> 
             failwith "linestyle change operation apply only to draw picture";;

let change_linewidth_picture lw {pict=pict; frame=f;
                            input_interface=ip; output_interface=op}
= match pict with
    (Draw(p,{linewidth=_;linecap=lc;linejoin=lj;dashpattern=dp},c,n))
     -> {pict=Draw(p,{linewidth=lw;linecap=lc;linejoin=lj;dashpattern=dp},c,n);
         frame=f;
         input_interface=ip; 
         output_interface=op}
    |  _  -> 
       failwith "linewidth change operation apply only to draw picture";;


let set_picture_interfaces {pict=p ; frame=f; _} (p1,p2) =
    {pict=p ; frame=f; input_interface=p1; output_interface=p2} ;;



(* Operations on interfaces            *)

let join_interfaces = function
       No_port,p2 -> p2
    |  p1,_       -> p1;;

let rec transform_interface t = function
       No_port ->  No_port
    |  One_port (pt1,pt2) -> One_port(transform_point t pt1,
                                      transform_point t pt2)
    |  Ports l -> Ports (map (function n,p ->
                              (n,transform_interface t p))
                           l);;


let rec find_handle p l = 
  match (p,l) with 
      No_port,_ -> failwith "Could not find the right port"
   |  One_port p,[] ->  p
   |  One_port p,_ -> failwith "Could not find the right port"
   |  Ports l,[] -> failwith "Could not find the right port"
   |  Ports l, a::ll -> find_handle (assoc a l) ll;;


let rec update l a v =
  match l with
    [] -> failwith "update"
  | (a1,v1)::l1 ->
      if a1=a then (a,v)::l1 else (a1,v1)::update l1 a v;;

let rec subst_interfaces p1 occ p2  =
  match (p1,occ) with
      No_port,_ -> failwith "Could not find the right port"
   |  One_port p,[] ->  p2
   |  One_port p,_ -> failwith "Could not find the right port"
   |  Ports l,[]  -> failwith "Could not find the right port"
   |  Ports l,a::ll ->
           let p = try assoc a l 
                   with failure _ -> failwith "Could not find the right port"
           in
             Ports(update l a (subst_interfaces p ll p2));;
  

(* To make pictures from sketches  *)

let make_draw_picture ((lsty:linestyle),col) 
                {path=p; frame = {xmin=a;xmax=b;ymin=c;ymax=d}; size=sz}  
=
  let dx = lsty.linewidth/2.0 in
      {pict= Draw(p,lsty,col,sz) ; 
       frame = {xmin=a-dx;xmax=b+dx;ymin=c-dx;ymax=d+dx};
       input_interface=No_port;
       output_interface=No_port};;

let make_default_draw_picture =
    make_draw_picture (default_linestyle(),default_color());;

let make_fill_picture (fsty,col) {path=p ; frame = f; size=sz} =
      {pict= Fill(p,fsty,col,sz) ; frame = f;
       input_interface=No_port;
       output_interface=No_port};;

let make_default_fill_picture  =
   make_fill_picture (default_fillstyle(),default_color());;

let clip_picture clipsty sk p =
  {pict=Clip(clipsty,sk.path,p.pict,sk.size); 
   frame = sk.frame;
   input_interface=p.input_interface; 
   output_interface=p.output_interface};;


(* To make pictures from bitmaps  *)

let make_bitmap_picture b =
 {pict= Bitmap b; 
  frame = bitmap_frame b;
  input_interface=No_port;
  output_interface=No_port}  ;;


(* To make pictures from texts  *)

let make_text_picture str font color = 
  let t = make_text  str font
  in
    {pict=Text (t,color); 
     frame= text_frame t;
     input_interface=No_port;
     output_interface=No_port}  ;;

let make_default_text_picture str = 
  let t = make_default_text str 
  in
    {pict=Text (t,default_color()); 
     frame= text_frame t;
     input_interface=No_port;
     output_interface=No_port}  ;;


(* To make pictures from frames  *)

let make_frame_picture (lsty,col) =
  (make_draw_picture (lsty,col)) o frame_sketch;;

let make_default_frame_picture  =
  make_default_draw_picture o frame_sketch;;


(* Operations on pictures   *)

let join_pictures  {pict=pict1 ; frame=f1; input_interface= ip1; 
                                           output_interface=op1}
                   {pict=pict2 ; frame=f2; input_interface= ip2; 
                                           output_interface=op2}
=
      {pict = Cpict(pict1,pict2) ; 
       frame= compose_frames f1 f2;
       input_interface = join_interfaces (ip1, ip2);
       output_interface = join_interfaces (op1, op2)};;

let p1 JPICT p2 = join_pictures p1 p2;;

let join_picture_list (p::pl) =
  it_list join_pictures p pl;;

let transform_picture t {pict= p ; frame =f; 
                         input_interface=inp; output_interface=outp} =
  let f' = transform_frame t f
  in
   match p with (Tpict (t',p')) -> {pict=(Tpict (t CT t',p')) ; 
                                 frame= f';
                                 input_interface=inp; output_interface=outp}

           | p -> {pict = Tpict(t,p); frame= f';
                   input_interface=inp; output_interface=outp};;


let scale_picture (hscale,vscale) (pict:picture) =
  let a= pict.frame.xmin and c=pict.frame.ymin
  in let T1 = translation (-a,-c)
     and S = scaling (hscale,vscale)
     and T2 = translation (a,c)
     in
       transform_picture (T2 CT S CT T1) pict;;

let fit_picture_in_frame (pict:picture) f =   
   let t = frame_to_frame_transform (pict.frame) f
   in  transform_picture t pict;;

let force_picture_in_frame f p =   
   {pict=p.pict;frame=f;input_interface=p.input_interface;
                        output_interface=p.output_interface};;

let add_frame_to_picture lc p =
  p JPICT (make_frame_picture lc (picture_frame p));;

let vflip_picture pict =
   let s = vsymetry (picture_center pict).xc
   in  transform_picture s pict;;

let hflip_picture pict =
   let s = hsymetry (picture_center pict).yc
   in  transform_picture s pict;;

let rotate_picture a pict =
     transform_picture (rotation (picture_center pict) a ) pict;;  
   
    
let besides_picture (pict1:picture) (pict2:picture) =    
    let {xmin=a; xmax=b; ymin=c; ymax=d} = pict1.frame  
    and  width2 = pict2.frame.xmax - pict2.frame.xmin 
in pict1 JPICT (fit_picture_in_frame  pict2 
                       {xmin=b;xmax=b+width2;ymin=c;ymax=d})  ;;

let over_picture (pict1:picture) (pict2:picture) =    
    let {xmin=a; xmax=b; ymin=c; ymax=d} = pict1.frame
    and  height2 = pict2.frame.ymax - pict2.frame.ymin 
in pict1 JPICT (fit_picture_in_frame  pict2
                      {xmin=a;xmax=b;ymin=c-height2;ymax=c})  ;;


let pict1 BPICT pict2 = besides_picture pict1 pict2;;
let pict1 OPICT pict2 = over_picture pict1 pict2;;

let extend_picture_frame str k {pict=p;frame=fr;
                                input_interface=ip;output_interface=op} =
    {pict=p;frame=extend_frame str k fr;
     input_interface=ip;output_interface=op};;

let (pict1:picture) BPICT' (pict2:picture) =
     let h1 = pict1.frame.ymax - pict1.frame.ymin
     and h2 = pict2.frame.ymax - pict2.frame.ymin 

in pict1 BPICT (extend_picture_frame  Vertic_ext ((h1-h2)/(2.0*h2)) pict2 );;

let (pict1:picture) OPICT' (pict2:picture) =
     let w1 = pict1.frame.xmax - pict1.frame.xmin
     and w2 = pict2.frame.xmax - pict2.frame.xmin 

in pict1 OPICT (extend_picture_frame  Horiz_ext ((w1-w2)/(2.0*w2)) pict2 );;
          


(* Operations on pictures using interfaces *)

let named_attach_pictures (p1,p2) (a1,a2) =
  let handle1 = find_handle p1.output_interface a1
  and handle2 = find_handle  p2.input_interface a2
  in
    let T = handle_transform handle1 handle2
    in
      {pict=Cpict(p1.pict,Tpict(T,p2.pict));
       frame=compose_frames p1.frame (transform_frame T p2.frame);
       input_interface= p1.input_interface;
       output_interface= subst_interfaces 
                             p1.output_interface 
                             a1 
                             (transform_interface T p2.output_interface)}
;;

let attach_pictures (p1,p2) =
   match (p1.output_interface,p2.input_interface)
   with  (One_port handle1,One_port handle2)
         ->     let T = handle_transform handle1 handle2
        in
      {pict=Cpict(p1.pict,Tpict(T,p2.pict));
       frame=compose_frames p1.frame (transform_frame T p2.frame);
       input_interface= p1.input_interface;
       output_interface= transform_interface T p2.output_interface}
    |   (No_port,_) | (_,No_port) -> failwith "Port missing"
    |   _ -> failwith "Use \"named_attach_pictures\" for pictures with named ports"
;;

let p1 APICT p2 = attach_pictures (p1,p2);;

end module
with
type picture
 and pict
 and interface;
value picture_input_interface 
  and picture_output_interface
  and set_picture_interfaces
  and change_color_picture
  and change_linestyle_picture
  and change_linewidth_picture
  and make_draw_picture
  and make_default_draw_picture
  and make_fill_picture
  and make_default_fill_picture
  and make_frame_picture
  and make_default_frame_picture
  and make_bitmap_picture
  and make_text_picture
  and make_default_text_picture
  and picture_center
  and picture_frame
  and extend_picture_frame
  and clip_picture
  and join_pictures
  and prefix JPICT
  and join_picture_list
  and transform_picture
  and fit_picture_in_frame
  and force_picture_in_frame
  and add_frame_to_picture
  and rotate_picture
  and scale_picture
  and vflip_picture
  and hflip_picture
  and besides_picture
  and over_picture
  and prefix BPICT
  and prefix OPICT
  and prefix BPICT'
  and prefix OPICT'
  and named_attach_pictures
  and attach_pictures
  and prefix APICT;;

