(*************************************************************************)
(*                                                                       *)
(*                     Projet      Formel                                *)
(*                                                                       *)
(*                    CAML: users' library                               *)
(*                                                                       *)
(*************************************************************************)
(*                                                                       *)
(*                            LIENS                                      *)
(*                        45 rue d'Ulm                                   *)
(*                         75005 PARIS                                   *)
(*                            France                                     *)
(*                                                                       *)
(*************************************************************************)

(* geometry.ml     basic geometric objects                              *)
(*              Emmanuel Chailloux & Guy Cousineau                       *)

module geometry;;

#pragma infix "CT";;

#arith float;;

type point = {xc:float;yc:float};;

type geom_element =
  Seg of point list
| Arc of point * float * float * float   
        (* center,radius,start_angle,end_angle*)
| Curve of point * point * point * point;;
        (* start,control1,control2,end *)
type transformation = {m11:float;m12:float;m13:float;
                       m21:float;m22:float;m23:float};;


let point(x,y) = {xc=x ; yc =y};;
let O = point(0.0,0.0);;

let make_transformation (a,b,c,d,e,f) =
           {m11=a; m12=b; m13=c; m21=d; m22=e; m23=f};;

let id_trans =  {m11=1.0;m12=0.0;m13=0.0;m21=0.0;m22=1.0;m23=0.0};;

let transform_point {m11=a;m12=b;m13=c;m21=d;m22=e;m23=f} {xc=x;yc=y} =
      {xc = a*x+b*y+c ; yc = d*x+e*y+f};;

let compose_transformation {m11=a11;m12=a12;m13=a13;m21=a21;m22=a22;m23=a23} 
                           {m11=b11;m12=b12;m13=b13;m21=b21;m22=b22;m23=b23} 
=

{m11=a11*b11+a12*b21 ; m12=a11*b12 + a12*b22 ; m13=a11*b13+a12*b23+a13;
 m21=a21*b11+a22*b21 ; m22=a21*b12+a22*b22 ; m23 = a21*b13+a22*b23+a23};;

let t1 CT t2 = compose_transformation t1 t2;;

let inverse_transformation {m11=a;m12=b;m13=c;m21=d;m22=e;m23=f} =
  let det = a*e - d*b 
  in
     if det =0.0 then failwith "determinant nul"
               else  {m11=e/det ; m12=-b/det ; m13=(f*b-c*e)/det ;
                      m21 = -d/det ; m22 = a/det ; m23=(d*c-a*f)/det};;

let INV = inverse_transformation;;

let handle_transform ({xc=x1;yc=y1},{xc=x2;yc=y2}) 
                     ({xc=x3;yc=y3},{xc=x4;yc=y4})
=

 let a'=x2-x1
 and b'=y2-y1
 and a''=x4-x3
 and b''=y4-y3
in
 let det = a''*a'' + b''*b'' 
in

 if det = 0.0 
   then failwith "Wrong handle"
   else
     let c = (a'*a''+b'*b'')/det
     and d = (b'*a''-a'*b'')/det 
      in
        {m11=c;m12=-d;m13=-c*x3+d*y3+x1;
         m21=d;m22=c;m23=-d*x3-c*y3+y1};;


(* let newCTM = matrix (c,d,-d,c,-c*x3+d*y3+x1,-d*x3-c*y3+y1) *)


let Pi = 2.0*(acos 0.0);;
let cva a = Pi*a/180.0;;
let sinus = sin o cva;;
let cosinus = cos o cva;;


let translation (a,b) = {m11=1.0 ; m12=0.0 ; m13=a ; m21=0.0 ; m22=1.0 ; m23 = b};;

let Orotation alpha = 
  {m11=cosinus alpha ; m12= - (sinus alpha); m13=0.0 ; 
   m21=sinus alpha ; m22=cosinus alpha ; m23 = 0.0};;

let rotation {xc=x;yc=y} alpha =
  (translation(x,y)) CT (Orotation alpha) CT (translation(-x,-y));;

let scaling (a,b) = {m11=a ; m12=0.0 ; m13=0.0 ; m21=0.0 ; m22=b ; m23 = 0.0};;

let symetry (a,b) =        (* symetrie par rapport a la droite y-ax+b  *)
      if a=0.0 then failwith "a=0: use hsymetry"
             else let A = a*a+1.0 in
   {m11=-((a*a-1.0)/A); m12=2.0*a/A; m13=-(2.0*a*b/A);
    m21=2.0/(a*A); m22=(a*a-1.0)/A;  m23=-(2.0*b/A)};;  

let vsymetry a =      (* symetrie par rapport a la droite x=a   *)
   {m11=-1.0; m12=0.0; m13=2.0*a; m21=0.0; m22=1.0; m23=0.0};;

let hsymetry a =      (* symetrie par rapport a la droite y=a   *)  
   {m11=1.0; m12=0.0; m13=0.0 ; m21=0.0; m22=-1.0; m23=2.0*a};;
  
end module with

type point and geom_element and transformation;
value point
and   id_trans
and   transform_point
and   make_transformation
and   compose_transformation
and   inverse_transformation
and   prefix CT
and   INV
and   handle_transform
and   sinus
and   cosinus
and   translation
and   Orotation
and   rotation
and   scaling
and   symetry
and   hsymetry
and   vsymetry;;
