





(*************************************************************************)
(*                                                                       *)
(*                     Projet      Formel                                *)
(*                                                                       *)
(*                    CAML: users' library                               *)
(*                                                                       *)
(*************************************************************************)
(*                                                                       *)
(*                            Inria                                      *)
(*                      Domaine de Voluceau                              *)
(*                      78150  Rocquencourt                              *)
(*                            France                                     *)
(*                                                                       *)
(*************************************************************************)

(* cps_str.ml      Structured graphics data types visualised with CPS    *)
(*                 Emmanuel Chailloux  (INRIA-ENS)                       *)

#fast arith false;;
#directive infix "MX";;
#directive infix "AP";;
#directive infix "AFIG";;
#directive infix "ATILE";;

module strCPS

(*             Import of strCPS module  from CPS.ml                      *)
(*             _______________________                                   *)

using
type ps_int     = PS_INT of int;
type ps_num     = PS_NUM of num;
type ps_bool    = PS_BOOL of bool;
type ps_array   = PS_ARRAY of num list;
type ps_string  = PS_STRING of string;
type ps_matrix  = PS_MATRIX of num list;
type ps_font    = PS_FONT of string;
type ps_vm      = PS_VM of string;
type ps_image   = PS_IMAGE of string;
type ps_channel =   PS_CHANNEL of in_channel&out_channel&out_channel&string
                  | PS_FILE    of out_channel&string;


value moveto_PS       : (num * num) -> unit;
value rmoveto_PS      : (num * num) -> unit;
value lineto_PS       : (num * num) -> unit;
value rlineto_PS      : (num * num) -> unit;
value arc_PS          : (num * num) -> num -> num -> num -> unit;
value arcn_PS         : (num * num) -> num -> num -> num -> unit;
value curveto_PS      : (num * num) -> (num * num) -> (num * num) -> unit;
value rcurveto_PS     : (num * num) -> (num * num) -> (num * num) -> unit;
value charpath_PS     : string -> bool -> unit;

value closepath_PS    : unit -> unit;
value setgray_PS      : num -> unit;
value setrgbcolor_PS  : num list -> unit;
value sethsbcolor_PS  : num list -> unit;
value newpath_PS      : unit -> unit;
value setlinewidth_PS : num -> unit;
value setlinecap_PS   : num -> unit;
value setlinejoin_PS  : num -> unit;
value setdash_PS      : num list -> num -> unit;

value stroke_PS       : unit -> unit;
value fill_PS         : unit -> unit;
value eofill_PS       : unit -> unit;
value clip_PS         : unit -> unit;
value eoclip_PS       : unit -> unit;
value image_PS        : num -> num -> num ->
                           ps_matrix -> ps_image -> num -> unit;
value show_PS         : string -> unit;
value F_PS            : string -> num -> unit;
value gsave_PS        : unit -> unit;
value grestore_PS     : unit -> unit;
value concat_PS       : ps_matrix -> unit

from "cps"
;;



(*             Body of strCPS module                                     *)
(*             _____________________                                     *)


(*       graphics data types                                            *)
(*       -------------------                                            *)



type point = couple of num&num;;

type elt_of_path = moveto of point
                 | rmoveto of point
                 | lineto of point
                 | rlineto of point
                 | arc of point&num&num&num
                 | arcn of point&num&num&num
                 | curveto of point&point&point
                 | rcurveto of point&point&point
                 | charpath of string&bool
;;

type CTM = matrix of num&num&num&num&num&num;;

type open_path == elt_of_path list;;

type path = opat  of open_path
          | cpat  of open_path
          | tpat  of CTM&path
          | lpat  of path list;;

type color = rgb of num&num&num
           | hsb of num&num&num
           | gra of num;;

type linecap = currentcap | buttcap | squarecap | roundcap;;

type linejoin = currentjoin | beveljoin | roundjoin | miterjoin;;

type linestyle == num&linecap&linejoin&num list;;

type fillstyle  = nzfill | eofill;;

type clipstyle  = nzclip | eoclip;;


type form == num&num&string;;


type font = font of string ;;

type stringstyle == font&num;;

type text == string&stringstyle;;

type figure = stroke  of  path&linestyle&color
             | fill   of  path&fillstyle&color
             | clip   of  path&clipstyle&figure
             | bitmap of form&color
             | shows  of text&color
             | transg of CTM&figure
             | appends of figure list
;;





type tile    == point&point&point&point&figure;;

(*      convertion functions with CPS                              *)
(*      -----------------------------                              *)


let matrix_PS_of_matrix M =
  match M with matrix (a,b,c,d,e,f) -> PS_MATRIX [a;b;c;d;e;f]
;;

let array_PS_of_num_list NL = PS_ARRAY NL;;


(*       general functions                                          *)
(*       -----------------                                          *)

let rec mkinterval a b = if a = b then [b] else a::(mkinterval (a+1) b);;

let range f a b = map f (mkinterval a b);;

let rec rinsert (f,s,u) = match s with  [] ->  u
                           |  h::t         -> f (h,(rinsert (f,t,u)));;

let rec mapapp l s = match l with [] -> []
                       |      h::t   -> (h s)@(mapapp t s);;



(*       current transformation matrix (CTM)                        *)
(*       -----------------------------------                        *)


let mk_CTM (a,b,c,d,e,f) = matrix (a,b,c,d,e,f);;

let ICTM = mk_CTM (1,0,0,1,0,0);;

let prod_CTM A B =
  match A with
       matrix (1,0,0,1,0,0) -> B
    |  matrix (a,b,c,d,e,f) ->
         (match B with
               matrix (1,0,0,1,0,0) -> A
            |  matrix (a',b',c',d',e',f') ->
                    matrix (a*a'+c*b' , b*a' + d*b',
                            a*c'+c*d' , b*c'+d*d',
                            a*e'+c*f'+e , b*e'+d*f'+f)) ;;

let x MX y = prod_CTM x y ;;

let inv_CTM A =
 match A with
       matrix (1,0,0,1,0,0) -> ICTM
   |   matrix (a,b,c,d,e,f) -> let det = a*d-b*c in
         if det=0 then failwith "determinant nul"
                  else matrix (d/det,-b/det,-c/det,a/det,
                               (f*c-e*d)/det,(b*e-a*f)/det);;

let translation (e,f) = mk_CTM (1,0,0,1,e,f);;

let scaling  (a,d) = mk_CTM (a,0,0,d,0,0);;

let PI = 2 * (acos 0) in
let rad2deg  n = 180*n/PI
and deg2rad  n = PI*n/180
;;

let rotation  teta  =
let teta' = deg2rad teta in
 mk_CTM (cos teta', sin teta', -(sin teta'), cos teta',0,0);;

let transformation (s1,s2) teta (t1,t2) =
prod_CTM (  mk_CTM (s1,0,0,s2,t1,t2)) ( rotation teta);;

let translate_path (e,f) p = tpat (translation (e,f),p);;

let scale_path (a,d) p = tpat (scaling (a,d) , p);;

let rotate_path n p = tpat (rotation n,p);;

let prod_path n p = tpat (n,p);;


(*      point                                                     *)



let prod_CTM_point A p = match A with matrix (1,0,0,1,0,0) -> p
                                |   matrix (a,b,c,d,e,f) ->
                match p with couple (x,y) -> couple (a*x +c*y+e,b*x+d*y+f);;

let coord_point p = match p with couple (x,y) -> (x,y);;





(*      path                                                      *)
(*      ----                                                      *)


let transform_point A p = match p with couple (x,y) -> prod_CTM_point A (couple(
x,y));;

let transform_elt A e = match e with moveto p -> moveto (transform_point A p)
                         |   rmoveto p -> rmoveto (transform_point A p)
                         |   lineto p -> lineto (transform_point A p)
                         |   rlineto p -> rlineto (transform_point A p)
                         |   arc (p,r,a1,a2) -> arc (transform_point A p,r, a1,a
2)
                         |   arcn (p,r,a1,a2) -> arcn (transform_point A p,r, a1
,a2)
                         |   curveto (p1,p2,p3) ->
                                         curveto (transform_point A p1,transform
_point A p2,transform_point A p3)
                         |   rcurveto (p1,p2,p3) ->
                                         rcurveto (transform_point A p1,transfor
m_point A p2,transform_point A p3)
                         |   l -> l
;;

let transform_open_path A l = map (transform_elt A) l;;

let open_path P = match P with cpat P1 -> opat P1
                           |      _    -> P;;


let simplify_path P = simplify_path' ICTM P
where rec simplify_path' I P =
  match P with tpat (X,P1) -> simplify_path' (prod_CTM I X) P1
            | lpat l       -> (match l with
  [] -> lpat []
   | [P1] ->  simplify_path' I P1
   | P1::P2 -> lpat (map (simplify_path' I) l))
            |       _      -> if (I=ICTM) then P else tpat (I,P)
;;

let rec append_path P1 P2 = match (P1,P2) with
 (opat P1', opat P2')             -> opat (P1'@P2')
|( tpat (M1',P1'),tpat (M2',P2')) ->
   if (M1'=M2') then tpat (M1', append_path P1' P2')
                else lpat [ P1; P2]
|           _                     -> lpat [ P1;P2];;

let flat_path P =  opat (flat_path' ICTM P)
where rec flat_path' I P = match P with opat P1 -> transform_open_path I P1
                         |   cpat P1 -> transform_open_path I P1
                         |   tpat (X,P1) -> flat_path' (prod_CTM I X) P1
                         |   lpat l -> (match l with
[] -> []
| [P1] -> flat_path' I P1
| P1::P2 -> (flat_path' I P1)@( flat_path' I (lpat P2))
);;

let transform_path M P = simplify_path (tpat (M, P));;


(*
let TP (M,P) = tpat (M,P);;
#directive infix "TP";;
*)

let x AP y = append_path x y;;


(*      figure                                                   *)
(*      -------                                                   *)



let append_figure P1 P2 = appends [P1;P2];;

let x AFIG y = append_figure x y;;

let flat_figure P = let rec flat_figure' I P' = match P' with
                           fill (a,b,c) -> fill (flat_path a,b, c)
                       |   stroke (a,b,c) -> stroke (flat_path a,b,c)
                       |   clip (a,b,c) -> clip (flat_path a,b,c)
                       |   transg (X,p)   -> flat_figure' (prod_CTM I X) p
                       |   appends l       -> appends (map (flat_figure' I) l)
                       |        f          -> f
in flat_figure' ICTM P;;


let rec transform_figure f  FIG = match FIG with stroke (p,b,c) -> stroke (f p,b
,c)
                             |  fill (p,b,c) -> fill (f p,b,c)
                             |  clip (p,b,c) -> clip (f p,b,c)
                             |  transg (m,p) -> transform_figure(f o (prod_path
m)) p
                             |  appends l    -> appends (map (transform_figure f
) l)
                             |   l           -> l;;

let rec simplify_figure l =

match l with
  transg (M,f) ->
    (match f with  transg (M',f') -> simplify_figure (transg (M' MX M,f'))
               | appends m        -> appends (
                                map (simplify_figure o (curry transg M )) m)
               |   _ -> l
    )
 | appends m -> appends (map simplify_figure m)
 |     _  -> l
;;


let rotate_figure n p = transg (rotation n,p);;

let translate_figure c p = transg  (translation c,p) ;;

let scale_figure s p = transg ( scaling s,p);;





(*      str                                                       *)
(*      ---                                                       *)


(*      tile                                                      *)
(*      ----                                                      *)



let transform_tile M T = match T with (a,b,c,d,p) ->

(transform_point M a,transform_point M b,transform_point M c,transform_point M d
, transg (M,p));;

let append_tile (T1:tile) (T2:tile) =
match T1 with (q1,q2,p1,p2,P1) ->
match T2 with (p3,p4,q3,q4,P2) ->
let ((x1,y1),(x2,y2),(x3,y3),(x4,y4))=
        (coord_point p1,coord_point p2,coord_point p3,coord_point p4) in
let a'=x2-x1
and b'=y2-y1
and a''=x4-x3
and b''=y4-y3
in
let det = a''*a'' + b''*b'' in

if det = 0 then failwith "Not a segment"
           else
let c = (a'*a''+b'*b'')/det
and d = (b'*a''-a'*b'')/det in


let newCTM = matrix (c,d,-d,c,-c*x3+d*y3+x1,-d*x3-c*y3+y1) in

(q1,q2,transform_point newCTM q3,transform_point newCTM q4,
 P1 AFIG (transg (newCTM,P2)))
;;

let x ATILE y = append_tile x y;;


let perm_tile (a,b,c,d) T = match T with (p1,p2,p3,p4,P) ->
let l = [p1;p2;p3;p4] in
  (nth l  a,nth l b,nth l c ,nth l d ,P);;






(*  Color & Picture                                                    *)





let mkrgbcolor (a,b,c) = rgb (a,b,c);;

let mkhsbcolor (a,b,c) = hsb (a,b,c);;

let mkgray a = gra a;;

let  mklinestyle (width:num) (lc:linecap) (lj:linejoin) (dash:num list) =
    (width,lc,lj,dash);;

let standardlinestyle = mklinestyle 1 currentcap currentjoin [];;
let standardfillstyle = nzfill;;
let standardclipstyle = nzclip;;

let standardgray = gra 0;;
let standardcolor = mkrgbcolor (1,1,1);;




let strokefigure p l c = stroke (p,l,c);;
let fillfigure p s c  = fill (p,s,c);;
let clipfigure path c fig  = clip (path,c,fig);;

let standardstroke p = stroke (p, standardlinestyle,standardgray);;
let standardfill p = fill (p, standardfillstyle,standardgray);;

let overpaint fig1 fig2 = appends  [fig1;fig2];;

let figure_nil =  stroke (opat[],standardlinestyle,mkrgbcolor (0,0,0));;

(* Display                                                              *)

let display_elt P = match P with
moveto (couple (x,y)) -> moveto_PS (x,y)
|rmoveto (couple (x,y)) -> rmoveto_PS (x,y)
|lineto (couple (x,y)) -> lineto_PS (x,y)
|rlineto (couple (x,y)) -> rlineto_PS (x,y)
|arc (couple(x,y),r,a1,a2) -> arc_PS (x,y) r a1 a2
|arcn (couple(x,y),r,a1,a2) -> arcn_PS (x,y) r a1 a2
|curveto (couple (x1,y1),couple (x2,y2),couple (x3,y3)) ->
             curveto_PS (x1,y1) (x2,y2) (x3,y3)
|rcurveto (couple (x1,y1),couple (x2,y2),couple (x3,y3)) ->
             rcurveto_PS (x1,y1) (x2,y2) (x3,y3)
|charpath (s,b) -> charpath_PS s b
;;
let display_point P = display_elt (moveto P);;

let rec display_path P =
match P with
  opat P'     -> map display_elt P';()
| cpat P'     -> map display_elt P';closepath_PS()
| tpat (M,P') -> concat_PS (matrix_PS_of_matrix M);display_path P';
                 concat_PS (matrix_PS_of_matrix (inv_CTM M))
| lpat l      -> map display_path l;()
;;

let rec display_figure P =
(match P with
  fill (p,b,c) ->
    ((match c with gra u -> setgray_PS u
               |  rgb (x,y,z) -> setrgbcolor_PS [x;y;z]
               |  hsb (x,y,z) -> sethsbcolor_PS [x;y;z]);
     newpath_PS ();
     display_path p;
     (match b with nzfill -> fill_PS()
                |  eofill -> eofill_PS());
     closepath_PS())
| stroke (p,b,c) ->
    ((match c with gra u -> setgray_PS u
               |  rgb (x,y,z) -> setrgbcolor_PS [x;y;z]
               |  hsb (x,y,z) -> sethsbcolor_PS [x;y;z]);
     (match b with (w,e,j,d) ->
          (setlinewidth_PS w;
           (match e with currentcap-> ()
                      |  buttcap   -> setlinecap_PS 0
                      |  squarecap -> setlinecap_PS 2
                      |  roundcap  -> setlinecap_PS 1);
           (match j with currentjoin-> ()
                      |  beveljoin  -> setlinejoin_PS 2
                      |  roundjoin  -> setlinejoin_PS 1
                      |  miterjoin  -> setlinejoin_PS 0);
           if d<>[] then setdash_PS d 0;
           newpath_PS();
           display_path p;
           stroke_PS()));
           closepath_PS())
| clip (p,c,fig) -> gsave_PS();
                    newpath_PS();
                    display_path (flat_path p);
                   (match c with nzclip -> clip_PS()
                              |  eoclip -> eoclip_PS());
                    closepath_PS ();
                    display_figure fig;
                    grestore_PS()
|bitmap ((w,h,s),c) ->    ((match c with gra u -> setgray_PS u
               |  rgb (x,y,z) -> setrgbcolor_PS [x;y;z]
               |  hsb (x,y,z) -> sethsbcolor_PS [x;y;z]);
                        moveto_PS (0,0);
                        image_PS w h 1 (PS_MATRIX  [1;0;0;-1;0;1])
                                (PS_IMAGE s)
                                (num_of_int
--More--(91%)
                                 (quo_int
                                   (add_int ((int_of_num w),#7),#8))) )
|shows ((s,(font f,n)),c)  ->   ((match c with gra u -> setgray_PS u
               |  rgb (x,y,z) -> setrgbcolor_PS [x;y;z]
               |  hsb (x,y,z) -> sethsbcolor_PS [x;y;z]);
                        moveto_PS(0,0);
                        F_PS   f n;
                        show_PS s)
| transg (M,fig)  -> gsave_PS();
                  concat_PS (matrix_PS_of_matrix M); display_figure fig;
                 grestore_PS()
| appends l       -> map display_figure l;());
()
;;




let display_tile (P:tile) = match P with (a,b,c,d,t) -> display_figure t;;

(*                  Output interface of strCPS module                      *)
(*                  _________________________________                      *)

end module
with type  point and elt_of_path and open_path and CTM and path
           and color
           and  linecap and linejoin and linestyle and fillstyle and clipstyle
           and font and stringstyle and text
           and  figure and tile ;
     value mk_CTM and prod_CTM and prefix MX and inv_CTM
           and translation and scaling
           and rotation and transformation
           and flat_path and translate_path and scale_path and rotate_path
           and prod_path and transform_path and append_path and prefix AP
           and flat_figure and transform_figure and scale_figure
           and rotate_figure and translate_figure
           and append_figure and prefix AFIG
           and transform_tile and append_tile and prefix ATILE
           and display_path and display_figure and display_tile
;;

