(*************************************************************************)
(*                                                                       *)
(*                     Projet      Formel                                *)
(*                                                                       *)
(*                    CAML: users' library                               *)
(*                                                                       *)
(*************************************************************************)
(*                                                                       *)
(*                            Inria                                      *)
(*                      Domaine de Voluceau                              *)
(*                      78150  Rocquencourt                              *)
(*                            France                                     *)
(*                                                                       *)
(*************************************************************************)

(* cps.ml       A interface between Caml and PostScript                  *)
(*              Emmanuel Chailloux                                       *)

(*              Last modification  :  21 March 1990                      *)

module cps;;

(*                           Body of module CPS                          *)
(*                           __________________                          *)

(*                            New Types                                  *)

type ps_int	= PS_INT of int;;
type ps_float    	= PS_FLOAT of float;;
type ps_bool	= PS_BOOL of bool;;
type ps_array	= PS_ARRAY of float list;;
type ps_string 	= PS_STRING of string;;
type ps_matrix 	= PS_MATRIX of float list;;
type ps_font 	= PS_FONT of string;;
type ps_vm	= PS_VM of string;;
type ps_image	= PS_IMAGE of string;;
type ps_channel =   PS_CHANNEL of in_channel&out_channel&out_channel&string
		  | PS_FILE    of out_channel&string;;
type ps_dialog  =   PS_OK   of string
		  | PS_ANSWER  of string
		  | PS_ERROR  of string
 		  | PS_INFO of string;;


(*                      Strings convertions                              *)

(* prend 2 fonctions et 2 chaines 
         calcule f ps1 s1 s2 sur tous les elements de s1
         et ramene g s1 s2
*)

let map_string f g s1 s2 = 
  map_string_int 0( length_string s1) 
  where rec map_string_int p  l = 
    if p=l then (g s1 s2) else (f p s1 s2 ;map_string_int (p+1) l)
;;


let n_replace_string s2 s' p =
let k = ref 0 
and l = length_string s' 
in
 while (!k<l) do (set_nth_ascii(p+!k,s2,(nth_ascii(!k,s')));incr k) done;
 s2
;;

let substitute_lchar lchar lchar' =
  subst_rec where rec subst_rec  s1 = 
    let s2 = make_string (2*(length_string s1)) ` `
    and p2 = ref 0 
    in
      let f p s1 s2 = 
        let c = nth_ascii (p,s1) 
        in 
          let ch = ascii c 
          in
          if mem ch lchar 
          then 
            let s' = nth lchar' (index ch lchar)
            in 
              (replace_string  s2 s' !p2;
               p2:=!p2+(length_string s');
               s2)

          else (set_nth_ascii (!p2, s2,c);incr p2;s2)
      and g s1 s2 = s2
      in
      map_string f g s1 s2
;;

let rec old_substitute_lchar lchar lchar' =
  subst_rec where rec subst_rec s     =
    match lscan_string  s lchar with
	  (_,"") -> s
	| (n,c)  ->(extract_string s 0 (pred n))^
                     (nth lchar' (index c lchar))^ 
		    (subst_rec (extract_string s (n+(length_string c))
					       (length_string s)) );;

let substitute_brackets s = substitute_lchar ["[";"]"] ["";""] s;;

let substitute_space s    = substitute_char " " "" s 0;;

let substitute_brackets_and_space s = 
	substitute_lchar  ["[";"]";" "] ["";"";""] s;; 

let filter_RCLF s =
	substitute_lchar ["\R";"\L"] ["";""] s;;


(*                       Convertion to stringP	                          *)

let stringPS_of_int (i:int) = string_of_int  i;;

let stringPS_of_float n = 
   let s = string_of_float n in
	let p = index_string s "/" 0 in
	if (p=-1) then s
	   	  else (set_nth_ascii (p,s,32);(s^" div"));;

let stringPS_of_string s =  
	let gs =  substitute_lchar  ["("  ;")"  ;"\T" ;"\R" ;"\L" ;"\S"] 
				    ["\\(";"\\)";"\\t";"\\r";"\\n";" "] s in
   "("^gs^")";;

let stringPS_of_bool t = string_of_bool t;;

let rec length_stringlist = 
  function [] -> 0 
       | h::l -> (length_string h)+ (length_stringlist l)
;;


let rec stringPS_of_float_list = function
    [] -> ""
  | h::t -> ((stringPS_of_float h)^" "^(stringPS_of_float_list t));;

let stringPS_of_array l = ("[ "^(stringPS_of_float_list l)^" ]");;

let stringPS_of_matrix  = fun (PS_MATRIX l) -> (
	if (length l) = 6 then  (stringPS_of_array l)
			  else failwith " string_of_matrixPS" );;

let stringPS_of_point = fun  (x,y) -> 
	((stringPS_of_float x)^" "^(stringPS_of_float y));;

let stringPS_of_angle a = (stringPS_of_float a)^" ca ";;

let stringPS_of_font = fun (PS_FONT s) -> s;;

let stringPS_of_vm = fun (PS_VM v) -> v;;

let stringPS_of_image = fun (PS_IMAGE im) -> im;;




(*                            Convertion from stringPS                    *)


let int_of_stringPS s = int_of_string (substitute_space s);;

let float_of_stringPS s =  float_of_string (substitute_space s);;

let string_of_stringPS s =  
   old_substitute_lchar	["\\(";"\\)";"\\t";"\\r";"\\n";" " ;"\R";"\L"]
	     		["("  ;")"  ;"\T" ;"\R" ;"\L" ;"\S";""  ;""] s ;;

let bool_of_stringPS s = bool_of_string s;;

let array_of_stringPS s = map float_of_string (words s);;

let matrix_of_stringPS  s = PS_MATRIX (map float_of_string (words s));;

let point_of_stringPS  s =
    let l = map float_of_string (words s) in (nth l 1,nth l 2);;

let points_of_stringPS s = 
    points (array_of_stringPS s) where 
       points = function
              h1::h2::h3::h4::[] -> ( (h1,h2), (h3,h4))
             | _        -> failwith "Bad conversion in points_of_stringPS";;

let angle_of_stringPS s = float_of_string s;;

let font_of_stringPS  s = PS_FONT (substitute_char "/" " " s 0);;

let vm_of_stringPS  s = PS_VM s;;

let image_of_stringPS  s = PS_IMAGE s;;

let string_of_dialogPS = function
		  (PS_OK s) -> s
		| (PS_ANSWER r) -> r
		| (PS_ERROR e) -> e
		| (PS_INFO i) -> i;;



(*              communication functions  PS <--> CAML                   *)
(*              ---------------------------------------	                *)

(* with                                                                 *) 

(*      default_channel_PS : the PostScript channel                     *)
(*      ps_loaded : to know if the communication is good		*)
(*      debug_mode_PS : a flag to know the current state            	*)
(*      (no_)debug_string : two string constants                    	*)
(*      end_debug_nohandshake_mode_PS : default mode without handshake  *)
(*      begin_debug_mode_PS   : to debug slowly                     	*)
(*      error_PS : this function traps PostScript errors                *)
(*      copy_stream : to copy a file to a open channel                  *)
(*      send_file_PS : to copy a file to the PostScript channel         *)
(*      send_and_flush_command_PS : to send a string as a PS command    *)    
(*      exec_PS  : to execute a PostScript command                      *)
(*      exec_and_send_file_PS : to execute a command and to send a file *)
(*      ask_PS   : to execute a PostScript command and wait for an answer *)
(*	ask_and_exec_PS : ask and execute realy the command		*)


let CPS_directory = lib_directory;;
let CPS_inter_prelude     = (lib_directory^"cps_inter.ps");;
let CPS_file_prelude      = (lib_directory^"cps_file.ps");;
let default_channel_PS = ref (PS_FILE (std_out,""));;
let ps_loaded          = ref false;;
let debug_mode_PS  = ref true;;

let no_debug_string =  ("{/"^"Ok "^" {Ok2} def Ok1} t");;
let debug_string    =  ("{/"^"Ok "^" {Ok1} def} t");;

let rec read_linef i = filter_RCLF (read_line i);;

let read_line_cin i = 
	if !debug_mode_PS then 
	( match (read_linef i) with 
		  "Ok"     -> (PS_OK  "Ok")
		|"Answer"  -> (PS_ANSWER "Answer") 
		|"Error"   -> (PS_ERROR "Error") 
		| s	   -> (PS_INFO s))
                              else (PS_OK "Ok")
;;

let immediate_output_line x y = output_line x y ; flush x;;

let end_debug_mode_PS () =
  match !default_channel_PS  with
    PS_CHANNEL (cin,cout,ps,name) -> 
      (immediate_output_line cout no_debug_string;
 	match (read_line_cin cin) with
	   (PS_OK s) -> debug_mode_PS := false
	|	 _   -> failwith ("Bad handshake in"^
                                         " end_debug_mode_PS"))
   | _ -> failwith ("no PostScript channel")
;;

let begin_debug_mode_PS () =
  match !default_channel_PS  with
    PS_CHANNEL (cin,cout,ps,name) -> 
      (immediate_output_line cout debug_string;
        (match (read_line_cin cin) with
	    (PS_OK s)->( debug_mode_PS := true ;
			 match (read_line_cin cin) with (PS_OK r) -> true 
						               | _ -> false)
		|_   -> failwith ("Bad handshake in"^
                                          "begin_debug_mode_PS")))
   | _ -> failwith ("no PostScript channel")
;;


let error_PS  s =
 match !default_channel_PS  with
  PS_CHANNEL (cin,cout,ps,name) -> 
    (match (read_line_cin cin) with 
	(PS_ANSWER r)  -> failwith ("Error in "^s^" with "^(read_linef cin))
	| _         ->	failwith ("Fatal Error in "^s^" no Answer"))
   | _ -> failwith ("no PostScript channel")
;;

let copy_stream f b = 
   let a = inter_open_in f in
	( while (not (end_of_channel a)) 
	do output_line b(read_line a) done ; flush b;close_in a)
;;

let send_file_PS file =
  match !default_channel_PS with
    PS_CHANNEL (cin,cout,ps,name) -> copy_stream file cout
  | PS_FILE    (ps,name)	  -> copy_stream file ps
;;

let send_and_flush_command_PS  chan com = 
  immediate_output_line chan ("{"^com^"}t")
;;

let exec_PS  s =
  match !default_channel_PS with
  PS_CHANNEL (cin,cout,ps,name) -> 	
    (send_and_flush_command_PS cout s;
     match (read_line_cin cin) with
	   (PS_OK r)    -> output_line ps s
	 | (PS_ANSWER r)   -> error_PS  (read_linef cin)
	 | (PS_ERROR r)   -> error_PS  s
	 |     _        -> failwith ("Fatal Error in "^s))
| PS_FILE (ps,name)	-> output_line ps s
;;

(*  last modifications : send a postscript comment   :  send_comment_PS
 *                       define a postscript symbol  : beginproc_PS, endproc_PS
 *                       execute a postscript symbol : callproc_PS
 *                       write a string on the output PS channel  : 
                                                       output_line_PS 
 *)

let send_comment_PS s = 
  match !default_channel_PS with
  PS_FILE (ps,name)  -> output_line ps ("%"^s)
|   _                -> ()
;;

let beginproc_PS  s =
  match !default_channel_PS with
  PS_CHANNEL (cin,cout,ps,name) -> 	
    (immediate_output_line cout ("/"^s);
     immediate_output_line cout "{";
     match (read_line_cin cin) with
	   (PS_OK r)    -> output_line ps ("/"^s);output_line ps "{"
	 | (PS_ANSWER r)   -> error_PS  (read_linef cin)
	 | (PS_ERROR r)   -> error_PS  s
	 |     _        -> failwith ("Fatal Error in "^s))
| PS_FILE (ps,name)	-> output_line ps ("/"^s);output_line ps "{"
;;

let endproc_PS  () =
  match !default_channel_PS with
  PS_CHANNEL (cin,cout,ps,name) -> 	
    (immediate_output_line cout "} def";
     match (read_line_cin cin) with
	   (PS_OK r)    -> output_line ps "} def"
	 | (PS_ANSWER r)   -> error_PS  (read_linef cin)
	 | (PS_ERROR r)   -> error_PS  "} def"
	 |     _        -> failwith ("Fatal Error in "^"} def"))
| PS_FILE (ps,name)	-> output_line ps "} def"
;;

let callproc_PS = exec_PS;;

let output_line_PS s = 
  match !default_channel_PS with
  PS_CHANNEL (cin,cout,ps,name) -> 	
    (immediate_output_line cout s;
     output_line ps s)
| PS_FILE (ps,name)	-> output_line ps s
;;


(* end of the modifications *)


let exec_and_send_file_PS  s file  = 
 find_file file;
 match !default_channel_PS with
 PS_CHANNEL (cin,cout,ps,name) -> 	
   (send_and_flush_command_PS cout s;
   output_line ps s;
   send_file_PS  file;
   let rep = read_line_cin cin in
     match rep with
	(PS_OK r)  -> copy_stream file ps
      | (PS_ANSWER r) -> error_PS  (read_linef cin)
      | (PS_ERROR r) -> error_PS   s
      |     _      -> failwith ("Fatal Error in "^s))
| PS_FILE (ps,name) -> (output_line ps s; send_file_PS file)
;;


let  gen_ask_PS  s fl =
  match !default_channel_PS with
    PS_CHANNEL (cin,cout,ps,name) -> 
(      let flag_mode = !debug_mode_PS in
( 
      if !debug_mode_PS then true else  begin_debug_mode_PS  ();
      send_and_flush_command_PS cout s;
      if fl then output_line ps s;
      match (read_line_cin cin)  with
         (PS_ANSWER r) ->
    	     (let rep = substitute_brackets(read_linef cin)	
	      in 
              (if flag_mode          then true  
				     else end_debug_mode_PS ();
               rep))
         | (PS_ERROR r) -> error_PS  s
         |     _   -> failwith ("Fatal Error in "^s)))
   | PS_FILE(ps,name) -> failwith ("no PostScript channel")
in
let ask_PS s = gen_ask_PS s false
and ask_and_exec_PS s = gen_ask_PS s true
;;


(* Open and close the communication channel between Caml <-> PS		*)
(* ------------------------------------------------------------		*)

let close_PS () =
  let close_psfile ps = 
	(output_line ps " showpage ";
         flush ps;
	 close_out ps) in
    match !default_channel_PS with  
  PS_CHANNEL (cin,cout,ps,name) ->
    ( close_psfile ps;
      default_channel_PS:= PS_CHANNEL (cin,cout,std_out,""))
| PS_FILE (ps,name) -> (close_psfile ps;
		        PS_FILE (std_out,""));;


let init_PS filename =
  let psfile = (filename^".ps") in 
        match !default_channel_PS with  
  PS_CHANNEL (cin,cout,ps,name) ->
		(default_channel_PS := PS_CHANNEL  (cin,cout,open_out psfile,filename);
		 immediate_output_line cout "{CPS_init}t";	
		 while not ((read_linef cin) = "CPS_init") do () done;
		 read_linef cin;
		 debug_mode_PS := true;
     	         end_debug_mode_PS  ();
		 !ps_loaded)
| PS_FILE (ps,name) -> failwith "no PostScript channel"	;;

let see_PS filename =
 match !default_channel_PS with 
  PS_CHANNEL (cin,cout,ps,name) -> copy_stream (filename^".ps") ps
| PS_FILE    (ps,name)          -> copy_stream (filename^".ps") ps;;

let old_get_var symb = substitute_string "|" "" (get_var symb) 0;;


let open_PS filename = 
  let ipty = (old_get_var "IPTY")
  and opty = (old_get_var "OPTY")
  and psfile = (filename^".ps") 
   in
   (if opty<>"" & ipty<>""  then

    (if !ps_loaded then 
        (match !default_channel_PS with 
          PS_CHANNEL (cin,cout,ps,name) -> 
           default_channel_PS:=PS_CHANNEL (cin,cout,std_out,"")
        | PS_FILE (ps,name) ->
           default_channel_PS:= PS_CHANNEL (inter_open_in ipty, open_out opty,
                                  std_out,""))
                   else
           default_channel_PS:= PS_CHANNEL (inter_open_in ipty, open_out opty,
                                  std_out,""))
		else
          default_channel_PS:=PS_FILE (std_out,""));

   (match !default_channel_PS with 
     PS_CHANNEL (cin,cout,ps,name) -> 
        (immediate_output_line cout "";
         immediate_output_line cout "";
         print_string "Initializing PostScript : ";
         print_newline();
	 immediate_output_line cout "";
	 if !ps_loaded=false then immediate_output_line cout 
		("("^CPS_inter_prelude^") run executive"); 
	      init_PS filename;())
    | PS_FILE (ps,name) -> 
        (default_channel_PS:=PS_FILE(open_out psfile,filename);()));
   
   match !default_channel_PS with 
     PS_CHANNEL (_,_,ps,_) ->  ps_loaded:=true;
                               copy_stream CPS_file_prelude ps;
                               !default_channel_PS
   | PS_FILE (ps,_)        ->  copy_stream CPS_file_prelude ps;
                               !default_channel_PS;;




(*		    Arithmetic and math operators			*)


let add_PS a b=float_of_stringPS(ask_PS 
 	((stringPS_of_float_list (a::[b]))^" add"));;

let div_PS a b=float_of_stringPS(ask_PS 
 	((stringPS_of_float_list (a::[b]))^" div"));;

let idiv_PS a b=int_of_stringPS(ask_PS 
 	((stringPS_of_float_list ((float_of_int a)::[(float_of_int b)]))^" idiv"));;

let mod_PS a b=int_of_stringPS(ask_PS 
 	((stringPS_of_float_list ((float_of_int a)::[(float_of_int b)]))^" mod"));;


let mul_PS a b=float_of_stringPS(ask_PS 
 	((stringPS_of_float_list (a::[b]))^" mul"));;

let sub_PS a b=float_of_stringPS(ask_PS 
 	((stringPS_of_float_list (a::[b]))^" sub"));;


let abs_PS a = float_of_stringPS (ask_PS ( (stringPS_of_float a)^" abs"));;

let neg_PS a = float_of_stringPS (ask_PS ( (stringPS_of_float a)^" neg"));;

let ceiling_PS a = 
	float_of_stringPS (ask_PS ( (stringPS_of_float a)^" ceiling"));;

let floor_PS a = float_of_stringPS (ask_PS ( (stringPS_of_float a)^" floor"));;

let round_PS a = float_of_stringPS (ask_PS ( (stringPS_of_float a)^" round"));;

let truncate_PS a = 
	float_of_stringPS (ask_PS ( (stringPS_of_float a)^" truncate"));;

let sqrt_PS a = float_of_stringPS (ask_PS ( (stringPS_of_float a)^" sqrt"));;

let atan_PS a b=float_of_stringPS(ask_PS 
 	((stringPS_of_float_list (a::[b]))^" atan"));;

let cos_PS a = float_of_stringPS (ask_PS ( (stringPS_of_float a)^" cos"));;

let sin_PS a = float_of_stringPS (ask_PS ( (stringPS_of_float a)^" sin"));;

let exp_PS a b=float_of_stringPS(ask_PS 
 	((stringPS_of_float_list (a::[b]))^" exp"));;

let ln_PS a = float_of_stringPS (ask_PS ( (stringPS_of_float a)^" ln"));;

let log_PS a = float_of_stringPS (ask_PS ( (stringPS_of_float a)^" log"));;

let rand_PS () = float_of_stringPS (ask_PS " rand");;

let srand_PS a = exec_PS ( (stringPS_of_int a)^" srand");;

let rrand_PS () = float_of_stringPS (ask_PS " rrand");;


(*			   Virtual memory operators			*)
(*			   ------------------------			*)

let save_PS () = vm_of_stringPS ( ask_PS " save sts") ;;

let restore_PS v = exec_PS ((stringPS_of_vm v)^" restore");;

let vmstatus_PS () =array_of_stringPS(ask_PS "vmstatus");;


(*			  Miscellaneous operators			*)
(*			  -----------------------			*)


let version_PS () = string_of_stringPS (ask_PS "version");;


(*			  Graphics state operators			*)
(*			  ------------------------			*)


let gsave_PS () =  exec_PS "gsave";;

let grestore_PS () = exec_PS "grestore";;

let grestoreall_PS () = exec_PS "grestoreall";;

let initgraphics_PS () = exec_PS "initgraphics";;

let setlinewidth_PS n = exec_PS ((stringPS_of_float n)^" setlinewidth");;

let currentlinewidth_PS () = float_of_stringPS ( ask_PS "currentlinewidth");;

let setlinecap_PS n = exec_PS ((stringPS_of_int (int_of_float n))^" setlinecap");;

let currentlinecap_PS () = (float_of_int(int_of_stringPS ( ask_PS "currentlinecap")));;

let setlinejoin_PS n = exec_PS ((stringPS_of_int (int_of_float n))^" setlinejoin");;

let currentlinejoin_PS () = (float_of_int(int_of_stringPS (ask_PS "currentlinejoin")));;

let setmiterlimit_PS n = exec_PS ((stringPS_of_float n)^" setmiterlimit");;

let currentmiterlimit_PS () = float_of_stringPS (ask_PS "currentmiterlimit");;

let setdash_PS arr off = exec_PS ((stringPS_of_array arr)^
                                   (stringPS_of_float off)^" setdash");;

let currentdash_PS () =  array_of_stringPS (ask_PS "currentdash");;

let setgray_PS n = exec_PS ((stringPS_of_float n)^" setgray");;

let currentgray_PS () = float_of_stringPS (ask_PS "currentgray");;

let sethsbcolor_PS l  = 
 match l with [h;s;b] ->exec_PS((stringPS_of_float_list [h;s;b])^" sethsbcolor")
           |      _   ->failwith "Bad argument in sethsbcolor";;

let currenthsbcolor_PS () = array_of_stringPS (ask_PS "currenthsbcolor");;

let setrgbcolor_PS l  = 
 match l with [r;g;b] ->exec_PS((stringPS_of_float_list [r;g;b])^" setrgbcolor")
           |      _   ->failwith "Bad argument in setrgbcolor";;

let currentrgbcolor_PS () = array_of_stringPS (ask_PS "currentrgbcolor");;

(*		Coordinate system and matrix operators			*)
(*		--------------------------------------			*)


let matrix_PS () = let r = ask_PS "matrix" in
	matrix_of_stringPS r
;;

let initmatrix_PS () = exec_PS "initmatrix";;

let identmatrix_PS m = let r = ask_PS ((stringPS_of_matrix m)^" identmatrix")
	in matrix_of_stringPS r
;;

let defaultmatrix_PS m = let r = ask_PS ((stringPS_of_matrix m)^
						" defaultmatrix")
	in matrix_of_stringPS r
;;

let currentmatrix_PS m = let r = ask_PS ((stringPS_of_matrix m)^
						" currentmatrix")
	in matrix_of_stringPS r
;;

let setmatrix_PS m =  exec_PS ((stringPS_of_matrix m)^" setmatrix")
;;

let translate_PS (x,y) = exec_PS 
	((stringPS_of_float x)^" "^(stringPS_of_float y)^" translate");;

let translatem_PS (x,y) l = let r = ask_PS 
	((stringPS_of_float x)^" "^(stringPS_of_float y)^" "^
	 (stringPS_of_matrix l)^" translate")
in
	matrix_of_stringPS r
;;

let scale_PS (x,y) = exec_PS 
        ((stringPS_of_float x)^" "^(stringPS_of_float y)^" scale");;


let scalem_PS (x,y) l = let r = ask_PS 
	((stringPS_of_float x)^" "^(stringPS_of_float y)^" "^
	 (stringPS_of_matrix l)^" scale")
in
	matrix_of_stringPS r
;;

let rotate_PS x = exec_PS 
	((stringPS_of_float x)^" rotate");;


let rotatem_PS x l = let r = ask_PS 
	((stringPS_of_float x)^" "^
	 (stringPS_of_matrix l)^" rotate")
in
	matrix_of_stringPS r
;;


let concat_PS l = exec_PS ((stringPS_of_matrix l)^" concat")
;;

let concatmatrix_PS l1 l2 l3 = let r = ask_PS 
	((stringPS_of_matrix l1)^" "^
	 (stringPS_of_matrix l2)^" "^
	 (stringPS_of_matrix l3)^" concatmatrix")
in
	matrix_of_stringPS r
;;

let transform_PS (x,y)  = let r = ask_PS 
	((stringPS_of_float x)^" "^(stringPS_of_float y)^" "^" transform")
in
	point_of_stringPS r
;;


let transformm_PS (x,y) l = let r = ask_PS 
	((stringPS_of_float x)^" "^(stringPS_of_float y)^" "^
	 (stringPS_of_matrix l)^" transform")
in
	point_of_stringPS r
;;

let dtransform_PS (x,y)  = let r = ask_PS 
	((stringPS_of_float x)^" "^(stringPS_of_float y)^" "^" dtransform")
in
	point_of_stringPS r
;;

let dtransformm_PS (x,y) l = let r = ask_PS 
	((stringPS_of_float x)^" "^(stringPS_of_float y)^" "^
	 (stringPS_of_matrix l)^" dtransform")
in
	point_of_stringPS r
;;

let itransform_PS (x,y) l = let r = ask_PS 
	((stringPS_of_float x)^" "^(stringPS_of_float y)^" "^" itransform")
in
	point_of_stringPS r
;;

let itransformm_PS (x,y) l = let r = ask_PS 
	((stringPS_of_float x)^" "^(stringPS_of_float y)^" "^
	 (stringPS_of_matrix l)^" itransform")
in
	point_of_stringPS r
;;

let idtransform_PS (x,y) l = let r = ask_PS 
	((stringPS_of_float x)^" "^(stringPS_of_float y)^" "^" idtransform")
in
	point_of_stringPS r
;;

let idtransformm_PS (x,y) l = let r = ask_PS 
	((stringPS_of_float x)^" "^(stringPS_of_float y)^" "^
	 (stringPS_of_matrix l)^" idtransform")
in
	point_of_stringPS r
;;

let invertmatrix_PS l1 l2 = let r = ask_PS 
	((stringPS_of_matrix l1)^" "^(stringPS_of_matrix l2)^" invertmatrix")
in
	matrix_of_stringPS r
;;


(*			Path construction operators			*)
(*			---------------------------			*)


let newpath_PS   () = exec_PS "newpath"
;;

let currentpoint_PS () =
 let s = "currentpoint" in point_of_stringPS (ask_PS s)
;;

let moveto_PS p = 
 let s = ((stringPS_of_point p)^" moveto") in  exec_PS s
;;

let rmoveto_PS p = 
 let s = ((stringPS_of_point p)^" rmoveto") in    exec_PS s
;;

let lineto_PS p =
 let s = ((stringPS_of_point p)^" lineto") in exec_PS s
;;

let rlineto_PS p = 
 let s = ((stringPS_of_point p)^" rlineto") in exec_PS s
;;

let arc_PS p r a1 a2 = 
 let s = ((stringPS_of_point p)^" "^(stringPS_of_float r)^" "^
          (stringPS_of_angle a1)^" "^(stringPS_of_angle a2)^" arc") in
	exec_PS s
;;

let arcn_PS p r a1 a2 = 
 let s = ((stringPS_of_point p)^" "^(stringPS_of_float r)^" "^
          (stringPS_of_angle a1)^" "^(stringPS_of_angle a2)^" arcn") in
	exec_PS s
;;


let arcto_PS p1 p2 r =  
 let s = ((stringPS_of_point p1)^" "^(stringPS_of_point p2)^" "^
          (stringPS_of_float r)^" arcto") in
	(points_of_stringPS (ask_PS s));;




let curveto_PS p1 p2 p3 = 
 let s = ((stringPS_of_point p1)^" "^(stringPS_of_point p2)^" "^
          (stringPS_of_point p3)^" curveto") in
        exec_PS s
;;

let rcurveto_PS p1 p2 p3 = 
 let s = ((stringPS_of_point p1)^" "^(stringPS_of_point p2)^" "^
          (stringPS_of_point p3)^" rcurveto") in
        exec_PS s
;;

let closepath_PS () = exec_PS "closepath";;

let flattenpath_PS () = exec_PS "flattenpath";;

let reversepath_PS () = exec_PS "reversepath";;

let strokepath_PS () = exec_PS "strokepath";;

let charpath_PS s b = exec_PS ((stringPS_of_string s)^" "^
			        (stringPS_of_bool b)^" charpath");;

let clippath_PS () = exec_PS "clippath";;

let pathbbox_PS () = 
 let s = "pathbbox" in
	points_of_stringPS (ask_PS s);;

let initclip_PS () = exec_PS "initclip";;

let clip_PS () = exec_PS "clip";;

let eoclip_PS () = exec_PS "eoclip";;


(*			Painting Operator				*)
(*			-----------------				*)

let erasepage_PS () = exec_PS " erasepage";;

let fill_PS () = exec_PS " fill";;

let eofill_PS () = exec_PS "eofill";;

let stroke_PS () = exec_PS " stroke";;

let image_PS w h b m str p = exec_and_send_file_PS 
       (("/picstr "^(stringPS_of_int p)^" string def ")^
	(stringPS_of_int w)^" "^
	(stringPS_of_int h)^" "^
	(stringPS_of_int b)^" "^
	(stringPS_of_matrix m )^" newimage " )
	(stringPS_of_image str) ?
        failwith "bitmap file not exist"
;;


    
let imagemask_PS w h i m str p = exec_and_send_file_PS 
       (("/picstr "^(stringPS_of_int p)^" string def ")^
	(stringPS_of_int w)^" "^
	(stringPS_of_int h)^" "^
        (stringPS_of_bool i)^" "^
	(stringPS_of_matrix m )^" newimagemask ") 
	(stringPS_of_image str)  ?
        failwith "bitmap file not exist"
;;


(* *)

let newimage_PS w h b m p = 
       output_line_PS 
       (("/picstr "^(stringPS_of_int p)^" string def ")^
	(stringPS_of_int w)^" "^
	(stringPS_of_int h)^" "^
	(stringPS_of_int b)^" "^
	(stringPS_of_matrix m )^" newimage " )
;;

let newimagemask_PS w h i m p = output_line_PS
       (("/picstr "^(stringPS_of_int p)^" string def ")^
	(stringPS_of_int w)^" "^
	(stringPS_of_int h)^" "^
        (stringPS_of_bool i)^" "^
	(stringPS_of_matrix m )^" newimagemask ") 
;;

(*		Device setup and output operators			*)
(*		---------------------------------			*)


let showpage_PS  () = exec_PS "showpage";;

let copypage_PS  () = exec_PS "copypage";;


(*			Character and font operators			*)
(*			----------------------------			*)


let findfont_PS k =
 let s = ("/"^k^" findfont sts") in
  (font_of_stringPS ( ask_and_exec_PS s)) ;;


let scalefont_PS f n =
 let s = ((stringPS_of_font f) ^ " "^
	(stringPS_of_float_list [n])^" scalefont sts") 
   in (font_of_stringPS(ask_and_exec_PS s))
;;

let setfont_PS f =
 let s = ((stringPS_of_font f)^" setfont") in exec_PS s
;;

let show_PS s = 
  exec_PS ((stringPS_of_string s)^" show");;

let ashow_PS s (ax,ay) = 
  exec_PS ((stringPS_of_float ax)^" "^(stringPS_of_float ay)^" "^
           (stringPS_of_string s)^" ashow");;

let widthshow_PS s c (cx,cy) = 
  exec_PS ((stringPS_of_float cx)^" "^(stringPS_of_float cy)^" "^
           (stringPS_of_float c)^" "^(stringPS_of_string s)^" widthshow");;

let awidthshow_PS s (ax,ay) c (cx,cy) = 
  exec_PS ((stringPS_of_float cx)^" "^(stringPS_of_float cy)^" "^
           (stringPS_of_float c)^" "^
           (stringPS_of_float ax)^" "^(stringPS_of_float ay)^" "^
           (stringPS_of_string s)^" awidthshow");;

let stringwidth_PS s =
  let com = ((stringPS_of_string s)^" stringwidth ") in 
    let u = words (ask_PS com) in
  ( float_of_stringPS (nth u 1),float_of_stringPS  (nth u 2))
;;



(*             Extentions                                                  *)
(*             ----------                                                  *)

let F_PS s n = exec_PS ((string_of_float n)^" /"^s^" F");;

let name_PS () = string_of_stringPS (ask_PS "nameps");;


(*                  Output interface of CPS module                         *)
(*                  ______________________________                         *)


end module
with abstype ps_int and ps_float and ps_bool and ps_array and ps_string
     and     ps_font and ps_vm and ps_image;
     type    ps_matrix and ps_array 
     and     ps_channel and ps_dialog;  
     value   open_PS and close_PS and begin_debug_mode_PS and end_debug_mode_PS
     and     add_PS and div_PS and idiv_PS and mod_PS 
     and     mul_PS and sub_PS and abs_PS and neg_PS
     and     ceiling_PS and floor_PS and round_PS 
     and     truncate_PS and sqrt_PS and atan_PS
     and     cos_PS and sin_PS and exp_PS and ln_PS 
     and     log_PS and rand_PS and srand_PS and rrand_PS
     and     save_PS and restore_PS and vmstatus_PS
     and     gsave_PS and grestore_PS and grestoreall_PS and initgraphics_PS
     and     setlinewidth_PS and currentlinewidth_PS and setlinecap_PS 
     and     currentlinecap_PS and setlinejoin_PS
     and     currentlinejoin_PS and setmiterlimit_PS
     and     currentmiterlimit_PS and setdash_PS 
     and     currentdash_PS and setgray_PS 
     and     currentgray_PS and sethsbcolor_PS 
     and     currenthsbcolor_PS and setrgbcolor_PS 
     and     currentrgbcolor_PS
     and     matrix_PS and initmatrix_PS and identmatrix_PS 
     and     defaultmatrix_PS and currentmatrix_PS
     and     setmatrix_PS and translate_PS and translatem_PS 
     and     scale_PS and scalem_PS
     and     rotate_PS and rotatem_PS and concat_PS 
     and     concatmatrix_PS and transform_PS
     and     transformm_PS and dtransform_PS 
     and     dtransformm_PS and itransform_PS
     and     itransformm_PS and idtransform_PS 
     and     idtransformm_PS and invertmatrix_PS
     and     newpath_PS and currentpoint_PS 
     and     moveto_PS and rmoveto_PS and lineto_PS  
     and     rlineto_PS and arc_PS and arcn_PS 
     and     arcto_PS and curveto_PS and rcurveto_PS
     and     closepath_PS and flattenpath_PS 
     and     reversepath_PS and strokepath_PS
     and     charpath_PS and clippath_PS and pathbbox_PS 
     and     initclip_PS and clip_PS and eoclip_PS
     and     erasepage_PS and fill_PS and eofill_PS 
     and     stroke_PS and image_PS and imagemask_PS
     and     showpage_PS and copypage_PS
     and     findfont_PS and scalefont_PS and setfont_PS 
     and     show_PS and ashow_PS and widthshow_PS
     and     awidthshow_PS and stringwidth_PS and F_PS
     and     name_PS and version_PS
     and     send_comment_PS and beginproc_PS and endproc_PS and callproc_PS
     and     output_line_PS and newimage_PS and newimagemask_PS 
;;
