(*************************************************************************)
(*                                                                       *)
(*                     Projet      Formel                                *)
(*                                                                       *)
(*                    CAML: users' library                               *)
(*                                                                       *)
(*************************************************************************)
(*                                                                       *)
(*                            LIENS                                      *)
(*                        45 rue d'Ulm                                   *)
(*                         75005 PARIS                                   *)
(*                            France                                     *)
(*                                                                       *)
(*************************************************************************)

(* bitmaps.ml
              Guy Cousineau                         
              29 septembre 1990                                         *)

module bitmap
using type point = {xc:float ; yc:float};
      type frame = {xmin:float;xmax:float;ymin:float;ymax:float}
;;

#arith float;;
#arith int;;

type bitmap = {b_width:int;
               b_height:int;
               b_depth:int;
               b_bits:string vect};;

(* Functions to access components of a bitmap                            *)

let bitmap_width (b:bitmap) = b.b_width;;
let bitmap_height (b:bitmap) = b.b_height;;
let bitmap_depth (b:bitmap) = b.b_depth;;

let create_bitmap w h d =
  if (not (mem d [1;2;4;8;16]))
    then failwith
           ("Bitmap: create_bitmap: bad depth ("^
             (string_of_int d)^") for bitmap, only 1,2,4,8 or 16 are ok")
  if not (w*d mod 8=0)
     then failwith ("create_bitmap: width*depth should be a multiple of 8")
    else
     let hexaw=2*((w*d+7) quo 8)
     and sl= ref ([]:string list)
     in 
       for i=0 to h-1 do
        sl:= (make_string hexaw `0`) :: !sl
       done;
       let v = (vector of !sl)
       in
         {b_width=w; b_height=h; b_depth=d; b_bits=v}
;;

let ascii_0 = ascii_code "0"
and ascii_9 = ascii_code "9"
and ascii_a = ascii_code "a"
and ascii_f = ascii_code "f"
and ascii_A = ascii_code "A"
and ascii_F = ascii_code "F"
;;

let conv_four_bits c =
     if ascii_0 <= c && c <= ascii_9
        then c-ascii_0
        else if ascii_a <= c && c <= ascii_f
                then c-ascii_a+10
                else if ascii_A <= c && c <= ascii_F
                        then  c-ascii_A+10
                        else  failwith "conv_four_bits"
;;

let iconv_four_bits n =
      if 0<=n && n<=9
         then ascii_0 + n
         else if 10<=n && n<=15
                 then ascii_a + n - 10
                 else  failwith "iconv_four_bits"
;;

let nth (n,s) = conv_four_bits (nth_ascii(n,s));;
let set_nth (n,s,c) = set_nth_ascii (n,s,iconv_four_bits c);;

let char_map_bitmap f b =
  let b' = create_bitmap b.b_width b.b_height b.b_depth
  in let p = b.b_bits
     and p'= b'.b_bits
     in
      let w = length_string p.(0)
      and h = b.b_height
      in
       for i=0 to h-1
        do
         for j=0 to w-1
          do
           set_nth (j,p'.(i),f (nth(j,p.(i))))
          done
        done;
       b'   
;;


let sub_bitmap b (y1,hh) (x1,ww) =
 let w,h,d,bits = b.b_width,b.b_height,b.b_depth,b.b_bits
 in
  if not (0<hh && 0<ww) 
     then failwith "sub_bitmap: width and height should be >0"
  if not (0<=y1 && y1+hh<=h && 0<=x1 && x1+ww<=w)
     then failwith "sub_bitmap: some coordinate is out of the bitmap"
  if not (ww*d mod 8 = 0 && x1*d mod 8=0)
     then failwith ("sub_bitmap: x coordinates should be such that"
                   ^"x*depth is a multiple of 8")
     else
      let sl= ref ([]:string list)
      in 
      for i=y1 to y1+hh-1
       do
       sl:= sub_string bits.(i) (x1*d/4) (ww*d/4) :: !sl
       done;
      {b_width=ww; b_height=hh;b_depth=d;
       b_bits=(vector of (rev !sl))};;

let copy_bitmap b = sub_bitmap b (0,b.b_height) (0,b.b_width);;


let mask0001 = 1;;
let mask0010 = 2;;
let mask0100 = 4;;
let mask1000 = 8;;
let mask0011 = 3;;
let mask1100 = 10;;
let mask1110 = 14;;
let mask1101 = 13;;
let mask1011 = 11;;
let mask0111 = 7;;
let mask1111 = 15;;

let invert_bitmap = char_map_bitmap 
                      (fun x -> land(lnot x,mask1111));;



let val1 s char_pos bit_pos =
    let v= nth(char_pos,s)
    in match bit_pos
       with 0 -> land(mask0001,lshift(v,-3))
        |   1 -> land(mask0001,lshift(v,-2))
        |   2 -> land(mask0001,lshift(v,-1))
        |   3 -> land(mask0001,v);;

let val2 s char_pos bit_pos =
    let v= nth(char_pos,s)
    in match bit_pos
       with 0 -> land(mask0011,lshift(v,-2))
        |   1 -> land(mask0011,v);;

let val4 s char_pos  =
   nth(char_pos,s);;

let val8 s char_pos =
   lor(lshift(nth(char_pos,s),4),nth(char_pos+1,s));;

let val16 s char_pos =
   lor(lshift(lor(lshift(lor(lshift(nth(char_pos,s),4),
                             nth(char_pos+1,s)),
                        4),
                 nth(char_pos+2,s)),
              4),
        nth(char_pos+3,s))
;;


let change_val1 s char_pos bit_pos bit_val =
  if not (bit_val=0 || bit_val=1)
    then failwith ("Attempt to assign value "
                    ^ string_of_int bit_val
                    ^ " to a binary bitmap")
    else
    let v= nth(char_pos,s)
    in set_nth
           (char_pos,s,
               match bit_pos
               with 0 -> lor(lshift(bit_val,3),
                             land(v,mask0111))
                |   1 -> lor(lshift(bit_val,2),
                             land(v,mask1011))
                |   2 -> lor(lshift(bit_val,1),
                             land(v,mask1101))
                |   3 -> lor(bit_val,
                             land(v,mask1110)))
;;

let change_val2 s char_pos bit_pos pix_val =
  if not (pix_val>=0 && pix_val<=3)
    then failwith ("Attempt to assign value "
                    ^ string_of_int pix_val
                    ^ " to a bitmap with depth 2")
    else
    let v= nth(char_pos,s)
    in set_nth
           (char_pos,s,
               match bit_pos
               with 0 -> lor(lshift(pix_val,2),
                             land(v,mask0011))
                |   1 -> lor(pix_val,
                             land(v,mask1100)))
;;

let change_val4 s char_pos  pix_val =
  if not (pix_val>=0 && pix_val<=15)
    then failwith ("Attempt to assign value "
                    ^ string_of_int pix_val
                    ^ " to a bitmap with depth 4")
    else
        set_nth (char_pos,s,pix_val);;

let change_val8 s char_pos  pix_val =
  if not (pix_val>=0 && pix_val<=255)
    then failwith ("Attempt to assign value "
                    ^ string_of_int pix_val
                    ^ " to a bitmap with depth 8")
    else
       set_nth(char_pos,s,lshift(pix_val,-4));
       set_nth(char_pos+1,s,land(pix_val,mask1111))
;;

let change_val16 s char_pos  pix_val =
       set_nth(char_pos,s,lshift(pix_val,-12));
       set_nth(char_pos+1,s,land(lshift(pix_val,-8),
                                        mask1111));
       set_nth(char_pos+2,s,land(lshift(pix_val,-4),
                                        mask1111));
       set_nth(char_pos+3,s,land(pix_val,mask1111))
;;

(* To set a pixel in a bitmap *)



let set_pixel1 b x y v =
      let m,n = x*b.b_depth quo 4 , x*b.b_depth mod 4 
      and p = b.b_height-1-y
      in  change_val1 b.b_bits.(p) m n v
;;

let set_pixel2 b x y v =
      let m,n = x*b.b_depth quo 4 , x*b.b_depth mod 4 
      and p = b.b_height-1-y
      in  change_val2 b.b_bits.(p) m n v
;;

let set_pixel4 b x y v =
      let m = x
      and p = b.b_height-1-y
      in    change_val4 b.b_bits.(p) m v
;;

let set_pixel8 b x y v =
      let m = 2*x
      and p = b.b_height-1-y
      in    change_val8 b.b_bits.(p) m v
;;

let set_pixel16 b x y v =
      let m = 4*x
      and p = b.b_height-1-y
      in    change_val16 b.b_bits.(p) m v
;;

let set_pixel b x y v =
 if x<0 || x>b.b_width-1 || y<0 || y>b.b_height-1
    then failwith "set_pixel: wrong bitmap coordinates"
    else
        match b.b_depth
         with  1  ->  set_pixel1 b x y v
           |   2  ->  set_pixel2 b x y v
           |   4  ->  set_pixel4 b x y v
           |   8  ->  set_pixel8 b x y v
           |   16  -> set_pixel16 b x y v
;;


(* To get a pixel value in a bitmap *)


let get_pixel1 b x y =
      let m,n = x*b.b_depth quo 4 , x*b.b_depth mod 4 
      and p = b.b_height-1-y
      in  val1 b.b_bits.(p) m n
;;

let get_pixel2 b x y =
      let m,n = x*b.b_depth quo 4 , x*b.b_depth mod 4 
      and p = b.b_height-1-y
      in  val2 b.b_bits.(p) m n
;;

let get_pixel4 b x y =
      let m = x
      and p = b.b_height-1-y
      in    val4 b.b_bits.(p) m
;;

let get_pixel8 b x y =
      let m = 2*x
      and p = b.b_height-1-y
      in    val8 b.b_bits.(p) m
;;

let get_pixel16 b x y =
      let m = 4*x
      and p = b.b_height-1-y
      in    val16 b.b_bits.(p) m
;;

let get_pixel b x y =
 if x<0 || x>b.b_width-1 || y<0 || y>b.b_height-1
    then failwith "set_pixel: wrong bitmap coordinates"
    else
        match b.b_depth
         with  1  ->  get_pixel1 b x y
           |   2  ->  get_pixel2 b x y
           |   4  ->  get_pixel4 b x y
           |   8  ->  get_pixel8 b x y
           |   16  -> get_pixel16 b x y 
;;


let map_hexabyte1 f hex = 
   let bit1 = f (lshift(hex,-3))
   and bit2 = f (land(lshift(hex,-2),mask0001))
   and bit3 = f (land(lshift(hex,-1),mask0001))
   and bit4 = f (land(hex,mask0001))
   in
       lor(lshift(lor(lshift(lor(lshift(bit4,1),bit3),1),bit2),1),bit1);;

let map_hexabyte2 f hex = 
   let b1 = f (lshift(hex,-2))
   and b2 = f (land(hex,mask0001))
   in
       lor(lshift(b1,2),b2);;

let map_bitmap8 f b = 
    let w = b.b_width
    and h = b.b_height
    in
      let b' = create_bitmap w h 8
      in 
       for i=0 to h-1
        do
         for j=0 to 2 step w/8-1
          do
           let v = f (lor(lshift(nth(j,b.b_bits.(i)),4),nth(j+1,b.b_bits.(i))))
           in set_nth(j,b'.b_bits.(i),(lshift(v,-4)));
              set_nth(j+1,b'.b_bits.(i),(land(v,mask1111)))
          done
        done;
       b'
;;

let convert_bitmap (d,f) b =
    let w = b.b_width
    and h = b.b_height
    and (get,set) = match (b.b_depth , d)
                    with  (1,1)  ->  (get_pixel1,set_pixel1)
                      |   (1,2)  ->  (get_pixel1,set_pixel2)
                      |   (1,4)  ->  (get_pixel1,set_pixel4)
                      |   (1,8)  ->  (get_pixel1,set_pixel8)
                      |   (1,16) ->  (get_pixel1,set_pixel16)
                      |   (2,1)  ->  (get_pixel2,set_pixel1)
                      |   (2,2)  ->  (get_pixel2,set_pixel2)
                      |   (2,4)  ->  (get_pixel2,set_pixel4)
                      |   (2,8)  ->  (get_pixel2,set_pixel8)
                      |   (2,16) ->  (get_pixel2,set_pixel16)
                      |   (4,1)  ->  (get_pixel4,set_pixel1)
                      |   (4,2)  ->  (get_pixel4,set_pixel2)
                      |   (4,4)  ->  (get_pixel4,set_pixel4)
                      |   (4,8)  ->  (get_pixel4,set_pixel8)
                      |   (2,16) ->  (get_pixel4,set_pixel16)
                      |   (8,1)  ->  (get_pixel8,set_pixel1)
                      |   (8,2)  ->  (get_pixel8,set_pixel2)
                      |   (8,4)  ->  (get_pixel8,set_pixel4)
                      |   (8,8)  ->  (get_pixel8,set_pixel8)
                      |   (8,16) ->  (get_pixel8,set_pixel16)
                      |   (16,1)  ->  (get_pixel16,set_pixel1)
                      |   (16,2)  ->  (get_pixel16,set_pixel2)
                      |   (16,4)  ->  (get_pixel16,set_pixel4)
                      |   (16,8)  ->  (get_pixel16,set_pixel8)
                      |   (16,16) ->  (get_pixel16,set_pixel16)

     in
       let b' = create_bitmap w h d
         in
           for i=0 to w-1
            do
             for j=0 to h-1
              do
               set b' i j (f (get b i j))
              done
            done;
           b'
;;

let map_bitmap f  b =
    match b.b_depth
    with  1  ->  char_map_bitmap (map_hexabyte1 f) b
      |   2  ->  char_map_bitmap (map_hexabyte2 f) b
      |   4  ->  char_map_bitmap f b
      |   8  ->  map_bitmap8 f b
      |   16 ->  convert_bitmap(b.b_depth,f) b
;;



let read_bitmap depth filename=
 let channel = inter_open_in filename
 in let line= read_line channel
    in let w= length_string line
       and ll= ref [line]
       in
         while (not (end_of_channel channel))
             do ll:= read_line channel :: !ll done;
         close_in channel;
         {b_width=w*4/depth; b_height=length !ll; b_depth=depth;
          b_bits=(vector of (rev !ll))}
;;

let write_bitmap b filename =
  let channel = open_out filename in
  let out = output channel
  in
    for i=0 to b.b_height-1
     do out b.b_bits.(i)
     done;
    close_out channel
;;


let bitmap_frame {b_width=w; b_height=h;_} =
   {xmin=0.0;xmax=float_of_int w;ymin=0.0;ymax=float_of_int h};;


end module
with
type bitmap;
value bitmap_width and bitmap_height and bitmap_depth
and bitmap_frame and copy_bitmap
and create_bitmap and char_map_bitmap and invert_bitmap
and set_pixel and get_pixel and sub_bitmap
and convert_bitmap and map_bitmap
and read_bitmap and write_bitmap;;
