(*************************************************************************)
(*                                                                       *)
(*                     Projet      Formel                                *)
(*                                                                       *)
(*                            CAML                                       *)
(*                                                                       *)
(*************************************************************************)
(*                                                                       *)
(*                            Inria                                      *)
(*                      Domaine de Voluceau                              *)
(*                      78150  Rocquencourt                              *)
(*                            France                                     *)
(*                                                                       *)
(*************************************************************************)

(* TypeRewrite.ml Rewrite gtypes to canonical form                       *)
(*		Roberto Di Cosmo and Pierre Weis			 *)

#standard arith true;;

module Rewrite;;

(*\

\begin{caml_eval}
latex_set_pretty false;;
\end{caml_eval}

In the  CAML system,  the availability  of  a user  level grammar to  describe
expressions and  types of the language makes the first task very easy.  A type
can be described with  the usual concrete syntax: it just suffices  to declare
it as a  type expression (which is recognized by the grammar  {\tt gtype})  to
the system by quoting it with {\tt <:gtype< >>}.  Even better, we declare that
the standard grammar will be {\tt gtype}.  In this way, we  can write our CAML
code  for  the  type  rewriting  function  by  cases  almost  exactly   as  in
Definition~\ref{d:type_rewrite}.

\*)

#set default grammar gtype:gtype;;

let full_iso = ref true;;

let rec rew_type =
  function  (* rewrite : congruence closure *)
  | <<^x * ^y>>  -> rew_type_irr <<^(rew_type x) * ^(rew_type y)>>
  | <<^x -> ^y>> -> rew_type_irr <<^(rew_type x) -> ^(rew_type y)>>
  |          x   -> x

and rew_type_irr =
  function  (* rewrite : reduce expressions with irreducible subexpressions *)
    <<^x -> unit>> as t -> if !full_iso then <<unit>> else t (* Eliminate unit types *)
  | <<unit -> ^x>>      -> x
  | <<^x * unit>>       -> x
  | <<unit * ^x>>       -> x
  | <<(^x * ^y) -> ^z>> -> rew_type_irr <<^x -> ^(rew_type_irr <<^y -> ^z>>)>>   (* currying can produce new currying redexes *)
  | <<^x -> (^y * ^z)>>  -> 
        <<^(rew_type_irr <<^x -> ^y>>) * ^(rew_type_irr <<^x -> ^z>>)>>
  | x                    -> x (* done *);;

(*\ 

   We keep the  coordinates in  a flat  list  where  we  forget about  the
   product structure, and we keep track of the  number of  coordinates, to
   perform more efficient comparisons later on.

\*)

type Type_Coords == int * gtype list;;

let TypeRewrite t = (flatten (rew_type t))
where rec flatten = function
    <<^x * ^y>> -> let (lgt1,l1) = flatten x and (lgt2,l2) = flatten y 
                   in (lgt1+lgt2,append l1 l2)
  |       x     -> (1,[x]);;
(*\

   Then we procede to rename the variables in  the coordinates  of a type.
   We also keep track of this renaming, so that, later on, we can not only
   say if two types are equal, but also up to which renaming of variables.

   The  function  {\tt  split\_vars}  splits  type  variables  in  a  type
   (represented as a  coordinate  list)  by  renaming  them  consecutively
   starting  from {\tt start}  and  keeping  track  of the renamings  in a
   renamings list {\tt envlist}.

   It  uses  the  function  {\tt  shift\_compact\_vars},  which  takes   a
   coordinate  and  returns  a  copy  with  the   type  variables  renamed
   consecutively starting from  {\tt  start}  and  keeping  track  of  the
   renaming in {\tt env}.

\*)
   type VarRenaming == (int * int) list;;

let split_vars start ((lgt:int), coords) =
    (let (ren,newcoords) = fold shifter (start, []) coords
     in ren,lgt,newcoords)
    where shifter (start,envlist) coord =
          (let ((ends,env),newcoord) = shift_compact_vars (start,[]) coord
          in (ends,append env envlist),newcoord)
    where rec shift_compact_vars (start,env) = function
              <<(^l) ^n>> ->
                let (start_env, l') = fold shift_compact_vars (start,env) l 
                in start_env,<<(^l') ^n>>
            | <<'^i>> ->
                ((start,env),<<'^(assq i env)>> ? 
                 (start+1,(i,start)::env),<<'^start>>);;
(*\ 

   We can now combine our TypeRewrite function with the renaming functions
   to bring a type to {\em split-normal-form}.
%%   Again,   we   have two   versions  for  rewriting just a  
%% type or a whole  list  of  types to {\em   split-normal-form}.

\*)
(* Rewrite one type to split normal with variables starting from START *)

let SplitTR start typ = split_vars start (TypeRewrite typ);;

end module with
 value SplitTR   (* : int -> gtype -> (int * (int * int) list) * int * gtype list *)
 and   full_iso  (* : bool ref *)
;;
