(*************************************************************************)
(*                                                                       *)
(*                     Projet      Formel                                *)
(*                                                                       *)
(*                            CAML                                       *)
(*                                                                       *)
(*************************************************************************)
(*                                                                       *)
(*                            Inria                                      *)
(*                      Domaine de Voluceau                              *)
(*                      78150  Rocquencourt                              *)
(*                            France                                     *)
(*                                                                       *)
(*************************************************************************)

(* latex.ml	A filter to provide automatic treatment of               *)
(*              caml parts in latex files                                *)
(*		Pierre Weis						 *)

#standard arith true;;
#fast arith true;;
#open compilation true;;

directive use (lib_directory ^ "automat");;
directive use (lib_directory ^ "automat_actions");;
directive use (lib_directory ^ "top_string");;
directive use (lib_directory ^ "latex_value");;
directive use (lib_directory ^ "latex_type");;

(* This is a very useful procedure to read from a file
    which must be filtered using latex_file: *)
let latex_ask s =
    print_string s; print_flush();
    do_seq
     [do_until_char "\n" copy_accumulate; do_thru_char_found copy] ();
    accu_flush ()
;;

(* One can use it with:
\begin{caml_example}
ask"nom";;
2
\end{caml_example}
*)

(* String Constants *)
let begin_caml_example = ref "\\begin{verbatim}"
and begin_caml_example_newline = ref "\\begin{verbatim}\n";;

let end_caml_example = ref "\\end{verbatim}"
and end_caml_example_newline = ref "\\end{verbatim}\n";;

let begin_caml_type_of_newline = ref !begin_caml_example_newline
and end_caml_type_of_newline = ref !end_caml_example_newline;;

let begin_caml_primitive = ref
            "\n\\vbox{\\vspace {2.855cm}}\n\
             \\vskip -2cm\n\
             \\noindent\\llap{\\hbox to 0.5cm {$\\clubsuit$\\hfill}}\n\
             \\vskip -0.855cm\n\
             \\vskip 0cm\\nobreak\n\
             \\begin{verbatim}"
and end_caml_primitive = ref !end_caml_example
;;

let begin_caml_verify_newline = ref !begin_caml_example_newline
and end_caml_verify_newline = ref !end_caml_example_newline
;;

let begin_caml_include_newline = ref !begin_caml_example_newline
and end_caml_include_newline = ref !end_caml_example_newline
;;

let begin_caml_newline = ref !begin_caml_example_newline
and end_caml_newline = ref !end_caml_example_newline
;;

(* Ada stuff *)
let begin_ada_example = ref !begin_caml_example_newline
and begin_ada_result = ref !begin_caml_example_newline
and begin_ada_verify = ref !begin_caml_example_newline
;;

let end_ada_example = ref !end_caml_example_newline
and end_ada_result = ref !end_caml_example_newline
and end_ada_verify = ref !end_caml_example_newline
;;

let begin_ada = ref !begin_caml_example_newline
and end_ada = ref !end_caml_example_newline
;;

let begin_ada_include = ref !begin_caml_example_newline
and end_ada_include = ref !end_caml_example_newline
;;

(* Commands to call ada compiler:
   may be redefined by the user, these one are suited for Alsys Ada compiler *)
let ada_source_name = ref "example.ada";;
let ada_target_name = ref "example.out";;
let call_ada_compiler =
 ref "ada compile library=adalib source=example.ada > example.out 2>&1";;
let call_ada_linker =
 ref "ada bind library=adalib program=example >> example.out 2>&1";;
let call_ada_run =
 ref "./example >> example.out 2>&1";;
let ada_example_treatment = ref
 (function () ->
   comline !call_ada_compiler;
   comline !call_ada_linker;
   comline !call_ada_run
 );;
let ada_header = ref
    "with TEXT_IO, INTEGER_TEXT_IO;\n\
      use TEXT_IO, INTEGER_TEXT_IO;\n\
     procedure EXAMPLE is\n";;

let ada_footer = ref
    "end EXAMPLE; \n"
;;

let ada_verify_treatment = ref
 (function () ->
   comline !call_ada_compiler;
   comline !call_ada_linker
 );;

let user_syntax_terminator = ref ";;"
and user_syntax_prefix = ref "I(<<"
and user_syntax_postfix = ref ">>);;"
and user_syntax_prompt = ref "#"
and user_begin_example = ref !begin_caml_example_newline
and user_end_example = ref !end_caml_example_newline;;

let begin_caml_marker = "\\begin{caml";;

(* Ada stuff *)
let begin_ada_marker = "\\begin{ada";;

let latex_margin = ref 65;;

let set_latex_margin n =
 if n > 0 || n > 256
  then failwith"set_latex_margin"
  else (latex_margin := n;
        set_margin !latex_margin;
        set_echo_margin !latex_margin);;

(* To print current prompt *)
let print_prompt () =
 toplevel_print_newline();display_string (prompt());print_flush();;


let print_user_syntax_prompt () =
    display_string !user_syntax_prompt;print_flush();;


(* The tml loop for a string tml_string is in top_string.ml *)
(* The eml_string (which evaluates a string but does not print) too *)

(* Toplevel for latex_value :
   evaluates silently then, in the case of an expression, updates ``it'' and
   prints ``it'' using latex_print_value *)
let latex_it_value_string =
    eml_it_string (fun (obj,ty) -> latex_print_value ty obj);;
(* Print ``it'' and its type *)
let it_string =
    eml_it_string (fun (obj,ty) -> print_valtype (ty,obj);print_newline());;
(* Prints the value of it without its type *)
let it_value_string =
    eml_it_string (fun (obj,ty) -> print_value ty obj;print_newline());;

(* Verifies syntax and typechecking *)
let vml_string s =
reset_line_counter();
(match parse_string s
 with	(MLdecl D)      -> typing_decl D;()
 |	(ML E)          -> typing_exp E;()
 |      (MLgrammar g)   -> type_grammar_decl g;()
 |      (MLsyn_grammar (new,old as p)) -> typing_syn_grammar p;()
 |	(MLdirective (Pragmaexp E)) -> do_switching_dir_env typing_exp E;()
 |	(MLdirective (Pragmadecl D)) -> do_switching_dir_env typing_decl D;()
 |	(MLpragma (Pragmaexp E)) -> do_switching_prag_env typing_exp E;()
 |	(MLpragma (Pragmadecl D)) -> do_switching_prag_env typing_decl D;()
(**************
Should be:
 |      (MLbegin_mod _) ->
 |       MLsys_begin_mod (s,arg) -> of string & (MLspec list & string list)
 |       MLend_mod of MLspec list
 |       MLautoload of ((string & MLtype) & (string & MLtype) list) & ML
 |       MLautoload_grammar of string & ((string & string) list) & ML
instead of:
***************)
 |      _ -> ()
      );
clear_v_stack()
;;

(* Returns the type of an expression *)
let tyml_string s =
 reset_line_counter();
 (match parse_string s with
    (ML E) -> print_type (abbreviate (snd(snd(typing_exp E))))
  | (MLdirective (Pragmaexp E)) ->
        print_type (abbreviate (snd(snd(do_switching_dir_env typing_exp E))))
  | (MLpragma (Pragmaexp E)) ->
        print_type (abbreviate (snd(snd(do_switching_prag_env typing_exp E))))
  | _ -> error_message "caml_type_of needs an expression as argument !");
 clear_v_stack()
;;

(* tml_gen executes a step of CAML toplevel as usual
   caml_gen does the same but does not print anything *)
let latex_default_example_printer s =
    print_prompt ();
    display_string (substitute_char "\n" ("\n"^prompt()) s 0);
    (s^"\n");;

let pretty_print t =
    pretty_print_MLsyntax t;
    print_newline();
    message";;"
;;

let latex_example_pretty_printer s = pretty_print (parse_string s);s;;

let latex_example_printer = ref latex_default_example_printer;;

(* if latex_pretty_flag is true then caml text is pretty_printed *)
let latex_pretty_flag = ref false;;

let latex_set_pretty_example b =
    latex_example_printer :=
     if b then latex_example_pretty_printer else latex_default_example_printer;
    latex_pretty_flag := b;;

let caml_example =
    tml_gen (fun s -> tml_string (!latex_example_printer s))

and caml_print_it_value =
    tml_gen (fun s -> it_value_string (!latex_example_printer s))

and caml_verify =
 let verify_string s =
 display_message s; (s^"\n") in
tml_gen (vml_string o verify_string);;

let latex_tyml_string s =
 reset_line_counter();
 (match parse_string s with
    (ML E) -> latex_print_type (abbreviate (snd(snd(typing_exp E))))
  | (MLdirective (Pragmaexp E)) ->
        latex_print_type
         (abbreviate (snd(snd(do_switching_dir_env typing_exp E))))
  | (MLpragma (Pragmaexp E)) ->
        latex_print_type
         (abbreviate (snd(snd(do_switching_prag_env typing_exp E))))
  | _ -> error_message "caml_type_of needs an expression as argument !");
 clear_v_stack()
;;

let caml_type_of,caml_primitive =
 let type_string s =
  open_hovbox 0;print_string (s^" :");print_break(1,0);
  tyml_string (s^";;\n") in
 let type_message s = type_string s;print_newline() in
 let do_caml_primitive s =
  let prims = words2 "\n" s in
  do_list (fun p -> display_message ("\\index{\\verb\""^p^"\"}");
                    display_string "\\glossary{\\verb\"";
                    type_string p;message"\"}")
          prims;
  display_message !begin_caml_primitive;
  do_list type_message prims;
  display_message !end_caml_primitive in
 (tml_gen type_message),(tml_gen do_caml_primitive);;

let caml_function,caml_function_star,caml_function_star_star,
    caml_function_star_star_star =
 let type_string s =
  display_string"\\camlname{";display_string s;display_string"}";
  display_string"{";
  latex_tyml_string (s^";;\n");
  display_string"}" in
 let type_message s = type_string s;display_newline() in
 let do_caml_primitive_gen caml_fun_message s =
  let prims = words2 "\n" s in
  display_message caml_fun_message;
  do_list type_message prims;
  display_message "}\n" in
 let do_caml_primitive = do_caml_primitive_gen "\\camlfunction{"
 and do_caml_primitive_star = do_caml_primitive_gen "\\camlfunction[i]{"
 and do_caml_primitive_star_star = do_caml_primitive_gen "\\camlfunction[m]{"
 and do_caml_primitive_star_star_star =
     do_caml_primitive_gen "\\camlfunction[]{" in
 (tml_gen do_caml_primitive),
 (tml_gen do_caml_primitive_star),
 (tml_gen do_caml_primitive_star_star),
 (tml_gen do_caml_primitive_star_star_star)
;;

let caml_primitive_ref = ref caml_primitive;;

(*
let print_declared_type s =
    let (Tbinding (stamp,nvars,weakl,tags,variant)) = get_global_type s in
    match variant with
     Concrete_variant (consts,morphs,superfluous,constructors) ->
      ....
;;

let caml_print_type
*)

(* Evaluates but does not print in the output file *)
let caml_eval = caml_gen eml_string;;

(* Evaluates but does not print the results in the output file:
   printing side-effects are written into the output *)
let caml_eval_star = tml_gen eml_string;;

(* Prints the result of the evaluation with latex_print_value *)
let caml_latex_value = tml_gen latex_it_value_string
(* Prints the result of the evaluation with print_value and the type of it *)
and caml_print_it = tml_gen it_string
(* Prints the result of the evaluation with print_value *)
and caml_print_it_value_star = tml_gen it_value_string;;

(* caml_user_syntax_example 
    - echoes each phrase with a user_syntax_prompt
    - concats it with both a prefix string and a postfix string
      and sends it to the CAML toplevel.
   It can be used to treat user examples run on his own) toplevel with his own 
   syntax, setting:
     user_syntax_terminator := "TERMINATOR";;
     user_syntax_prefix := "user_toplevel(<:user_value_syntax<";;
     user_syntax_postfix := ">>);;";;
     user_syntax_prompt := "PROMPT: ";;
   in a caml_eval latex environment. *)
let caml_user_syntax_example =
 let example_string s =
  print_user_syntax_prompt ();
  display_string
   (substitute_char "\n" ("\n"^(!user_syntax_prompt)) s 0);
  display_string "\n";
  implode [!user_syntax_prefix;s;!user_syntax_postfix;"\n"] in
 tml_gen (eml_string o example_string);;

let copy_file =
 let buff = make_string 1024 ` ` in
 (fun os is ->
   try
   while true do
    buf_input is buff 1024;
    output os buff done
   with ml_eof n -> output os (sub_string buff 0 n); close_in is)
;;

(* Include a file printing the strings before and after
   if the file is not empty (due to latex verbatim limitations!) *)
let caml_include_gen before after s =
 try system_handler
     (function s ->
       let is = open_in s in
        if lookahead_ascii is <> -1 then
         begin
          output !auto_outs before;
          copy_file !auto_outs is;
          output !auto_outs after
         end
        else close_in is) s
 with _ -> ();;

let caml_include = caml_include_gen "" "";;

let bips = make_string 2 `\007`;;

let latex_parano_flag = ref true;;

let warning_ignored s =
    warning ("latex filter: INCORRECT INPUT\n"^bips^s^" was ignored");;

let warning_input_ignored s =
    if s <> "" && (skip_space_return s <> "") then warning_ignored s;;

let warning_unknown_environment s =
    if !latex_parano_flag then
     do_printing_in_error_mode
      (fun () ->
        print_string("% Warning: Latex filter"^bips^" %");
        print_string("% ill used (or unknown) Caml environment \""^s^"\" %");
        message
         ("% (be careful with blanks: use exactly \\begin{caml_example}\\n) %")
      )
      ()
;;

(* When simulating a toplevel \n are replaced by a prompt
   thus we ellipse the terminal \n from the strings extracted from the file *)
let treat_caml_phrases treat_phrase end_phrases =
 treat where rec treat =
 do_seq
  [do_until [";;\n";";; \n";end_phrases] accumulate;
   do_when_found
    (fun (";;\n" | ";; \n") -> 
         do_seq [do_thru_pattern ignore;
                 accumulate_string ";;";
                 treat_phrase o accu_flush;
                 treat]
     |  _       ->
        warning_input_ignored (accu_flush());
        do_thru_pattern ignore)];;

let treat_caml_example =
 treat_caml_phrases caml_example "\\end{caml_example}\n"
and treat_caml_verify =
 treat_caml_phrases caml_verify "\\end{caml_verify}\n"
;;

(* Ada stuff *)
(*
   Ada example works as follows:
   when encountering
    ada_prelude 
    ada_example
    ada_postlude
   This feature needs a
   \begin{caml_eval}
   init_ada();;
   \end{caml_eval}
   at the beginning of your latex file, and a
   \begin{caml_eval}
   close_ada();;
   \end{caml_eval}
   at the end of the source file
*)
let ada_out = ref std_out;;
let ada_in = ref std_in;;
let prelude_ada = ref false;;

let init_ada () =
    ada_out := open_out !ada_source_name;
    output !ada_out !ada_header;;

let ada_current_treatment = ref !ada_example_treatment;;

let include_ada_result () =
    output !ada_out !ada_footer;
    flush !ada_out;
    close_out !ada_out;
    !ada_current_treatment ();
    caml_include_gen !begin_ada_result !end_ada_result !ada_target_name
;;

(*let ada_gen copy_flag treat_fun s =
 if not !prelude_ada then init_ada();
 if copy_flag then
  begin copy_string s ();
        copy_string !begin_ada_example()
  end;
 output !ada_out s;
 ada_current_treatment := !treat_fun;
 if not !prelude_ada then include_ada_result();;
*)
let ada_example s =
 if not !prelude_ada then init_ada();
 copy_string !begin_ada_example();
 copy_string s ();
 output !ada_out s;
 ada_current_treatment := !ada_example_treatment;
 copy_string !end_ada_example();
 if not !prelude_ada then include_ada_result()
;;

let ada_example_star s =
 if not !prelude_ada then init_ada();
 output !ada_out s;
 ada_current_treatment := !ada_example_treatment;
 if not !prelude_ada then include_ada_result()
;;

let ada_verify s =
 if not !prelude_ada then init_ada();
 copy_string !begin_ada_verify();
 copy_string s ();
 copy_string !end_ada_verify();
 output !ada_out s;
 ada_current_treatment := !ada_verify_treatment;
 if not !prelude_ada then include_ada_result()
;;

let warning_pre_post_lude s =
    warning_input_ignored s;
    warning "ill prelude-postlude balance (should be nested like bracket)";;

let ada_prelude s =
  if !prelude_ada
   then warning_pre_post_lude s
  else
   begin
    prelude_ada:=true;
    init_ada();
    output !ada_out s
   end
and ada_postlude s =
  if not !prelude_ada
   then warning_pre_post_lude s
  else
   begin
    prelude_ada:=false;
    output !ada_out s;
    include_ada_result ()
   end
;;

let treat_ada_phrases treat_phrase end_phrases =
 do_seq
  [do_until [end_phrases] accumulate;
   do_when_found
    (fun x ->
     if x = end_phrases then
      do_seq [do_thru_pattern ignore;
              treat_phrase o accu_flush]
      else
       begin
       warning_input_ignored (accu_flush());
       do_thru_pattern ignore
       end
     )];;

let treat_ada_example =
 treat_ada_phrases ada_example "\\end{ada_example}\n"
and treat_ada_verify =
 treat_ada_phrases ada_verify "\\end{ada_verify}\n"
;;

let treat_ada_example_star =
 treat_ada_phrases ada_example_star "\\end{ada_example*}\n"
;;
let treat_ada_example_star_star =
 treat_ada_phrases ada_example_star "\\end{ada_example**}\n"
;;

let treat_ada_prelude =
 treat_ada_phrases ada_prelude "\\end{ada_prelude}\n"
and treat_ada_postlude =
 treat_ada_phrases ada_postlude "\\end{ada_postlude}\n"
;;

let treat_ada =
 do_seq
  [do_until ["\\end{ada}\n"] copy;
   do_thru_pattern ignore]
;;

let treat_single_caml_phrase_gen treat_phrase end_phrases =
 treat where rec treat =
 do_seq
  [do_until_word end_phrases accumulate;
   do_thru_pattern ignore;
   treat_phrase o accu_flush];;

let treat_single_caml_phrase =
  treat_single_caml_phrase_gen caml_example "\\end{caml_phrase}\n";;
let treat_single_caml_phrase_star =
  treat_single_caml_phrase_gen caml_print_it "\\end{caml_phrase*}\n";;
let treat_single_caml_phrase_star_star =
  treat_single_caml_phrase_gen caml_eval_star "\\end{caml_phrase**}\n";;


(* treat_caml_user_syntax_phrases must work with current user syntax
   environment, thus we must delay the treat function to get the proper
   dereferencing of user_syntax_terminator *)
let treat_caml_user_syntax_phrases treat_phrase end_phrases =
fun() -> 
 let rec treat = 
 do_seq
  [do_until [!user_syntax_terminator^"\n";end_phrases] accumulate;
   do_when_found
    (fun x ->
      if x = (!user_syntax_terminator^"\n") then
       do_seq [do_thru_pattern ignore;
               accumulate_string !user_syntax_terminator;
               treat_phrase o accu_flush;
               treat]
      else
       (warning_input_ignored(accu_flush());
        do_thru_pattern ignore)
      )]
 in treat();;

let treat_caml_user_syntax_example =
 treat_caml_user_syntax_phrases
   caml_user_syntax_example "\\end{caml_user_syntax_example}\n";;

(* Normal case caml_phrases are extracted until the terminal newline *)
let treat_caml_gen treat end_phrases =
 treat_rec where rec treat_rec =
 do_seq
  [do_until [";;\n";";; \n";end_phrases] accumulate;
   do_when_found
   (fun (";;\n" | ";; \n") -> 
        do_seq [do_thru_pattern accumulate;
                treat o accu_flush;
		treat_rec]
     |  _       ->
        warning_input_ignored(accu_flush());
        do_thru_pattern ignore)];;

let treat_caml_eval =
 treat_caml_gen caml_eval "\\end{caml_eval}\n"
and treat_caml_print_latex =
 treat_caml_gen caml_eval_star "\\end{caml_print_latex}\n"
and treat_caml_latex_value =
 treat_caml_gen caml_latex_value "\\end{caml_latex_value}\n"
and treat_caml_example_star =
 treat_caml_gen caml_print_it "\\end{caml_example*}\n"
and treat_caml_example_star_star =
 treat_caml_gen caml_eval_star "\\end{caml_example**}\n"
and treat_caml_print_it_value =
 treat_caml_gen caml_print_it_value "\\end{caml_print_it_value}\n"
and treat_caml_print_it_value_star =
 treat_caml_gen caml_print_it_value_star "\\end{caml_print_it_value*}\n"
;;

(* There the terminator is only a newline *)
let treat_gen treat end_phrases =
 treat_rec where rec treat_rec =
 do_seq
  [do_until ["\n";end_phrases] accumulate;
   do_when_found
   (fun "\n" -> do_seq [do_thru_pattern ignore;
                        treat o accu_flush;
                        treat_rec]
     |  x       ->
        warning_input_ignored (accu_flush());
        do_thru_pattern ignore)];;

let treat_caml_type_of = treat_gen caml_type_of "\\end{caml_type_of}\n" 
and treat_caml_include = treat_gen caml_include "\\end{caml_include}\n";;

(* just copy ! (in the future may be more sophisticated) *)
let treat_caml =
 do_seq
  [do_until ["\\end{caml}\n"] copy;
   do_thru_pattern ignore]
;;

let treat_caml_primitive_gen end_prim treat_prim =
 do_seq [do_until_word end_prim accumulate;
         do_thru_pattern ignore;
         treat_prim o accu_flush]
;;


let end_prim_marker = "\\end{caml_primitive}\n";;
let end_prim_star_marker = "\\end{caml_primitive*}\n";;
let end_prim_star_star_marker = "\\end{caml_primitive**}\n";;
let end_prim_star_star_star_marker = "\\end{caml_primitive***}\n";;

let treat_caml_parts =
 [do_find_pat_thru
          ["_example}\n"; "_type_of}\n";
           "_primitive}\n"; "_primitive*}\n"; "_primitive**}\n";
           "_primitive***}\n";
           "_eval}\n"; "_verify}\n";
           "_include}\n"; "_print_latex}\n"; "_latex_value}\n";
           "_example*}\n";
           "_example**}\n";
           "_print_it_value}\n";
           "_print_it_value*}\n";
           "_user_syntax_example}\n";
           "_phrase}\n";
           "_phrase*}\n";
           "_phrase**}\n";
           "_ignore}\n";
           "}\n";
           "} \n"]
          ignore;
  do_when_found
   (function
        "_example}\n" ->
        do_seq [copy_string !begin_caml_example;
                treat_caml_example;
                copy_string !end_caml_example_newline]
     |  "_type_of}\n" ->
        do_seq [copy_string !begin_caml_type_of_newline;
                treat_caml_type_of;
                copy_string !end_caml_type_of_newline]
     |  "_primitive}\n" ->
          treat_caml_primitive_gen end_prim_marker !caml_primitive_ref
     |  "_primitive*}\n" ->
          treat_caml_primitive_gen end_prim_star_marker caml_function_star
     |  "_primitive**}\n" ->
          treat_caml_primitive_gen end_prim_star_star_marker
                                   caml_function_star_star
     |  "_primitive***}\n" ->
          treat_caml_primitive_gen end_prim_star_star_star_marker
                                   caml_function_star_star_star
     |  "_eval}\n" -> treat_caml_eval
     |  "_verify}\n" ->
        do_seq [copy_string !begin_caml_example_newline;
                treat_caml_verify;
                copy_string !end_caml_example_newline]
     |  "_include}\n" ->
        do_seq [copy_string !begin_caml_include_newline;
                treat_caml_include;
                copy_string !end_caml_include_newline]
     |  "_user_syntax_example}\n" -> 
        do_seq [copy_string !user_begin_example;
                treat_caml_user_syntax_example;
                copy_string !user_end_example]
     | "_phrase}\n" ->
        do_seq [copy_string !begin_caml_example;
                treat_single_caml_phrase;
                copy_string !end_caml_example_newline]
     | "_phrase*}\n" ->
        do_seq [copy_string !begin_caml_example;
                treat_single_caml_phrase_star;
                copy_string !end_caml_example_newline]
     | "_phrase**}\n" -> treat_single_caml_phrase_star_star
     | "_print_latex}\n" -> treat_caml_print_latex
     | "_latex_value}\n" -> treat_caml_latex_value
     | "_example*}\n" ->
        do_seq [copy_string !begin_caml_example_newline;
                treat_caml_example_star;
                copy_string !end_caml_example_newline]
     | "_example**}\n" -> treat_caml_example_star_star
     |  "_print_it_value}\n" -> treat_caml_print_it_value
     |  "_print_it_value*}\n" -> treat_caml_print_it_value_star
     |  "_ignore}\n" -> do_thru_word "\\end{caml_ignore}\n" ignore
     |  ("}\n" | "} \n") ->
        do_seq [copy_string !begin_caml_newline;
                treat_caml;
                copy_string !end_caml_newline]
     |  s (* should be "" (nothing found) *) ->
        do_seq [(* Find the wrong environment *)
               do_until ["}"] accumulate;
               warning_unknown_environment o accu_dump;
               copy_string begin_caml_marker;
               copy_flush_accu]);
  accu_clear];;

(* Ada stuff *)
let treat_ada_include =
 treat_gen caml_include "\\end{ada_include}\n";;

let treat_ada_parts =
  [do_find_pat_thru
          ["_example}\n";
           "_verify}\n";
           "_prelude}\n";
           "_postlude}\n";
           "_include}\n";
           "_example*}\n";
           "_example**}\n";
	   "}\n"]
	  ignore;
  do_when_found
   (fun "_example}\n" -> treat_ada_example
     |  "_verify}\n" -> treat_ada_verify
     |  "_prelude}\n" -> treat_ada_prelude
     |  "_postlude}\n" -> treat_ada_postlude
     |  "_include}\n" ->
        do_seq [copy_string !begin_ada_include;
                treat_ada_include;
                copy_string !end_ada_include]
     | "_example*}\n" -> treat_ada_example_star
     | "_example**}\n" -> treat_ada_example_star_star
     |  "}\n" ->
        do_seq [copy_string !begin_ada;
                treat_ada;
                copy_string !end_ada]
     |  s (* should be "" (nothing found) *) ->
        do_seq [(* Find the wrong environment *)
               do_until ["}"] accumulate;
               warning_unknown_environment o accu_dump;
               copy_string begin_ada_marker;
               copy_flush_accu]);
  accu_clear];;

let treat_parts =
 [do_find_pat_thru [begin_caml_marker;begin_ada_marker] ignore;
  do_when_found
   (fun
      "\\begin{caml" -> do_seq treat_caml_parts
    | "\\begin{ada" -> do_seq treat_ada_parts
    |         s     -> failwith ("treat parts"^s))];;

let make_latex_automat treat_file =
 make_automaton_with
  [do_act (fun () ->
            open_user_print !auto_outs;
            open_system_print !auto_outs;
            set_margin !latex_margin;
            set_echo_margin !latex_margin;
            load_flag:=false;
            show_filename_on_error:=false;
            open_error_print (broadcast[std_out;!auto_outs])) ()]
  treat_file
  [do_act (fun () ->
            close_user_print !auto_outs;
            close_system_print !auto_outs;
            all_print_to std_out) ()];;

let latex_gen =
 let treat_file = 
  (do_until [begin_caml_marker;begin_ada_marker] copy) :: treat_parts in
 make_latex_automat treat_file;;

let latex_file_gen (s1,s2) =
 let is = open_in s1 in
 let os = try open_out s2 with reraise -> close_in is; raise reraise in
 system_message ("Processing "^s1);
 latex_gen (is,os);
 system_message (s1^" processed");;

let latex_fil_gen p =
 open_env "Latex";
 try latex_file_gen p; close_env "Latex"
 with reraise -> close_env "Latex"; raise reraise;;

let latex_files_of s =
 let file = find_file (s^".tex") in
 file,((remove_ext ".tex" file)^".ml.tex");;

let (latex_file, latex_fil) = 
 latex_file_gen o latex_files_of, latex_fil_gen o latex_files_of;;

(* To process a file only when source is newer than object *)
let use_latex_file,use_latex_fil =
 let use_gen use_fun s =
 let (f1,f2) = latex_files_of s in
 if (newest f1 f2) = f2 then () else use_fun (f1,f2) in
 (use_gen latex_file_gen),
 (use_gen latex_fil_gen);;

let (latex_file_from_to,latex_fil_from_to) = latex_file_gen,latex_fil_gen;;


export_from_sys_to_usr
    <:Caml:Export_list<
  value caml_primitive_ref
    and latex_parano_flag
    and latex_print_type

    and caml_primitive
    and caml_function
    and latex_ask

    and begin_caml_example
    and end_caml_example
    and begin_caml_example_newline
    and end_caml_example_newline

    and begin_caml_type_of_newline
    and end_caml_type_of_newline

    and begin_caml_primitive
    and end_caml_primitive

    and begin_caml_verify_newline
    and end_caml_verify_newline

    and begin_caml_include_newline
    and end_caml_include_newline

    and begin_caml_newline
    and end_caml_newline

    and user_syntax_terminator
    and user_syntax_prefix
    and user_syntax_postfix
    and user_syntax_prompt
    and user_begin_example
    and user_end_example

    and set_latex_margin
    and latex_margin

    (* Ada stuff *)
    and ada_source_name
    and ada_target_name
    and call_ada_compiler
    and call_ada_linker
    and call_ada_run
    and ada_header
    and ada_footer

    and begin_ada_example
    and end_ada_example

    and begin_ada_result
    and end_ada_result

    and begin_ada_verify
    and end_ada_verify

    and begin_ada_include
    and end_ada_include

    and begin_ada
    and end_ada
>>;;
