;;
;;  newer version of PostScript vbm
;;
;;********************************** globals **********************************
;;


(defvar #:sys-package:bitmap '#:display:ps)

(defvar #:sys-package:colon #:sys-package:bitmap)

(unless (boundp ':apple-lw-fontnames)
	(defvar :apple-lw-fontnames
	  '("Courier" "Courier-Bold" "Courier-Oblique"
	    "Courier-BoldOblique" "Times-Roman" "Times-Bold"
	    "Times-Italic" "Times-BoldItalic" "Helvetica"
	    "Helvetica-Bold" "Helvetica-Oblique"
	    "Helvetica-BoldOblique" "Symbol")))

(unless (boundp ':rgb) (defvar :rgb #[() () ()]))

(unless (boundp ':ps-font-dir)
	(defvar :ps-font-dir "/udd/halles/rodine/psbv/afm/"))

(defvar :a4h 200)
(defvar :a4v 287)
(defvar :a5h 100)
(defvar :a5v 145)

;;
;;******************************** structures ********************************
;;

(defstruct #:display:ps
  reread
  line-style-vector
  pattern-vector
  font-vector
  font-metrics
  mode-vector
  ps-current-ge
  version
  filename
  hchannel
  ochannel
  printer-id
  engine-id
  ps-id
  epsf-id
  scale-factor
  setscrlib
  abbrev
  orient
  origin
  margins
  paper-size
  mxn-pages)

(defstruct #:ps:extend
  view-rect
  offset-x
  offset-y
  visible
  pswindow
  psgraph-env
  current-pattern
  font-y
  font-h)

;(defstruct #:ps:psgraph-env
;  ctm
;  color
;  position
;  path
;  cpath
;  font
;  lwidth
;  lcap
;  ljoin
;  screen
;  transfer
;  flatness
;  mlimit
;  dash
;  device)

(defstruct #:ps:bitmap
  window
  bitstr)

;;
;;*****************************************************************************
;;**************************** display functions ******************************
;;*****************************************************************************
;;

(defun psprologue olist
  (when #:display:all-bitmaps
	(let* ((package (cassq '|PS| #:display:all-bitmaps))
	       (display (if (and package
				 (getfn1 package 'make)
				 (subtypep package 'display))
			    (new package))))
	                    ;; bitprologue has "else (new 'display)" - needed?
	  (ifn display
	       (error 'psprologue 'erroob "ps not initialized"))
	  (#:display:name display '|PS|)
	  (#:display:package display package)
	  (:psdefaults display)
	  (while (setq opt (nextl olist))
	    (let ((oname (car opt))
		  (oval  (cdr opt)))
	      (print "opt: " oname ", val: " oval)
	      (selectq oname
		       ((max scale) (:pssetmax display oval))
		       (orient (:psorient display oval))
		       ((file output) (:psfilename display oval))
		       (origin (:psorigin display oval))
		       (margins (:psmargins display oval))
		       ((paper format) (:pspapersize display oval))
		       (mxn-pages (:psmxnpages display oval))
		       (abbrev (:psabbrev display oval)))))
	  (:bitprologue display)
	  (newl #:display:all-displays display)
	  (#:display:prologuep display t)
	  (unless (current-display) (current-display display))
;;  do these further down in the code -- see work/x11.ll example
;;	  (catcherror () (make-named-color "red"))
;;	  (catcherror () (make-named-color "blue"))
;;	  (catcherror () (make-named-color "green"))
;;	  (catcherror () (make-named-color "yellow"))
;;	  (catcherror () (make-named-color "magenta"))
;;	  (catcherror () (make-named-color "cyan"))
;;	  (catcherror () (make-named-color "grey"))
;;	  (catcherror () (make-named-color "lightgrey"))
	  display)))

(defun :psdefaults (display)
  (let ((xmax 1151)
	(ymax 899))
    (:filename display (or (getenv "PSDISPLAY")
			   "psbv.ps"))
    (:abbrev display t)
    (:orient display 'h)
    (:origin display '(0 . 0))
    (:margins display (list 0 0 0 0))
    (:paper-size display (cons :a4h :a4v))
    (:mxn-pages display '(1 . 1))
    (#:display:xmax display xmax)
    (#:display:ymax display ymax)
    (:scale-factor display (:psscale display))))

;;
;;******************** some ps parameter init functions ********************
;;

(defun :pssetmax (display val)
  (let ((xval (car val))
	(yval (cdr val)))
    (when (and (numberp xval) (numberp yval))
	  (#:display:xmax display xval)
	  (#:display:ymax display yval))
    (:scale-factor display (:psscale display))))

(defun :psorient (display orientation)
  (let ((o (selectq (car orientation)
		    ((h hor l land landscape) 'h)
		    ((v ver p port portrait)  'v))))
    (ifn o (error 'psorient erroob orientation))
    (:orient display o)
    (:scale-factor display (:psscale display))))

(defun :psfilename (display fnl)
  (:filename display (car fnl)))

(defun :psorigin (display origin)
  (let ((xo (car origin))
	(yo (cdr origin)))
    (ifn (and (numberp xo) (numberp yo))
	 (error 'psorigin 'erroob origin))
    (:origin display origin)))

(defun :psmargins (display margins)
  (let ((lft (nextl margins))
	(top (nextl margins))
	(rgt (nextl margins))
	(btm (nextl margins)))
    (ifn (and (numberp lft) (numberp top)
	      (numberp rgt) (numberp btm))
	 (error 'psmargins 'erroob margin))
    (:margins display margins)))

;; must verify that the a4,a5,etc. values below (in mm) are correct

(defun :pspapersize (display type)
  (let ((orient (:orient display))
	(fpair (selectq type
			(a4 (cons :a4h :a4v))
			(a5 (cons :a5h :a5v)))))
    (if fpair
	(:paper-size display fpair)
      (error 'pspapersize 'erroob type))))

(defun :psmxnpages (display mn)
  (let ((m (car mn))
	(n (cdr mn)))
    (ifn (and (numberp m) (numberp n))
	 (error 'psmxnpages 'erroob mn))
    (:margins display mn)))

(defun :psabbrev (display ind)
  (let ((v (car ind)))
    (ifn (or (equal v t) (equal v ()))
	 (error 'psabbrev 'erroob ind))
  (:abbrev display v)))

(defun :psscale (display)
  (let* ((pagedim (:paper-size display))
	 (paperh (car pagedim))
	 (paperv (cdr pagedim))
	 (hpts (* 72 (/ paperh 25.4)))
	 (vpts (* 72 (/ paperv 25.4)))
	 (xmax (#:display:xmax display))
	 (ymax (#:display:ymax display)))
    (if (equal (:orient display) 'h)
	(min (/ hpts (1+ ymax))
	     (/ vpts (1+ xmax)))
      (min (/ hpts (1+ xmax))
	   (/ vpts (1+ ymax))))))


;;
;;******************************** etc. ********************************
;;

(defun :bitprologue (display)
  (unless (#:display:prologuep display)
	  (:initialize display)
	  (:init-color display)
	  (:init-font display)
	  (:init-line-style display)
	  (:init-pattern display)
	  (:init-window display)
	  (:init-psfile display)
	  (:reread display ())
	  display)))

(defun :initialize (display)
  (let* ((given-path (pathname (:filename display)))
	 (of-path (merge-pathnames #p".ps" given-path))
	 (hd-path (merge-pathnames #p".pr" given-path))
	 (dir-path (pathname (current-directory)))
	 (abs-of-path (absolute-path dir-path of-path))
	 (abs-hd-path (absolute-path dir-path hd-path))
	 (sys-abs-of-path (sys-abs-path-string abs-of-path))
	 (sys-abs-hd-path (sys-abs-path-string abs-hd-path)))
    (#:display:device display abs-of-path)
    (:filename display (file-namestring of-path))
    (:hchannel display (openo sys-abs-hd-path))
    (:ochannel display (openo sys-abs-of-path))

;;  the next four are hard-wired "pour l'instant"; they
;;  can be initialized with more active functions later.
;;  (ps printers will divulge their names in response to
;;   the printername command).

    (:printer-id display "apple-lw")
    (:engine-id display "canon")
    (:epsf-id display "epsf1.2")
    (:ps-id display "PS-Adobe-2.0")))

(defun :bitepilogue (display)
  (when (#:display:prologuep display)
          (:font-vector display ())
          (:line-style-vector display ())
          (:pattern-vector display ())
	  (:filename display ())
	  (#:display:colors display ())
	  (:psprint display "showpage")
	  (sys-merge-files (cadr (channel (:hchannel display)))
			      (cadr (channel (:ochannel display))))
          (close (:hchannel display))
	  (close (:ochannel display))))

(defun :bitmap-save (display)
  (:bitepilogue display))

(defun :bitmap-restore (display)
  (:bitprologue display))

(defun :bitmap-refresh (display)
  ())

(defun :bitmap-flush (display)
  ())

(defun :bitmap-sync (display)
  ())

;;
;;***************************** default functions *****************************
;;

(defun :standard-roman-font (display) 0)

(defun :standard-bold-font (display) 1)

(defun :small-roman-font (display) 2)

(defun :large-roman-font (display) 3)

(defun :standard-background-pattern (display) 0)

(defun :standard-foreground-pattern (display) 1)

(defun :standard-medium-gray-pattern (display) 2)

(defun :standard-light-gray-pattern (display) 3)

(defun :standard-dark-gray-pattern (display) 4)

(defun :standard-lelisp-cursor (display) 0)

(defun :standard-gc-cursor (display) 1)

(defun :standard-busy-cursor (display) 2)

;;
;;****************************** font functions ******************************
;;

(defun :init-font (display)
  (:font-vector display #[])
  (:font-metrics display #[])
  (let* ((default-name
	   (car (selectq (:printer-id display)
			 ("apple-lw" :apple-lw-fontnames)
			 ("qms-800"  :qms-800-fontnames)
			 ("linotronic-300" :lino-300-fontnames))))
	 (default-size 12))
    (:add-a-font display "font"
		 default-name
		 default-size)
    (:add-a-font display "attributefont"
		 (:attribute-font-name default-name)
		 default-size)
    (:add-a-font display "smallfont"
		 default-name
		 (:smaller-font default-size))
    (:add-a-font display "largefont"
		 default-name
		 (:larger-font default-size))
    (:add-a-font display "boldfont"
		 (:attribute-font-name default-name)
		 default-size)
    (:add-a-font display "italicfont"
		 (:italic-font-name default-name)
		 default-size)
    (:add-a-font display "bolditalicfont"
		 (:bolditalic-font-name default-name)
		 default-size)))

(defun :font-family (namestring)
  (let ((where (scanstring namestring "-0123456789")))
    (substring namestring 0 where)))

(defun :font-namestr (namestring)
  (let ((where (scanstring namestring "0123456789")))
    (substring namestring 0 where)))

(defun :font-sizestr (namestring)
  (let ((where (scanstring namestring "0123456789")))
    (substring namestring where (slen namestring))))

(defun :attribute-font-name (family)
  (catenate family "-Bold"))

(defun :italic-font-name (family)
  (catenate family
	    (if (equal family "Times")
		"-Italic"
	      "-Oblique")))

(defun :bolditalic-font-name (family)
  (catenate family
	    (if (equal family "Times")
		"-BoldItalic"
	      "-BoldOblique")))

(defun :smaller-font (size)
  (let ((small-size (truncate (* 0.75 size))))
    (if (= 0 (logand small-size #$1))
	small-size
      (1- small-size))))

(defun :larger-font (size)
  (let ((large-size (truncate (* 1.5 size))))
    (if (= 0 (logand large-size #$1))
	large-size
      (1+ large-size))))

(defun :add-a-font (display item fname fsize)
  (let* ((dfile-list (get-ps-default display "lelisp" item))
	 (name (or (car dfile-list) fname))
	 (size (or (cadr dfile-list) fsize))
	 (full-string (catenate name size))
	 (ge (#:display:graph-env display)))
    (#:display:font-names display
			  (acons full-string
				 (:load-font display ge full-string)
				 (#:display:font-names display)))))

(defun :font-max (display ge)
  (sub1 (vlength (:font-vector display))))

(defun :load-font (display ge fontname)
  (let ((nbfont (vlength (:font-vector display))))
    (ifn (member (:font-namestr fontname) :apple-lw-fontnames)
	 (error ':load-font 'erroob fontname))
    (:font-vector display
		  (bltvector (makevector (add1 nbfont) 0)
			     0 (:font-vector display) 
			     0 nbfont))
    (vset (:font-vector display) nbfont fontname)
    (:font-metrics display
		   (bltvector (makevector (add1 nbfont) 0)
			      0 (:font-metrics display) 
			      0 nbfont))
    (vset (:font-metrics display)
	  nbfont
	  (:read-font-metrics fontname))
    nbfont))

(defun :current-font (display ge font-index))

(defun :read-font-metrics (fname)
  (let* ((family (:font-family fname))
	 (name (:font-namestr fname))
	 (inch (openi (catenate :ps-font-dir family "/" name ".pafm")))
	 (mvector (makevector 96 ())))
    (ifn inch
	 (error ':read-font-metrics 'erroob fname)
       (with ((inchan inch))
	     (for (i 0 1 95)
		  (vset mvector i (read)))))
    (close inch)
    mvector))

;;
;;  Font metrics files are stored in $PSDIR/afm/font-family/font-name.afm
;;

(defun :ps-font-ascent (display font)
  (let ((size (:fsize display font))
	(mxvect (vref (:font-metrics display) font)))
    (sub size (:round (* (/ (vref mxvect 0) 1000) size)))))

(defun :round (n)
  (let ((ntrunc (truncate n)))
    (if (>= (- n ntrunc) 0.5)
	(1+ ntrunc)
      ntrunc)))

(defun :fsize (display font)
  (let ((#:system:print-for-read ()))
    (implode
     (explode
      (:font-sizestr
       (car (rassq font (#:display:font-names display))))))))

(defun :width-substring (display ge string start length)
  (let* ((the-string (:no-control (substring string start length)))
	 (findex (#:graph-env:font ge))
	 (fname (vref (:font-vector display) findex))
	 (fsize (:fsize display findex))
	 (width
	  (if (equal (:font-family fname) "Courier")
	      (mul (slen the-string) 600)
	    (let ((fm (vref (:font-metrics display) findex))
		  (expl (explode the-string)))
	      (apply '+ (mapcar '(lambda (x)
				   (car (vref fm x))) expl))))))
    (:round (* fsize (/ width 1000)))))

(defun :no-control (str)
  (let ((lim (1- (slen str)))
	(cleanstr "")
	cn)
    (for (i 0 1 lim)
	 (when (ge (setq cn (sref str i)) 32)
	       (setq cleanstr (catenate cleanstr (ascii cn)))))
    cleanstr))

(defun :height-substring (display ge string start length)
  (:fsize display (current-font)))

(defun :x-base-substring (display ge string start length)
  (let* ((the-string (:no-control (substring string start length)))
	 (findex (#:graph-env:font ge))
	 (fname (vref (:font-vector display) findex))
	 (fsize (:fsize display findex))
	 (charcn (sref string start))
	 (cindex (differ charcn 31))
	 (mxvect (vref (:font-metrics display) findex)))
    (:round
     (*
      (/ (cdr (vref (:font-metrics display) cindex)) 1000) fsize))))

(defun :y-base-substring (display ge string start length)
  (:ps-font-ascent display (current-font)))

(defun :x-inc-substring (display ge string start length)
  (differ (:width-substring display ge string start length)
	  (:x-base-substring display ge string start 0)))

(defun :y-inc-substring (display ge string start length)
    0)

(defun :font-ascent (display)
  (:ps-font-ascent display (current-font)))

(defun :font-height (display)
  (:fsize display (current-font)))

;;
;;*************************** line-style functions ***************************
;;

(defun :init-line-style (display)
  (:line-style-vector display
			#[(0) (0 2) (0 3) (0 4 2) (0 3 5) (0 2 1) (0 1 2)]))

(defun :line-style-max (display ge)
    (sub1 (vlength (:line-style-vector display))))

(defun :current-line-style (display ge line-style))

;;
;;***************************** pattern functions *****************************
;;

(defun :init-pattern (display)
    (:pattern-vector display #[])
    (:add-a-pattern display 16 16
                   #[#*0000 #*0000 #*0000 #*0000
                   #*0000 #*0000 #*0000 #*0000
                   #*0000 #*0000 #*0000 #*0000
                   #*0000 #*0000 #*0000 #*0000])
    (:add-a-pattern display 16 16
                   #[#*ffff #*ffff #*ffff #*ffff
                   #*ffff #*ffff #*ffff #*ffff
                   #*ffff #*ffff #*ffff #*ffff
                   #*ffff #*ffff #*ffff #*ffff])
    (:add-a-pattern display 16 16
                   #[#*aaaa #*5555 #*aaaa #*5555
                   #*aaaa #*5555 #*aaaa #*5555
                   #*aaaa #*5555 #*aaaa #*5555
                   #*aaaa #*5555 #*aaaa #*5555])
    (:add-a-pattern display 16 16
                   #[#*8888 #*2222 #*8888 #*2222
                   #*8888 #*2222 #*8888 #*2222
                   #*8888 #*2222 #*8888 #*2222
                   #*8888 #*2222 #*8888 #*2222])
    (:add-a-pattern display 16 16
                   #[#*7777 #*dddd #*7777 #*dddd
                   #*7777 #*dddd #*7777 #*dddd
                   #*7777 #*dddd #*7777 #*dddd
                   #*7777 #*dddd #*7777 #*dddd]))

(defun :add-a-pattern (display w h bits)
    (let ((bitmap (#:bitmap:make)))
      (#:bitmap:w bitmap w)
      (#:bitmap:h bitmap h)
      (#:bitmap:display bitmap display)
      (:create-bitmap display bitmap)
      (#:bitmap:bits bitmap bits)
      (:make-pattern display (#:display:graph-env display) bitmap)))

(defun :pattern-max (display ge)
    (sub1 (vlength (:pattern-vector display))))

(defun :make-pattern (display ge bitmap)
    (let ((nbpattern (vlength (:pattern-vector display))))
      (:pattern-vector display
                       (bltvector (makevector (add1 nbpattern) 0)
                                  0 (:pattern-vector display) 
                                  0 nbpattern))
      (vset (:pattern-vector display) nbpattern bitmap)
      nbpattern))

(defun :current-pattern (display ge pattern))

;;
;;***************************** color functions ******************************
;;

(defun :init-color (display)
  (let ((forename (get-ps-default (:filename display)
				  "lelisp"
				  "foreground"))
	(backname (get-ps-default (:filename display)
				  "lelisp"
				  "background"))
	(lelispreversevideo
	 (get-ps-default (:filename display) "lelisp" "reversevideo"))
	(fore (#:color:make))
	(back (#:color:make))
	planemask)
    (when (eqstring forename "")
	  (setq forename "black"))
    (when (eqstring backname "")
	  (setq backname "white"))
    (when (eqstring lelispreversevideo "on")
	  (setq forename "white" backname "black"))
    (#:color:name fore forename)
    (#:color:display fore display)
    (:make-named-color display fore forename)
    (#:color:name back backname)
    (#:color:display back display)
    (:make-named-color display back backname)
    (#:display:foreground display fore)
    (#:display:background display back)

;;
;;  PostScript color stuff will go here. The mode-vector will be used
;;  for memory-to-memory bitblt a` la X11.
;;
    (:mode-vector display #[0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15])))


(defun :make-named-color (display color s)
  (let ((c (:make-named-pscolor display color s)))
    (when (eq c -1)
	  (error ':make-named-color 'errnomorecolors s))
    (:do-color display color c)))

;;
;;  this is a very temporary hack, full PS color coming soon.
;;

(defun :make-named-pscolor (display color name)
  (cond ((eqstring name "black") 
	 (#:color:red color 0)
	 (#:color:green color 0)
	 (#:color:blue color 0)
	 0)
	((eqstring name "white")
	 (#:color:red color 32767)
	 (#:color:green color 32767)
	 (#:color:blue color 32767)
	 1)
	((eqstring name "grey")
	 (#:color:red color 24000)
	 (#:color:green color 24000)
	 (#:color:blue color 24000)
	 2)
	(t (- 1))))

(defun :do-color (display color pscolor)
  (or (any (lambda (c) (when (eq (:pscolor c) pscolor) c))
	   (#:display:colors display))
      (progn
	(:pscolor color pscolor)
	(#:display:colors display (cons color (#:display:colors display)))
	color)))

;;
;;***************************** window functions *****************************
;;

(defun :init-window (display)
  (let ((root (#:window:make))
	(extend (#:ps:extend:make))
	ge)
    (#:window:left root 0)
    (#:window:top root 0)
    (#:window:width root (#:display:xmax display))
    (#:window:height root (#:display:ymax display))
    (#:window:title root "root-window")
    (#:window:hilited root 0)
    (#:window:visible root 1)
    (#:window:display root display)
    (#:window:extend root extend)
    (:pswindow root t)
    (:create-graph-env display root)
    (setq ge (#:window:graph-env root))
    (#:display:root-window display root)
    (#:display:main-graph-env display ge)
    (#:graph-env:foreground ge (#:display:foreground display))
    (#:graph-env:background ge (#:display:background display))
    (#:display:window display root)
    (#:display:windows display (list root))
    (#:display:graph-env display (#:display:main-graph-env display))
    (:ps-current-ge display ge)))

(de :top-window (win)
    (if (#:window:father win)
        (:top-window (#:window:father win))
      win))

(defun :create-window (display win)
  (let ((x (#:window:left win))
	(y (#:window:top win))
	(w (#:window:width win))
	(h (#:window:height win))
	(ti (#:window:title win))
	(hi (#:window:hilited win))
	(vi (#:window:visible win)))
    (#:window:extend win (#:ps:extend:make))
    (:pswindow win t)
    (send 'set-window-hints win)
    (:create-graph-env display win)
    win))

(de :create-subwindow (display win)
    (let ((x (#:window:left win))
          (y o(#:window:top win))
          (w (#:window:width win))
          (h (#:window:height win))
          (vi (#:window:visible win))
          (father (#:window:father win)))
      (#:window:extend win (#:ps:extend:make))
      (:pswindow win t)
      (:create-graph-env display win)
      win))

(defun :current-window (display win))

;;(de :current-window (display win)
;;;;    (:drawing-flag display (:visible win))
;;    (let ((ge (#:window:graph-env win)))
;;      (:psprint display 
;;		(#:image:rectangle:x (:view-rect ge))
;;		(#:image:rectangle:y (:view-rect ge))
;;		(#:image:rectangle:w (:view-rect ge))
;;		(#:image:rectangle:h (:view-rect ge))
;;		(add (:offset-x win) (#:window:left (:top-window win)))
;;		(add (:offset-y win) (#:window:top (:top-window win)))
;;		'curwindow)
;;      (unless (send 'father win)
;;	      (let ((w (#:image:rectangle:w (:view-rect ge)))
;;		    (h (#:image:rectangle:h (:view-rect ge))))
;;	      (:psrect display
;;		       0 0 w h ())))))

(de :uncurrent-window (display win))
;;    (:drawing-flag display ()))

(de :modify-window (display win x y w h title hilited visible)
    (when x (#:window:left win x))
    (when y (#:window:top win y))
    (when w (#:window:width win w))
    (when h (#:window:height win h))
    (when title (#:window:title win title))
    (when hilited (#:window:hilited win hilited))
    (when visible (#:window:visible win visible))
    (:set-clip-for-subwindows display win)
    (when (eq win (#:display:window display))
          (:pscurwindow display x y w h win))
    (send 'modify-window-hints win))

(de :update-window (display win x y w h)
    (when x (#:window:left win x))
    (when y (#:window:top win y))
    (when w (#:window:width win w))
    (when h (#:window:height win h))
    (:set-clip-for-subwindows display win)
    (when (eq win (#:display:window display))
          (:pscurwindow display x y w h win))
    (send 'modify-window-hints win))

(de :kill-window (display win)
    (:pswindow win ())
    (:psgraph-env win ()))

(de :pop-window (display win))

(de :move-behind-window (display win1 win2))

(de :current-keyboard-focus-window (display win))

(de :uncurrent-keyboard-focus-window (display win))

(de :find-window (display x y))

(de :map-window (display win :x :y :lx :ly))

;;
;;********************** graphics environment functions **********************
;;

(defun :create-graph-env (display win)
  (let ((ge (#:graph-env:make)))
    (#:window:graph-env win ge)
    (#:graph-env:pattern ge 1)
    (#:graph-env:clip-x ge 0)
    (#:graph-env:clip-y ge 0)
    (#:graph-env:clip-w ge (#:window:width win))
    (#:graph-env:clip-h ge (#:window:height win))
    (when (#:display:main-graph-env display)
	  (#:graph-env:foreground ge
				  (#:graph-env:foreground
				   (#:display:main-graph-env display)))
	  (#:graph-env:background ge
				  (#:graph-env:background
				   (#:display:main-graph-env display))))
    (#:graph-env:display ge display)
    (#:graph-env:extend ge (#:window:extend win))
    (let ((font (#:graph-env:font ge))
	  (line-style (vref (:line-style-vector display)
			    (#:graph-env:line-style ge)))
	  (pattern (vref (:pattern-vector display) (#:graph-env:pattern ge)))
	  (mode (vref (:mode-vector display) (#:graph-env:mode ge))))
;;  the following will initialize a PostScript ge ..
      (:psgraph-env win t)
      (:font-y ge (:ps-font-ascent display font))
      (:font-h ge (:fsize display font)))
    (:view-rect ge (#:image:rectangle:make))
    (:set-clip-for-window display win)
    ge))

;;  This function will take on more and more importance as the
;;  bv->ps model becomes more clear to me. (Some of the extend stuff
;;  will be integrated into it.)

(defun :create-psgraph-env (args)
  ())

;;
;;******************************** clipping ********************************
;;

(de :set-clip-for-window (display win)
    (:set-offsets win)
    (:set-view-rectangle win)
    (:set-visibility win))

(de :set-clip-for-subwindows (display win)
    (:set-clip-for-window display win)
    (mapc (lambda (w) (:set-clip-for-subwindows display w))
          (#:window:subwindows win)))

(de :intersect (rect x y w h)
    (let ((x0 (#:image:rectangle:x rect))
          (y0 (#:image:rectangle:y rect))
          (w0 (#:image:rectangle:w rect))
          (h0 (#:image:rectangle:h rect))
          (u (add x w))
          (v (add y h)))
      (#:image:rectangle:x rect (or (gt x0 x) x)) 
      (#:image:rectangle:y rect (or (gt y0 y) y)) 
      (#:image:rectangle:w rect (sub (or (lt (add x0 w0) u) u)
                                     (#:image:rectangle:x rect)))
      (#:image:rectangle:h rect (sub (or (lt (add y0 h0) v) v)
                                     (#:image:rectangle:y rect)))
      rect))

(de :set-offsets (win)
    (let ((father (#:window:father win)))
      (:offset-x win (if father
                         (add (#:window:left win) (:offset-x father))
                       0))
      (:offset-y win (if father
                         (add (#:window:top win) (:offset-y father))
                       0))))

(de :set-view-rectangle (win)
    (let* ((ge (#:window:graph-env win))
           (fa (#:window:father win))
           (rect (:view-rect ge)))
      (#:image:rectangle:x rect 0)
      (#:image:rectangle:y rect 0)
      (#:image:rectangle:w rect (#:window:width win))
      (#:image:rectangle:h rect (#:window:height win))
      (:view-rect ge (:intersect rect
                                 (#:graph-env:clip-x ge)
                                 (#:graph-env:clip-y ge)
                                 (#:graph-env:clip-w ge)
                                 (#:graph-env:clip-h ge)))
      (when fa
            (:view-rect ge
                        (:intersect rect
                                    (sub (#:image:rectangle:x 
                                          (:view-rect (#:window:graph-env fa)))
                                         (#:window:left win))
                                    (sub (#:image:rectangle:y 
                                          (:view-rect (#:window:graph-env fa)))
                                         (#:window:top win))
                                    (#:image:rectangle:w 
                                     (:view-rect (#:window:graph-env fa)))
                                    (#:image:rectangle:h 
                                     (:view-rect (#:window:graph-env fa))))))))

(de :set-visibility (win)
    (let ((father (#:window:father win))
          (viewrect (:view-rect (#:window:graph-env win))))
      (:visible win (if father 
                        (and (:visible father)
                             (eq 1 (#:window:visible win))
                             (gt (#:image:rectangle:w viewrect) 0)
                             (gt (#:image:rectangle:h viewrect) 0))
                      (eq 1 (#:window:visible win))))))

;;
;;***************************** drawing functions *****************************
;;

(defun :draw-cn (display ge x y cn)
  (:psge display)
  (let ((ymax (#:display:ymax display)))
    (:psprint display
	      x (:pstransy y ymax) 'moveto
	      (:psstring (ascii cn))
	      'show))
  0)

(defun :draw-substring (display ge x y string start length)
  (:psge display)
  (let* ((sl (slen string))
	 (maxle (sub sl start)))
    (when (le start sl)
	  (if (gt length maxle) (setq length maxle))
	  (let ((the-string (substring string start length))
		(ymax (#:display:ymax display)))
	    (:psprint display
		      x (:pstransy y ymax) 'moveto
		      (:psstring the-string)
		      'show))))
  0)

(defun :draw-point (display ge x y)
  (:psge display)
  (let ((ymax (#:display:ymax display)))
    (:psprint display
	      'newpath x (:pstransy y ymax) 'moveto))
  (:psprint display
	    0 1 'rlineto)
  (:psprint display
	    1 0 'rlineto)
  (:psprint display
	    0 -1 'rlineto)
  (:psprint display
	    'closepath)
  (:psblackfill display)
  0)
	    
(defun :draw-polymarker (display ge n vx vy)
  (let ((ymax (#:display:ymax display)))
    (for (i 0 1 (sub1 n))
	 (:draw-point display
		      (vref vx i)
		      (:pstransy (vref vy i) ymax))))
  0)

(defun :draw-line (display ge x0 y0 x1 y1)
  (:psge display)
  (let ((ymax (#:display:ymax display)))
    (:psprint display
	      'newpath)
    (:psprint display
	      x0 (:pstransy y0 ymax) 'moveto)
    (:psprint display
	      x1 (:pstransy y1 ymax) 'lineto)
    (:psprint display
	      'stroke))
  0)

(defun :draw-rectangle (display ge x y w h)
  (:psrect display x y w h ())
  0)

(defun :draw-polyline (display ge n vx vy)
  (:pspolyline display n vx vy ())
  0)

(defun :draw-ellipse (display ge x y rx ry)
  (:psellipse display x y rx ry ())
  0)

(defun :draw-circle (display ge x y r)
  (:pscircle display x y r ())
  0)

;;
;;****************************** fill functions ******************************
;;

(defun :fill-rectangle (display ge x y w h)
  (:psrect display x y w h t)
  0)
	    
(defun :fill-area (display ge n vx vy)
  (:pspolyline display n vx vy t)
  0)

(defun :fill-ellipse (display ge x y rx ry)
  (:psellipse display x y rx ry t)
  0)

(defun :fill-circle (display ge x y r)
  (:pscircle display x y r t)
  0)

;;
;;********************************** bitmaps **********************************
;;

(defun :create-bitmap (display bitmap)
  (#:bitmap:extend bitmap (#:ps:bitmap:make))
  (:psbitstr bitmap (makestring 
		     (mul (#:bitmap:h bitmap)
			  (mul 2 (div (add 7 (#:bitmap:w bitmap)) 8)))
		     48))
  bitmap)

(defun :create-window-bitmap (display window bitmap)
  (#:bitmap:extend bitmap (#:ps:bitmap:make))
  (:window-bitmap bitmap window)
  bitmap)


(defun :kill-bitmap (display bitmap)
  ())

(defun :bmref (display bitmap i j) 
  (let* ((pos (add (mul (mul (div (add 7 (#:bitmap:w bitmap)) 8) 2) j) 
		   (div i 4)))
	 (digit (sref (:psbitstr bitmap) pos)))
    (if (load-byte-test 
	 (:ascii-to-digit digit)
	 (sub 3 (modulo i 4)) 1)
	1 0)))


(defun :ascii-to-digit (c)
  (let ((v (sub c 48)))
    (when (gt v 9)
	  (setq v (sub v 7)))
    (when (gt v 15)
	  (setq v (sub v 32)))
    (logxor 15 v)))

(defun :digit-to-ascii (v)
  (setq v (logxor v 15))
  (let ((c (add v 48)))
    (when (gt v 9)
	  (setq c (add c 7)))
    c))


(defun :bmset (display bitmap i j b)
  (let* ((line (:psbitstr bitmap))
	 (pos (add (mul (mul (div (add 7 (#:bitmap:w bitmap)) 8) j) 2) 
		   (div i 4))))
    (sset line pos
	  (:digit-to-ascii
	   (deposit-byte (:ascii-to-digit (sref line pos))
			 (sub 3 (modulo i 4)) 1 b)))))

(defun :get-bit-line (display bitmap y bitvector)
  (let ((w (#:bitmap:w bitmap))
	(wb (mul (div (add 7 (#:bitmap:w bitmap)) 8) 2))
	(psbits (:psbitstr bitmap)))
    (for (i 0 2 (sub1 wb))
	 (sset bitvector 
	       (div i 2)
	       (add (:ascii-to-digit 
		     (sref psbits (add (mul wb y) (add1 i))))
		    (mul 16 (:ascii-to-digit 
			     (sref psbits (add (mul wb y) i)))))))))

(defun :set-bit-line (display bitmap y bitvector)
  (let ((w (#:bitmap:w bitmap))
	(wb (mul (div (add 7 (#:bitmap:w bitmap)) 8) 2))
	(psbits (:psbitstr bitmap)))
    (for (i 0 2 (sub1 wb))
	 (sset psbits (add i (mul wb y))
	       (:digit-to-ascii (div (sref bitvector (div i 2)) 16)))
	 (sset psbits (add1 (add i (mul wb y)))
	       (:digit-to-ascii (modulo (sref bitvector (div i 2)) 16))))))

(defun :get-byte-line (display bitmap y bytevector)
  ())
(defun :set-byte-line (display bitmap y bytevector)
  ())

(defun :byteref (display bitmap x y)
  ())

(defun :byteset (display bitmap x y byte)
  ())

(de :bitblit (display b1 b2 x1 y1 x2 y2 w h)
    (if (:window-bitmap b2)
	(error 'bitblit "from pixmap, not implemented" b2)
      (if (:window-bitmap b1)
	  (let* ((win (:window-bitmap b1))
		 (ge (#:window:graph-env win))
		 (x0 (#:image:rectangle:x (:view-rect ge)))
		 (w0 (#:image:rectangle:w (:view-rect ge)))		 
		 (w2 (#:bitmap:w b2))
		 (xa (max x0 (sub x1 (min 0 x2))))
		 (wa (sub (min (add x0 w0)
			       (sub (add x1 w)
				    (max 0 (sub (add x2 w) w2))))
			  xa))
		 (xb (sub x1 x2))
		 (y0 (#:image:rectangle:y (:view-rect ge)))
		 (h0 (#:image:rectangle:h (:view-rect ge)))		 
		 (h2 (#:bitmap:h b2))
		 (ya (max y0 (sub y1 (min 0 y2))))
		 (ha (sub (min (add y0 h0)
			       (sub (add y1 h)
				    (max 0 (sub (add y2 h) h2))))
			  ya))
		 (yb (sub y1 y2)))
	    
	    (:pscurwindow display
			  xa ya wa ha 
			  win)
	    (let ((ymax (#:display:ymax display))
		  (bits (:psbitstr b2)))
	      (:psprint display
			xb (:pstransy yb ymax) 'translate
			w2 h2 'scale
			w2 h2 1 (:psarray (list w2 0 0 (- h2) 0 0))
			(:psexstr bits)
			'image))))
      (current-window (current-window))))

;;
;;****************************** menu functions ******************************
;;

(defun :menuwindow (le to wi he ti)
  ())

(defun :create-menu (display menu) 
  ())

(defun :kill-menu (display menu))

(defun :activate-menu (display menu x y)
  ())

(defun :menu-insert-item (display menu choix index name active value))

(defun :menu-insert-item-list (display menu choix name active))

(defun :menu-delete-item-list (display menu choix))

(defun :menu-delete-item (display menu choix index))

(defun :menu-modify-item-list (display menu choix name active))

(defun :menu-modify-item (display menu choix index name active value))

;;
;;*****************************************************************************
;;********************************* utilities *********************************
;;*****************************************************************************
;;

;;
;;**************************** default file acess *****************************
;;

;;  The idea is to mimick the Xdefault retrieval function: search for the value
;;  associated with key in $HOME/.PSdefaults, and maybe some other files..

(defun get-ps-default (display appli key)
  ())

;;
;;****************************** ge maintenance ******************************
;;

(defun :psge (display)
  (let* ((dspwin (#:display:window display))
	 (winge (#:display:graph-env display))
	 (winfont (#:graph-env:font winge))
	 (winpat (#:graph-env:pattern winge))
	 (winline (#:graph-env:line-style winge))
	 (psge (:ps-current-ge display))
	 (psfont (#:graph-env:font psge))
	 (pspat (#:graph-env:pattern psge))
	 (psline (#:graph-env:line-style psge)))
    (unless (equal winfont psfont)
	    (:ps-newfont display winfont)
	    (#:graph-env:font psge winfont))
    (unless (equal winpat pspat)
	    (#:graph-env:pattern psge winpat))
    (unless (equal winline psline)
	    (:ps-newlinestyle display winline)
	    (#:graph-env:line-style psge winline))	    
    (:ps-newwin display dspwin)
    (:ps-current-ge display psge)))

(defun :ps-newfont (display font)
  (let* ((newfn (vref (:font-vector display) font))
	 (newname (:font-namestr newfn))
	 (newsize (:font-sizestr newfn)))
  (:font-y ge (:ps-font-ascent display font))
  (:font-h ge (:fsize display font))
  (:psprint display
	    newsize (concat '/ newname) 'fnt)))


(defun :ps-newlinestyle (display linestyle)
  (let* ((style (vref (:line-style-vector display) linestyle))
	 (offsetval (car style))
	 (arrayval (catenate "["
			     (if (null (cdr style))
				 ""
			       (format () "~{~A ~}" (cdr style)))
			     "]")))
    (:psprint display arrayval offsetval 'setdash)))

(defun :ps-newwin (display win)
  (let* ((ge (#:window:graph-env win))
	 (rect (:view-rect ge))
	 (x (#:image:rectangle:x rect))
	 (y (#:image:rectangle:y rect))
	 (w (#:image:rectangle:w rect))
	 (h (#:image:rectangle:h rect)))
    (:pscurwindow display
		  x y w h
		  win)
    (unless (send 'father win)
	    (:psprint display
		      'stroke))))

;;
;;******************************** printing ps ********************************
;;
;; Several of these tasks must be done in :bitepilogue to be properly done..
;; (e.g., CreationDate, BoundingBox, DocumentFonts, other Font comments, etc.)
;; They will soon be written into a temp file, which can be supplemented by
;; :bitepilogue..
;;

(defun :init-psfile (display)
  (with ((outchan (:hchannel display))
	 (rmargin 257)
	 (lmargin 0))
	(:pspreamble)
	(:psfont-widget)
	(:psinit-widget display)
	(:psctm-widget display)
	(when (:abbrev display)
	      (:pstransy-widget (#:display:ymax display))
	      (:psstart-widget)
	      (:psrestroot-widget)
	      (:pscurwindow-widget))
	(print "12 /Courier fnt")))

(defun :pspreamble ()
  (print (:pscomment 'bang (:ps-id display) (:epsf-id display)))
  (print (:pscomment 'pcnt "Creator:" "get-user-name-from-/etc/passwd"))
  (print (:pscomment 'pcnt "Title:" (:filename display)))
  (print (:pscomment 'pcnt "CreationDate:" (date)))
  (print (:pscomment 'pcnt "EndComments"))
  (print))

(defun :psinit-widget (display)
  (let ((scale (:scale-factor display)))
    (print "[] 0 setdash")
    (print (* 0.8 scale) " setlinewidth")
    ;; ? other ps init things here: linejoin, miterlimit, linecap, etc.
    (print)))

(defmacro :pstransy (y ymax)
  `(- (sub ,y (1+ ,ymax))))

(defun :pstransy-widget (ymax)
  (let ((fcnstr (:psfcn (list (add1 ymax) 'sub 'neg))))
    (prinf "~@{~A ~}" '/transy fcnstr 'bind 'def))
  (print))

(defun :psstart-widget ()
  (let ((fcnstr (:psfcn (list 'transy 'moveto))))
    (prinf "~@{~A ~}" '/startat fcnstr 'bind 'def))
  (print))

(defun :psfont-widget ()
  (let ((fcnstr (:psfcn (list 'findfont 'exch 'scalefont 'setfont))))
    (prinf "~@{~A ~}" '/fnt fcnstr 'bind 'def)
    (print)))

(defun :pscurwindow-widget ()
  (let ((fcnstr (:psfcn (list 'restoreroot 'neg 'translate 'newpath
			      '/height 'exch 'def '/width 'exch 'def
			      'startat
			      'width 0 'rlineto
			      0 'height 'neg 'rlineto
			      'width 'neg 0 'rlineto
			      0 'height 'rlineto
			      'clip)))) 
    (prinf "~@{~A ~}" '/curwindow fcnstr 'bind 'def))
  (print))

(defun :psrestroot-widget ()
  (let ((fcnstr (:psfcn (list 'rootmatrix 'setmatrix 'initclip))))
    (print "/rootmatrix matrix currentmatrix def")
    (prinf "~@{~A ~}" '/restoreroot fcnstr 'bind 'def))
  (print))

(defun :psctm-widget (display)
  (let* ((ymax (#:display:ymax display))
	 (sfact (:scale-factor display))
	 (orient (:orient display)))
    (prinf "~@{~A ~}" sfact sfact 'scale)
    (when (equal orient 'h)
	  (print 90 " rotate")
	  (print 50 " " (- (add 50 ymax)) " translate")
	  (print))
    (prinf "~@{~A ~}" '/rootmtrx 'matrix 'currentmatrix 'def)
    (print)))
    
(defun :pscomment (type . stuff)
  (catenate "%"
	    (cond ((equal type 'bang) "!")
		  ((equal type 'pcnt) "%")
		  ((equal type 'cntd) "%+"))
	    (ifn (null stuff)
		 (:list-to-strings stuff))))

(defun :pscurwindow (display x y w h win)
  (let ((ymax (#:display:ymax display))
	(xcorner (add (:offset-x win) (#:window:left (:top-window win))))
	(ycorner (add (:offset-y win) (#:window:top (:top-window win)))))
    (if (:abbrev display)
	(:psprint display
		  x y w h xcorner ycorner 'curwindow)
      (:psprint display
		'rootmtrx 'setmatrix 'initclip)
      (ifn (and (zerop xcorner) (zerop ycorner))
	   (:psprint display
		     xcorner (- ycorner) 'translate))
      (:psprint display
		'newpath)
      (:psprint display
		x (:pstransy y ymax) 'moveto)
      (:psprint display
		w 0 'rlineto)
      (:psprint display
		0 (- h) 'rlineto)
      (:psprint display
		(- w) 0  'rlineto)
      (:psprint display
		'closepath 'clip))))

(defun :pscircle (display x y r fill)
  (:psge display)
  (:psprint display
	    'newpath)
  (let ((ymax (#:display:ymax display)))
    (:psprint display
	      x (:pstransy y ymax) r
	      0 360 'arc)
    (if fill
	(:psfill display)
      (:psprint display 'stroke))))

(defun :psellipse (display x y xr yr fill)
  (:psge display)
  (:psprint display
	    '/sm 'matrix 'currentmatrix 'def)
  (let ((ymax (#:display:ymax display)))
    (:psprint display
	      'newpath x (:pstransy y ymax) 'translate))
  (:psprint display
	    xr yr 'scale)
  (:psprint display
	    0 0 1 0 360 'arc)
  (:psprint display
	    'sm 'setmatrix)

  (if fill
      (:psfill display)
    (:psprint display 'stroke)))

(defun :psrect (display x y w h fill)
  (:psge display)
  (:psprint display
	    'newpath)
  (let ((ymax (#:display:ymax display)))
    (:psprint display
	      x (:pstransy y ymax) 'moveto))
  (:psprint display
	    w 0 'rlineto)
  (:psprint display
	    0 h 'neg 'rlineto)
  (:psprint display
	    w 'neg 0 'rlineto)
  (:psprint display
	    'closepath)
  (if fill
      (:psfill display)
    (:psprint display 'stroke))))


(defun :pspolyline (display n vx ny fill)
  (:psge display)
  (let ((ymax (#:display:ymax display)))
    (:psprint display
	      'newpath)
    (:psprint display
	      (vref vx 0)
	      (:pstransy (vref vy 0) ymax)
	      'moveto)
    (for (i 1 1 (sub1 n))
	 (:psprint display
		   (vref vx i)
		   (:pstransy (vref vy i) ymax)
		   'lineto))
    (if fill
	(:psfill display)
      (:psprint display 'stroke))))

(defun :psfill (display)
  (let ((grays '((0 1) (1 0) (2 0.5) (3 0.75) (4 0.25)))
	(pattern (#:graph-env:pattern (:ps-current-ge display))))
    (if (le pattern 4)
	(:psprint display
		  'currentgray
		  (car (cassq pattern grays)) 'setgray
		  'fill
		  'setgray)
      (let* ((bmap (vref (:pattern-vector display) pattern))
	     (bw (#:bitmap:w bmap))
	     (bh (#:bitmap:h bmap))
	     (bstring (:psbitstring bmap)))
	(if (nequal bw bh)
	    (error 'current-pattern 'erroob bmap))
	;; load up the pssetscrlib.ps file
	(unless (:pssetscrlib display)
		(:add-pssetscrlib display)
		(:pssetscrlib display t))
	(:psprint display 'currentgray)
	(:psprint display 
		  1 'setgray)
	(:psprint display
		  bw bstring 'setpattern 'fill 1 'setgray 'setgray)))))

;;(defun :psstroke (display)
;;  (let ((ge (#:display:graph-env display)))
;;    (if (equal (#:graph-env:font ge) 1)
;;	(:psprint display 'stroke)
;;      (:psprint display 'currentgray 0 'setgray 'stroke 'setgray))))

(defun :add-pssetscrlib (display)
  (sys-merge-files (cadr (channel (:hchannel display)))
		      "./setscrlib.ps"))

(defun :psblackfill (display)
  (:psprint display
	    'currentgray 0 'setgray 'fill 'setgray))

(defun :pswhitefill (display)
  (:psprint display
	    'currentgray 1 'setgray 'fill 'setgray))

(defun :psstring (str)
  (let ((the-string str)
	(the-length (slength str))
	(nparens (+ (:noccurs #/( str)
		    (:noccurs #/)  str))))
    (if (gt nparens 0)
	(setq the-string (:psparens nparens the-string)))
    (catenate "(" the-string ")" )))

(defun :psparens (np str)
  (let* ((sl (slen str))
	 (new-str (makestring (+ sl np) #\SP))
	 (dst 0))
    (for (i 0 1 (1- sl))
	 (let ((cn (sref str i)))
	   (when (or (equal cn #/( )
		     (equal cn #/) ))
		 (sset new-str dst #/\ )
		 (incr dst))
	   (sset new-str dst cn)
	   (incr dst)))
    new-str))

(defun :noccurs (cn str)
  (let ((expl (explode str)))
    (- (length expl) (length (remove cn expl)))))
  
(defun :psfcn (l)
  (catenate "{ " (:list-to-strings l) "}"))
  
(defun :psexstr (s)
  (catenate "{<" s ">}"))
  
(defun :psarray (l)
  (catenate "[" (:list-to-strings l) "]"))
  
;; Beef this up so that it can deal with rather long lists.. and break lines
;; correctly, etc. It would be nice to have a "readable function" option..

(defun :list-to-strings (l)
  (format () "~{~A ~}" l))

(defun :psprint (psdispl . l)
  (with ((outchan (:ochannel psdispl))
	 (rmargin 257)
	 (lmargin 0))
	(print (:list-to-strings l))))

(defun :psfin (display)
  (:psprint display 'showpage)
  (flush))

;;
;;***************************** bitmap utilities *****************************
;;

(defun bitmap-to-str (bitmap)
  (let ((str))
    (mapvector (lambda (x)
		 (setq str (catenate str (bitv-to-str x))))
	       (#:bitmap:bits bitmap))
    str))

(defun bitv-to-str (bv)
  (let ((str))
    (for (i 0 1 (1- (slen bv)))
       (setq str (catenate str (format () "~@{~2,'0X~}" (sref bv i)))))
    str))

;;
;;*********************** absolute pathname calculation ***********************
;;

(defun absolute-path (cdpath fnpath)
  (let ((fn-dirlist (pathname-directory fnpath))
	(cd-dirlist (pathname-directory cdpath)))
    (cond ((member '#:pathname:wild fn-dirlist)
	   (error 'absolute-path 'erroob fn-dirlist))
	  ((or (equal '#:pathname:current (car fn-dirlist))
	       (null (pathname-directory fnpath)))
	   (merge-pathnames cdpath fnpath))
	  ((member '#:pathname:up fn-dirlist)
	   (make-pathname () ()
			  (resolve-rel cd-dirlist
				       (cut-rel () fn-dirlist))
			  (pathname-name fnpath)
			  (pathname-type fnpath)
			  ()))
	  (t (error 'absolute-path 'erroob "bad file or directory path")))))

;;
;;  note that any '#:pathname:up symbols will occur at the head
;;  of the list returned by (cut-rel). this justifies the simple-
;;  minded behavior of the next function (the length calculations
;;  and the append..).
;;

(defun resolve-rel (dirlist reldirlist)
  (ifn (equal '#:pathname:up (car reldirlist))
       (append dirlist reldirlist)
     (let* ((sans (remove '#:pathname:up reldirlist))
	    (sanslength (length sans))
	    (dirlength (length dirlist)))
       (if (<= dirlength sanslength)
	   sans
	 (append (firstn (- dirlength sanslength) dirlist) sans)))))

(defun cut-rel (hd tl)
  (let ((el0 (car (last hd)))
	(el1 (car tl)))
    (cond ((null tl) hd)
	  ((equal el1 '*)
	   (if (and (nequal el0 '*) (nequal el0 ()))
	       (cut-rel (firstn (1- (length hd)) hd) (cdr tl))
	     (cut-rel (append1 hd el1) (cdr tl))))
	  (t (cut-rel (append1 hd el1) (cdr tl))))))

;;
;;********************** system-dependent pathname stuff **********************
;;

;; these must one day be made robust (selectq on 'system-id)
;; right now they will work for unix (/root/dir0/../dirN/file.ext)

(defun sys-abs-path-string (abs-path)
  (catenate (format () "~{/~A~}" (pathname-directory abs-path))
	    "/"
	    (file-namestring abs-path)))

;; use file-namestring instead of the following fcn (??)

;;(defun sys-fn-string (filepath)
;;  (catenate (pathname-name filepath)
;;	    "."
;;	    (pathname-type filepath)))

(defun sys-merge-files (fn0 fn1)
  (ifn (and (stringp fn0) (stringp fn1))
       (error 'sys-merge-files 'erroob (cons fn0 . fn1))
    (let ((msg (format () "~@{~A ~}" 'cat fn1 '>> fn0 ";" 'mv fn0 fn1)))
			(comline msg))))

;;
;;****************************************************************************
;;**************************** waiting for DspPS *****************************
;;****************************************************************************
;;

;;
;;***************************** cursor functions *****************************
;;

(defun :standard-lelisp-cursor (display)
  ())

(defun :standard-gc-cursor (display)
  ())

(defun :standard-busy-cursor (display)
  ())

(defun :cursor-max (display)
  ())

(defun :make-cursor (display b1 b2 x y)
  ())

(defun :current-cursor (display cursor)
  ())

(defun :move-cursor (display x y)
  ())

(defun :draw-cursor (display ge x y st)
  ())

;;
;;****************************** window functions ****************************
;;


;;
;;****************************** event functions *****************************
;;

(defun :event-mode (display mode)
  ())

(defun :flush-event (display)
  ())

(defun :eventp (display)
  ())

(defun :read-event (display event)
  event)

(defun :peek-event (display event)
  event)

(defun :read-mouse (display event)
  event)

(defun :add-event (display event)
  ())

(defun :grab-event (display event)
  ())

(defun :ungrab-event (display)
  ())

(defun :itsoft-event (display)
  ())

;;
;;********************** graphical environment functions *********************
;;

(defun :clear-graph-env (display ge)
  (let* ((rect (:view-rect ge))
	 (x (#:image:rectangle:x rect))
	 (y (#:image:rectangle:y rect))
	 (w (#:image:rectangle:w rect))
	 (h (#:image:rectangle:h rect))
	 (restore (current-pattern)))
    (with ((current-pattern 0))
	  (:psrect display
		   x y w h t))
    (current-pattern restore))
  0)

(defun :current-clip (display ge x y w h)
  ())

;;
;;********************************** color **********************************
;;

(defun :make-color (display color r g b)
  ())

(defun :make-mutable-color (display color r g b)
  ())

(defun :kill-color (display c)
  ())

(defun :current-foreground (display ge fore)
  ())

(defun :current-background (display ge back)
  ())

(defun :red-component (display color red)
  ())

(defun :green-component (display color green)
  ())

(defun :blue-component (display color blue)
  ())

;;
;;**************************** mode functions (?) ****************************
;;

(defun :current-mode (display ge mode)
  ())

;;
;;****************************************************************************
;;********************************** macros **********************************
;;****************************************************************************
;;

(dmd :window-bitmap (bitmap . val)
     (if val
         `(#:ps:bitmap:window (#:bitmap:extend ,bitmap) ,. val)
       `(#:ps:bitmap:window (#:bitmap:extend ,bitmap))))

(defmacro :pswindow (win . val)
  (if val
         `(#:ps:extend:pswindow (#:window:extend ,win) ,. val)
       `(#:ps:extend:pswindow (#:window:extend ,win))))

(defmacro :psgraph-env (win . val)
  (if val
      `(#:ps:extend:psgraph-env (#:window:extend ,win) ,. val)
    `(#:ps:extend:psgraph-env (#:window:extend ,win))))

(defmacro :pscolor (c . val)
  (if val
      `(#:color:extend ,c ,. val)
    `(#:color:extend ,c)))

(defmacro :view-rect (ge . val)
  (if val
      `(#:ps:extend:view-rect (#:graph-env:extend ,ge) ,. val)
    `(#:ps:extend:view-rect (#:graph-env:extend ,ge))))

(defmacro :psbitstr (bitmap . val)
  (if val
      `(#:ps:bitmap:bitstr (#:bitmap:extend ,bitmap) ,. val)
    `(#:ps:bitmap:bitstr (#:bitmap:extend ,bitmap))))

(defmacro :current-font-name (ge . val)
  (if val
      `(#:ps:extend:current-font-name (#:graph-env:extend ,ge) ,. val)
    `(#:ps:extend:current-font-name (#:graph-env:extend ,ge))))

(defmacro :current-font-size (ge . val)
  (if val
      `(#:ps:extend:current-font-size (#:graph-env:extend ,ge) ,. val)
    `(#:ps:extend:current-font-size (#:graph-env:extend ,ge))))

(defmacro :current-pspattern (ge . val)
  (if val
      `(#:ps:extend:current-pspattern (#:graph-env:extend ,ge) ,. val)
    `(#:ps:extend:current-pspattern (#:graph-env:extend ,ge))))

(defmacro :current-linestyle (ge . val)
  (if val
      `(#:ps:extend:current-linestyle (#:graph-env:extend ,ge) ,. val)
    `(#:ps:extend:current-linestyle (#:graph-env:extend ,ge))))

(defmacro :offset-x (win . val)
  (if val
      `(#:ps:extend:offset-x (#:window:extend ,win) ,. val)
    `(#:ps:extend:offset-x (#:window:extend ,win))))

(defmacro :offset-y (win . val)
  (if val
      `(#:ps:extend:offset-y (#:window:extend ,win) ,. val)
    `(#:ps:extend:offset-y (#:window:extend ,win))))

(defmacro :visible (win . val)
  (if val
      `(#:ps:extend:visible (#:window:extend ,win) ,. val)
    `(#:ps:extend:visible (#:window:extend ,win))))

(defmacro :font-y (ge . val)
  (if val
      `(#:ps:extend:font-y (#:graph-env:extend ,ge) ., val)
    `(#:ps:extend:font-y (#:graph-env:extend ,ge))))

(defmacro :font-h (ge . val)
  (if val
      `(#:ps:extend:font-h (#:graph-env:extend ,ge) ., val)
    `(#:ps:extend:font-h (#:graph-env:extend ,ge))))

(dmd :psbitmap (bitmap . val)
     (if val
         `(#:bitmap:extend ,bitmap ,. val)
       `(#:bitmap:extend ,bitmap)))
