(*************************************************************************)
(*                                                                       *)
(*                     Projet      Formel                                *)
(*                                                                       *)
(*                            CAML                                       *)
(*                                                                       *)
(*************************************************************************)
(*                                                                       *)
(*                            Inria                                      *)
(*                      Domaine de Voluceau                              *)
(*                      78150  Rocquencourt                              *)
(*                            France                                     *)
(*                                                                       *)
(*************************************************************************)

(* dir_gram The grammar of directives and pragmas                        *)
(*          Michel Mauny                                                 *)

(*          (Last edit date : Tue Dec 20 16:40:21 1988)                  *)

#system module dir_gram;;

(********************************************)
(* Preliminaries: functions used in actions *)
(********************************************)

let eval_caml_pragma () =
    match eval_prag_syntax (parse_caml_pragma ())
    with dynamic (s: MLpragma) -> s
       | _ -> raise parse "syntax of pragmas is out of scope"
;;

let eval_caml_MLtype () =
    let gty =
    (try MLtype_to_gtype (parse_caml_typ ())
     with failure _ -> raise parse "Unknown type in default printer")
    in ML (MLapply (MLvar "default_printer", MLquote (dynamic gty),[]))
;;

let eval_macro_to_string e =
  match eval_prag_syntax e
  with dynamic (MLconst (mlstring name) : ML) -> name
     | dynamic (_ : ML) -> raise parse "file name must be a string"
     | _ -> raise parse "Ill typed macro (should be of type ML)"
;;

(*******************************)
(*                             *)
(*      The grammar itself     *)
(*                             *)
(*******************************)

grammar for values directives =

rule entry Commands =
    parse Literal "directive" (* To be executed in all modes *)
            -> MLdirective (eval_caml_pragma ())

        | Literal "pragma" -> MLpragma (eval_caml_pragma ())

        | Literal "autoload"; Literal "from"; Str_or_macro str;
          Straint_list str_l
            -> MLdirective
                (Pragmaexp
                   (MLapply
                    (MLvar "autoload",
                     MLconst(mlstring (abs_path str)),
                     [str_l])))

        | Literal "autoload"; Literal "grammar"; IDENT gname;
          Literal "with"; Entries_parsers epl;
          Literal "from"; Str_or_macro str; Literal ";;"
            -> MLdirective
                 (Pragmaexp
                  (MLapply
                    (MLvar "autoload_grammar",
                     MLconst(mlstring (abs_path str)),
                     [MLpair(MLconst(mlstring gname),
                             MLlist(uncons(rev epl)))])))

        | Literal "module"; IDENT name; Sig_imp sig
            -> MLpragma(Pragmaexp
               (if !module_flag 
                then raise parse "module already open"
                else MLapply(MLvar "begin_module", sig,[])))

        | Literal "system"; Literal "module"; IDENT name; Sig_imp sig
            -> MLpragma(Pragmaexp
               (if !module_flag 
               then raise parse "module already open"
               else MLapply(MLvar "begin_system_module", sig,[])))

        | Literal "end"; "module"; Sig_exp sig
            -> MLpragma(Pragmaexp
               (if not !module_flag
               then raise parse "no module open"
               else MLapply(MLvar "end_module", sig,[])))

        | Literal "use"; Str_or_macro str; Literal ";;"
            -> MLdirective
                (Pragmaexp(MLapply (MLvar "use",MLconst(mlstring str),[])))

        | Literal "use"; Literal "syntax"; Str_or_macro str; Literal ";;"
            -> MLdirective
                (Pragmaexp
                  (MLapply (MLvar "use_syntax",MLconst(mlstring str),[])))

        | Literal "load"; Str_or_macro str; Literal ";;"
            -> MLdirective
                (Pragmaexp(MLapply (MLvar "load", MLconst(mlstring str),[])))

        | Literal "compile"; Str_or_macro str; Literal ";;"
            -> MLdirective
                (Pragmaexp
                  (MLapply (MLvar "compile", MLconst(mlstring str),[])))

        | Literal "infix"; Str_as_id id; Literal ";;"
            -> MLdirective
                (Pragmaexp(MLapply (MLvar "infix",  id,[])))

        | Literal "uninfix"; Str_as_id id; Literal ";;"
            -> MLdirective
                (Pragmaexp(MLapply (MLvar "uninfix",  id,[])))

        | Literal "fast"; Literal "arith"; BOOL b; Literal ";;"
            -> MLdirective
                (Pragmaexp
                  (MLapply (MLvar "fast_arith",  MLconst(mlbool b),[])))

        | Literal "open"; Literal "compilation"; BOOL b; Literal ";;"
            -> MLdirective
                (Pragmaexp (MLapply (MLvar "open_compilation",
                                     MLconst(mlbool b),[])))

        | Literal "set"; Literal "default"; Literal "grammar"; Syntax s;
          Literal ";;"
            -> MLpragma
                (Pragmaexp (MLapply (MLvar "set_default_grammar", s,[])))

        | Literal "default"; Literal "grammar"; Literal ";;"
            -> MLdirective
                (Pragmaexp
                  (MLapply (MLvar "default_grammar", MLconst mlnull,[])))

        | Literal "printer"; Str_as_id id; Literal ";;"
            -> ML(MLapply (MLvar "new_printer", id,
                           [MLconst(mlstring "")]))

        | Literal "printer"; Str_as_id id; Literal "for"; Syntax_for_printer s;
          Literal ";;"
            -> ML(MLapply (MLvar "new_printer", id,[s]))

        | Literal "default"; Literal "printer"; Literal "for";
          Literal "type"; {eval_caml_MLtype ()} e ; Literal ";;" -> e

        | Literal "evaluation"; Literal  "strict"; Literal ";;"
            -> MLdirective (Pragmaexp
              #(if LAZY
                then <:CAML:Expr<MLapply (MLvar "evaluation",
                                          MLinj0 "Strict",[])>>
                else <:CAML:Expr<warning "ignored strict directive";
                                 MLinj0 "Strict">>))

        | Literal "evaluation"; Literal "lazy"; Literal ";;"
            -> MLdirective (Pragmaexp
              #(if LAZY
                then <:CAML:Expr<MLapply (MLvar "evaluation",
                                          MLinj0 "Lazy",[])>>
                else <:CAML:Expr<warning "ignored lazy directive";
                                 MLinj0 "Lazy">>))

        | Literal "evaluation"; Literal "parallel"; Literal ";;"
            -> MLdirective (Pragmaexp
              #(if LAZY
                then <:CAML:Expr<MLapply (MLvar "evaluation",
                                          MLinj0 "Parallel",[])>>
                else <:CAML:Expr<warning "ignored parallel directive";
                                 MLinj0 "Parallel">>))

        | Literal "evaluation"; Literal "("; Literal ")"; Literal ";;"
            -> MLdirective (Pragmaexp
              #(if LAZY
                then <:CAML:Expr<MLapply (MLvar "default_evaluation",
                                 MLconst(mlnull),[])>>
                else <:CAML:Expr<MLinj0 "Strict">>))

        | Literal "eval"; Literal "when"; Literal "print"; BOOL b; Literal ";;"
            -> MLdirective
                (Pragmaexp (MLapply
                             (MLvar "eval_when_print",MLconst(mlbool b),[])))

        | Literal "quit"; Literal ";;"
            -> (quit();
                MLpragma
                 (Pragmaexp (MLapply (MLvar "quit", MLconst mlnull,[]))))



and Str_as_id = parse IDENT id -> MLconst (mlstring id)

and Sig_imp =
    parse Literal ";;" ->
            MLquote (dynamic(([],[]): (MLdecl list) & (string list)))
        | Literal "using"; {parse_caml_import_list ()} imp; Literal ";;"
            -> imp


and Sig_exp =
    parse Literal ";;" -> MLquote(dynamic ([]: string list list))
        | Literal "with"; {parse_caml_export_list ()} imp; Literal ";;"
            -> imp

and Str_or_macro =
    parse STRING s -> s
        | Literal "#"; IDENT x
            -> eval_macro_to_string (MLvar x)


and Straint_list = parse {parse_caml_straint_list()} strl; Literal ";;" -> strl

and Syntax =
    parse Str_as_id id -> MLpair( id,MLconst(mlstring ""))
        | Str_as_id syntax_name; Literal ":" ; Str_as_id entr
            -> MLpair(syntax_name, entr)

and Syntax_for_printer =
    parse Str_as_id id -> id
        | IDENT syntax_name; Literal ":" ; IDENT entr
            -> MLconst(mlstring(syntax_name^":"^entr))

and Entries_parsers =
    parse Entry_parser ep -> [ep]
        | Entries_parsers epl; Literal "and"; Entry_parser ep ->ep::epl

and Entry_parser =
    parse Str_as_id str1; Literal "at"; Literal "entry"; Str_as_id str2
            -> MLpair (str2,str1)

;;

let parse_dir = (directives "Commands").Parse_raw;;

let parse_directive () =
    match eval_syntax (parse_dir())
    with dynamic (S: MLsyntax) -> S
       | _ -> system_error "parse_directive"
;;

#end module with value parse_directive and directives;;
