(* topdec evaluator *)

(*
$File: Interpreter/EvalTopdec.sml $
$Date: 1992/04/07 14:58:00 $
$Revision: 1.14 $
$Locker:  $
*)

(*$EvalTopdec:
	TOPDEC_GRAMMAR MODULE_DYNOBJECT CORE_DYNOBJECT EVALDEC
	FLAGS PRETTYPRINT REPORT CRASH EVALTOPDEC
 *)

functor EvalTopdec(structure Grammar: TOPDEC_GRAMMAR

		   structure ModuleDynObject: MODULE_DYNOBJECT
		     sharing type ModuleDynObject.var = Grammar.id
		         and type ModuleDynObject.excon = Grammar.excon
			 and type ModuleDynObject.sigid = Grammar.sigid
		         and type ModuleDynObject.strid = Grammar.strid
		         and type ModuleDynObject.strexp = Grammar.strexp
			 and type ModuleDynObject.funid = Grammar.funid
			 and type ModuleDynObject.longstrid = Grammar.longstrid

		   structure CoreDynObject: CORE_DYNOBJECT
		     sharing type ModuleDynObject.Env = CoreDynObject.Env
		         and type ModuleDynObject.StrEnv = CoreDynObject.StrEnv
			 and type CoreDynObject.strid = Grammar.strid

		   structure EvalDec: EVALDEC
		     sharing type EvalDec.dec = Grammar.dec
			 and type EvalDec.Env = CoreDynObject.Env

		   structure Flags: FLAGS

		   structure PP: PRETTYPRINT
		     sharing type ModuleDynObject.StringTree
		       		  = CoreDynObject.StringTree
				  = PP.StringTree

		   structure Report: REPORT
		     sharing type PP.Report = Report.Report

		   structure Crash: CRASH
		  ): EVALTOPDEC =
  struct
    type topdec = Grammar.topdec

    structure C = CoreDynObject
          and M = ModuleDynObject	(* abbreviation... *)

    type DynamicBasis = M.Basis

    type StringTree = M.StringTree
    val layoutDynamicBasis = M.layoutBasis

    infix cut
    val op cut = M.Cut

    infix //
    val op // = Report.//

    fun ifPresent _ None = None
      | ifPresent (f, x) (Some y) = Some(f(x, y))

    open Grammar

   (* debug: the suspension is to avoid expensive prettyprinting. *)

    fun debug(title, f: unit -> StringTree) =
      if Flags.DEBUG_EVALTOPDEC
      then Report.print(Report.line title // PP.reportStringTree(f()))
      else ()

    fun varsOfValdesc valdesc =
      case valdesc
	of VALDESC(_, id, _, valdesc_opt) =>
	     EqSet.insert id (case valdesc_opt
				of Some valdesc => varsOfValdesc valdesc
				 | None => EqSet.empty
			     )

    fun exconsOfExdesc exdesc =
      case exdesc
	of EXDESC(_, id, _, exdesc_opt) =>
	     EqSet.insert id (case exdesc_opt
				of Some exdesc => exconsOfExdesc exdesc
				 | None => EqSet.empty
			     )

   (* The evalXXX functions allow EvalDec.UNCAUGHT to escape. We catch
      it when we wrap up (below). *)

    fun evalStrdesc(IB, strdesc): M.IntEnv =
      case strdesc
	of STRDESC(_, strid, sigexp, strdesc_opt) =>
	     let
	       val I = evalSigexp(IB, sigexp)

	       val IE_opt = ifPresent (evalStrdesc, IB) strdesc_opt
	     in
	       (fn IE => case IE_opt of Some IE' => M.IE_plus_IE(IE, IE')
				      | None => IE
	       ) (M.singleIE(strid, I))
	     end

    and evalSpec(IB, spec): M.Int =
      case spec
	of VALspec(_, valdesc) =>
	     M.Vars_in_Int(varsOfValdesc valdesc)

	 | EXCEPTIONspec(_, exdesc) =>
	     M.Excons_in_Int(exconsOfExdesc exdesc)

	 | STRUCTUREspec(_, strdesc) =>
	     M.IE_in_Int(evalStrdesc(IB, strdesc))

	 | LOCALspec(_, spec1, spec2) =>
	     let
	       val I1 = evalSpec(IB, spec1)
	       val I2 = evalSpec(M.IB_plus_IE(IB, M.IE_of_Int I1), spec2)
	     in
	       I2
	     end

	 | OPENspec(_, list) =>
	     let
	       fun f (WITH_INFO(_, longstrid)) Is =
		 M.Int_plus_Int(Is, M.lookup_LongStrId_IB(IB, longstrid))
	     in
	       List.foldR f M.emptyInt list
	     end

	 | INCLUDEspec(_, list) =>
	     let
	       fun f (WITH_INFO(_, sigid)) Is =
		 M.Int_plus_Int(Is, M.lookup_SigId(M.G_of_IB IB, sigid))
	     in
	       List.foldR f M.emptyInt list
	     end

	 | SEQspec(_, spec1, spec2) =>
	     let
	       val I1 = evalSpec(IB, spec1)
	       val I2 = evalSpec(M.IB_plus_IE(IB, M.IE_of_Int I1), spec2)
	     in
	       M.Int_plus_Int(I1, I2)
	     end

	 | _ =>	M.emptyInt		(* EMPTYspec and those not
					   concerned with evaluation. *)

    and evalSigexp(IB, sigexp): M.Int =
      case sigexp
	of SIGsigexp(_, spec) =>
	     evalSpec(IB, spec)

	 | SIGIDsigexp(_, sigid) =>
	     M.lookup_SigId(M.G_of_IB IB, sigid)

    and evalStrexp(B, strexp): M.Env =
      case strexp
	of STRUCTstrexp(_, strdec) =>
	     evalStrdec(B, strdec)

	 | LONGSTRIDstrexp(_, longstrid) =>
	     M.lookup_LongStrId_B(B, longstrid)

	 | APPstrexp(_, funid, strexp) =>
	     let
	       val ((strid, I), (strexp', I'_opt), B') =
		 M.unClosure(M.lookup_FunId(M.F_of_B B, funid))

	       val E = evalStrexp(B, strexp)

	       val _ = debug("StrExp.E", fn () => C.layoutEnv E)

	       val _ = debug("StrExp.(E cut I)", fn () => C.layoutEnv(E cut I))

	       val E' =
		 evalStrexp(M.B_plus_SE(B', C.singleSE(strid, E cut I)),
			    strexp'
			   )

	       val _ = debug("StrExp.E'", fn () => C.layoutEnv E')
	     in
	       case I'_opt of Some I' => E' cut I'
			    | None => E'
	     end

	 | LETstrexp(_, strdec, strexp) =>
	     let
	       val E = evalStrdec(B, strdec)
	       val E' = evalStrexp(M.B_plus_E(B, E), strexp)
	     in
	       E'
	     end

    and evalStrbind(B, strbind): C.StrEnv =
      case strbind
	of STRBIND(_, strid, sigexp_opt, strexp, strbind_opt) =>
	     let
	       val E = evalStrexp(B, strexp)
	       val I_opt = ifPresent (evalSigexp, M.InterB B) sigexp_opt
	       val SE_opt = ifPresent (evalStrbind, B) strbind_opt
	     in
	       (fn SE => case SE_opt of Some SE' => C.SE_plus_SE(SE, SE')
				      | None => SE
	       ) (C.singleSE(strid,
			     case I_opt of Some I => E cut I | None => E
			    )
		 )
	     end

    and evalStrdec(B, strdec): M.Env =
      case strdec
	of DECstrdec(_, dec) =>
	     EvalDec.eval(M.E_of_B B, dec)

         | STRUCTUREstrdec(_, strbind) =>
	     C.SE_in_E(evalStrbind(B, strbind))

         | LOCALstrdec(_, strdec1, strdec2) =>
	     let
	       val E1 = evalStrdec(B, strdec1)
	       val B' = M.B_plus_B(B, M.E_in_B E1)
	       val E2 = evalStrdec(B', strdec2)
	     in
	       E2
	     end

	 | EMPTYstrdec _ =>
	     C.emptyE

	 | SEQstrdec(_, strdec1, strdec2) =>
	     let
	       val E1 = evalStrdec(B, strdec1)
	       val B' = M.B_plus_B(B, M.E_in_B E1)
	       val E2 = evalStrdec(B', strdec2)
	     in
	       C.E_plus_E(E1, E2)
	     end

    and evalSigbind(IB, sigbind): M.SigEnv =
      case sigbind
	of SIGBIND(_, sigid, sigexp, sigbind_opt) =>
	     let
	       val I = evalSigexp(IB, sigexp)
	       val G_opt = ifPresent (evalSigbind, IB) sigbind_opt
	     in
	       (fn G => case G_opt of Some G' => M.G_plus_G(G, G')
				    | None => G
	       ) (M.singleG(sigid, I))
	     end

    and evalSigdec(IB, sigdec): M.SigEnv =
      case sigdec
	of SIGNATUREsigdec(_, sigbind) =>
	     evalSigbind(IB, sigbind)

	 | EMPTYsigdec _ =>
	     M.emptyG

	 | SEQsigdec(_, sigdec1, sigdec2) =>
	     let
	       val G1 = evalSigdec(IB, sigdec1)
	       val G2 = evalSigdec(M.IB_plus_G(IB, G1), sigdec2)
	     in
	       M.G_plus_G(G1, G2)
	     end

    and evalFundec(B, fundec): M.FunEnv =
      case fundec
	of FUNCTORfundec(_, funbind) =>
	     evalFunbind(B, funbind)

	 | EMPTYfundec _ =>
	     M.emptyF

	 | SEQfundec(_, fundec1, fundec2) =>
	     let
	       val F1 = evalFundec(B, fundec1)
	       val F2 = evalFundec(M.B_plus_F(B, F1), fundec2)
	     in
	       M.F_plus_F(F1, F2)
	     end

    and evalFunbind(B, funbind): M.FunEnv =
      case funbind
	of FUNBIND(_, funid, strid, sigexp, sigexp_opt, strexp, funbind_opt) =>
	  let
	    val I = evalSigexp(M.InterB B, sigexp)

	    val _ = debug("FunBind.I", fn () => M.layoutInt I);

	    val I'_opt =
	      ifPresent
	      (fn ((), sigexp') =>
	         evalSigexp(M.IB_plus_IE(M.InterB B, M.singleIE(strid, I)),
			    sigexp'
			   ),
	       ()
	      ) sigexp_opt		(* Sorry about the `()' stuff; just
					   trying to match `ifPresent'... *)

	    val F_opt = ifPresent (evalFunbind, B) funbind_opt
	  in
	    (fn F => case F_opt of Some F' => M.F_plus_F(F, F')
				 | None => F
	    ) (M.singleF(funid, M.mkClosure((strid, I), (strexp, I'_opt), B)))
	  end

   (* export the following: *)
    type Pack = EvalDec.Pack
    exception UNCAUGHT = EvalDec.UNCAUGHT
    val pr_Pack = EvalDec.pr_Pack
    val RE_RAISE = EvalDec.RE_RAISE
    val FAIL_USE = EvalDec.FAIL_USE

    fun eval(B, topdec) =
      case topdec
	of STRtopdec(_, strdec) =>
	     M.E_in_B(evalStrdec(B, strdec))
					(* UNCAUGHT(p) might propagate. *)

	(* SIGtopdec and FUNtopdec always succeed at evaluation, since
	   they aren't generative. *)

	 | SIGtopdec(_, sigdec) =>
	     M.G_in_B(evalSigdec(M.InterB B, sigdec))

	 | FUNtopdec(_, fundec) =>
	     M.F_in_B(evalFundec(B, fundec))
  end;
