(*
$File: Compiler/CompileDec.sml $
$Date: 1992/12/18 09:00:42 $
$Revision: 1.3 $
$Locker:  $
*)

(*$CompileDec:
	LAB VAR CON SCON EXCON DEC_GRAMMAR RESIDENT LVARS LAMBDA_EXP
	COMPILER_ENV MATCH_COMPILER OVERLOADING_INFO 
	GRAMMAR_INFO TYPE_INFO FINMAP CRASH COMPILE_DEC
 *)

functor CompileDec(structure Lab: LAB	(* Clean way of getting `eqtype lab'. *)
		   structure Var: VAR	(* Ditto for var. *)
		   structure Con: CON   (* Ditto for con. *)
		   structure SCon: SCON	(* Ditto for scon. *)
		   structure Excon: EXCON	(* For excon->string. *)

		   structure Grammar: DEC_GRAMMAR
		     sharing type Grammar.lab = Lab.lab
		         and type Grammar.scon = SCon.scon
			 and type Grammar.excon = Excon.excon

		   structure ResIdent: RESIDENT
		     sharing type ResIdent.longid = Grammar.longid
		         and type ResIdent.longvar = Var.longvar
		         and type ResIdent.longcon = Con.longcon
			 and type ResIdent.longexcon = Excon.longexcon

		   structure Lvars: LVARS

		   structure LambdaExp: LAMBDA_EXP
		     sharing type LambdaExp.lvar = Lvars.lvar

		   structure CompilerEnv: COMPILER_ENV
		     sharing type CompilerEnv.var = Var.var
		         and type CompilerEnv.longvar = Var.longvar
		         and type CompilerEnv.excon = Excon.excon
			 and type CompilerEnv.longexcon = Excon.longexcon
			 and type CompilerEnv.lvar = Lvars.lvar

		   structure MatchCompiler: MATCH_COMPILER
		     sharing type MatchCompiler.pat = Grammar.pat
		         and type MatchCompiler.var = Var.var
			 and type MatchCompiler.con = Con.con
			 and type MatchCompiler.lab = Lab.lab
			 and type MatchCompiler.scon = SCon.scon
			 and type MatchCompiler.longexcon = Excon.longexcon
			 and type MatchCompiler.lvar = Lvars.lvar
			 and type MatchCompiler.CEnv = CompilerEnv.CEnv

		   structure OverloadingInfo: OVERLOADING_INFO
		   structure GrammarInfo: GRAMMAR_INFO
		     sharing type Grammar.info = GrammarInfo.PostElabGrammarInfo
		         and type MatchCompiler.TypeInfo = GrammarInfo.TypeInfo
			 and type GrammarInfo.OverloadingInfo = OverloadingInfo.info
		   structure TypeInfo: TYPE_INFO
		     sharing type TypeInfo.lab = Lab.lab
			 and type TypeInfo.longcon = Con.longcon
(* =-> *)	         and type TypeInfo.info = GrammarInfo.TypeInfo

(* =->		   val nj_sml_bug: GrammarInfo.TypeInfo -> TypeInfo.info *)
			(* Trying to share these types induces an internal
			   exception in the SML/NJ typechecker. As a workround,
			   we keep the types distinct and use this coercion. *)

		   structure FinMap: FINMAP
		     sharing type MatchCompiler.map
		                  = LambdaExp.map = FinMap.map

		   structure Crash: CRASH
		  ): COMPILE_DEC =
  struct
    open ResIdent MatchCompiler LambdaExp CompilerEnv
    open Grammar

    infix plus
    val (op plus) = CompilerEnv.plus

   (* A note about exceptions. Exceptions are a hassle. The exception
      constructors have to be generative, and have to be available
      for pattern matching; and yet, they must behave as values of
      type exn (for non-value-carrying constructors) or as closures
      with type 'a -> exn, so that the following will work:

	exception E of int
	val f = E
	... f 3 ...

      We represent an exception constructor as a string ref; the
      string isn't essential, but it carries the exception name for
      printing. (Note: this gives us the same problem as New Jersey,
      in that `exception B = A; ... raise B ...' will report the
      exception name as A; this could be fixed using a different
      representation, but this would probably slow down handler
      matching.) It has to be a ref because exception declarations are
      generative, and this is the only way to produce new objects with
      an identity equality relation. So, the compiler environment
      maps lvars to string refs. Exception packets are pairs of the
      form ('a, string ref) where the 'a is the value carried, if any;
      this is a canonical representation so that we can locate the
      determiner for the root of a handler decision tree, regardless
      of the kind of exception in the packet.
         The dynamic semantics keeps exceptions as a distinct kind of
      object, and treats application of exception constructors to
      values as a special operation (V4 #113); see EvalDec. We don't
      want this clutter in the application operation, so we transform
      occurrences of exception identifiers in atomic expressions into
      the appropriate form; non-value-carrying ones are turned into
      packets, and value-carrying ones are turned into closures of the
      form fn x => (x, ref name). We can optimise direct applications
      like E(3), of course. Hence, the TypeInfo must tell us, for each
      occurrence of an excon in an atomic expression, whether this
      excon carries a value (easy to determine from the type; see
      ElabDec). This is going to make the top-level bind printing
      routines slightly tricky if they want to print exception names
      from dynamic values. I think we'll punt on that one.
         This is my third attempt at an implementation scheme for
      exceptions. I think I've thought of everything this time. *)

   (* MEMO: temporary things to raise Match and Bind (not properly;
      we'll do that later). *)

   (*
    val raiseMatch = RAISE(pair(VOID, VAR NeededLvars.matchLvar))
    and raiseBind  = RAISE(pair(VOID, VAR NeededLvars.bindLvar))
    *)

    val raiseMatch = RAISE VOID and raiseBind = RAISE VOID

   (* These are the hooks provided by TypeInfo. *)

    local
      open TypeInfo
    in
      fun whichLab info: int =
	case (*nj_sml_bug*) info
	  of LAB_INFO{index} => index
	   | _ => Crash.impossible "CompileDec.whichLab"

      fun conInfo info =
	case (*nj_sml_bug*) info
	  of CON_INFO x => x
	   | _ => Crash.impossible "CompileDec.conInfo"

      fun carriesValue info =
	case GrammarInfo.getPostElabTypeInfo info
	  of Some(EXCON_INFO{functional}) => functional
	   | _ => Crash.impossible "CompileDec.carriesValue"

      fun whichCon info = #index(conInfo info)
      fun isFunctional info = #functional(conInfo info)
    end

    fun makeList (f: 'a -> 'a Option) (x: 'a) =
      x :: (case f x
	      of Some y => makeList f y
	       | None => nil
	   )

   (* Equality is always prim #0. Here is the equality function: *)

    fun equals(lamb1, lamb2) = PRIM_APP(0, pair(lamb1, lamb2))

   (* nullary/unaryCon: create a CON with a specific tag and a (possibly
      irrelevant) value. *)

    fun unaryCon(tag, value) = pair(value, INTEGER tag)
    fun nullaryCon tag = unaryCon(tag, VOID)

   (* conClosure: we've encountered a unary constructor, but not in the context
      of an application, so we have to build a trivial closure containing it. *)

    fun conClosure tag =
      let
	val lv = Lvars.newLvar()
      in
	FN(lv, unaryCon(tag, VAR lv))
      end

   (* Decompose the absyn argument in a `prim(n, xxx)' expression. Notice
      that we're extremely intolerant of any deviation from this precise
      syntax. *)

    fun decomposePrimArg atexp: int * exp =
      case atexp
	of RECORDatexp(_,
	     Some(EXPROW(_, _, ATEXPexp(_, SCONatexp(_, SCon.INTEGER i)),
			       Some(EXPROW(_, _, exp2, None))
			)
		 )
	   ) => (i, exp2)

	 | _ => Crash.impossible "CompileDec.decomposePrimArg"

   (* two kinds of failure from a compiled match: RAISEMATCH means raise the
      Match exception, and RAISESELF means raise the original packet. *)

    datatype FailType = RAISEMATCH | RAISESELF

   (* envOfDecTree - for `val <pat> = <exp>' bindings, we want the environment
      to pass out for use in the scope of the declaration. So, we guddle
      around the decision tree to look for it. This scheme will fail for
      general pattern matching tasks in things like `fn <match>' if a rule
      can be reached via two or more decision paths; but, we do the environment
      management for that differently anyway. *)

    fun envOfDecTree tree: CEnv =
      let
	open MatchCompiler
	exception Gotcha of CEnv

	fun envOfDecTree' tree: unit =	(* raise Gotcha(env) when found. *)
	  case tree
	    of LAB_DECOMPOSE{child, ...} => envOfDecTree' child
	     | CON_DECOMPOSE{child, ...} => envOfDecTree' child
	     | EXCON_DECOMPOSE{child, ...} => envOfDecTree' child

	     | CON_SWITCH{selections, wildcard, ...} =>
		 (case wildcard
		    of None => ()
		     | Some w => envOfDecTree' w;

		  FinMap.fold
		    (fn ((_, t), _) => envOfDecTree' t) ()
		    (selections:
		       ((*eqtype*) Con.con, (TypeInfo * DecisionTree))
		         FinMap.map
		    )
		 )

	     | SCON_SWITCH{selections, wildcard, ...} =>
		 (envOfDecTree' wildcard;	(* most likely case... *)

		  FinMap.fold
		    (fn (t, _) => envOfDecTree' t) ()
		    (selections:
		       ((*eqtype*) SCon.scon, DecisionTree) FinMap.map
		    )
		 )

	     | EXCON_SWITCH{selections, wildcard, ...} =>
		 (envOfDecTree' wildcard;	(* most likely case... *)
		  map (fn (_, t) => envOfDecTree' t) selections;
		  ()
		 )

	     | END{environment, ...} =>
		 raise Gotcha environment

	     | FAIL => ()
      in
	(envOfDecTree' tree; Crash.impossible "CompileDec.envOfDecTree")
	handle Gotcha e => e
      end

   (* ifThenElse assumes the canonical representation of booleans as
      datatypes with a value and a tag. (Must change this some day...)
      We simply do an integer switch on the tag. *)

    fun ifThenElse(cond, t, f) =
      let
	val selections =
	  List.foldL (fn (x, y) => fn z => FinMap.add(x, y, z))
	  	     FinMap.empty [(0, f), (1, t)]
      in
	SWITCH_I(SWITCH{arg=SELECT(1, cond),
			selections=selections,
			wildcard=None
		       }
		)
      end




    fun compileAtexp env atexp =
      case atexp
	of SCONatexp(_, SCon.INTEGER x) => INTEGER x
	 | SCONatexp(_, SCon.STRING x) => STRING x
	 | SCONatexp(_, SCon.REAL x) => REAL x

	 | IDENTatexp(info, OP_OPT(longid, _)) =>
	     (case longid
	        of LONGVAR longvar =>
		     (case lookupLongvar env longvar
		        of LVAR lv => VAR lv
			 | _ =>
			     Crash.impossible "compileAtexp(PRIM/overloaded op)"
					(* We can only handle `prim(n, ...)' *)
(* XXX Unfinished --- this code implies that 
`val leq = (op < : int * int -> ool)' fails to evaluate, as the
lookup of < returns LESS (and not a LVAR) *)
		     )

		 | LONGCON longcon =>
		     let
		       val ti =
			 case GrammarInfo.getPostElabTypeInfo info
			   of Some ti => ti
			    | None => Crash.impossible "compileAtExp(ti)"

		       val functional = isFunctional ti
		       val index = whichCon ti
		     in
		       if functional then
			 conClosure index
		       else
			 nullaryCon index
		     end

		 | LONGEXCON longexcon =>
		     let
		       val refLv = lookupLongexcon env longexcon
		     in		(* refLv is the lvar for the string ref. *)
		       if carriesValue info then
			 let
			   val lv = Lvars.newLvar()
			 in
			   FN(lv, pair(VAR lv, VAR refLv))
			 end
		       else
			 pair(VOID, VAR refLv)
		     end
	     )

	(* records: the fields must be evaluated in their textual order,
	   but the resulting record object must have the fields in a
	   canonical order (we adopt alphabetic ordering). Hmm. Tricky.
	   Easiest way is to bind the record field expressions to lvars
	   and then build a record of the (appropriately ordered) lvars. *)

	 | RECORDatexp(_, Some exprow) =>
	     let
	       val rows = makeList (fn EXPROW(_, _, _, e) => e) exprow
	       val labs = map (fn EXPROW(_, l, _, _) => l) rows
	       val lvars = map (fn _ => Lvars.newLvar()) rows
	       val exps = map (fn EXPROW(_, _, e, _) => compileExp env e) rows

	       val scope =		(* The final record expression *)
		 let
		   val sortedLvarsXlabs =
		     ListSort.sort
		       (fn (_, l1) => fn (_, l2) => Lab.<(l1, l2))
		       (ListPair.zip(lvars, labs))
		 in
		   VECTOR(map (fn (lv, _) => VAR lv) sortedLvarsXlabs)
		 end

	     in
	       List.foldR (General.curry Let) scope (ListPair.zip(lvars, exps))
	     end

         | RECORDatexp(_, None) => VOID

	 | LETatexp(_, dec, exp) =>
	     let
	       val (env1, f) = compileDec env (false, dec)
	     in
	       f(compileExp (env plus env1) exp)
	     end

	 | PARatexp(_, exp) => compileExp env exp

    and compileExp env exp =
      case exp
	of ATEXPexp(_, atexp) => compileAtexp env atexp

	 | APPexp(_,
		  f as ATEXPexp(_, IDENTatexp(info, OP_OPT(longid, _))),
		  arg
		 ) =>
			(* We have to spot direct application of "prim" - apart
			   from that, we don't have to bother with constructors
			   and the like. They'll compile to functions, but the
			   optimiser will spot them later. *)

	   let 

	     (* resolve (i) returns 0 if if the resolved type is int
                and ~1 if the resolved type is real; crashes otherwise *)
	     fun resolve i =
	       case (GrammarInfo.getPostElabOverloadingInfo i) of
		  None => Crash.impossible "CompileExp resolve 1"
		| Some (OverloadingInfo.RESOLVED_INT) => 0
		| Some (OverloadingInfo.RESOLVED_REAL) => 1
		| Some (OverloadingInfo.UNRESOLVED _) => 
		    Crash.impossible "CompileExp resolve unresolved"

	     fun make_prim i =
	       PRIM_APP(~((resolve info) + i), 
			compileAtexp env arg)

	   in
	     (case longid
	        of LONGVAR longvar =>
		     (case lookupLongvar env longvar
			of LVAR lv =>	(* Not a primitive... *)
			     APP(VAR lv, compileAtexp env arg)

			     (* For the overloaded operators a PRIM_APP(n, ...)
			        is made, with negative n (the numbers must
				correspond to those found in /Common/Apply.sml) *)
			 | ABS => make_prim 1
			 | NEG => make_prim 3
			 | PLUS => make_prim 5
			 | MINUS => make_prim 7
			 | MUL => make_prim 9
			 | LESS => make_prim 11
			 | GREATER => make_prim 13
			 | LESSEQ => make_prim 15
			 | GREATEREQ => make_prim 17

			 | PRIM =>	(* Application of `prim'. We must now
					   disassemble the argument to get
					   the prim number - it must be
					   possible to do at compile-time. *)
			     let
			       val (n, arg') = decomposePrimArg arg
			     in
			       PRIM_APP(n, compileExp env arg')
			     end
		     )

		 | _ => APP(compileExp env f, compileAtexp env arg)
	     )
	   end
	 | APPexp(_, f, arg) =>		(* non-trivial function expression... *)
	     APP(compileExp env f, compileAtexp env arg)

	 | TYPEDexp(_, exp, _) => compileExp env exp

	 | HANDLEexp(_, exp, match) =>
	     HANDLE(compileExp env exp,
		    compileMatch env (match, false, RAISESELF)
		   )

	 | RAISEexp(_, exp) =>
	     RAISE(compileExp env exp)

	 | FNexp(_, match) =>
	     compileMatch env (match, true, RAISEMATCH)

	 | UNRES_INFIXexp _ =>
	     Crash.impossible "compileExp(UNRES_INFIX)"

   (* compileMatch - compiles a match into a FN expression; this is used
      for FNexp expressions and also for handlers. The failure argument
      indicates what to plant for non-matches; RAISEMATCH means plant a lambda
      which raises the Match exception, whereas RAISESELF means raise the
      original packet. `warn' is true if inexhaustiveness warnings are
      required (TRUE for case statement on excons, for example, but FALSE
      for the equivalent in a handler).
        compileMatch is a bit wasteful in that if a rule is reachable more
      than once through a decision tree, the RHS expression will get compiled
      each time. It's rather difficult to abstract this away, since different
      decision paths to a particular rule result in different bindings for
      the pattern variables. This isn't impossible to solve, but I don't
      want to bother doing it. *)

    and compileMatch env (match, warn, failure) =
       let
	 val matches = makeList (fn MATCH(_, _, m) => m) match
	 val pats = map (fn MATCH(_, MRULE(_, pat, _), _) => pat) matches
	 val exps = map (fn MATCH(_, MRULE(_, _, exp), _) => exp) matches
			  (* We need to compile each exp into a lambda,
			     for which we need the environment established
			     by the corresponding pattern. *)

	 val root = Lvars.newLvar()

	 val decTree =
	   matchCompiler(root, pats,
			 {warnInexhaustive=warn, warnNoBindings=false}
			)

	 fun f(n: int, e: CEnv): LambdaExp =
	   compileExp (env plus e) (List.nth (n-1) exps)

	 val exp =
	    compileDecTree env (decTree, f,
				case failure
				  of RAISEMATCH => raiseMatch
				   | RAISESELF => RAISE(VAR root)
			       )
       in
	 FN(root, exp)
       end

   (* compileDec - takes an enclosing environment and a declaration, and
      returns the environment *for this declaration only*, together with a
      function to apply to the declaration's scope to return the entire
      lambda term. The `topLevel' parameter is only needed because the
      match compiler is expected to report non-binding patterns for
      non top-level val bindings only. *)

    and compileDec env (topLevel, dec): (CEnv * (LambdaExp -> LambdaExp)) =
      case dec
	of VALdec(_, valbind) =>
	     compileValbind env (topLevel, valbind)

	 | UNRES_FUNdec _ =>
	     Crash.impossible "compileDec(UNRES_FUN)"

	 | TYPEdec _ => (emptyCEnv, fn x => x)	(* ignore type decs. *)
	 | DATATYPEdec _ => (emptyCEnv, fn x => x) (* ignore datatype decs. *)

	 | ABSTYPEdec(_, _, dec) =>
	     compileDec env (false, dec)	(* compile the with-part. *)

	 | EXCEPTIONdec(_, exbind) =>
	     compileExbind env exbind

	 | LOCALdec(_, dec1, dec2) =>
	     let
	       val (env1, f1) = compileDec env (false, dec1)
	       val (env2, f2) = compileDec (env plus env1) (false, dec2)
	     in
	       (env2, f1 o f2)
	     end

	 | OPENdec _ => Crash.unimplemented "compile(OPENdec)"

	 | SEQdec(_, dec1, dec2) =>
	     let
	       val (env1, f1) = compileDec env (topLevel, dec1)
	       val (env2, f2) = compileDec (env plus env1) (topLevel, dec2)
	     in
	       (env1 plus env2, f1 o f2)
	     end

	(* INFIX/NONFIX declarations have no effect on execution. *)

	 | INFIXdec _ => (emptyCEnv, fn x => x)
	 | INFIXRdec _ => (emptyCEnv, fn x => x)
	 | NONFIXdec _ => (emptyCEnv, fn x => x)

         | EMPTYdec _ => (emptyCEnv, fn x => x)

   (* compileValbind - although there may be `rec' prefixes nested anywhere
      in the valbind, the effect is a single non-recursive layer of
      binding, together with a second layer of several distinct recursive
      valbinds. I think. *)

    and compileValbind env (topLevel, valbind)
        : (CEnv * (LambdaExp -> LambdaExp)) =
      let
	fun flattenRecValbind vb: (pat * exp) list =
	  case vb
	    of PLAINvalbind(_, pat, exp, vbOpt) =>
	         (pat, exp) :: (case vbOpt of Some vb => flattenRecValbind vb
				            | None    => nil
			       )

	     | RECvalbind(_, vb) =>
		 flattenRecValbind vb
      in
	case valbind
	  of PLAINvalbind(_, pat, exp, None) =>
	       compileBinding env (topLevel, pat, exp)

	   | PLAINvalbind(_, pat, exp, Some vb) =>
	       let
		 val (env1, f1) = compileBinding env (topLevel, pat, exp)
		 val (envRest, f2) = compileValbind env (topLevel, vb)
	       in
		 (env1 plus envRest, f1 o f2)
	       end

	   | RECvalbind(_, vb) =>
	       let
		 val pairs = flattenRecValbind vb
	       in
		 compileREC env (ListPair.unzip pairs)
	       end
      end

   (* compileExbind - we're being a little blase about the environment
      mapping identifiers to lvars, since there's no need to keep the
      variable and exception bindings distinct. *)

    and compileExbind env exbind
        : (CEnv * (LambdaExp -> LambdaExp)) =
      case exbind
	of EXBIND(_, OP_OPT(excon, _), _, rest) =>
	     let
	       val (env1, f1) = compileNewExn excon

	       val (envRest, f2) =
		 case rest
		   of Some exbind' => compileExbind env exbind'
		    | None => (emptyCEnv, fn x => x)
	     in
	       (env1 plus envRest, f1 o f2)
	     end

	 | EXEQUAL(_, OP_OPT(excon1, _), OP_OPT(LONGEXCON excon2, _), rest) =>
	     let
	       val lv =	lookupLongexcon env excon2
	       val env1 = declareExcon(excon1, lv, emptyCEnv)
					(* Map new excon to same lvar. *)

	       val f1 = fn x => x	(* No new code. *)

	       val (envRest, f2) =
		 case rest
		   of Some exbind' => compileExbind env exbind'
		    | None => (emptyCEnv, fn x => x)
	     in
	       (env1 plus envRest, f1 o f2)
	     end

         | EXEQUAL _ => Crash.impossible "compileExbind(EXEQUAL(_, ?, ?, _))"

   (* compileNewExn - create a new exception, by binding a new
      lvar to (ref name). *)

    and compileNewExn excon =
      let
	val name = Excon.pr_excon excon
	val lv = Lvars.newLvar()
	val env = declareExcon(excon, lv, emptyCEnv)
      in
	(env, fn x => Let((lv, REF(STRING name)), x))
      end

   (* compileBinding - we're compiling something like
	`let val pat = exp in scope end'.
      compileBinding returns the environment established by `pat',
      and a function: LambdaExp->LambdaExp, which is applied to the
      `scope' expression to yield the entire let expression. This
      generalises to parallel (`and'-linked) bindings. As long as we
      get the environment handling right (and don't make the thing
      `rec' by mistake), we can finally just cascade together the result
      functions returned by compileBinding. *)

    and compileBinding env (topLevel, pat, exp)
        : (CEnv * (LambdaExp -> LambdaExp)) =
      let
	val root = Lvars.newLvar()	(* Root of the pattern. *)

	val decTree =
	  matchCompiler(root, [pat],
			{warnInexhaustive=false, warnNoBindings=not topLevel}
		       )

				(* Decision tree which takes the root
				   apart according to the pattern. *)

	val env1 = envOfDecTree decTree
				(* The identifier environment generated by
				   this (single) pattern. *)
      in
	(env1, fn scope =>
	         let
		   val exp' = compileExp env exp
		   fun f _ = scope
		 in
		   Let((root, exp'),
		       compileDecTree env (decTree, f, raiseBind)
		      )
		 end	(* Given the final scope of this declaration, we can
			   compile the declaration tree such that `scope'
			   appears in the scope of the pattern. Weird side-
			   effect that the compilation is passed out as a
			   suspension. *)
	)
      end

   (* compileREC - compile a list of `rec' pattern/expression pairs. The
      patterns must all be variables (and aren't even allowed to be
      bracketted??), and the RHS's must all be lambdas. Type constraints
      are allowed, though, I think.
        Returns the rec env only, plus `fn scope -> lexp'. *)

    and compileREC env (pats, exps): (CEnv * (LambdaExp -> LambdaExp)) =
      let
	fun varOfPat(TYPEDpat(_, pat, _)) = varOfPat pat
	  | varOfPat(ATPATpat(_, LONGIDatpat(_, OP_OPT(LONGVAR longvar, _)))) =
	      (case Var.decompose longvar
		 of (nil, var) => var
		  | _ => Crash.impossible("compileREC.varOfPat(long: "
					  ^ Var.pr_longvar longvar ^ ")"
					 )
	      )
	  | varOfPat _ = Crash.impossible "compileREC.varOfPat"

	val vars = map varOfPat pats
	val lvars = map (fn _ => Lvars.newLvar()) vars
	val varsAndLvars = ListPair.zip(vars, lvars)

	val recEnv: CEnv =
	  List.foldL (fn (var, lv) => fn env => declareVar(var, lv, env))
	             emptyCEnv varsAndLvars

	val lexps = map (compileExp (env plus recEnv)) exps
      in
	(recEnv, fn scope => FIX(lvars, lexps, scope))
      end

    and compileSconSwitch env (arg: lvar,
			       selections: (scon, DecisionTree) map,
			       wildcard: DecisionTree,
			       compiler: (int * CEnv) -> LambdaExp,
			       failure: LambdaExp
			      ): LambdaExp =
      let
	open LambdaExp

	exception Next			(* wrong type of scon tried, try
					   the next type. *)

	val foldIntegerMap =		(* change (scon, dtree) map into
					   (int, LambdaExp) map. *)
	  FinMap.Fold
	    (fn ((SCon.INTEGER x, t), map)
		  => FinMap.add(
		       x, compileDecTree env (t, compiler, failure), map
		     )

	      | _ => raise Next
	    ) FinMap.empty

	and foldStringMap =		(* change (scon, dtree) map into
					   (string, LambdaExp) map. *)
	  FinMap.Fold
	    (fn ((SCon.STRING x, t), map)
		  => FinMap.add(
		       x, compileDecTree env (t, compiler, failure), map
		     )

	      | _ => raise Next
	    ) FinMap.empty

	and foldRealMap =		(* change (scon, dtree) map into
					   (real, LambdaExp) map. *)
	  FinMap.Fold
	    (fn ((SCon.REAL x, t), map)
		  => FinMap.add(
		       x, compileDecTree env (t, compiler, failure), map
		     )

	      | _ => raise Next
	    ) FinMap.empty
      in
	SWITCH_I(SWITCH{arg=VAR arg,
			selections=foldIntegerMap selections,
			wildcard=Some(compileDecTree env
				        (wildcard, compiler, failure)
				     )
		       }
	        )
	handle Next =>
	SWITCH_S(SWITCH{arg=VAR arg,
			selections=foldStringMap selections,
			wildcard=Some(compileDecTree env
				        (wildcard, compiler, failure)
				     )
		       }
	        )
	handle Next =>
	SWITCH_R(SWITCH{arg=VAR arg,
			selections=foldRealMap selections,
			wildcard=Some(compileDecTree env
				        (wildcard, compiler, failure)
				     )
		       }
	        )
	handle Next => Crash.impossible "compileSconSwitch"
      end

   (* compileExconSwitch - rather like compileSconSwitch, but it has
      to be an if/then/else sequence in rule order, since we don't know
      that different exception constructors do in fact have distinct
      names. compileExconSwitch requires an environment in order
      to find the exception names. NB: compileExconSwitch takes, as first
      argument, the lvar of the reference part of the exception packet
      being matched; be sure to select this from the exception packet
      itself in all calls to compileExconSwitch. *)

    and compileExconSwitch env (name: lvar,
				selections: (longexcon * DecisionTree) list,
				wildcard: DecisionTree,
				compiler: (int * CEnv) -> LambdaExp,
				failure: LambdaExp
			       ): LambdaExp =
      case selections
	of (longexcon, tree) :: rest =>
	     ifThenElse(equals(VAR name, VAR(lookupLongexcon env longexcon)),
			compileDecTree env (tree, compiler, failure),
			compileExconSwitch env
			  (name, rest, wildcard, compiler, failure)
		       )

         | nil =>
	     compileDecTree env (wildcard, compiler, failure)

   (* compileDecTree - Since the decision trees are typed and the lambda code
      isn't, it's here that we have to interpret the record and constructor
      type information to get offsets and tag fields correct. The `failure' arg
      is the lambda expression to evaluate when pattern-matching fails.

        Convention for data representation:
	   Records are vectors of values, ordered corresponding to the
	   alphabetic ordering of the labels;
	   Datatype constructions are pairs, with pair[0] being the
	   constructed value and pair[1] being the tag, an integer from
	   0..n-1, assigned to respect the alphabetic ordering of the
	   constructor names. (see also `nullaryCon' above.)

      Unreachable rules are easy to flag:
      the decision tree has nodes which contain
      the rule reached by that decision route. We can just scan for the
      rules which don't feature in the tree.

      The `env' argument is needed to look up exception names. That's it
      (I think). Guess who took it out, only to discover...

      Oh: note that compileDecTree doesn't take a list of abstract syntax
      expressions as argument: we may not have one - compileDec isn't used
      in that way at top-level. Nor does it take a list of compiled lambda
      expressions: there is not necessarily a unique lambda for each RHS as
      an RHS might be reachable several ways through a decision tree, so
      the environment for generating the lambda isn't easily determined
      elsewhere. So, it seems that the best bet is to pass `compileDecTree'
      a function of type `(int * CEnv) -> LambdaExp', which it can call at
      each leaf point to generate a lambda for that rule with the decomposition
      environment.
    *)

    and compileDecTree env (tree,
			    compiler: (int * CEnv) -> LambdaExp,
			    failure
			   ): LambdaExp =
      case tree
	of LAB_DECOMPOSE{bind, parent, lab, child, info} =>
	     Let((bind, SELECT(whichLab info, VAR parent)),
		 compileDecTree env (child, compiler, failure)
		)

	 | CON_DECOMPOSE{bind, parent, child} =>
	     Let((bind, first(VAR parent)),
		 compileDecTree env (child, compiler, failure)
		)

	 | EXCON_DECOMPOSE{bind, parent, child} =>
	     Let((bind, first(VAR parent)),
		 compileDecTree env (child, compiler, failure)
		)

	 | CON_SWITCH{arg, selections, wildcard} =>
		(* CON_SWITCH despatches on the constructors. We need to alter
		   it to pick out the tag field (pair[1]) and dispatch on
		   it as an integer constant according to the constructors'
		   alphabetic ordering. *)
	     let
	       val tag = second(VAR arg)

	      (* we do a fold of `f' to turn the (con -> tree) map into
	         a (int -> LambdaExp) one. *)

	       fun f((con: (*eqtype*) Con.con, (info, tree: DecisionTree)),
		     map: (int, LambdaExp) map
		    ): (int, LambdaExp) map =
		 FinMap.add(whichCon info,
			    compileDecTree env (tree, compiler, failure),
			    map
			   )
	     in
	       SWITCH_I(
		 SWITCH{arg=tag,
			selections=FinMap.Fold f FinMap.empty selections,
			wildcard=compileDecTreeOpt env
				   (wildcard, compiler, failure)
		       }
	       )
	     end

	 | SCON_SWITCH{arg, selections, wildcard} =>
	     compileSconSwitch env
	       (arg, selections, wildcard, compiler, failure)

	 | EXCON_SWITCH{arg, selections, wildcard} =>
					(* We have to extract the exception
					   reference (name) from the packet
					   before going into the switch. *)
	     let
	       val name = Lvars.newLvar()
	     in
	       Let((name, second(VAR arg)),
		   compileExconSwitch env
		     (name, selections, wildcard, compiler, failure)
		  )
	       end

	 | END{ruleNum, environment} => compiler(ruleNum, environment)
         | FAIL => failure

    and compileDecTreeOpt env (None, _, _) = None
      | compileDecTreeOpt env (Some t, compiler, failure) =
	  Some(compileDecTree env (t, compiler, failure))


    val compileDec = fn env => fn dec => compileDec env (true, dec)
  end;
