(* The semantic objects of the static semantics of Modules;
   they are built on top of the semantic objects for the Core.
   Definition v3, p 31 ff *)

(*
$File: Common/ModuleStatObject.sml $
$Date: 1993/03/05 14:38:48 $
$Revision: 1.10 $
$Locker: birkedal $
*)

(*$ModuleStatObject:
	STRID SIGID FUNID TYCON TYNAME STATOBJECT_PROP ENVIRONMENTS_PROP
	ERROR_INFO FINMAP PRETTYPRINT FLAGS CRASH MODULE_STATOBJECT
 *)

functor ModuleStatObject(structure StrId  : STRID
			 structure SigId  : SIGID
			 structure FunId  : FUNID
			 structure TyName : TYNAME

			 structure TyCon : TYCON
			   sharing type TyCon.strid = StrId.strid
			       and type TyCon.tycon = TyName.tycon

			 structure StatObject : STATOBJECT_PROP
			   sharing type StatObject.TyName = TyName.TyName

			 structure C : ENVIRONMENTS_PROP
			   sharing type C.TypeFcn    = StatObject.TypeFcn
			       and type C.TyNameSet  = StatObject.TyNameSet
			       and type C.TyName     = TyName.TyName
			       and type C.Type       = StatObject.Type
			       and type C.TypeScheme = StatObject.TypeScheme
			       and type C.tycon      = TyCon.tycon
			       and type C.strid      = StrId.strid
			       and type C.tyrea      = StatObject.tyrea
			       and type C.id         = StatObject.id
			       and type C.SyntaxTyVar = StatObject.SyntaxTyVar

			 structure ErrorInfo: ERROR_INFO
			   sharing type ErrorInfo.id = StatObject.id
			       and type ErrorInfo.strid = StrId.strid
			       and type ErrorInfo.longstrid = StrId.longstrid
			       and type ErrorInfo.excon = C.excon
			       and type ErrorInfo.tycon = TyCon.tycon
			       and type ErrorInfo.longtycon = TyCon.longtycon
			       and type ErrorInfo.TypeFcn = StatObject.TypeFcn
			       and type ErrorInfo.TyName = StatObject.TyName
			       and type ErrorInfo.TyVar = StatObject.TyVar
			       and type ErrorInfo.Type = StatObject.Type

			 structure FinMap : FINMAP

			 structure PP : PRETTYPRINT
			   sharing type PP.StringTree = FinMap.StringTree
			       and type PP.StringTree = C.StringTree

			 structure Flags : FLAGS
			 structure Crash : CRASH
			): MODULE_STATOBJECT =
  struct
    (********
    Unqualified identifiers
    ********)

    type id = StatObject.id

    (********
    Variables
    ********)

    type var = StatObject.var

    val mk_var = StatObject.mk_var

    (********
    Syntactic type variables
    ********)

    type SyntaxTyVar = StatObject.SyntaxTyVar

    (*********
     Syntactic type expressions (abstract syntax tree for ty's)
     ********)

    type ty = C.ty 
    val syntaxtyvarsTy = C.syntaxtyvarsTy

    (********
    Type variables
    ********)

    type TyVar = StatObject.TyVar

    val mkExplicitTyVar = StatObject.mkExplicitTyVar

    (********
    Constructed types
    ********)

    type ConsType = StatObject.ConsType

    val mkConsType = StatObject.mkConsType

    (****
    Types
    ****)

    type Type = StatObject.Type

    val TypeExn        = StatObject.TypeExn
    and mkTypeTyVar    = StatObject.mkTypeTyVar
    and mkTypeConsType = StatObject.mkTypeConsType
    and mkTypeArrow    = StatObject.mkTypeArrow

    (********
    TypeSchemes
    ********)

    type TypeScheme = StatObject.TypeScheme

    val Type_in_TypeScheme = StatObject.Type_in_TypeScheme

    val TySch_generalises_TySch = StatObject.TySch_generalises_TySch
    and TySch_generalises_Type  = StatObject.TySch_generalises_Type

    (********
    Structure names
    ********)

    type StrName = C.StrName

    val freshStrName = C.freshStrName
    val bogus_StrName = C.bogus_StrName

    (********
    Structure name sets
    ********)

    type StrNameSet = C.StrNameSet

    val singleM = C.singleM
    val emptyStrNameSet = C.emptyStrNameSet
    val StrNameSetUnion = C.StrNameSetUnion
    val StrNameSetFold = C.StrNameSetFold

    (********
    Type names
    ********)

    type TyName = TyName.TyName

    val freshTyName = TyName.freshTyName

    (********
    TyName sets
    ********)

    type TyNameSet = StatObject.TyNameSet

    val emptyTyNameSet = StatObject.emptyTyNameSet
    val isemptyT       = StatObject.isemptyT
    val TyNameSetMinus = StatObject.TyNameSetMinus
    val TyNameSetUnion = StatObject.TyNameSetUnion
    val TyNameSetFold  = StatObject.TyNameSetFold
    val TyNamesTy      = StatObject.TyNamesTy
    val TyNamesTySch   = StatObject.TyNamesTySch
    val TyNamesTypeFcn = StatObject.TyNamesTypeFcn

    (********
    Name sets
    ********)

    type NameSet = C.NameSet

    val isIn_StrName  = C.isIn_StrName
    and isIn_TyName   = C.isIn_TyName
    and mkNameSet_Str = C.mkNameSet_Str

    val mkNameSet = C.mkNameSet
    and unNameSet = C.unNameSet
    and NameSetIntersect = C.NameSetIntersect

    (********
    Structure Identifiers
    ********)

    type strid = StrId.strid

    (********
    Qualified structure identifiers
    ********)

    type longstrid = StrId.longstrid
    val implode_longstrid = StrId.implode_longstrid

    (********
    Environments
    ********)

    type Env = C.Env

    val unEnv = C.unEnv
    and mkEnv = C.mkEnv


    (********
    Structure Environments
    ********)

    type StrEnv = C.StrEnv

    val SEmap = C.SEmap
    and SEFold = C.SEFold

    (********
    Type constructors
    ********)

    type tycon = TyCon.tycon
    type longtycon = TyCon.longtycon
    val implode_LongTyCon = TyCon.implode_LongTyCon

    (********
    Type Environments
    ********)

    type TyEnv = C.TyEnv

    val TEFold = C.TEFold

    (********
    Variable Environments
    ********)

    type VarEnv = C.VarEnv

    (********
    Exception Constructor Environments
    ********)

    type excon = C.excon
    type ExConEnv = C.ExConEnv

    (********
    Structures
    ********)

    type Str = C.Str

    val mkStr  = C.mkStr
    and unStr  = C.unStr
    and namesS = C.namesS

    (********
    Type functions
    ********)

    type TypeFcn = StatObject.TypeFcn

    val admits_equality   = StatObject.admits_equality
    and TyName_in_TypeFcn = StatObject.TyName_in_TypeFcn
    and arity_TypeFcn     = StatObject.arity_TypeFcn

    fun grounded_TypeFcn (typefcn : TypeFcn, nameset : NameSet) : bool =
      let
	val tynameset = C.T_of_N nameset
      in
	StatObject.grounded_TypeFcn(typefcn, tynameset)
      end

    fun unTyName_TypeFcn (typefcn : TypeFcn) : TyName Option =
      StatObject.unTyName_TypeFcn typefcn


    (********
    Constructor environments
    ********)

    type ConEnv = C.ConEnv

    val CEmap = C.CEmap

    (********
    Type structures
    ********)

    type TyStr = C.TyStr

    val mkTyStr      = C.mkTyStr
    and unTyStr      = C.unTyStr
    and TyStr_shares = C.TyStr_shares
    and Theta_of     = C.Theta_of
    and isTyName     = C.isTyName
    and CE_of_TyStr  = C.CE_of_TyStr

    val bogus_TyStr = C.bogus_TyStr

    (********
    Signatures
    ********)

    datatype Sig = SIGMA of {N : NameSet, S : Str}

    val trivSig = 
      let
	val m = freshStrName()
      in
	SIGMA {N = C.mkNameSet_Str (C.singleM m), S = mkStr(m, C.emptyE)}
      end

    fun unSig (SIGMA {N, S}) = (N, S)
    and mkSig (N,S) = SIGMA {N=N, S=S}

    fun namesSig (SIGMA {N, S}) =
      C.NameSetMinus(namesS S, N)

    val bogus_Sig = trivSig

    (********
    Functor signatures
    ********)

    datatype FunSig = FUNSIG of {N : NameSet, S : Str, N'S' : Sig}

    fun mkFunSig (N, S, N'S') = FUNSIG {N = N, S = S, N'S' = N'S'}
    and unFunSig (FUNSIG {N, S, N'S'}) = (N, S, N'S')

    fun namesFunSig (FUNSIG {N, S, N'S' = SIGMA {N = N', S = S'}}) =
      C.NameSetUnion
      (C.NameSetMinus(namesS S , N),
       C.NameSetMinus(namesS S', C.NameSetUnion(N, N')))


    (********
    Close a structure to get a signature
    *********
    The NameSet is the set of `rigid' names which cannot be quantified
    ********)

    fun closeStr (NofB, S) =
      SIGMA{N = C.NameSetMinus(namesS S, NofB), S = S}


    (********
    Type Realisations
    ********)
    type tyrea              = StatObject.tyrea
    val id_tyrea            = StatObject.id_tyrea
    val oo_tyrea            = StatObject.oo_tyrea
    val mktyrea             = StatObject.mktyrea
    val mktyrea_class       = StatObject.mktyrea_class
    val restrict_tyrea      = StatObject.restrict_tyrea

    (********
    Realisations
    ********)

    type Realisation = {TyRea : tyrea,
			StrRea: StrName -> StrName }
    (* definition sec. 5.6 page 33 *)
    type Rename = Realisation * (TyName -> TyName)

    fun id_strrea x = x

    val Id = {TyRea=id_tyrea, StrRea=id_strrea}

    fun strrea_in_rea strrea : Realisation =
	{TyRea = id_tyrea,
	 StrRea = strrea}

    fun tyrea_in_rea tyrea : Realisation =
	{TyRea = tyrea,
	 StrRea = id_strrea}

    (* create a realisation from a list of pairs of equivalence classes 
       and representatives *)

    fun mkstrrea_class (M,m) = fn m' =>
	if List.member m' M then m else m'

    fun mkRea (Cstr, Cty) =
      let
	val f = (fn c => fn strrea => strrea o mkstrrea_class c)
	val g = (fn c => fn tyrea => oo_tyrea (tyrea, mktyrea_class c))
      in
	{StrRea = List.foldL f id_strrea Cstr,
	 TyRea  = List.foldL g id_tyrea  Cty}
      end

    val bogus_Realisation = Id

    infix oo				(* composition of realisations *)
    fun (rea as {TyRea=tr,StrRea=sr}) oo (rea' as {TyRea=tr',StrRea=sr'})=
	     {TyRea = oo_tyrea (tr,tr'),
	      StrRea= sr o sr'}

    (* Realisation applications *)

    fun onStrName ({StrRea, ...} : Realisation) (m : StrName) : StrName =
      StrRea m

    and onTyName ({TyRea, ...} : Realisation) : TyName -> TypeFcn =
      StatObject.tyrea_on_TyName TyRea

    and onTypeFcn ({TyRea, ...} : Realisation) : TypeFcn -> TypeFcn =
      StatObject.tyrea_on_TypeFcn TyRea

    and onTypeScheme ({TyRea, ...} : Realisation) : TypeScheme -> TypeScheme =
      StatObject.tyrea_on_TypeScheme TyRea

    and onType ({TyRea, ...} : Realisation) : Type -> Type =
      StatObject.tyrea_on_Type TyRea

    and onCE ({TyRea, ...} : Realisation) : ConEnv -> ConEnv =
      C.tyrea_on_CE TyRea

    and onTyStr ({TyRea, ...} : Realisation) : TyStr -> TyStr =
      C.tyrea_on_TyStr TyRea

    and onTE ({TyRea, ...} : Realisation, TE : TyEnv) : TyEnv =
      C.tyrea_on_TE TyRea TE

    and onVE ({TyRea, ...} : Realisation, VE : VarEnv) : VarEnv =
      C.tyrea_on_VE TyRea VE

    and onEE ({TyRea, ...} : Realisation, EE : ExConEnv) : ExConEnv =
      C.tyrea_on_EE TyRea EE

    fun onS (rea, S : Str) : Str =
      let
	val (m, E) = unStr S
      in
	mkStr(onStrName rea m, onE(rea, E))
      end

    and onE (rea, E : Env) : Env =
      let
	val (SE, TE, VE, EE) = unEnv E
      in
	mkEnv(onSE(rea, SE), onTE(rea,TE), onVE(rea,VE), onEE(rea,EE))
      end

    and onSE (rea, SE : StrEnv) : StrEnv =
      SEmap (fn S => onS(rea, S)) SE

    and onNameSet (rea : Realisation, N : NameSet) : NameSet =
      let
	val (M,T) = unNameSet N
	fun strname_map (n,M) =
	  StrNameSetUnion (singleM (onStrName rea n), M)
	and tyname_map (t,T) =
	  TyNameSetUnion (TyNamesTypeFcn (onTyName rea t),T)
      in
	mkNameSet(StrNameSetFold strname_map emptyStrNameSet M,
		  TyNameSetFold  tyname_map  emptyTyNameSet  T)
      end


    fun restrict(rea as {TyRea, ...}, N : NameSet) : Realisation =
       (* ^ the restriction of rea to a name set *)

      let
	val (M,T) = unNameSet N
      in
	{StrRea = fn m => if isIn_StrName(m, N) then onStrName rea m else m,
	 TyRea = restrict_tyrea T TyRea}
      end

    (* Renaming: renaming of semantic objects *)

    fun renaming (N : NameSet) : Realisation  =
      (* make a renaming from a nameset *)
      let
	val (M, T) = unNameSet N
	val new_strnames : (StrName * StrName) list =
	  let
	    fun f (n : StrName, names) =
	      (n, freshStrName()) :: names
	  in
	    StrNameSetFold f nil M
	  end

	and new_tynames : (TyName * TyName) list =
	  let
	    fun f (n : TyName, names) =
	      let
		val n' =
		  TyName.freshTyName
		  {name = TyName.name n, arity= TyName.arity n,
		   equality= TyName.equality n}
	      in
		(n, n') :: names
	      end
	  in
	    TyNameSetFold f nil T
	  end

	fun mkmap (mold,mnew) f m = if m=mold then mnew else f m
	and mktyreas (m, m') tyrea =
	  oo_tyrea(tyrea, mktyrea(m, TyName_in_TypeFcn m'))
      in
	{TyRea  = List.foldL mktyreas id_tyrea new_tynames,
	 StrRea = List.foldL mkmap id_strrea new_strnames }
      end

    fun onSig' (rea : Realisation, Sigma as SIGMA {N, S}) : Sig * Realisation =
	  (*        ^ renaming of bound names *)
      let
	 (* only clashing bound names need to be renamed to preserve        *)
	 (* N insersection ( Supp rea union Yield rea) = EmptySet           *)
	 (****  realisations, Definition v4, page 33 sec. 5.7 ***           *)
	 val Nfree = namesSig Sigma  (* free names *)
	 val Nfree'= onNameSet (rea, Nfree)
	 val clashes = NameSetIntersect (N,Nfree')
	 val rename = renaming clashes
	 val rea_f = restrict (rea, Nfree)
	 val N' = onNameSet (rename, N)
	 val S' = onS (rea_f oo rename, S)
      in
	(SIGMA {N = N', S = S'}, rename)
      end

    fun onSig (rea, Sigma : Sig) : Sig = 
      case onSig'(rea, Sigma) of (Sigma', _) => Sigma'

    fun onFunSig (rea, funsig' as FUNSIG {N, S, N'S'}) : FunSig =
      let
	val (SIGMA {N = N1, S = S1}, rename) = onSig'(rea, SIGMA {N = N,S = S})
	val Sigma1' = onSig(rea oo rename, N'S')
      in
	FUNSIG {N = N1, S = S1, N'S' = Sigma1'}
      end

    (********
    instanceSig returns the structure from a signature where names bound by
    the formal parameter have been instantiated to fresh (generative) names
    ********)

    fun instanceSig(SIGMA {N, S}) : Str =
      onS(renaming N, S)

    (********
    Datatype used to return results of matches
    ********)

    open ErrorInfo
    type ErrorInfo = info

    datatype 'a MatchResult =
	OK of 'a 
      | ERROR of ErrorInfo

  (***** signature matching definition page 35 sec. 5.12 *****)

      fun sigMatchRea (SIGMA{N, S}, S') : Realisation MatchResult =
	let
	  exception Fail of Realisation MatchResult

	  fun implies(a,b) = (* a --> b *) not (a andalso (not b))
	  val (M,T) = unNameSet N

	  fun matchStr(S, S', path): Realisation =
	    let
	      val (m, E) = unStr S
	      val (m', E') = unStr S'
	    in 
	      if m=m' then
		matchEnv(E, E', path)
	      else if isIn_StrName(m, N) then
		let
		  val eps = strrea_in_rea(fn m0 => if m0=m then m' else m0)
		in
		  eps oo matchEnv(E, E', path)
		end
	      else		(* clash between different rigid structures *)
		let
		  val longstrid = case path of
		     (strid::path') => Some(implode_longstrid (rev path',strid))
		   | nil            => None
		in
		    raise Fail(ERROR(S_RIGIDSTRCLASH longstrid))
		end
	    end

	  and matchEnv(E, E', path): Realisation =
	    let
	      val (SE, TE, _, _) = unEnv E
	      val (SE', TE', _, _) = unEnv E'
	    in
	      matchSE(SE, SE', path) oo matchTE(TE, TE', path)
	    end

	  and matchSE(SE, SE', path): Realisation =
	    let
	      fun f((strid: (*eqtype*) StrId.strid, S), rea): Realisation =
		case C.lookupSE(SE', strid) of
		  Some(S') =>
		    let val Snew = onS(rea, S)
		    in matchStr(Snew,S',strid::path) oo rea end
		| None =>
		      raise Fail(ERROR(MISSINGSTR(
					 implode_longstrid(rev path, strid)
				       )
				      )
				)
	    in
	      SEFold f Id SE 
	    end

	  and matchTE(TE, TE', path): Realisation =
	      let
	       fun f((tycon: (*eqtype*) tycon, tystr), rea): Realisation=
	       case C.lookupTE(TE', tycon) of

	       Some(tystr') =>
	      (*** As Sigma is type explicit, Definition sec. 5.8
	       *** we know that for all flexible type names t (type names bound
	       *** by N of Sigma) there exist a tycon, such that 
	       *** TE(tycon) = (t, CE) for some CE. 
	       *** However, it is possible that
	       *** there exists a t' and tycon', s.t. TE(tycon') =
	       *** (t', CE') for some CE' _and_ t' is not flexible (not in 
	       *** N of Sigma).
	       ***)
	       let val theta = C.Theta_of tystr
	       in
		   case (unTyName_TypeFcn theta) of
		   Some(t) =>
		       let val theta' = C.Theta_of tystr'
			   fun err f = raise Fail(ERROR (f 
			       (implode_LongTyCon (rev path, tycon),
				(t,theta'))))
		       in
			 if (isIn_TyName(t, N)) then
                           (*** definition page 33 sec 5.6 Type Realisation ***)
                           if (TyName.arity t = arity_TypeFcn theta') 
			     then
			       if implies(TyName.equality t, admits_equality theta')
				 then  rea oo ((tyrea_in_rea o mktyrea) (t,theta'))
			       else err CONFLICTINGEQUALITY
			   else   err S_CONFLICTINGARITY
			 else (* t is rigid *)
			   rea 
		       end
		 | None => rea
	       end

		 | None => 
		     raise Fail(ERROR(MISSINGTYPE(
					implode_LongTyCon(rev path, tycon)
				      )
				     )
			       )
	     in
		 TEFold f Id TE
	     end
	in 
	  OK(matchStr(S, S', nil))
	  handle Fail x => x
	end (* sigMatchRea *)

  (**** Enrichment : definition page 34 sec. 5.11 *)

      local
	  exception Enrichment of bool MatchResult
	  fun fail reason = raise Enrichment(ERROR reason)

	  fun enrichesStr (S, S', path) : bool =
	      let
		  val (m,E) = unStr S
		  and (m',E') = unStr S'
	      in
		  if (m=m') then enrichesEnv(E,E',path)
		  else
		   let val longstrid = case path of
		     (strid::path') => Some(implode_longstrid (rev path',strid))
		    | nil           => None
		   in
		       fail (S_RIGIDSTRCLASH longstrid)
		   end
	      end

	  and enrichesEnv (E, E', path) : bool =
	      let val (SE,TE,VE,EE) = unEnv E
		  and (SE',TE',VE',EE') = unEnv E'
	      in
		  enrichesSE(SE,SE',path) andalso
		  enrichesTE(TE,TE',path) andalso
		  enrichesVE(VE,VE',path) andalso
		  enrichesEE(EE,EE',path) 
	      end

	  and enrichesSE (SE,SE',path) : bool =
	      let fun f ((strid: StrId.strid, S'), enr) : bool =
		  case C.lookupSE(SE,strid) of
		      Some(S) => enrichesStr(S,S',strid::path)
		    | None     => fail (MISSINGSTR (
				      implode_longstrid(rev path, strid)))
	      in
		  C.SEFold f true SE'
	      end

	  and enrichesTE (TE,TE', path) : bool =
	      let fun f ((tycon, tystr'), enr) =
		  let fun enrichesTyStr (tystr,tystr') : bool =
		      let
			fun err f = fail(f(implode_LongTyCon(rev path,tycon)))
			val (theta,CE) = C.unTyStr tystr
			and (theta',CE') = C.unTyStr tystr'
		      in
			  if (theta = theta') then
			      C.isemptyCE CE'    orelse
			      C.equalCE (CE,CE') orelse
			      err S_CONFLICTING_DOMCE
			  else err S_RIGIDTYCLASH
		      end
		  in
		   case C.lookupTE(TE,tycon) of
		      Some(tystr) => enrichesTyStr (tystr,tystr')
		    | None => fail (MISSINGTYPE (implode_LongTyCon
				      (rev path, tycon)))
		  end
	      in
		  TEFold f true TE'
	      end


	  and enrichesVE (VE,VE',path) : bool =
	      let fun f ((id,varenvrng'),enr) =
		  let fun err f =
			fail (f (path,id))

		  fun enrVER(C.LONGVAR ts, C.LONGVAR ts') = enrTS_TS(ts,ts')
		    | enrVER(C.LONGCON ts, C.LONGVAR ts') = enrTS_TS(ts,ts')
		    | enrVER(C.LONGVAR ts, C.LONGCON ts') = enrTS_TS(ts,ts')
		    | enrVER(C.LONGCON ts, C.LONGCON ts') = enrTS_TS(ts,ts')
		    | enrVER(C.LONGVAR ts, C.LONGEXCON t') = enrTS_T(ts,t')
		    | enrVER(C.LONGCON ts, C.LONGEXCON t') = enrTS_T(ts,t')
		    | enrVER(C.LONGEXCON t, C.LONGEXCON t') = enrT_T(t,t')
		    | enrVER _ = err NOTYENRICHMENT

		  and enrTS_TS ts_ts' =
		      TySch_generalises_TySch ts_ts' orelse err NOTYENRICHMENT
		  and enrTS_T ts_t' =
		      TySch_generalises_Type ts_t' orelse err NOTYENRICHMENT
		  and enrT_T (t,t') =
		      (t=t') orelse err NOTYENRICHMENT

		  in
		      case C.lookupVE(VE,id) of
		       Some (varenvrng)  => enrVER(varenvrng,varenvrng')
		     | None => err MISSINGVAR
		  end
	      in
		  C.VEFold f true VE'
	      end

	  and enrichesEE (EE,EE',path) : bool =
	      let fun f ((excon,t'),enr) =
		  case C.lookupEE(EE,excon) of
		      Some(t) => (t=t') orelse
				  fail (EXCNOTEQUAL (rev path,excon,(t,t')))
		    | None => fail (MISSINGEXC (rev path,excon))
	      in
		  C.EEFold f true EE'
	      end
      in
	  fun enrichesS (S,S') : bool MatchResult = 
	      OK(enrichesStr(S,S',nil))
	      handle Enrichment x => x
      end

    (********
    If S'= sigMatchStr(Sigma,S) succeeds, then S matches Sigma
    and S' is an instance of Sigma and S enriches S'
    ********)

      fun sigMatchStr(Sig as SIGMA{N, S}, S'): Str MatchResult =
  (**** Signature matching : definition page 35 sec. 5.12 ***)
  (*    S' matches Sigma if there exists S'' s.t. Sigma >= S''
	and S' enriches S'' *)

	case sigMatchRea(Sig,S') of
	    OK rea =>
		let val S'' = onS(rea,S)
		in
		    case enrichesS(S',S'')
		      of OK true  => OK S''
		       | OK _ => Crash.impossible "sigMatchStr"
		       | ERROR x => ERROR x
		end
	  | ERROR x => ERROR x


    (********
    If Sig'= funsigMatchStr(Phi,S) succeeds, then S matches the
    argument signature of the functor signature Phi and Sig' is 
    the signature of the actual result of the functor application
    ********)

      fun funsigMatchStr(FUNSIG{N=N, S=S, N'S'=Sig' as SIGMA{N=N',S=S'}}, S0)
	  : Sig MatchResult =
	case sigMatchRea(SIGMA{N=N,S=S}, S0) of
	    OK rea =>
		let
		  val S'' = onS(rea,S)
		in
		  case enrichesS(S0,S'')
		    of OK true =>
			 let
			   val rename = renaming N'
			 in
			   OK(onSig(rename oo rea,Sig'))
			 end

		     | OK _ => Crash.impossible "sigMatchRea"
		     | ERROR x => ERROR x
		end

	    | ERROR x => ERROR x


    (********
    Check a structure for type-explicitness
    *********
    The NameSet is the rigid NameSet of the Basis in which the check is made
    definition sec 5.8 page 33
    A signature (N)S is type explicit, if forall t in N, 
    if t occurs free in S then some substructure of S contains a TE s.t.
    TE(tycon) = (t,CE) for some tycon, CE
    ********)

    local
	fun tnamesS S =
	    let
		val (_,E) = unStr S
	    in
		tnamesE E
	    end

	and tnamesE E =
	    let
		val (SE,TE,VE,EE) = unEnv E
	    in
		tnamesSE SE o
		tnamesTE TE o
		tnamesVE VE o
		tnamesEE EE
	    end

	and tnamesSE SE p =
	    let
		fun f (S,p) = tnamesS S p
	    in
		C.SEfold f p SE
	    end

	and tnamesTE TE p =
	    let
		fun f (tystr,(T,Texpl)) =
		    let
			val (theta,CE) = unTyStr tystr
			val TofTheta = TyNamesTypeFcn theta
			val T' = TyNameSetUnion (TofTheta, T)
			val Texpl' = case unTyName_TypeFcn theta of
			    Some _ => TyNameSetUnion(TofTheta, Texpl)
			  | None   => Texpl
		    in
			tnamesCE CE (T',Texpl')
		    end
	    in
		C.TEfold f p TE
	    end

	and tnamesCE CE p =
	    let
		fun f (tysch, (T,Texpl)) =
		    (TyNameSetUnion(TyNamesTySch tysch,T), Texpl)
	    in
		C.CEfold f p CE
	    end

	and tnamesVE VE p =
	    let
		fun f (ver,(T,Texpl)) =
		    let
			fun g (C.LONGCON   ts)  = TyNamesTySch ts
			  | g (C.LONGEXCON tau) = TyNamesTy tau
			  | g (C.LONGVAR   ts)  = TyNamesTySch ts
		    in
			(TyNameSetUnion (g ver,T), Texpl)
		    end
	    in
		C.VEfold f p VE
	    end

	and tnamesEE EE p =
	    let
		fun f (tau,(T,Texpl)) = (TyNameSetUnion(TyNamesTy tau,T),Texpl)
	    in
		C.EEfold f p EE
	    end

    in
	fun type_explicit (NofB, S) : bool =
    (*******
     Bound names of S are names S \ NofB
     Type explicit if {Bound type names} = {Explicit type names} 
    *******)
	let
	    val (TofS, TofSexpl) = tnamesS S (emptyTyNameSet, emptyTyNameSet)
	    val TBound           = TyNameSetMinus(TofS, C.T_of_N NofB)
	    val nonExplicit      = TyNameSetMinus(TBound, TofSexpl)
	in
	    isemptyT nonExplicit
	end
    end


    (********
     Well-formedness of signature - Definition, Sec. 5.3, p. 32
     ********
     The NameSet is the rigid NameSet of the Basis in with the check is made
     ********)

    fun wellformedsig(NofB, S) : bool =
      (* Bound names (N) of S are names S \ NofB, i.e., the first requirement
         of well-formedness (N \subseteq (names S)) holds, and we
	 only need to check for each substructure (m, E) of S, if
	 m \notin N then (N \cap names E) = \emptyset *)

      let
	val (m, E) = C.unStr S
	val (SE, _, _, _) = C.unEnv E
	exception NotWellformed
	fun f (S, b) = 
	  let 
	    val (m, E) = C.unStr S
	    val (SE, _, _, _) = C.unEnv E
	  in
	    (if (C.isIn_StrName(m, NofB) 
                  (* m rigid *)
	        andalso
	        not (C.eqNameSet(C.NameSetMinus (C.namesE E, NofB), C.emptyN))
		  (* there exist a name below m, not rigid *)
		) then
	      raise NotWellformed
	     else true)
	    andalso
	    (C.SEfold f true SE)
	  end
      in
	C.SEfold f true SE
      end 
      handle NotWellformed => false 


    (******
    equality-principal signature
    ******)

    (****** definition sec. 5.13 page 35 ******)

    datatype EqPrincipalResult =
	PRINCIPAL of Str 
      | FAIL of longtycon list


   local
    fun violates_equality tys =
	StatObject.violates_equality (StatObject.emptyTyNameSet,tys)

    fun respects_equality (S:Str) : EqPrincipalResult =
	let
	    fun respects_equalityS (path,longtycons) S =
	    (*****
	     returns a list of all longtycons violating equality
	    *****)
		let
		    val (_,E) = unStr S
		    val (SE,TE,_,_) = unEnv E
		    val l = respects_equalityTE (path,longtycons) TE
		in
		    respects_equalitySE (path,l) SE 
		end

	    and respects_equalitySE (path,longtycons) SE =
		let
		    fun f ((strid,str),l) =
			respects_equalityS (strid::path,l) str
		in
		    SEFold f longtycons SE
		end

	    and respects_equalityTE (path,longtycons) TE =
		(*****
		 If any of the constructors of a type structure violate
		 equality, then the type cannot admit equality
		 *****)
		let
		    fun f ((tycon,tystr),l) =
			let
			    val (theta,CE) = C.unTyStr tystr
			    val violates =
				let
				  fun g (ts, b) = violates_equality ts orelse b
				in
				    C.CEfold g false CE
				end
			in
			    if violates andalso (admits_equality theta) then 
				implode_LongTyCon(rev path,tycon) :: l
			    else
				l
			end
		in
		    TEFold f longtycons TE
		end

	in
	    case respects_equalityS (nil,nil) S of
		nil        => PRINCIPAL S
	      | longtycons => FAIL longtycons
	end

  fun bound_datatype_names (NofB : NameSet, S : Str) : TyNameSet =
      (* return the set of bound type names in S which do not admit equality
	 and such that (t,CE) occurs in S for some CE <> EmptySet
	 This corresponds to T0 in definition sec 5.13 page 36 :
	 equality principal signatures *)

	 let
	     fun datatype_namesS (S,dnames) =
	     let
		 val (_,E) = unStr S
		 val (SE,TE,_,_) = unEnv E
		 val dnames' = datatype_namesTE (TE,dnames)
	     in
		 datatype_namesSE (SE,dnames')
	     end

	     and datatype_namesSE (SE,dnames) =
		 C.SEfold datatype_namesS dnames SE

	     and datatype_namesTE (TE,dnames) =
		 StatObject.TyNameSetUnion (C.flexible_tynames TE,dnames)
	 in
	     StatObject.TyNameSetMinus(datatype_namesS (S,emptyTyNameSet),
				       C.T_of_N NofB)
	 end


    fun equality_principal (NofB:NameSet, S0: Str)  =
    (****** result is equality-prinicipal for S0 in Basis B ******)
	let
	    fun maximise_equality (S,T) =
	    (******
	    accumulates the result of maximising equality on all of the TE in S
	    ******)
	    let
		val (_,E) = unStr S
		val (SE,TE,_,_) = unEnv E
		val T' = C.maximise_TE_equality (T,TE)
	    in
		C.SEfold maximise_equality T' SE
	    end

	    fun fixpoint T =
	    (******
	     calculate the maximum fixpoint of bound datatype name set of S0
	     which can be made to admit equality
	     ******)
	     let
		 val T' = maximise_equality (S0,T)
	     in
		 if StatObject.eqTyNameSet (T,T') then T else fixpoint T'
	     end


	     (******   definition sec 5.13 page 36  ******)

	     val T0 = bound_datatype_names (NofB,S0)
	     val T = fixpoint T0  (* this is the maximal subset of T0 
				     which can be made to admit equality *)
	     val rea = (tyrea_in_rea o StatObject.generate_tyrea) T
	in
	    onS(rea,S0)
	end
   in
       val equality_principal : NameSet * Str -> EqPrincipalResult =
	   respects_equality o equality_principal
  end


    (********
    PrettyPrinting
    ********)

    type StringTree = PP.StringTree

    val layoutSig =
      if Flags.DEBUG_STATOBJECTS then
	fn (SIGMA {N, S}) =>
	  let
	    val Ntree = PP.NODE{start="(", finish=")", indent=1,
				children=[C.layoutNameSet N], childsep=PP.NONE
				}
	  in
	    PP.NODE{start="", finish="", indent=0,
		    children=[Ntree, C.layoutStr S], childsep=PP.RIGHT " "
		    }
	  end
      else
	fn (SIGMA {N, S}) =>
	  C.layoutStr S

    fun layoutFunSig(FUNSIG{N, S, N'S'=Sig})=
      let
	val argsig = PP.NODE{start="(", finish="", indent=1,
			     children=[layoutSig(SIGMA{N=N, S=S})], 
			     childsep=PP.NONE
			     }
	val ressig = PP.NODE{start=") : ", finish="", indent=3,
			     children=[layoutSig Sig],
			     childsep=PP.NONE
			     }
      in
	PP.NODE{start="", finish="", indent=0, 
		children=[argsig, ressig],
		childsep=PP.RIGHT " "
		}
      end
  end;
