;; Eulisp Module
;; Author: pete broadbery
;; File: pvm-client.em
;; Date: 20/may/1991
;;
;; Project:
;; Description: 
;; functionality for a client 
;; similar to a net-client
;;

(defmodule pvm-client 
  (standard0
   list-fns
   scan-args
   waiters
   comm-low
   client
   pvm-support
   pvm ;; should not need to import if compiled.
   rshow
   )
  ()
  (expose client)
  ;; structures

  (defstruct pvm-client client
    ((msg-type initarg type reader pvm-client-msg-type)
     ;; local id (usually)
     (id initarg id reader pvm-client-id)
     (self initform (pvm-whoami) initarg self reader pvm-client-self)
     (waiter initform (make-waiter)
	     accessor pvm-client-waiter)
     (known-clients initform (make-table eq) 
		    reader pvm-client-known-clients))
    constructor (make-pvm-client type id . rest))

  (defstruct remote-pvm-client remote-client
    ((local-id initarg local-id 
	       accessor remote-pvm-client-id)
     (client initarg client 
	     accessor remote-pvm-client-local-client))
    constructor (make-remote-pvm-client local-id client))
  
  (export pvm-client client remote-pvm-client
	  pvm-client-self make-remote-pvm-client
	  remote-pvm-client-local-client
	  pvm-client-waiter)
  
  ;;; 
  ;; initialisation
  ;;

  (defmethod initialize-instance ((cl pvm-client) args)
    (let ((new-obj (call-next-method)))
      ((setter slot-value) new-obj 'id 
       (make-local-id (scan-args 'type args '%_+_%)))
      (add-client new-obj (obj-id new-obj)
		  (pvm-client-self new-obj))
      (start-dispatch-loop new-obj)
      new-obj))
  
  ;;
  ;; generics
  ;;
  
  ;; type of a particular pvm-client

  (defgeneric pvm-client-name (ob)
    methods ((((o object))
	      (default-pvm-name))))

  (defgeneric find-remote-client (ob name msg))

  (defgeneric object-reader (cl))
    
  (export object-reader)
  ;;
  ;; methods
  ;;

  (defmethod obj-id ((o pvm-client))
    (pvm-client-id o))

  (defmethod obj-id ((o remote-pvm-client))
    (remote-pvm-client-id o))
  
  (defmethod find-waiter ((o pvm-client))
    (pvm-client-waiter o))

  (defmethod find-waiter ((o remote-pvm-client))
    (find-waiter (remote-pvm-client-local-client o)))

  ;;
  ;; local id
  ;;

  (defun make-local-id (msg-type)
    (make-symbol (format nil "client-~a-~a" (local-id) msg-type)))

  ;;
  ;; making a client
  ;;

  (defun make-local-client (class initargs)
    (apply make-instance class 'id 
	   (make-local-id (scan-args 'type
				     initargs nil))
	   initargs))
  
  (defun make-remote-client (class id pvm initargs)
    (make-instance class `(id ,id self ,pvm ,@initargs)))
			       
  (export make-local-client make-remote-client)

  ;;
  ;; receiving
  ;;

  (defmethod recv-msg ((cl pvm-client))
    (pvm-recv (pvm-client-msg-type cl) () (object-reader cl)))

  (defmethod readable-p ((cl pvm-client))
    (pvm-probe (pvm-client-msg-type cl)))

  (export readable-p)
  ;;
  ;; sending
  ;;

  (defun broadcast-msg (client msg)
    (print 'BROADCAST)
    (show client)
    (print msg)
    (pvm-send (make-pvm-id (pvm-client-name client))
	      (pvm-client-msg-type client)
	      msg
	      (object-reader client))
    ;; broadcast includes self.
    (pvm-send (pvm-whoami) (pvm-client-msg-type client)
	      msg (object-reader  client)))

  
  (defun send-msg-to (client dest msg)
    (let ((target (table-ref (pvm-client-known-clients client)
			     dest)))
      ;;(format t "Send: Sending: ~a \n        to: ~a\n" 
      ;;msg dest)
      (if target
	  ;; probably wrong...
	  (pvm-send target
		    (pvm-client-msg-type client)
		    msg
		    (object-reader client))
	(thread-start (make-thread wait-a-while)
		      client dest msg))))
    

  (defun wait-a-while (client dest msg)
    (format t "Waiting for: ~a ~%" dest)
    (let ((target (table-ref (pvm-client-known-clients client)
			     dest)))
      (if target
	  (pvm-send target
		    (pvm-client-msg-type client)
		    msg
		    (object-reader client))
	(progn (thread-reschedule)
	       (wait-a-while client dest msg)))))
    

  (defmethod send-msg ((rc remote-pvm-client) msg)
    (send-msg-to (remote-pvm-client-local-client rc)
		 (remote-pvm-client-id rc) msg))

  ;; because I can...
  (defmethod send-msg ((client pvm-client) msg)
    (pvm-send (pvm-client-self client)
	      (pvm-client-msg-type client)
	      msg
	      (object-reader client)))

  ;; needed at initialisation
  (defun client-send-by-id (client id msg)
    ;;(format t "send-by-id: [~a] ~a~%" id msg)
    (pvm-send id (pvm-client-msg-type client) msg
	      (object-reader client)))
  
  (defun all-known-clients (client)
    (list (table-keys (pvm-client-known-clients client))
	  (table-parameters (pvm-client-known-clients client))))

  (export client-send-by-id send-msg-to broadcast-msg all-known-clients)
  ;; adding clients

  (defun add-client (client name new-client)
    (if (table-ref (pvm-client-known-clients client) name)
	t
      (progn ((setter table-ref)
	      (pvm-client-known-clients client) name new-client)
	     nil)))
  
  (export send-msg-to add-client)

  (defgeneric make-reply-dest (client))
    
  (defmethod make-reply-dest ((client pvm-client))
    (obj-id client))

  (defmethod make-reply-dest ((client remote-pvm-client))
    (make-reply-dest (remote-pvm-client-local-client client)))

  (export make-reply-dest)
  
  ;; non-blocking wait
  (defun wait-for-msg (client)
    (add-pvm-waiter (std-pvm-waiter) (current-thread) 
		    (pvm-client-msg-type client))
    (thread-suspend))
  (export wait-for-msg)
  ;;
  ;; encoding these things
  ;;
  (defmethod generic-encode ((pc pvm-client))
    (list '%_client
	  (pvm-client-self pc)
	  (pvm-client-msg-type pc)
	  (encode (pvm-client-self pc))))

  
  ;; Hacky...
  (defmethod generic-encode ((id pvm-id))
    (list '%-pvm-id-% id))

  ((setter *decoder) '%-pvm-id-%
   (lambda (msg)
     (pvm-make-id-from-pair (cadr msg))))
  
  
  ;; end module
  )
